/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container for information about the replication destination.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Destination implements SdkPojo, Serializable, ToCopyableBuilder<Destination.Builder, Destination> {
    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Destination::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<String> ACCOUNT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Destination::account))
            .setter(setter(Builder::account))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Account")
                    .unmarshallLocationName("Account").build()).build();

    private static final SdkField<String> STORAGE_CLASS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Destination::storageClassAsString))
            .setter(setter(Builder::storageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageClass")
                    .unmarshallLocationName("StorageClass").build()).build();

    private static final SdkField<AccessControlTranslation> ACCESS_CONTROL_TRANSLATION_FIELD = SdkField
            .<AccessControlTranslation> builder(MarshallingType.SDK_POJO)
            .getter(getter(Destination::accessControlTranslation))
            .setter(setter(Builder::accessControlTranslation))
            .constructor(AccessControlTranslation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlTranslation")
                    .unmarshallLocationName("AccessControlTranslation").build()).build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(Destination::encryptionConfiguration))
            .setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration")
                    .unmarshallLocationName("EncryptionConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, ACCOUNT_FIELD,
            STORAGE_CLASS_FIELD, ACCESS_CONTROL_TRANSLATION_FIELD, ENCRYPTION_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final String account;

    private final String storageClass;

    private final AccessControlTranslation accessControlTranslation;

    private final EncryptionConfiguration encryptionConfiguration;

    private Destination(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.account = builder.account;
        this.storageClass = builder.storageClass;
        this.accessControlTranslation = builder.accessControlTranslation;
        this.encryptionConfiguration = builder.encryptionConfiguration;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the bucket where you want Amazon S3 to store replicas of the object identified
     * by the rule.
     * </p>
     * <p>
     * If there are multiple rules in your replication configuration, all rules must specify the same bucket as the
     * destination. A replication configuration can replicate objects to only one destination bucket.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the bucket where you want Amazon S3 to store replicas of the object
     *         identified by the rule. </p>
     *         <p>
     *         If there are multiple rules in your replication configuration, all rules must specify the same bucket as
     *         the destination. A replication configuration can replicate objects to only one destination bucket.
     */
    public String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The account ID of the destination bucket. Currently, Amazon S3 verifies this value only if Access Control
     * Translation is enabled.
     * </p>
     * <p>
     * In a cross-account scenario, if you change replica ownership to the AWS account that owns the destination bucket
     * by adding the <code>AccessControlTranslation</code> element, this is the account ID of the owner of the
     * destination bucket.
     * </p>
     * 
     * @return The account ID of the destination bucket. Currently, Amazon S3 verifies this value only if Access Control
     *         Translation is enabled. </p>
     *         <p>
     *         In a cross-account scenario, if you change replica ownership to the AWS account that owns the destination
     *         bucket by adding the <code>AccessControlTranslation</code> element, this is the account ID of the owner
     *         of the destination bucket.
     */
    public String account() {
        return account;
    }

    /**
     * <p>
     * The class of storage used to store the object. By default Amazon S3 uses storage class of the source object when
     * creating a replica.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return The class of storage used to store the object. By default Amazon S3 uses storage class of the source
     *         object when creating a replica.
     * @see StorageClass
     */
    public StorageClass storageClass() {
        return StorageClass.fromValue(storageClass);
    }

    /**
     * <p>
     * The class of storage used to store the object. By default Amazon S3 uses storage class of the source object when
     * creating a replica.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return The class of storage used to store the object. By default Amazon S3 uses storage class of the source
     *         object when creating a replica.
     * @see StorageClass
     */
    public String storageClassAsString() {
        return storageClass;
    }

    /**
     * <p>
     * A container for information about access control for replicas.
     * </p>
     * <p>
     * Use this element only in a cross-account scenario where source and destination bucket owners are not the same to
     * change replica ownership to the AWS account that owns the destination bucket. If you don't add this element to
     * the replication configuration, the replicas are owned by same AWS account that owns the source object.
     * </p>
     * 
     * @return A container for information about access control for replicas. </p>
     *         <p>
     *         Use this element only in a cross-account scenario where source and destination bucket owners are not the
     *         same to change replica ownership to the AWS account that owns the destination bucket. If you don't add
     *         this element to the replication configuration, the replicas are owned by same AWS account that owns the
     *         source object.
     */
    public AccessControlTranslation accessControlTranslation() {
        return accessControlTranslation;
    }

    /**
     * <p>
     * A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is specified, you
     * must specify this element.
     * </p>
     * 
     * @return A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is
     *         specified, you must specify this element.
     */
    public EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(account());
        hashCode = 31 * hashCode + Objects.hashCode(storageClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlTranslation());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Destination)) {
            return false;
        }
        Destination other = (Destination) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(account(), other.account())
                && Objects.equals(storageClassAsString(), other.storageClassAsString())
                && Objects.equals(accessControlTranslation(), other.accessControlTranslation())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration());
    }

    @Override
    public String toString() {
        return ToString.builder("Destination").add("Bucket", bucket()).add("Account", account())
                .add("StorageClass", storageClassAsString()).add("AccessControlTranslation", accessControlTranslation())
                .add("EncryptionConfiguration", encryptionConfiguration()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "Account":
            return Optional.ofNullable(clazz.cast(account()));
        case "StorageClass":
            return Optional.ofNullable(clazz.cast(storageClassAsString()));
        case "AccessControlTranslation":
            return Optional.ofNullable(clazz.cast(accessControlTranslation()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Destination, T> g) {
        return obj -> g.apply((Destination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Destination> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the bucket where you want Amazon S3 to store replicas of the object
         * identified by the rule.
         * </p>
         * <p>
         * If there are multiple rules in your replication configuration, all rules must specify the same bucket as the
         * destination. A replication configuration can replicate objects to only one destination bucket.
         * </p>
         * 
         * @param bucket
         *        The Amazon Resource Name (ARN) of the bucket where you want Amazon S3 to store replicas of the object
         *        identified by the rule. </p>
         *        <p>
         *        If there are multiple rules in your replication configuration, all rules must specify the same bucket
         *        as the destination. A replication configuration can replicate objects to only one destination bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The account ID of the destination bucket. Currently, Amazon S3 verifies this value only if Access Control
         * Translation is enabled.
         * </p>
         * <p>
         * In a cross-account scenario, if you change replica ownership to the AWS account that owns the destination
         * bucket by adding the <code>AccessControlTranslation</code> element, this is the account ID of the owner of
         * the destination bucket.
         * </p>
         * 
         * @param account
         *        The account ID of the destination bucket. Currently, Amazon S3 verifies this value only if Access
         *        Control Translation is enabled. </p>
         *        <p>
         *        In a cross-account scenario, if you change replica ownership to the AWS account that owns the
         *        destination bucket by adding the <code>AccessControlTranslation</code> element, this is the account ID
         *        of the owner of the destination bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder account(String account);

        /**
         * <p>
         * The class of storage used to store the object. By default Amazon S3 uses storage class of the source object
         * when creating a replica.
         * </p>
         * 
         * @param storageClass
         *        The class of storage used to store the object. By default Amazon S3 uses storage class of the source
         *        object when creating a replica.
         * @see StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageClass
         */
        Builder storageClass(String storageClass);

        /**
         * <p>
         * The class of storage used to store the object. By default Amazon S3 uses storage class of the source object
         * when creating a replica.
         * </p>
         * 
         * @param storageClass
         *        The class of storage used to store the object. By default Amazon S3 uses storage class of the source
         *        object when creating a replica.
         * @see StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageClass
         */
        Builder storageClass(StorageClass storageClass);

        /**
         * <p>
         * A container for information about access control for replicas.
         * </p>
         * <p>
         * Use this element only in a cross-account scenario where source and destination bucket owners are not the same
         * to change replica ownership to the AWS account that owns the destination bucket. If you don't add this
         * element to the replication configuration, the replicas are owned by same AWS account that owns the source
         * object.
         * </p>
         * 
         * @param accessControlTranslation
         *        A container for information about access control for replicas. </p>
         *        <p>
         *        Use this element only in a cross-account scenario where source and destination bucket owners are not
         *        the same to change replica ownership to the AWS account that owns the destination bucket. If you don't
         *        add this element to the replication configuration, the replicas are owned by same AWS account that
         *        owns the source object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlTranslation(AccessControlTranslation accessControlTranslation);

        /**
         * <p>
         * A container for information about access control for replicas.
         * </p>
         * <p>
         * Use this element only in a cross-account scenario where source and destination bucket owners are not the same
         * to change replica ownership to the AWS account that owns the destination bucket. If you don't add this
         * element to the replication configuration, the replicas are owned by same AWS account that owns the source
         * object.
         * </p>
         * This is a convenience that creates an instance of the {@link AccessControlTranslation.Builder} avoiding the
         * need to create one manually via {@link AccessControlTranslation#builder()}.
         *
         * When the {@link Consumer} completes, {@link AccessControlTranslation.Builder#build()} is called immediately
         * and its result is passed to {@link #accessControlTranslation(AccessControlTranslation)}.
         * 
         * @param accessControlTranslation
         *        a consumer that will call methods on {@link AccessControlTranslation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlTranslation(AccessControlTranslation)
         */
        default Builder accessControlTranslation(Consumer<AccessControlTranslation.Builder> accessControlTranslation) {
            return accessControlTranslation(AccessControlTranslation.builder().applyMutation(accessControlTranslation).build());
        }

        /**
         * <p>
         * A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is specified,
         * you must specify this element.
         * </p>
         * 
         * @param encryptionConfiguration
         *        A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is
         *        specified, you must specify this element.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is specified,
         * you must specify this element.
         * </p>
         * This is a convenience that creates an instance of the {@link EncryptionConfiguration.Builder} avoiding the
         * need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String account;

        private String storageClass;

        private AccessControlTranslation accessControlTranslation;

        private EncryptionConfiguration encryptionConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(Destination model) {
            bucket(model.bucket);
            account(model.account);
            storageClass(model.storageClass);
            accessControlTranslation(model.accessControlTranslation);
            encryptionConfiguration(model.encryptionConfiguration);
        }

        public final String getBucket() {
            return bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        public final String getAccount() {
            return account;
        }

        @Override
        public final Builder account(String account) {
            this.account = account;
            return this;
        }

        public final void setAccount(String account) {
            this.account = account;
        }

        public final String getStorageClassAsString() {
            return storageClass;
        }

        @Override
        public final Builder storageClass(String storageClass) {
            this.storageClass = storageClass;
            return this;
        }

        @Override
        public final Builder storageClass(StorageClass storageClass) {
            this.storageClass(storageClass.toString());
            return this;
        }

        public final void setStorageClass(String storageClass) {
            this.storageClass = storageClass;
        }

        public final AccessControlTranslation.Builder getAccessControlTranslation() {
            return accessControlTranslation != null ? accessControlTranslation.toBuilder() : null;
        }

        @Override
        public final Builder accessControlTranslation(AccessControlTranslation accessControlTranslation) {
            this.accessControlTranslation = accessControlTranslation;
            return this;
        }

        public final void setAccessControlTranslation(AccessControlTranslation.BuilderImpl accessControlTranslation) {
            this.accessControlTranslation = accessControlTranslation != null ? accessControlTranslation.build() : null;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public Destination build() {
            return new Destination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
