/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.RequiredTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class RenameObjectRequest extends S3Request implements
        ToCopyableBuilder<RenameObjectRequest.Builder, RenameObjectRequest> {
    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(RenameObjectRequest::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build(), RequiredTrait.create()).build();

    private static final SdkField<String> KEY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Key")
            .getter(getter(RenameObjectRequest::key))
            .setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.GREEDY_PATH).locationName("Key")
                    .unmarshallLocationName("Key").build(), RequiredTrait.create()).build();

    private static final SdkField<String> RENAME_SOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RenameSource")
            .getter(getter(RenameObjectRequest::renameSource))
            .setter(setter(Builder::renameSource))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-rename-source")
                    .unmarshallLocationName("x-amz-rename-source").build(), RequiredTrait.create()).build();

    private static final SdkField<String> DESTINATION_IF_MATCH_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DestinationIfMatch")
            .getter(getter(RenameObjectRequest::destinationIfMatch))
            .setter(setter(Builder::destinationIfMatch))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("If-Match")
                    .unmarshallLocationName("If-Match").build()).build();

    private static final SdkField<String> DESTINATION_IF_NONE_MATCH_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DestinationIfNoneMatch")
            .getter(getter(RenameObjectRequest::destinationIfNoneMatch))
            .setter(setter(Builder::destinationIfNoneMatch))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("If-None-Match")
                    .unmarshallLocationName("If-None-Match").build()).build();

    private static final SdkField<Instant> DESTINATION_IF_MODIFIED_SINCE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DestinationIfModifiedSince")
            .getter(getter(RenameObjectRequest::destinationIfModifiedSince))
            .setter(setter(Builder::destinationIfModifiedSince))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("If-Modified-Since")
                    .unmarshallLocationName("If-Modified-Since").build()).build();

    private static final SdkField<Instant> DESTINATION_IF_UNMODIFIED_SINCE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DestinationIfUnmodifiedSince")
            .getter(getter(RenameObjectRequest::destinationIfUnmodifiedSince))
            .setter(setter(Builder::destinationIfUnmodifiedSince))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("If-Unmodified-Since")
                    .unmarshallLocationName("If-Unmodified-Since").build()).build();

    private static final SdkField<String> SOURCE_IF_MATCH_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceIfMatch")
            .getter(getter(RenameObjectRequest::sourceIfMatch))
            .setter(setter(Builder::sourceIfMatch))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-rename-source-if-match")
                    .unmarshallLocationName("x-amz-rename-source-if-match").build()).build();

    private static final SdkField<String> SOURCE_IF_NONE_MATCH_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceIfNoneMatch")
            .getter(getter(RenameObjectRequest::sourceIfNoneMatch))
            .setter(setter(Builder::sourceIfNoneMatch))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-rename-source-if-none-match")
                    .unmarshallLocationName("x-amz-rename-source-if-none-match").build()).build();

    private static final SdkField<Instant> SOURCE_IF_MODIFIED_SINCE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SourceIfModifiedSince")
            .getter(getter(RenameObjectRequest::sourceIfModifiedSince))
            .setter(setter(Builder::sourceIfModifiedSince))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER)
                    .locationName("x-amz-rename-source-if-modified-since")
                    .unmarshallLocationName("x-amz-rename-source-if-modified-since").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.RFC_822)).build();

    private static final SdkField<Instant> SOURCE_IF_UNMODIFIED_SINCE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SourceIfUnmodifiedSince")
            .getter(getter(RenameObjectRequest::sourceIfUnmodifiedSince))
            .setter(setter(Builder::sourceIfUnmodifiedSince))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER)
                    .locationName("x-amz-rename-source-if-unmodified-since")
                    .unmarshallLocationName("x-amz-rename-source-if-unmodified-since").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.RFC_822)).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(RenameObjectRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-client-token")
                    .unmarshallLocationName("x-amz-client-token").build(), DefaultValueTrait.idempotencyToken()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, KEY_FIELD,
            RENAME_SOURCE_FIELD, DESTINATION_IF_MATCH_FIELD, DESTINATION_IF_NONE_MATCH_FIELD,
            DESTINATION_IF_MODIFIED_SINCE_FIELD, DESTINATION_IF_UNMODIFIED_SINCE_FIELD, SOURCE_IF_MATCH_FIELD,
            SOURCE_IF_NONE_MATCH_FIELD, SOURCE_IF_MODIFIED_SINCE_FIELD, SOURCE_IF_UNMODIFIED_SINCE_FIELD, CLIENT_TOKEN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String bucket;

    private final String key;

    private final String renameSource;

    private final String destinationIfMatch;

    private final String destinationIfNoneMatch;

    private final Instant destinationIfModifiedSince;

    private final Instant destinationIfUnmodifiedSince;

    private final String sourceIfMatch;

    private final String sourceIfNoneMatch;

    private final Instant sourceIfModifiedSince;

    private final Instant sourceIfUnmodifiedSince;

    private final String clientToken;

    private RenameObjectRequest(BuilderImpl builder) {
        super(builder);
        this.bucket = builder.bucket;
        this.key = builder.key;
        this.renameSource = builder.renameSource;
        this.destinationIfMatch = builder.destinationIfMatch;
        this.destinationIfNoneMatch = builder.destinationIfNoneMatch;
        this.destinationIfModifiedSince = builder.destinationIfModifiedSince;
        this.destinationIfUnmodifiedSince = builder.destinationIfUnmodifiedSince;
        this.sourceIfMatch = builder.sourceIfMatch;
        this.sourceIfNoneMatch = builder.sourceIfNoneMatch;
        this.sourceIfModifiedSince = builder.sourceIfModifiedSince;
        this.sourceIfUnmodifiedSince = builder.sourceIfUnmodifiedSince;
        this.clientToken = builder.clientToken;
    }

    /**
     * <p>
     * The bucket name of the directory bucket containing the object.
     * </p>
     * <p>
     * You must use virtual-hosted-style requests in the format
     * <code>Bucket-name.s3express-zone-id.region-code.amazonaws.com</code>. Path-style requests are not supported.
     * Directory bucket names must be unique in the chosen Availability Zone. Bucket names must follow the format
     * <code>bucket-base-name--zone-id--x-s3 </code> (for example, <code>amzn-s3-demo-bucket--usw2-az1--x-s3</code>).
     * For information about bucket naming restrictions, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-bucket-naming-rules.html">Directory bucket
     * naming rules</a> in the <i>Amazon S3 User Guide</i>.
     * </p>
     * 
     * @return The bucket name of the directory bucket containing the object.</p>
     *         <p>
     *         You must use virtual-hosted-style requests in the format
     *         <code>Bucket-name.s3express-zone-id.region-code.amazonaws.com</code>. Path-style requests are not
     *         supported. Directory bucket names must be unique in the chosen Availability Zone. Bucket names must
     *         follow the format <code>bucket-base-name--zone-id--x-s3 </code> (for example,
     *         <code>amzn-s3-demo-bucket--usw2-az1--x-s3</code>). For information about bucket naming restrictions, see
     *         <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-bucket-naming-rules.html">
     *         Directory bucket naming rules</a> in the <i>Amazon S3 User Guide</i>.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * Key name of the object to rename.
     * </p>
     * 
     * @return Key name of the object to rename.
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * Specifies the source for the rename operation. The value must be URL encoded.
     * </p>
     * 
     * @return Specifies the source for the rename operation. The value must be URL encoded.
     */
    public final String renameSource() {
        return renameSource;
    }

    /**
     * <p>
     * Renames the object only if the ETag (entity tag) value provided during the operation matches the ETag of the
     * object in S3. The <code>If-Match</code> header field makes the request method conditional on ETags. If the ETag
     * values do not match, the operation returns a <code>412 Precondition Failed</code> error.
     * </p>
     * <p>
     * Expects the ETag value as a string.
     * </p>
     * 
     * @return Renames the object only if the ETag (entity tag) value provided during the operation matches the ETag of
     *         the object in S3. The <code>If-Match</code> header field makes the request method conditional on ETags.
     *         If the ETag values do not match, the operation returns a <code>412 Precondition Failed</code> error.</p>
     *         <p>
     *         Expects the ETag value as a string.
     */
    public final String destinationIfMatch() {
        return destinationIfMatch;
    }

    /**
     * <p>
     * Renames the object only if the destination does not already exist in the specified directory bucket. If the
     * object does exist when you send a request with <code>If-None-Match:*</code>, the S3 API will return a
     * <code>412 Precondition Failed</code> error, preventing an overwrite. The <code>If-None-Match</code> header
     * prevents overwrites of existing data by validating that there's not an object with the same key name already in
     * your directory bucket.
     * </p>
     * <p>
     * Expects the <code>*</code> character (asterisk).
     * </p>
     * 
     * @return Renames the object only if the destination does not already exist in the specified directory bucket. If
     *         the object does exist when you send a request with <code>If-None-Match:*</code>, the S3 API will return a
     *         <code>412 Precondition Failed</code> error, preventing an overwrite. The <code>If-None-Match</code>
     *         header prevents overwrites of existing data by validating that there's not an object with the same key
     *         name already in your directory bucket.</p>
     *         <p>
     *         Expects the <code>*</code> character (asterisk).
     */
    public final String destinationIfNoneMatch() {
        return destinationIfNoneMatch;
    }

    /**
     * <p>
     * Renames the object if the destination exists and if it has been modified since the specified time.
     * </p>
     * 
     * @return Renames the object if the destination exists and if it has been modified since the specified time.
     */
    public final Instant destinationIfModifiedSince() {
        return destinationIfModifiedSince;
    }

    /**
     * <p>
     * Renames the object if it hasn't been modified since the specified time.
     * </p>
     * 
     * @return Renames the object if it hasn't been modified since the specified time.
     */
    public final Instant destinationIfUnmodifiedSince() {
        return destinationIfUnmodifiedSince;
    }

    /**
     * <p>
     * Renames the object if the source exists and if its entity tag (ETag) matches the specified ETag.
     * </p>
     * 
     * @return Renames the object if the source exists and if its entity tag (ETag) matches the specified ETag.
     */
    public final String sourceIfMatch() {
        return sourceIfMatch;
    }

    /**
     * <p>
     * Renames the object if the source exists and if its entity tag (ETag) is different than the specified ETag. If an
     * asterisk (<code>*</code>) character is provided, the operation will fail and return a
     * <code>412 Precondition Failed</code> error.
     * </p>
     * 
     * @return Renames the object if the source exists and if its entity tag (ETag) is different than the specified
     *         ETag. If an asterisk (<code>*</code>) character is provided, the operation will fail and return a
     *         <code>412 Precondition Failed</code> error.
     */
    public final String sourceIfNoneMatch() {
        return sourceIfNoneMatch;
    }

    /**
     * <p>
     * Renames the object if the source exists and if it has been modified since the specified time.
     * </p>
     * 
     * @return Renames the object if the source exists and if it has been modified since the specified time.
     */
    public final Instant sourceIfModifiedSince() {
        return sourceIfModifiedSince;
    }

    /**
     * <p>
     * Renames the object if the source exists and hasn't been modified since the specified time.
     * </p>
     * 
     * @return Renames the object if the source exists and hasn't been modified since the specified time.
     */
    public final Instant sourceIfUnmodifiedSince() {
        return sourceIfUnmodifiedSince;
    }

    /**
     * <p>
     * A unique string with a max of 64 ASCII characters in the ASCII range of 33 - 126. <code>RenameObject</code>
     * supports idempotency using a client token. To make an idempotent API request using <code>RenameObject</code>,
     * specify a client token in the request. You should not reuse the same client token for other API requests. If you
     * retry a request that completed successfully using the same client token and the same parameters, the retry
     * succeeds without performing any further actions. If you retry a successful request using the same client token,
     * but one or more of the parameters are different, the retry fails and an <code>IdempotentParameterMismatch</code>
     * error is returned.
     * </p>
     * 
     * @return A unique string with a max of 64 ASCII characters in the ASCII range of 33 - 126.
     *         <code>RenameObject</code> supports idempotency using a client token. To make an idempotent API request
     *         using <code>RenameObject</code>, specify a client token in the request. You should not reuse the same
     *         client token for other API requests. If you retry a request that completed successfully using the same
     *         client token and the same parameters, the retry succeeds without performing any further actions. If you
     *         retry a successful request using the same client token, but one or more of the parameters are different,
     *         the retry fails and an <code>IdempotentParameterMismatch</code> error is returned.
     */
    public final String clientToken() {
        return clientToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(renameSource());
        hashCode = 31 * hashCode + Objects.hashCode(destinationIfMatch());
        hashCode = 31 * hashCode + Objects.hashCode(destinationIfNoneMatch());
        hashCode = 31 * hashCode + Objects.hashCode(destinationIfModifiedSince());
        hashCode = 31 * hashCode + Objects.hashCode(destinationIfUnmodifiedSince());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIfMatch());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIfNoneMatch());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIfModifiedSince());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIfUnmodifiedSince());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RenameObjectRequest)) {
            return false;
        }
        RenameObjectRequest other = (RenameObjectRequest) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(key(), other.key())
                && Objects.equals(renameSource(), other.renameSource())
                && Objects.equals(destinationIfMatch(), other.destinationIfMatch())
                && Objects.equals(destinationIfNoneMatch(), other.destinationIfNoneMatch())
                && Objects.equals(destinationIfModifiedSince(), other.destinationIfModifiedSince())
                && Objects.equals(destinationIfUnmodifiedSince(), other.destinationIfUnmodifiedSince())
                && Objects.equals(sourceIfMatch(), other.sourceIfMatch())
                && Objects.equals(sourceIfNoneMatch(), other.sourceIfNoneMatch())
                && Objects.equals(sourceIfModifiedSince(), other.sourceIfModifiedSince())
                && Objects.equals(sourceIfUnmodifiedSince(), other.sourceIfUnmodifiedSince())
                && Objects.equals(clientToken(), other.clientToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RenameObjectRequest").add("Bucket", bucket()).add("Key", key())
                .add("RenameSource", renameSource()).add("DestinationIfMatch", destinationIfMatch())
                .add("DestinationIfNoneMatch", destinationIfNoneMatch())
                .add("DestinationIfModifiedSince", destinationIfModifiedSince())
                .add("DestinationIfUnmodifiedSince", destinationIfUnmodifiedSince()).add("SourceIfMatch", sourceIfMatch())
                .add("SourceIfNoneMatch", sourceIfNoneMatch()).add("SourceIfModifiedSince", sourceIfModifiedSince())
                .add("SourceIfUnmodifiedSince", sourceIfUnmodifiedSince()).add("ClientToken", clientToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "RenameSource":
            return Optional.ofNullable(clazz.cast(renameSource()));
        case "DestinationIfMatch":
            return Optional.ofNullable(clazz.cast(destinationIfMatch()));
        case "DestinationIfNoneMatch":
            return Optional.ofNullable(clazz.cast(destinationIfNoneMatch()));
        case "DestinationIfModifiedSince":
            return Optional.ofNullable(clazz.cast(destinationIfModifiedSince()));
        case "DestinationIfUnmodifiedSince":
            return Optional.ofNullable(clazz.cast(destinationIfUnmodifiedSince()));
        case "SourceIfMatch":
            return Optional.ofNullable(clazz.cast(sourceIfMatch()));
        case "SourceIfNoneMatch":
            return Optional.ofNullable(clazz.cast(sourceIfNoneMatch()));
        case "SourceIfModifiedSince":
            return Optional.ofNullable(clazz.cast(sourceIfModifiedSince()));
        case "SourceIfUnmodifiedSince":
            return Optional.ofNullable(clazz.cast(sourceIfUnmodifiedSince()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Bucket", BUCKET_FIELD);
        map.put("Key", KEY_FIELD);
        map.put("x-amz-rename-source", RENAME_SOURCE_FIELD);
        map.put("If-Match", DESTINATION_IF_MATCH_FIELD);
        map.put("If-None-Match", DESTINATION_IF_NONE_MATCH_FIELD);
        map.put("If-Modified-Since", DESTINATION_IF_MODIFIED_SINCE_FIELD);
        map.put("If-Unmodified-Since", DESTINATION_IF_UNMODIFIED_SINCE_FIELD);
        map.put("x-amz-rename-source-if-match", SOURCE_IF_MATCH_FIELD);
        map.put("x-amz-rename-source-if-none-match", SOURCE_IF_NONE_MATCH_FIELD);
        map.put("x-amz-rename-source-if-modified-since", SOURCE_IF_MODIFIED_SINCE_FIELD);
        map.put("x-amz-rename-source-if-unmodified-since", SOURCE_IF_UNMODIFIED_SINCE_FIELD);
        map.put("x-amz-client-token", CLIENT_TOKEN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RenameObjectRequest, T> g) {
        return obj -> g.apply((RenameObjectRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends S3Request.Builder, SdkPojo, CopyableBuilder<Builder, RenameObjectRequest> {
        /**
         * <p>
         * The bucket name of the directory bucket containing the object.
         * </p>
         * <p>
         * You must use virtual-hosted-style requests in the format
         * <code>Bucket-name.s3express-zone-id.region-code.amazonaws.com</code>. Path-style requests are not supported.
         * Directory bucket names must be unique in the chosen Availability Zone. Bucket names must follow the format
         * <code>bucket-base-name--zone-id--x-s3 </code> (for example, <code>amzn-s3-demo-bucket--usw2-az1--x-s3</code>
         * ). For information about bucket naming restrictions, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-bucket-naming-rules.html">Directory
         * bucket naming rules</a> in the <i>Amazon S3 User Guide</i>.
         * </p>
         * 
         * @param bucket
         *        The bucket name of the directory bucket containing the object.</p>
         *        <p>
         *        You must use virtual-hosted-style requests in the format
         *        <code>Bucket-name.s3express-zone-id.region-code.amazonaws.com</code>. Path-style requests are not
         *        supported. Directory bucket names must be unique in the chosen Availability Zone. Bucket names must
         *        follow the format <code>bucket-base-name--zone-id--x-s3 </code> (for example,
         *        <code>amzn-s3-demo-bucket--usw2-az1--x-s3</code>). For information about bucket naming restrictions,
         *        see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-bucket-naming-rules.html">
         *        Directory bucket naming rules</a> in the <i>Amazon S3 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * Key name of the object to rename.
         * </p>
         * 
         * @param key
         *        Key name of the object to rename.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Specifies the source for the rename operation. The value must be URL encoded.
         * </p>
         * 
         * @param renameSource
         *        Specifies the source for the rename operation. The value must be URL encoded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder renameSource(String renameSource);

        /**
         * <p>
         * Renames the object only if the ETag (entity tag) value provided during the operation matches the ETag of the
         * object in S3. The <code>If-Match</code> header field makes the request method conditional on ETags. If the
         * ETag values do not match, the operation returns a <code>412 Precondition Failed</code> error.
         * </p>
         * <p>
         * Expects the ETag value as a string.
         * </p>
         * 
         * @param destinationIfMatch
         *        Renames the object only if the ETag (entity tag) value provided during the operation matches the ETag
         *        of the object in S3. The <code>If-Match</code> header field makes the request method conditional on
         *        ETags. If the ETag values do not match, the operation returns a <code>412 Precondition Failed</code>
         *        error.</p>
         *        <p>
         *        Expects the ETag value as a string.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationIfMatch(String destinationIfMatch);

        /**
         * <p>
         * Renames the object only if the destination does not already exist in the specified directory bucket. If the
         * object does exist when you send a request with <code>If-None-Match:*</code>, the S3 API will return a
         * <code>412 Precondition Failed</code> error, preventing an overwrite. The <code>If-None-Match</code> header
         * prevents overwrites of existing data by validating that there's not an object with the same key name already
         * in your directory bucket.
         * </p>
         * <p>
         * Expects the <code>*</code> character (asterisk).
         * </p>
         * 
         * @param destinationIfNoneMatch
         *        Renames the object only if the destination does not already exist in the specified directory bucket.
         *        If the object does exist when you send a request with <code>If-None-Match:*</code>, the S3 API will
         *        return a <code>412 Precondition Failed</code> error, preventing an overwrite. The
         *        <code>If-None-Match</code> header prevents overwrites of existing data by validating that there's not
         *        an object with the same key name already in your directory bucket.</p>
         *        <p>
         *        Expects the <code>*</code> character (asterisk).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationIfNoneMatch(String destinationIfNoneMatch);

        /**
         * <p>
         * Renames the object if the destination exists and if it has been modified since the specified time.
         * </p>
         * 
         * @param destinationIfModifiedSince
         *        Renames the object if the destination exists and if it has been modified since the specified time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationIfModifiedSince(Instant destinationIfModifiedSince);

        /**
         * <p>
         * Renames the object if it hasn't been modified since the specified time.
         * </p>
         * 
         * @param destinationIfUnmodifiedSince
         *        Renames the object if it hasn't been modified since the specified time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationIfUnmodifiedSince(Instant destinationIfUnmodifiedSince);

        /**
         * <p>
         * Renames the object if the source exists and if its entity tag (ETag) matches the specified ETag.
         * </p>
         * 
         * @param sourceIfMatch
         *        Renames the object if the source exists and if its entity tag (ETag) matches the specified ETag.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIfMatch(String sourceIfMatch);

        /**
         * <p>
         * Renames the object if the source exists and if its entity tag (ETag) is different than the specified ETag. If
         * an asterisk (<code>*</code>) character is provided, the operation will fail and return a
         * <code>412 Precondition Failed</code> error.
         * </p>
         * 
         * @param sourceIfNoneMatch
         *        Renames the object if the source exists and if its entity tag (ETag) is different than the specified
         *        ETag. If an asterisk (<code>*</code>) character is provided, the operation will fail and return a
         *        <code>412 Precondition Failed</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIfNoneMatch(String sourceIfNoneMatch);

        /**
         * <p>
         * Renames the object if the source exists and if it has been modified since the specified time.
         * </p>
         * 
         * @param sourceIfModifiedSince
         *        Renames the object if the source exists and if it has been modified since the specified time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIfModifiedSince(Instant sourceIfModifiedSince);

        /**
         * <p>
         * Renames the object if the source exists and hasn't been modified since the specified time.
         * </p>
         * 
         * @param sourceIfUnmodifiedSince
         *        Renames the object if the source exists and hasn't been modified since the specified time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIfUnmodifiedSince(Instant sourceIfUnmodifiedSince);

        /**
         * <p>
         * A unique string with a max of 64 ASCII characters in the ASCII range of 33 - 126. <code>RenameObject</code>
         * supports idempotency using a client token. To make an idempotent API request using <code>RenameObject</code>,
         * specify a client token in the request. You should not reuse the same client token for other API requests. If
         * you retry a request that completed successfully using the same client token and the same parameters, the
         * retry succeeds without performing any further actions. If you retry a successful request using the same
         * client token, but one or more of the parameters are different, the retry fails and an
         * <code>IdempotentParameterMismatch</code> error is returned.
         * </p>
         * 
         * @param clientToken
         *        A unique string with a max of 64 ASCII characters in the ASCII range of 33 - 126.
         *        <code>RenameObject</code> supports idempotency using a client token. To make an idempotent API request
         *        using <code>RenameObject</code>, specify a client token in the request. You should not reuse the same
         *        client token for other API requests. If you retry a request that completed successfully using the same
         *        client token and the same parameters, the retry succeeds without performing any further actions. If
         *        you retry a successful request using the same client token, but one or more of the parameters are
         *        different, the retry fails and an <code>IdempotentParameterMismatch</code> error is returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3Request.BuilderImpl implements Builder {
        private String bucket;

        private String key;

        private String renameSource;

        private String destinationIfMatch;

        private String destinationIfNoneMatch;

        private Instant destinationIfModifiedSince;

        private Instant destinationIfUnmodifiedSince;

        private String sourceIfMatch;

        private String sourceIfNoneMatch;

        private Instant sourceIfModifiedSince;

        private Instant sourceIfUnmodifiedSince;

        private String clientToken;

        private BuilderImpl() {
        }

        private BuilderImpl(RenameObjectRequest model) {
            super(model);
            bucket(model.bucket);
            key(model.key);
            renameSource(model.renameSource);
            destinationIfMatch(model.destinationIfMatch);
            destinationIfNoneMatch(model.destinationIfNoneMatch);
            destinationIfModifiedSince(model.destinationIfModifiedSince);
            destinationIfUnmodifiedSince(model.destinationIfUnmodifiedSince);
            sourceIfMatch(model.sourceIfMatch);
            sourceIfNoneMatch(model.sourceIfNoneMatch);
            sourceIfModifiedSince(model.sourceIfModifiedSince);
            sourceIfUnmodifiedSince(model.sourceIfUnmodifiedSince);
            clientToken(model.clientToken);
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final String getRenameSource() {
            return renameSource;
        }

        public final void setRenameSource(String renameSource) {
            this.renameSource = renameSource;
        }

        @Override
        public final Builder renameSource(String renameSource) {
            this.renameSource = renameSource;
            return this;
        }

        public final String getDestinationIfMatch() {
            return destinationIfMatch;
        }

        public final void setDestinationIfMatch(String destinationIfMatch) {
            this.destinationIfMatch = destinationIfMatch;
        }

        @Override
        public final Builder destinationIfMatch(String destinationIfMatch) {
            this.destinationIfMatch = destinationIfMatch;
            return this;
        }

        public final String getDestinationIfNoneMatch() {
            return destinationIfNoneMatch;
        }

        public final void setDestinationIfNoneMatch(String destinationIfNoneMatch) {
            this.destinationIfNoneMatch = destinationIfNoneMatch;
        }

        @Override
        public final Builder destinationIfNoneMatch(String destinationIfNoneMatch) {
            this.destinationIfNoneMatch = destinationIfNoneMatch;
            return this;
        }

        public final Instant getDestinationIfModifiedSince() {
            return destinationIfModifiedSince;
        }

        public final void setDestinationIfModifiedSince(Instant destinationIfModifiedSince) {
            this.destinationIfModifiedSince = destinationIfModifiedSince;
        }

        @Override
        public final Builder destinationIfModifiedSince(Instant destinationIfModifiedSince) {
            this.destinationIfModifiedSince = destinationIfModifiedSince;
            return this;
        }

        public final Instant getDestinationIfUnmodifiedSince() {
            return destinationIfUnmodifiedSince;
        }

        public final void setDestinationIfUnmodifiedSince(Instant destinationIfUnmodifiedSince) {
            this.destinationIfUnmodifiedSince = destinationIfUnmodifiedSince;
        }

        @Override
        public final Builder destinationIfUnmodifiedSince(Instant destinationIfUnmodifiedSince) {
            this.destinationIfUnmodifiedSince = destinationIfUnmodifiedSince;
            return this;
        }

        public final String getSourceIfMatch() {
            return sourceIfMatch;
        }

        public final void setSourceIfMatch(String sourceIfMatch) {
            this.sourceIfMatch = sourceIfMatch;
        }

        @Override
        public final Builder sourceIfMatch(String sourceIfMatch) {
            this.sourceIfMatch = sourceIfMatch;
            return this;
        }

        public final String getSourceIfNoneMatch() {
            return sourceIfNoneMatch;
        }

        public final void setSourceIfNoneMatch(String sourceIfNoneMatch) {
            this.sourceIfNoneMatch = sourceIfNoneMatch;
        }

        @Override
        public final Builder sourceIfNoneMatch(String sourceIfNoneMatch) {
            this.sourceIfNoneMatch = sourceIfNoneMatch;
            return this;
        }

        public final Instant getSourceIfModifiedSince() {
            return sourceIfModifiedSince;
        }

        public final void setSourceIfModifiedSince(Instant sourceIfModifiedSince) {
            this.sourceIfModifiedSince = sourceIfModifiedSince;
        }

        @Override
        public final Builder sourceIfModifiedSince(Instant sourceIfModifiedSince) {
            this.sourceIfModifiedSince = sourceIfModifiedSince;
            return this;
        }

        public final Instant getSourceIfUnmodifiedSince() {
            return sourceIfUnmodifiedSince;
        }

        public final void setSourceIfUnmodifiedSince(Instant sourceIfUnmodifiedSince) {
            this.sourceIfUnmodifiedSince = sourceIfUnmodifiedSince;
        }

        @Override
        public final Builder sourceIfUnmodifiedSince(Instant sourceIfUnmodifiedSince) {
            this.sourceIfUnmodifiedSince = sourceIfUnmodifiedSince;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RenameObjectRequest build() {
            return new RenameObjectRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
