/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.Response;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.xml.AwsXmlProtocolFactory;
import software.amazon.awssdk.protocols.xml.XmlOperationMetadata;
import software.amazon.awssdk.services.s3control.model.BadRequestException;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.CreateJobRequest;
import software.amazon.awssdk.services.s3control.model.CreateJobResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.DescribeJobRequest;
import software.amazon.awssdk.services.s3control.model.DescribeJobResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.IdempotencyException;
import software.amazon.awssdk.services.s3control.model.InternalServiceException;
import software.amazon.awssdk.services.s3control.model.InvalidNextTokenException;
import software.amazon.awssdk.services.s3control.model.InvalidRequestException;
import software.amazon.awssdk.services.s3control.model.JobStatusException;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse;
import software.amazon.awssdk.services.s3control.model.ListJobsRequest;
import software.amazon.awssdk.services.s3control.model.ListJobsResponse;
import software.amazon.awssdk.services.s3control.model.NoSuchPublicAccessBlockConfigurationException;
import software.amazon.awssdk.services.s3control.model.NotFoundException;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.S3ControlException;
import software.amazon.awssdk.services.s3control.model.S3ControlRequest;
import software.amazon.awssdk.services.s3control.model.TooManyRequestsException;
import software.amazon.awssdk.services.s3control.model.TooManyTagsException;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityResponse;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusResponse;
import software.amazon.awssdk.services.s3control.paginators.ListAccessPointsPublisher;
import software.amazon.awssdk.services.s3control.paginators.ListJobsPublisher;
import software.amazon.awssdk.services.s3control.transform.CreateAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeletePublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyStatusRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetPublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListAccessPointsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutPublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.UpdateJobPriorityRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.UpdateJobStatusRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link S3ControlAsyncClient}.
 *
 * @see S3ControlAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultS3ControlAsyncClient implements S3ControlAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultS3ControlAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsXmlProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultS3ControlAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an access point and associates it with the specified bucket.
     * </p>
     *
     * @param createAccessPointRequest
     * @return A Java Future containing the result of the CreateAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.CreateAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessPointResponse> createAccessPoint(CreateAccessPointRequest createAccessPointRequest) {
        try {

            HttpResponseHandler<Response<CreateAccessPointResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(CreateAccessPointResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<CreateAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessPointRequest, CreateAccessPointResponse>()
                            .withOperationName("CreateAccessPoint")
                            .withMarshaller(new CreateAccessPointRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(createAccessPointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon S3 batch operations job.
     * </p>
     *
     * @param createJobRequest
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException</li>
     *         <li>BadRequestException</li>
     *         <li>IdempotencyException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {
        try {

            HttpResponseHandler<Response<CreateJobResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    CreateJobResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<CreateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>().withOperationName("CreateJob")
                            .withMarshaller(new CreateJobRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(createJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified access point.
     * </p>
     *
     * @param deleteAccessPointRequest
     * @return A Java Future containing the result of the DeleteAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessPointResponse> deleteAccessPoint(DeleteAccessPointRequest deleteAccessPointRequest) {
        try {

            HttpResponseHandler<Response<DeleteAccessPointResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteAccessPointResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<DeleteAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessPointRequest, DeleteAccessPointResponse>()
                            .withOperationName("DeleteAccessPoint")
                            .withMarshaller(new DeleteAccessPointRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(deleteAccessPointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the access point policy for the specified access point.
     * </p>
     *
     * @param deleteAccessPointPolicyRequest
     * @return A Java Future containing the result of the DeleteAccessPointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessPointPolicyResponse> deleteAccessPointPolicy(
            DeleteAccessPointPolicyRequest deleteAccessPointPolicyRequest) {
        try {

            HttpResponseHandler<Response<DeleteAccessPointPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteAccessPointPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<DeleteAccessPointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessPointPolicyRequest, DeleteAccessPointPolicyResponse>()
                            .withOperationName("DeleteAccessPointPolicy")
                            .withMarshaller(new DeleteAccessPointPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(deleteAccessPointPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete the tags on a Amazon S3 batch operations job, if any.
     * </p>
     *
     * @param deleteJobTaggingRequest
     * @return A Java Future containing the result of the DeleteJobTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobTaggingResponse> deleteJobTagging(DeleteJobTaggingRequest deleteJobTaggingRequest) {
        try {

            HttpResponseHandler<Response<DeleteJobTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteJobTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<DeleteJobTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobTaggingRequest, DeleteJobTaggingResponse>()
                            .withOperationName("DeleteJobTagging")
                            .withMarshaller(new DeleteJobTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(deleteJobTaggingRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param deletePublicAccessBlockRequest
     * @return A Java Future containing the result of the DeletePublicAccessBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeletePublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeletePublicAccessBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePublicAccessBlockResponse> deletePublicAccessBlock(
            DeletePublicAccessBlockRequest deletePublicAccessBlockRequest) {
        try {

            HttpResponseHandler<Response<DeletePublicAccessBlockResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeletePublicAccessBlockResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<DeletePublicAccessBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePublicAccessBlockRequest, DeletePublicAccessBlockResponse>()
                            .withOperationName("DeletePublicAccessBlock")
                            .withMarshaller(new DeletePublicAccessBlockRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(deletePublicAccessBlockRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the configuration parameters and status for a batch operations job.
     * </p>
     *
     * @param describeJobRequest
     * @return A Java Future containing the result of the DescribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DescribeJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobResponse> describeJob(DescribeJobRequest describeJobRequest) {
        try {

            HttpResponseHandler<Response<DescribeJobResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    DescribeJobResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<DescribeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                            .withOperationName("DescribeJob").withMarshaller(new DescribeJobRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(describeJobRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration information about the specified access point.
     * </p>
     *
     * @param getAccessPointRequest
     * @return A Java Future containing the result of the GetAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointResponse> getAccessPoint(GetAccessPointRequest getAccessPointRequest) {
        try {

            HttpResponseHandler<Response<GetAccessPointResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<GetAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointRequest, GetAccessPointResponse>()
                            .withOperationName("GetAccessPoint")
                            .withMarshaller(new GetAccessPointRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(getAccessPointRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the access point policy associated with the specified access point.
     * </p>
     *
     * @param getAccessPointPolicyRequest
     * @return A Java Future containing the result of the GetAccessPointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointPolicyResponse> getAccessPointPolicy(
            GetAccessPointPolicyRequest getAccessPointPolicyRequest) {
        try {

            HttpResponseHandler<Response<GetAccessPointPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<GetAccessPointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointPolicyRequest, GetAccessPointPolicyResponse>()
                            .withOperationName("GetAccessPointPolicy")
                            .withMarshaller(new GetAccessPointPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(getAccessPointPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Indicates whether the specified access point currently has a policy that allows public access. For more
     * information about public access through access points, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html">Managing Data Access with Amazon S3
     * Access Points</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     *
     * @param getAccessPointPolicyStatusRequest
     * @return A Java Future containing the result of the GetAccessPointPolicyStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointPolicyStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicyStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointPolicyStatusResponse> getAccessPointPolicyStatus(
            GetAccessPointPolicyStatusRequest getAccessPointPolicyStatusRequest) {
        try {

            HttpResponseHandler<Response<GetAccessPointPolicyStatusResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointPolicyStatusResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<GetAccessPointPolicyStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointPolicyStatusRequest, GetAccessPointPolicyStatusResponse>()
                            .withOperationName("GetAccessPointPolicyStatus")
                            .withMarshaller(new GetAccessPointPolicyStatusRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(getAccessPointPolicyStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieve the tags on a Amazon S3 batch operations job.
     * </p>
     *
     * @param getJobTaggingRequest
     * @return A Java Future containing the result of the GetJobTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobTaggingResponse> getJobTagging(GetJobTaggingRequest getJobTaggingRequest) {
        try {

            HttpResponseHandler<Response<GetJobTaggingResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    GetJobTaggingResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<GetJobTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobTaggingRequest, GetJobTaggingResponse>()
                            .withOperationName("GetJobTagging")
                            .withMarshaller(new GetJobTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(getJobTaggingRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param getPublicAccessBlockRequest
     * @return A Java Future containing the result of the GetPublicAccessBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchPublicAccessBlockConfigurationException Amazon S3 throws this exception if you make a
     *         <code>GetPublicAccessBlock</code> request against an account that doesn't have a
     *         <code>PublicAccessBlockConfiguration</code> set.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPublicAccessBlockResponse> getPublicAccessBlock(
            GetPublicAccessBlockRequest getPublicAccessBlockRequest) {
        try {

            HttpResponseHandler<Response<GetPublicAccessBlockResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetPublicAccessBlockResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<GetPublicAccessBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPublicAccessBlockRequest, GetPublicAccessBlockResponse>()
                            .withOperationName("GetPublicAccessBlock")
                            .withMarshaller(new GetPublicAccessBlockRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(getPublicAccessBlockRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), then the response will include a continuation token that you can use
     * to list the additional access points.
     * </p>
     *
     * @param listAccessPointsRequest
     * @return A Java Future containing the result of the ListAccessPoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessPointsResponse> listAccessPoints(ListAccessPointsRequest listAccessPointsRequest) {
        try {

            HttpResponseHandler<Response<ListAccessPointsResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(ListAccessPointsResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<ListAccessPointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessPointsRequest, ListAccessPointsResponse>()
                            .withOperationName("ListAccessPoints")
                            .withMarshaller(new ListAccessPointsRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(listAccessPointsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), then the response will include a continuation token that you can use
     * to list the additional access points.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsPublisher publisher = client.listAccessPointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsPublisher publisher = client.listAccessPointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation.</b>
     * </p>
     *
     * @param listAccessPointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAccessPointsPublisher listAccessPointsPaginator(ListAccessPointsRequest listAccessPointsRequest) {
        return new ListAccessPointsPublisher(this, applyPaginatorUserAgent(listAccessPointsRequest));
    }

    /**
     * <p>
     * Lists current jobs and jobs that have ended within the last 30 days for the AWS account making the request.
     * </p>
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException</li>
     *         <li>InternalServiceException</li>
     *         <li>InvalidNextTokenException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        try {

            HttpResponseHandler<Response<ListJobsResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    ListJobsResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(listJobsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists current jobs and jobs that have ended within the last 30 days for the AWS account making the request.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3control.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3control.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException</li>
     *         <li>InternalServiceException</li>
     *         <li>InvalidNextTokenException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListJobsPublisher listJobsPaginator(ListJobsRequest listJobsRequest) {
        return new ListJobsPublisher(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * Associates an access policy with the specified access point. Each access point can have only one policy, so a
     * request made to this API replaces any existing policy associated with the specified access point.
     * </p>
     *
     * @param putAccessPointPolicyRequest
     * @return A Java Future containing the result of the PutAccessPointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccessPointPolicyResponse> putAccessPointPolicy(
            PutAccessPointPolicyRequest putAccessPointPolicyRequest) {
        try {

            HttpResponseHandler<Response<PutAccessPointPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutAccessPointPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<PutAccessPointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccessPointPolicyRequest, PutAccessPointPolicyResponse>()
                            .withOperationName("PutAccessPointPolicy")
                            .withMarshaller(new PutAccessPointPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(putAccessPointPolicyRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Replace the set of tags on a Amazon S3 batch operations job.
     * </p>
     *
     * @param putJobTaggingRequest
     * @return A Java Future containing the result of the PutJobTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>TooManyTagsException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutJobTaggingResponse> putJobTagging(PutJobTaggingRequest putJobTaggingRequest) {
        try {

            HttpResponseHandler<Response<PutJobTaggingResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    PutJobTaggingResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<PutJobTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutJobTaggingRequest, PutJobTaggingResponse>()
                            .withOperationName("PutJobTagging")
                            .withMarshaller(new PutJobTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(putJobTaggingRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or modifies the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param putPublicAccessBlockRequest
     * @return A Java Future containing the result of the PutPublicAccessBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutPublicAccessBlockResponse> putPublicAccessBlock(
            PutPublicAccessBlockRequest putPublicAccessBlockRequest) {
        try {

            HttpResponseHandler<Response<PutPublicAccessBlockResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutPublicAccessBlockResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<PutPublicAccessBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPublicAccessBlockRequest, PutPublicAccessBlockResponse>()
                            .withOperationName("PutPublicAccessBlock")
                            .withMarshaller(new PutPublicAccessBlockRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(putPublicAccessBlockRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing job's priority.
     * </p>
     *
     * @param updateJobPriorityRequest
     * @return A Java Future containing the result of the UpdateJobPriority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.UpdateJobPriority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobPriority" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobPriorityResponse> updateJobPriority(UpdateJobPriorityRequest updateJobPriorityRequest) {
        try {

            HttpResponseHandler<Response<UpdateJobPriorityResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(UpdateJobPriorityResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<UpdateJobPriorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobPriorityRequest, UpdateJobPriorityResponse>()
                            .withOperationName("UpdateJobPriority")
                            .withMarshaller(new UpdateJobPriorityRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(updateJobPriorityRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status for the specified job. Use this operation to confirm that you want to run a job or to cancel
     * an existing job.
     * </p>
     *
     * @param updateJobStatusRequest
     * @return A Java Future containing the result of the UpdateJobStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>JobStatusException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.UpdateJobStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobStatusResponse> updateJobStatus(UpdateJobStatusRequest updateJobStatusRequest) {
        try {

            HttpResponseHandler<Response<UpdateJobStatusResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(UpdateJobStatusResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<UpdateJobStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobStatusRequest, UpdateJobStatusResponse>()
                            .withOperationName("UpdateJobStatus")
                            .withMarshaller(new UpdateJobStatusRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).withInput(updateJobStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private AwsXmlProtocolFactory init() {
        return AwsXmlProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("JobStatusException")
                                .exceptionBuilderSupplier(JobStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotencyException")
                                .exceptionBuilderSupplier(IdempotencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchPublicAccessBlockConfiguration")
                                .exceptionBuilderSupplier(NoSuchPublicAccessBlockConfigurationException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(S3ControlException::builder).build();
    }

    private <T extends S3ControlRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
