/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.s3control.model.BadRequestException;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.CreateJobRequest;
import software.amazon.awssdk.services.s3control.model.CreateJobResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.DescribeJobRequest;
import software.amazon.awssdk.services.s3control.model.DescribeJobResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.IdempotencyException;
import software.amazon.awssdk.services.s3control.model.InternalServiceException;
import software.amazon.awssdk.services.s3control.model.InvalidNextTokenException;
import software.amazon.awssdk.services.s3control.model.InvalidRequestException;
import software.amazon.awssdk.services.s3control.model.JobStatusException;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse;
import software.amazon.awssdk.services.s3control.model.ListJobsRequest;
import software.amazon.awssdk.services.s3control.model.ListJobsResponse;
import software.amazon.awssdk.services.s3control.model.NoSuchPublicAccessBlockConfigurationException;
import software.amazon.awssdk.services.s3control.model.NotFoundException;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.S3ControlException;
import software.amazon.awssdk.services.s3control.model.TooManyRequestsException;
import software.amazon.awssdk.services.s3control.model.TooManyTagsException;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityResponse;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusResponse;
import software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable;
import software.amazon.awssdk.services.s3control.paginators.ListJobsIterable;

/**
 * Service client for accessing AWS S3 Control. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * AWS S3 Control provides access to Amazon S3 control plane operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface S3ControlClient extends SdkClient {
    String SERVICE_NAME = "s3";

    /**
     * Create a {@link S3ControlClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static S3ControlClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link S3ControlClient}.
     */
    static S3ControlClientBuilder builder() {
        return new DefaultS3ControlClientBuilder();
    }

    /**
     * <p>
     * Creates an access point and associates it with the specified bucket.
     * </p>
     *
     * @param createAccessPointRequest
     * @return Result of the CreateAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.CreateAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateAccessPointResponse createAccessPoint(CreateAccessPointRequest createAccessPointRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an access point and associates it with the specified bucket.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateAccessPointRequest.Builder} avoiding the need
     * to create one manually via {@link CreateAccessPointRequest#builder()}
     * </p>
     *
     * @param createAccessPointRequest
     *        A {@link Consumer} that will call methods on {@link CreateAccessPointRequest.Builder} to create a request.
     * @return Result of the CreateAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.CreateAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    default CreateAccessPointResponse createAccessPoint(Consumer<CreateAccessPointRequest.Builder> createAccessPointRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        return createAccessPoint(CreateAccessPointRequest.builder().applyMutation(createAccessPointRequest).build());
    }

    /**
     * <p>
     * Creates an Amazon S3 batch operations job.
     * </p>
     *
     * @param createJobRequest
     * @return Result of the CreateJob operation returned by the service.
     * @throws TooManyRequestsException
     * @throws BadRequestException
     * @throws IdempotencyException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateJobResponse createJob(CreateJobRequest createJobRequest) throws TooManyRequestsException, BadRequestException,
            IdempotencyException, InternalServiceException, AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an Amazon S3 batch operations job.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateJobRequest.Builder} avoiding the need to
     * create one manually via {@link CreateJobRequest#builder()}
     * </p>
     *
     * @param createJobRequest
     *        A {@link Consumer} that will call methods on {@link CreateJobRequest.Builder} to create a request.
     * @return Result of the CreateJob operation returned by the service.
     * @throws TooManyRequestsException
     * @throws BadRequestException
     * @throws IdempotencyException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.CreateJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateJobResponse createJob(Consumer<CreateJobRequest.Builder> createJobRequest) throws TooManyRequestsException,
            BadRequestException, IdempotencyException, InternalServiceException, AwsServiceException, SdkClientException,
            S3ControlException {
        return createJob(CreateJobRequest.builder().applyMutation(createJobRequest).build());
    }

    /**
     * <p>
     * Deletes the specified access point.
     * </p>
     *
     * @param deleteAccessPointRequest
     * @return Result of the DeleteAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteAccessPointResponse deleteAccessPoint(DeleteAccessPointRequest deleteAccessPointRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified access point.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteAccessPointRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteAccessPointRequest#builder()}
     * </p>
     *
     * @param deleteAccessPointRequest
     *        A {@link Consumer} that will call methods on {@link DeleteAccessPointRequest.Builder} to create a request.
     * @return Result of the DeleteAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteAccessPointResponse deleteAccessPoint(Consumer<DeleteAccessPointRequest.Builder> deleteAccessPointRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        return deleteAccessPoint(DeleteAccessPointRequest.builder().applyMutation(deleteAccessPointRequest).build());
    }

    /**
     * <p>
     * Deletes the access point policy for the specified access point.
     * </p>
     *
     * @param deleteAccessPointPolicyRequest
     * @return Result of the DeleteAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteAccessPointPolicyResponse deleteAccessPointPolicy(DeleteAccessPointPolicyRequest deleteAccessPointPolicyRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the access point policy for the specified access point.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteAccessPointPolicyRequest.Builder} avoiding
     * the need to create one manually via {@link DeleteAccessPointPolicyRequest#builder()}
     * </p>
     *
     * @param deleteAccessPointPolicyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteAccessPointPolicyRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteAccessPointPolicyResponse deleteAccessPointPolicy(
            Consumer<DeleteAccessPointPolicyRequest.Builder> deleteAccessPointPolicyRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {
        return deleteAccessPointPolicy(DeleteAccessPointPolicyRequest.builder().applyMutation(deleteAccessPointPolicyRequest)
                .build());
    }

    /**
     * <p>
     * Delete the tags on a Amazon S3 batch operations job, if any.
     * </p>
     *
     * @param deleteJobTaggingRequest
     * @return Result of the DeleteJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteJobTaggingResponse deleteJobTagging(DeleteJobTaggingRequest deleteJobTaggingRequest)
            throws InternalServiceException, TooManyRequestsException, NotFoundException, AwsServiceException,
            SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Delete the tags on a Amazon S3 batch operations job, if any.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteJobTaggingRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteJobTaggingRequest#builder()}
     * </p>
     *
     * @param deleteJobTaggingRequest
     *        A {@link Consumer} that will call methods on {@link DeleteJobTaggingRequest.Builder} to create a request.
     * @return Result of the DeleteJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeleteJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteJobTaggingResponse deleteJobTagging(Consumer<DeleteJobTaggingRequest.Builder> deleteJobTaggingRequest)
            throws InternalServiceException, TooManyRequestsException, NotFoundException, AwsServiceException,
            SdkClientException, S3ControlException {
        return deleteJobTagging(DeleteJobTaggingRequest.builder().applyMutation(deleteJobTaggingRequest).build());
    }

    /**
     * <p>
     * Removes the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param deletePublicAccessBlockRequest
     * @return Result of the DeletePublicAccessBlock operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeletePublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeletePublicAccessBlock"
     *      target="_top">AWS API Documentation</a>
     */
    default DeletePublicAccessBlockResponse deletePublicAccessBlock(DeletePublicAccessBlockRequest deletePublicAccessBlockRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeletePublicAccessBlockRequest.Builder} avoiding
     * the need to create one manually via {@link DeletePublicAccessBlockRequest#builder()}
     * </p>
     *
     * @param deletePublicAccessBlockRequest
     *        A {@link Consumer} that will call methods on {@link DeletePublicAccessBlockRequest.Builder} to create a
     *        request.
     * @return Result of the DeletePublicAccessBlock operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DeletePublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeletePublicAccessBlock"
     *      target="_top">AWS API Documentation</a>
     */
    default DeletePublicAccessBlockResponse deletePublicAccessBlock(
            Consumer<DeletePublicAccessBlockRequest.Builder> deletePublicAccessBlockRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {
        return deletePublicAccessBlock(DeletePublicAccessBlockRequest.builder().applyMutation(deletePublicAccessBlockRequest)
                .build());
    }

    /**
     * <p>
     * Retrieves the configuration parameters and status for a batch operations job.
     * </p>
     *
     * @param describeJobRequest
     * @return Result of the DescribeJob operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DescribeJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobResponse describeJob(DescribeJobRequest describeJobRequest) throws BadRequestException,
            TooManyRequestsException, NotFoundException, InternalServiceException, AwsServiceException, SdkClientException,
            S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the configuration parameters and status for a batch operations job.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeJobRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeJobRequest#builder()}
     * </p>
     *
     * @param describeJobRequest
     *        A {@link Consumer} that will call methods on {@link DescribeJobRequest.Builder} to create a request.
     * @return Result of the DescribeJob operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.DescribeJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    default DescribeJobResponse describeJob(Consumer<DescribeJobRequest.Builder> describeJobRequest) throws BadRequestException,
            TooManyRequestsException, NotFoundException, InternalServiceException, AwsServiceException, SdkClientException,
            S3ControlException {
        return describeJob(DescribeJobRequest.builder().applyMutation(describeJobRequest).build());
    }

    /**
     * <p>
     * Returns configuration information about the specified access point.
     * </p>
     *
     * @param getAccessPointRequest
     * @return Result of the GetAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPoint" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAccessPointResponse getAccessPoint(GetAccessPointRequest getAccessPointRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns configuration information about the specified access point.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccessPointRequest.Builder} avoiding the need to
     * create one manually via {@link GetAccessPointRequest#builder()}
     * </p>
     *
     * @param getAccessPointRequest
     *        A {@link Consumer} that will call methods on {@link GetAccessPointRequest.Builder} to create a request.
     * @return Result of the GetAccessPoint operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPoint
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPoint" target="_top">AWS API
     *      Documentation</a>
     */
    default GetAccessPointResponse getAccessPoint(Consumer<GetAccessPointRequest.Builder> getAccessPointRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        return getAccessPoint(GetAccessPointRequest.builder().applyMutation(getAccessPointRequest).build());
    }

    /**
     * <p>
     * Returns the access point policy associated with the specified access point.
     * </p>
     *
     * @param getAccessPointPolicyRequest
     * @return Result of the GetAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccessPointPolicyResponse getAccessPointPolicy(GetAccessPointPolicyRequest getAccessPointPolicyRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the access point policy associated with the specified access point.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccessPointPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link GetAccessPointPolicyRequest#builder()}
     * </p>
     *
     * @param getAccessPointPolicyRequest
     *        A {@link Consumer} that will call methods on {@link GetAccessPointPolicyRequest.Builder} to create a
     *        request.
     * @return Result of the GetAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccessPointPolicyResponse getAccessPointPolicy(
            Consumer<GetAccessPointPolicyRequest.Builder> getAccessPointPolicyRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {
        return getAccessPointPolicy(GetAccessPointPolicyRequest.builder().applyMutation(getAccessPointPolicyRequest).build());
    }

    /**
     * <p>
     * Indicates whether the specified access point currently has a policy that allows public access. For more
     * information about public access through access points, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html">Managing Data Access with Amazon S3
     * Access Points</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     *
     * @param getAccessPointPolicyStatusRequest
     * @return Result of the GetAccessPointPolicyStatus operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPointPolicyStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicyStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default GetAccessPointPolicyStatusResponse getAccessPointPolicyStatus(
            GetAccessPointPolicyStatusRequest getAccessPointPolicyStatusRequest) throws AwsServiceException, SdkClientException,
            S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Indicates whether the specified access point currently has a policy that allows public access. For more
     * information about public access through access points, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html">Managing Data Access with Amazon S3
     * Access Points</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccessPointPolicyStatusRequest.Builder} avoiding
     * the need to create one manually via {@link GetAccessPointPolicyStatusRequest#builder()}
     * </p>
     *
     * @param getAccessPointPolicyStatusRequest
     *        A {@link Consumer} that will call methods on {@link GetAccessPointPolicyStatusRequest.Builder} to create a
     *        request.
     * @return Result of the GetAccessPointPolicyStatus operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetAccessPointPolicyStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicyStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default GetAccessPointPolicyStatusResponse getAccessPointPolicyStatus(
            Consumer<GetAccessPointPolicyStatusRequest.Builder> getAccessPointPolicyStatusRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {
        return getAccessPointPolicyStatus(GetAccessPointPolicyStatusRequest.builder()
                .applyMutation(getAccessPointPolicyStatusRequest).build());
    }

    /**
     * <p>
     * Retrieve the tags on a Amazon S3 batch operations job.
     * </p>
     *
     * @param getJobTaggingRequest
     * @return Result of the GetJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    default GetJobTaggingResponse getJobTagging(GetJobTaggingRequest getJobTaggingRequest) throws InternalServiceException,
            TooManyRequestsException, NotFoundException, AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieve the tags on a Amazon S3 batch operations job.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetJobTaggingRequest.Builder} avoiding the need to
     * create one manually via {@link GetJobTaggingRequest#builder()}
     * </p>
     *
     * @param getJobTaggingRequest
     *        A {@link Consumer} that will call methods on {@link GetJobTaggingRequest.Builder} to create a request.
     * @return Result of the GetJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    default GetJobTaggingResponse getJobTagging(Consumer<GetJobTaggingRequest.Builder> getJobTaggingRequest)
            throws InternalServiceException, TooManyRequestsException, NotFoundException, AwsServiceException,
            SdkClientException, S3ControlException {
        return getJobTagging(GetJobTaggingRequest.builder().applyMutation(getJobTaggingRequest).build());
    }

    /**
     * <p>
     * Retrieves the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param getPublicAccessBlockRequest
     * @return Result of the GetPublicAccessBlock operation returned by the service.
     * @throws NoSuchPublicAccessBlockConfigurationException
     *         Amazon S3 throws this exception if you make a <code>GetPublicAccessBlock</code> request against an
     *         account that doesn't have a <code>PublicAccessBlockConfiguration</code> set.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    default GetPublicAccessBlockResponse getPublicAccessBlock(GetPublicAccessBlockRequest getPublicAccessBlockRequest)
            throws NoSuchPublicAccessBlockConfigurationException, AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPublicAccessBlockRequest.Builder} avoiding the
     * need to create one manually via {@link GetPublicAccessBlockRequest#builder()}
     * </p>
     *
     * @param getPublicAccessBlockRequest
     *        A {@link Consumer} that will call methods on {@link GetPublicAccessBlockRequest.Builder} to create a
     *        request.
     * @return Result of the GetPublicAccessBlock operation returned by the service.
     * @throws NoSuchPublicAccessBlockConfigurationException
     *         Amazon S3 throws this exception if you make a <code>GetPublicAccessBlock</code> request against an
     *         account that doesn't have a <code>PublicAccessBlockConfiguration</code> set.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.GetPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    default GetPublicAccessBlockResponse getPublicAccessBlock(
            Consumer<GetPublicAccessBlockRequest.Builder> getPublicAccessBlockRequest)
            throws NoSuchPublicAccessBlockConfigurationException, AwsServiceException, SdkClientException, S3ControlException {
        return getPublicAccessBlock(GetPublicAccessBlockRequest.builder().applyMutation(getPublicAccessBlockRequest).build());
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), then the response will include a continuation token that you can use
     * to list the additional access points.
     * </p>
     *
     * @param listAccessPointsRequest
     * @return Result of the ListAccessPoints operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAccessPointsResponse listAccessPoints(ListAccessPointsRequest listAccessPointsRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), then the response will include a continuation token that you can use
     * to list the additional access points.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAccessPointsRequest.Builder} avoiding the need
     * to create one manually via {@link ListAccessPointsRequest#builder()}
     * </p>
     *
     * @param listAccessPointsRequest
     *        A {@link Consumer} that will call methods on {@link ListAccessPointsRequest.Builder} to create a request.
     * @return Result of the ListAccessPoints operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAccessPointsResponse listAccessPoints(Consumer<ListAccessPointsRequest.Builder> listAccessPointsRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        return listAccessPoints(ListAccessPointsRequest.builder().applyMutation(listAccessPointsRequest).build());
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), then the response will include a continuation token that you can use
     * to list the additional access points.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client.listAccessPointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client
     *             .listAccessPointsPaginator(request);
     *     for (software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client.listAccessPointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation.</b>
     * </p>
     *
     * @param listAccessPointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAccessPointsIterable listAccessPointsPaginator(ListAccessPointsRequest listAccessPointsRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), then the response will include a continuation token that you can use
     * to list the additional access points.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client.listAccessPointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client
     *             .listAccessPointsPaginator(request);
     *     for (software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsIterable responses = client.listAccessPointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListAccessPointsRequest.Builder} avoiding the need
     * to create one manually via {@link ListAccessPointsRequest#builder()}
     * </p>
     *
     * @param listAccessPointsRequest
     *        A {@link Consumer} that will call methods on {@link ListAccessPointsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListAccessPoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListAccessPointsIterable listAccessPointsPaginator(Consumer<ListAccessPointsRequest.Builder> listAccessPointsRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        return listAccessPointsPaginator(ListAccessPointsRequest.builder().applyMutation(listAccessPointsRequest).build());
    }

    /**
     * <p>
     * Lists current jobs and jobs that have ended within the last 30 days for the AWS account making the request.
     * </p>
     *
     * @param listJobsRequest
     * @return Result of the ListJobs operation returned by the service.
     * @throws InvalidRequestException
     * @throws InternalServiceException
     * @throws InvalidNextTokenException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsResponse listJobs(ListJobsRequest listJobsRequest) throws InvalidRequestException, InternalServiceException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists current jobs and jobs that have ended within the last 30 days for the AWS account making the request.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListJobsRequest#builder()}
     * </p>
     *
     * @param listJobsRequest
     *        A {@link Consumer} that will call methods on {@link ListJobsRequest.Builder} to create a request.
     * @return Result of the ListJobs operation returned by the service.
     * @throws InvalidRequestException
     * @throws InternalServiceException
     * @throws InvalidNextTokenException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsResponse listJobs(Consumer<ListJobsRequest.Builder> listJobsRequest) throws InvalidRequestException,
            InternalServiceException, InvalidNextTokenException, AwsServiceException, SdkClientException, S3ControlException {
        return listJobs(ListJobsRequest.builder().applyMutation(listJobsRequest).build());
    }

    /**
     * <p>
     * Lists current jobs and jobs that have ended within the last 30 days for the AWS account making the request.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.s3control.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     * @throws InternalServiceException
     * @throws InvalidNextTokenException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsIterable listJobsPaginator(ListJobsRequest listJobsRequest) throws InvalidRequestException,
            InternalServiceException, InvalidNextTokenException, AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists current jobs and jobs that have ended within the last 30 days for the AWS account making the request.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     *     for (software.amazon.awssdk.services.s3control.model.ListJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsIterable responses = client.listJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListJobsRequest.Builder} avoiding the need to
     * create one manually via {@link ListJobsRequest#builder()}
     * </p>
     *
     * @param listJobsRequest
     *        A {@link Consumer} that will call methods on {@link ListJobsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     * @throws InternalServiceException
     * @throws InvalidNextTokenException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.ListJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListJobsIterable listJobsPaginator(Consumer<ListJobsRequest.Builder> listJobsRequest) throws InvalidRequestException,
            InternalServiceException, InvalidNextTokenException, AwsServiceException, SdkClientException, S3ControlException {
        return listJobsPaginator(ListJobsRequest.builder().applyMutation(listJobsRequest).build());
    }

    /**
     * <p>
     * Associates an access policy with the specified access point. Each access point can have only one policy, so a
     * request made to this API replaces any existing policy associated with the specified access point.
     * </p>
     *
     * @param putAccessPointPolicyRequest
     * @return Result of the PutAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default PutAccessPointPolicyResponse putAccessPointPolicy(PutAccessPointPolicyRequest putAccessPointPolicyRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Associates an access policy with the specified access point. Each access point can have only one policy, so a
     * request made to this API replaces any existing policy associated with the specified access point.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutAccessPointPolicyRequest.Builder} avoiding the
     * need to create one manually via {@link PutAccessPointPolicyRequest#builder()}
     * </p>
     *
     * @param putAccessPointPolicyRequest
     *        A {@link Consumer} that will call methods on {@link PutAccessPointPolicyRequest.Builder} to create a
     *        request.
     * @return Result of the PutAccessPointPolicy operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutAccessPointPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutAccessPointPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    default PutAccessPointPolicyResponse putAccessPointPolicy(
            Consumer<PutAccessPointPolicyRequest.Builder> putAccessPointPolicyRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {
        return putAccessPointPolicy(PutAccessPointPolicyRequest.builder().applyMutation(putAccessPointPolicyRequest).build());
    }

    /**
     * <p>
     * Replace the set of tags on a Amazon S3 batch operations job.
     * </p>
     *
     * @param putJobTaggingRequest
     * @return Result of the PutJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws TooManyTagsException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    default PutJobTaggingResponse putJobTagging(PutJobTaggingRequest putJobTaggingRequest) throws InternalServiceException,
            TooManyRequestsException, NotFoundException, TooManyTagsException, AwsServiceException, SdkClientException,
            S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Replace the set of tags on a Amazon S3 batch operations job.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutJobTaggingRequest.Builder} avoiding the need to
     * create one manually via {@link PutJobTaggingRequest#builder()}
     * </p>
     *
     * @param putJobTaggingRequest
     *        A {@link Consumer} that will call methods on {@link PutJobTaggingRequest.Builder} to create a request.
     * @return Result of the PutJobTagging operation returned by the service.
     * @throws InternalServiceException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws TooManyTagsException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutJobTagging
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    default PutJobTaggingResponse putJobTagging(Consumer<PutJobTaggingRequest.Builder> putJobTaggingRequest)
            throws InternalServiceException, TooManyRequestsException, NotFoundException, TooManyTagsException,
            AwsServiceException, SdkClientException, S3ControlException {
        return putJobTagging(PutJobTaggingRequest.builder().applyMutation(putJobTaggingRequest).build());
    }

    /**
     * <p>
     * Creates or modifies the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     *
     * @param putPublicAccessBlockRequest
     * @return Result of the PutPublicAccessBlock operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    default PutPublicAccessBlockResponse putPublicAccessBlock(PutPublicAccessBlockRequest putPublicAccessBlockRequest)
            throws AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates or modifies the <code>PublicAccessBlock</code> configuration for an Amazon Web Services account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutPublicAccessBlockRequest.Builder} avoiding the
     * need to create one manually via {@link PutPublicAccessBlockRequest#builder()}
     * </p>
     *
     * @param putPublicAccessBlockRequest
     *        A {@link Consumer} that will call methods on {@link PutPublicAccessBlockRequest.Builder} to create a
     *        request.
     * @return Result of the PutPublicAccessBlock operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.PutPublicAccessBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutPublicAccessBlock" target="_top">AWS
     *      API Documentation</a>
     */
    default PutPublicAccessBlockResponse putPublicAccessBlock(
            Consumer<PutPublicAccessBlockRequest.Builder> putPublicAccessBlockRequest) throws AwsServiceException,
            SdkClientException, S3ControlException {
        return putPublicAccessBlock(PutPublicAccessBlockRequest.builder().applyMutation(putPublicAccessBlockRequest).build());
    }

    /**
     * <p>
     * Updates an existing job's priority.
     * </p>
     *
     * @param updateJobPriorityRequest
     * @return Result of the UpdateJobPriority operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.UpdateJobPriority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobPriority" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateJobPriorityResponse updateJobPriority(UpdateJobPriorityRequest updateJobPriorityRequest)
            throws BadRequestException, TooManyRequestsException, NotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an existing job's priority.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateJobPriorityRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateJobPriorityRequest#builder()}
     * </p>
     *
     * @param updateJobPriorityRequest
     *        A {@link Consumer} that will call methods on {@link UpdateJobPriorityRequest.Builder} to create a request.
     * @return Result of the UpdateJobPriority operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.UpdateJobPriority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobPriority" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateJobPriorityResponse updateJobPriority(Consumer<UpdateJobPriorityRequest.Builder> updateJobPriorityRequest)
            throws BadRequestException, TooManyRequestsException, NotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, S3ControlException {
        return updateJobPriority(UpdateJobPriorityRequest.builder().applyMutation(updateJobPriorityRequest).build());
    }

    /**
     * <p>
     * Updates the status for the specified job. Use this operation to confirm that you want to run a job or to cancel
     * an existing job.
     * </p>
     *
     * @param updateJobStatusRequest
     * @return Result of the UpdateJobStatus operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws JobStatusException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.UpdateJobStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobStatus" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateJobStatusResponse updateJobStatus(UpdateJobStatusRequest updateJobStatusRequest) throws BadRequestException,
            TooManyRequestsException, NotFoundException, JobStatusException, InternalServiceException, AwsServiceException,
            SdkClientException, S3ControlException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the status for the specified job. Use this operation to confirm that you want to run a job or to cancel
     * an existing job.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateJobStatusRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateJobStatusRequest#builder()}
     * </p>
     *
     * @param updateJobStatusRequest
     *        A {@link Consumer} that will call methods on {@link UpdateJobStatusRequest.Builder} to create a request.
     * @return Result of the UpdateJobStatus operation returned by the service.
     * @throws BadRequestException
     * @throws TooManyRequestsException
     * @throws NotFoundException
     * @throws JobStatusException
     * @throws InternalServiceException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3ControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3ControlClient.UpdateJobStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobStatus" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateJobStatusResponse updateJobStatus(Consumer<UpdateJobStatusRequest.Builder> updateJobStatusRequest)
            throws BadRequestException, TooManyRequestsException, NotFoundException, JobStatusException,
            InternalServiceException, AwsServiceException, SdkClientException, S3ControlException {
        return updateJobStatus(UpdateJobStatusRequest.builder().applyMutation(updateJobStatusRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("s3-control");
    }
}
