/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class PutJobTaggingRequest extends S3ControlRequest implements
        ToCopyableBuilder<PutJobTaggingRequest.Builder, PutJobTaggingRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(PutJobTaggingRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-account-id")
                    .unmarshallLocationName("x-amz-account-id").build()).build();

    private static final SdkField<String> JOB_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(PutJobTaggingRequest::jobId))
            .setter(setter(Builder::jobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("id").unmarshallLocationName("id")
                    .build()).build();

    private static final SdkField<List<S3Tag>> TAGS_FIELD = SdkField
            .<List<S3Tag>> builder(MarshallingType.LIST)
            .getter(getter(PutJobTaggingRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags")
                    .unmarshallLocationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            JOB_ID_FIELD, TAGS_FIELD));

    private final String accountId;

    private final String jobId;

    private final List<S3Tag> tags;

    private PutJobTaggingRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.jobId = builder.jobId;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The account ID for the Amazon Web Services account associated with the Amazon S3 batch operations job you want to
     * replace tags on.
     * </p>
     * 
     * @return The account ID for the Amazon Web Services account associated with the Amazon S3 batch operations job you
     *         want to replace tags on.
     */
    public String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The ID for the job whose tags you want to replace.
     * </p>
     * 
     * @return The ID for the job whose tags you want to replace.
     */
    public String jobId() {
        return jobId;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of tags to associate with the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The set of tags to associate with the job.
     */
    public List<S3Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(jobId());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PutJobTaggingRequest)) {
            return false;
        }
        PutJobTaggingRequest other = (PutJobTaggingRequest) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(jobId(), other.jobId())
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PutJobTaggingRequest").add("AccountId", accountId()).add("JobId", jobId()).add("Tags", tags())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "JobId":
            return Optional.ofNullable(clazz.cast(jobId()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PutJobTaggingRequest, T> g) {
        return obj -> g.apply((PutJobTaggingRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3ControlRequest.Builder, SdkPojo, CopyableBuilder<Builder, PutJobTaggingRequest> {
        /**
         * <p>
         * The account ID for the Amazon Web Services account associated with the Amazon S3 batch operations job you
         * want to replace tags on.
         * </p>
         * 
         * @param accountId
         *        The account ID for the Amazon Web Services account associated with the Amazon S3 batch operations job
         *        you want to replace tags on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The ID for the job whose tags you want to replace.
         * </p>
         * 
         * @param jobId
         *        The ID for the job whose tags you want to replace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobId(String jobId);

        /**
         * <p>
         * The set of tags to associate with the job.
         * </p>
         * 
         * @param tags
         *        The set of tags to associate with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<S3Tag> tags);

        /**
         * <p>
         * The set of tags to associate with the job.
         * </p>
         * 
         * @param tags
         *        The set of tags to associate with the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(S3Tag... tags);

        /**
         * <p>
         * The set of tags to associate with the job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<S3Tag>.Builder} avoiding the need to create
         * one manually via {@link List<S3Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<S3Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<S3Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<S3Tag>)
         */
        Builder tags(Consumer<S3Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3ControlRequest.BuilderImpl implements Builder {
        private String accountId;

        private String jobId;

        private List<S3Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PutJobTaggingRequest model) {
            super(model);
            accountId(model.accountId);
            jobId(model.jobId);
            tags(model.tags);
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getJobId() {
            return jobId;
        }

        @Override
        public final Builder jobId(String jobId) {
            this.jobId = jobId;
            return this;
        }

        public final void setJobId(String jobId) {
            this.jobId = jobId;
        }

        public final Collection<S3Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(S3Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<S3Tag> tags) {
            this.tags = S3TagSetCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(S3Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<S3Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> S3Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<S3Tag.BuilderImpl> tags) {
            this.tags = S3TagSetCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public PutJobTaggingRequest build() {
            return new PutJobTaggingRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
