/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The container for the Outposts bucket lifecycle rule and operator.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LifecycleRuleAndOperator implements SdkPojo, Serializable,
        ToCopyableBuilder<LifecycleRuleAndOperator.Builder, LifecycleRuleAndOperator> {
    private static final SdkField<String> PREFIX_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Prefix")
            .getter(getter(LifecycleRuleAndOperator::prefix))
            .setter(setter(Builder::prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Prefix")
                    .unmarshallLocationName("Prefix").build()).build();

    private static final SdkField<List<S3Tag>> TAGS_FIELD = SdkField
            .<List<S3Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(LifecycleRuleAndOperator::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags")
                    .unmarshallLocationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PREFIX_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String prefix;

    private final List<S3Tag> tags;

    private LifecycleRuleAndOperator(BuilderImpl builder) {
        this.prefix = builder.prefix;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Prefix identifying one or more objects to which the rule applies.
     * </p>
     * 
     * @return Prefix identifying one or more objects to which the rule applies.
     */
    public String prefix() {
        return prefix;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * All of these tags must exist in the object's tag set in order for the rule to apply.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return All of these tags must exist in the object's tag set in order for the rule to apply.
     */
    public List<S3Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(prefix());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LifecycleRuleAndOperator)) {
            return false;
        }
        LifecycleRuleAndOperator other = (LifecycleRuleAndOperator) obj;
        return Objects.equals(prefix(), other.prefix()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LifecycleRuleAndOperator").add("Prefix", prefix()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Prefix":
            return Optional.ofNullable(clazz.cast(prefix()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LifecycleRuleAndOperator, T> g) {
        return obj -> g.apply((LifecycleRuleAndOperator) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LifecycleRuleAndOperator> {
        /**
         * <p>
         * Prefix identifying one or more objects to which the rule applies.
         * </p>
         * 
         * @param prefix
         *        Prefix identifying one or more objects to which the rule applies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefix(String prefix);

        /**
         * <p>
         * All of these tags must exist in the object's tag set in order for the rule to apply.
         * </p>
         * 
         * @param tags
         *        All of these tags must exist in the object's tag set in order for the rule to apply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<S3Tag> tags);

        /**
         * <p>
         * All of these tags must exist in the object's tag set in order for the rule to apply.
         * </p>
         * 
         * @param tags
         *        All of these tags must exist in the object's tag set in order for the rule to apply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(S3Tag... tags);

        /**
         * <p>
         * All of these tags must exist in the object's tag set in order for the rule to apply.
         * </p>
         * This is a convenience that creates an instance of the {@link List<S3Tag>.Builder} avoiding the need to create
         * one manually via {@link List<S3Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<S3Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<S3Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<S3Tag>)
         */
        Builder tags(Consumer<S3Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String prefix;

        private List<S3Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LifecycleRuleAndOperator model) {
            prefix(model.prefix);
            tags(model.tags);
        }

        public final String getPrefix() {
            return prefix;
        }

        @Override
        public final Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public final void setPrefix(String prefix) {
            this.prefix = prefix;
        }

        public final Collection<S3Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(S3Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<S3Tag> tags) {
            this.tags = S3TagSetCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(S3Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<S3Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> S3Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<S3Tag.BuilderImpl> tags) {
            this.tags = S3TagSetCopier.copyFromBuilder(tags);
        }

        @Override
        public LifecycleRuleAndOperator build() {
            return new LifecycleRuleAndOperator(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
