/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.arns.Arn;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.Response;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.interceptor.trait.HttpChecksumRequired;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.xml.AwsXmlProtocolFactory;
import software.amazon.awssdk.protocols.xml.XmlOperationMetadata;
import software.amazon.awssdk.services.s3.internal.resource.S3AccessPointResource;
import software.amazon.awssdk.services.s3.internal.resource.S3Resource;
import software.amazon.awssdk.services.s3control.internal.S3ArnableField;
import software.amazon.awssdk.services.s3control.internal.S3ControlArnConverter;
import software.amazon.awssdk.services.s3control.internal.S3ControlInternalExecutionAttribute;
import software.amazon.awssdk.services.s3control.model.BadRequestException;
import software.amazon.awssdk.services.s3control.model.BucketAlreadyExistsException;
import software.amazon.awssdk.services.s3control.model.BucketAlreadyOwnedByYouException;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.CreateBucketRequest;
import software.amazon.awssdk.services.s3control.model.CreateBucketResponse;
import software.amazon.awssdk.services.s3control.model.CreateJobRequest;
import software.amazon.awssdk.services.s3control.model.CreateJobResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.DeleteBucketLifecycleConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketLifecycleConfigurationResponse;
import software.amazon.awssdk.services.s3control.model.DeleteBucketPolicyRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketPolicyResponse;
import software.amazon.awssdk.services.s3control.model.DeleteBucketRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketResponse;
import software.amazon.awssdk.services.s3control.model.DeleteBucketTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketTaggingResponse;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.DeleteStorageLensConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.DeleteStorageLensConfigurationResponse;
import software.amazon.awssdk.services.s3control.model.DeleteStorageLensConfigurationTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteStorageLensConfigurationTaggingResponse;
import software.amazon.awssdk.services.s3control.model.DescribeJobRequest;
import software.amazon.awssdk.services.s3control.model.DescribeJobResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointConfigurationForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointConfigurationForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusResponse;
import software.amazon.awssdk.services.s3control.model.GetAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointResponse;
import software.amazon.awssdk.services.s3control.model.GetBucketLifecycleConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketLifecycleConfigurationResponse;
import software.amazon.awssdk.services.s3control.model.GetBucketPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketPolicyResponse;
import software.amazon.awssdk.services.s3control.model.GetBucketRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketResponse;
import software.amazon.awssdk.services.s3control.model.GetBucketTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketTaggingResponse;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.GetStorageLensConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.GetStorageLensConfigurationResponse;
import software.amazon.awssdk.services.s3control.model.GetStorageLensConfigurationTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetStorageLensConfigurationTaggingResponse;
import software.amazon.awssdk.services.s3control.model.IdempotencyException;
import software.amazon.awssdk.services.s3control.model.InternalServiceException;
import software.amazon.awssdk.services.s3control.model.InvalidNextTokenException;
import software.amazon.awssdk.services.s3control.model.InvalidRequestException;
import software.amazon.awssdk.services.s3control.model.JobStatusException;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse;
import software.amazon.awssdk.services.s3control.model.ListJobsRequest;
import software.amazon.awssdk.services.s3control.model.ListJobsResponse;
import software.amazon.awssdk.services.s3control.model.ListRegionalBucketsRequest;
import software.amazon.awssdk.services.s3control.model.ListRegionalBucketsResponse;
import software.amazon.awssdk.services.s3control.model.ListStorageLensConfigurationsRequest;
import software.amazon.awssdk.services.s3control.model.ListStorageLensConfigurationsResponse;
import software.amazon.awssdk.services.s3control.model.NoSuchPublicAccessBlockConfigurationException;
import software.amazon.awssdk.services.s3control.model.NotFoundException;
import software.amazon.awssdk.services.s3control.model.PutAccessPointConfigurationForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointConfigurationForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyForObjectLambdaResponse;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyResponse;
import software.amazon.awssdk.services.s3control.model.PutBucketLifecycleConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketLifecycleConfigurationResponse;
import software.amazon.awssdk.services.s3control.model.PutBucketPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketPolicyResponse;
import software.amazon.awssdk.services.s3control.model.PutBucketTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketTaggingResponse;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingResponse;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockResponse;
import software.amazon.awssdk.services.s3control.model.PutStorageLensConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.PutStorageLensConfigurationResponse;
import software.amazon.awssdk.services.s3control.model.PutStorageLensConfigurationTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutStorageLensConfigurationTaggingResponse;
import software.amazon.awssdk.services.s3control.model.S3ControlException;
import software.amazon.awssdk.services.s3control.model.S3ControlRequest;
import software.amazon.awssdk.services.s3control.model.TooManyRequestsException;
import software.amazon.awssdk.services.s3control.model.TooManyTagsException;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityResponse;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusResponse;
import software.amazon.awssdk.services.s3control.paginators.ListAccessPointsForObjectLambdaPublisher;
import software.amazon.awssdk.services.s3control.paginators.ListAccessPointsPublisher;
import software.amazon.awssdk.services.s3control.paginators.ListJobsPublisher;
import software.amazon.awssdk.services.s3control.paginators.ListRegionalBucketsPublisher;
import software.amazon.awssdk.services.s3control.paginators.ListStorageLensConfigurationsPublisher;
import software.amazon.awssdk.services.s3control.transform.CreateAccessPointForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.CreateAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.CreateBucketRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointPolicyForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteBucketLifecycleConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteBucketRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteBucketTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeletePublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteStorageLensConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DeleteStorageLensConfigurationTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.DescribeJobRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointConfigurationForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyStatusForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointPolicyStatusRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetAccessPointRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetBucketLifecycleConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetBucketRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetBucketTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetPublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetStorageLensConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.GetStorageLensConfigurationTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListAccessPointsForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListAccessPointsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListRegionalBucketsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.ListStorageLensConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutAccessPointConfigurationForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutAccessPointPolicyForObjectLambdaRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutAccessPointPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutBucketLifecycleConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutBucketTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutJobTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutPublicAccessBlockRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutStorageLensConfigurationRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.PutStorageLensConfigurationTaggingRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.UpdateJobPriorityRequestMarshaller;
import software.amazon.awssdk.services.s3control.transform.UpdateJobStatusRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.HostnameValidator;

/**
 * Internal implementation of {@link S3ControlAsyncClient}.
 *
 * @see S3ControlAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultS3ControlAsyncClient implements S3ControlAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultS3ControlAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsXmlProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultS3ControlAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an access point and associates it with the specified bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points.html">Managing Data Access with Amazon
     * S3 Access Points</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p/>
     * <note>
     * <p>
     * S3 on Outposts only supports VPC-style Access Points.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">
     * Accessing Amazon S3 on Outposts using virtual private cloud (VPC) only Access Points</a> in the <i>Amazon Simple
     * Storage Service User Guide</i>.
     * </p>
     * </note>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html#API_control_CreateAccessPoint_Examples"
     * >Examples</a> section.
     * </p>
     * <p/>
     * <p>
     * The following actions are related to <code>CreateAccessPoint</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html">GetAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html">DeleteAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html">ListAccessPoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createAccessPointRequest
     * @return A Java Future containing the result of the CreateAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.CreateAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessPointResponse> createAccessPoint(CreateAccessPointRequest createAccessPointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessPoint");

            HttpResponseHandler<Response<CreateAccessPointResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(CreateAccessPointResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = createAccessPointRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = createAccessPointRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                createAccessPointRequest = createAccessPointRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(createAccessPointRequest.accountId(), "AccountId",
                    "createAccessPointRequest");
            String resolvedHostExpression = String.format("%s.", createAccessPointRequest.accountId());

            CompletableFuture<CreateAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessPointRequest, CreateAccessPointResponse>()
                            .withOperationName("CreateAccessPoint")
                            .withMarshaller(new CreateAccessPointRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(createAccessPointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAccessPointRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateAccessPointResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Object Lambda Access Point. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/transforming-objects.html">Transforming objects with
     * Object Lambda Access Points</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * The following actions are related to <code>CreateAccessPointForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointForObjectLambda.html">
     * DeleteAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointForObjectLambda.html">
     * GetAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForObjectLambda.html">
     * ListAccessPointsForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createAccessPointForObjectLambdaRequest
     * @return A Java Future containing the result of the CreateAccessPointForObjectLambda operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.CreateAccessPointForObjectLambda
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateAccessPointForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAccessPointForObjectLambdaResponse> createAccessPointForObjectLambda(
            CreateAccessPointForObjectLambdaRequest createAccessPointForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createAccessPointForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccessPointForObjectLambda");

            HttpResponseHandler<Response<CreateAccessPointForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(CreateAccessPointForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(createAccessPointForObjectLambdaRequest.accountId(), "AccountId",
                    "createAccessPointForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", createAccessPointForObjectLambdaRequest.accountId());

            CompletableFuture<CreateAccessPointForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAccessPointForObjectLambdaRequest, CreateAccessPointForObjectLambdaResponse>()
                            .withOperationName("CreateAccessPointForObjectLambda")
                            .withMarshaller(new CreateAccessPointForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAccessPointForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createAccessPointForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<CreateAccessPointForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action creates an Amazon S3 on Outposts bucket. To create an S3 bucket, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html">Create Bucket</a> in the <i>Amazon
     * Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Creates a new Outposts bucket. By creating the bucket, you become the bucket owner. To create an Outposts bucket,
     * you must have S3 on Outposts. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * Not every string is an acceptable bucket name. For information on bucket naming restrictions, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/BucketRestrictions.html#bucketnamingrules">Working
     * with Amazon S3 Buckets</a>.
     * </p>
     * <p>
     * S3 on Outposts buckets support:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * <li>
     * <p>
     * LifecycleConfigurations for deleting expired objects
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a complete list of restrictions and Amazon S3 feature limitations on S3 on Outposts, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3OnOutpostsRestrictionsLimitations.html"> Amazon S3
     * on Outposts Restrictions and Limitations</a>.
     * </p>
     * <p>
     * For an example of the request syntax for Amazon S3 on Outposts that uses the S3 on Outposts endpoint hostname
     * prefix and <code>x-amz-outpost-id</code> in your API request, see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateBucket.html#API_control_CreateBucket_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>CreateBucket</code> for Amazon S3 on Outposts:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html">PutObject</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucket.html">GetBucket</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucket.html">DeleteBucket</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html">CreateAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html">
     * PutAccessPointPolicy</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createBucketRequest
     * @return A Java Future containing the result of the CreateBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BucketAlreadyExistsException The requested Outposts bucket name is not available. The bucket
     *         namespace is shared by all users of the AWS Outposts in this Region. Select a different name and try
     *         again.</li>
     *         <li>BucketAlreadyOwnedByYouException The Outposts bucket you tried to create already exists, and you own
     *         it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.CreateBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateBucket" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBucketResponse> createBucket(CreateBucketRequest createBucketRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBucket");

            HttpResponseHandler<Response<CreateBucketResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    CreateBucketResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));

            CompletableFuture<CreateBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBucketRequest, CreateBucketResponse>()
                            .withOperationName("CreateBucket")
                            .withMarshaller(new CreateBucketRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.HTTP_CHECKSUM_REQUIRED,
                                    HttpChecksumRequired.create()).withInput(createBucketRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createBucketRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateBucketResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * You can use S3 Batch Operations to perform large-scale batch actions on Amazon S3 objects. Batch Operations can
     * run a single action on lists of Amazon S3 objects that you specify. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">S3 Batch Operations</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * This action creates a S3 Batch Operations job.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html">DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html">ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html">UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html">UpdateJobStatus</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_JobOperation.html">JobOperation</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createJobRequest
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TooManyRequestsException</li>
     *         <li>BadRequestException</li>
     *         <li>IdempotencyException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.CreateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");

            HttpResponseHandler<Response<CreateJobResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    CreateJobResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(createJobRequest.accountId(), "AccountId", "createJobRequest");
            String resolvedHostExpression = String.format("%s.", createJobRequest.accountId());

            CompletableFuture<CreateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>().withOperationName("CreateJob")
                            .withMarshaller(new CreateJobRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(createJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createJobRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateJobResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified access point.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html#API_control_DeleteAccessPoint_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>DeleteAccessPoint</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html">CreateAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html">GetAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html">ListAccessPoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAccessPointRequest
     * @return A Java Future containing the result of the DeleteAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessPointResponse> deleteAccessPoint(DeleteAccessPointRequest deleteAccessPointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessPoint");

            HttpResponseHandler<Response<DeleteAccessPointResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteAccessPointResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String name = deleteAccessPointRequest.name();
            Arn arn = null;
            if (name != null && name.startsWith("arn:")) {
                arn = Arn.fromString(name);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3AccessPointResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3AccessPointResource resource = (S3AccessPointResource) s3Resource;
                String accountId = deleteAccessPointRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                deleteAccessPointRequest = deleteAccessPointRequest.toBuilder().name(resource.accessPointName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteAccessPointRequest.accountId(), "AccountId",
                    "deleteAccessPointRequest");
            String resolvedHostExpression = String.format("%s.", deleteAccessPointRequest.accountId());

            CompletableFuture<DeleteAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessPointRequest, DeleteAccessPointResponse>()
                            .withOperationName("DeleteAccessPoint")
                            .withMarshaller(new DeleteAccessPointRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(deleteAccessPointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAccessPointRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAccessPointResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified Object Lambda Access Point.
     * </p>
     * <p>
     * The following actions are related to <code>DeleteAccessPointForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPointForObjectLambda.html">
     * CreateAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointForObjectLambda.html">
     * GetAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForObjectLambda.html">
     * ListAccessPointsForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAccessPointForObjectLambdaRequest
     * @return A Java Future containing the result of the DeleteAccessPointForObjectLambda operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteAccessPointForObjectLambda
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPointForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessPointForObjectLambdaResponse> deleteAccessPointForObjectLambda(
            DeleteAccessPointForObjectLambdaRequest deleteAccessPointForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteAccessPointForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessPointForObjectLambda");

            HttpResponseHandler<Response<DeleteAccessPointForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteAccessPointForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteAccessPointForObjectLambdaRequest.accountId(), "AccountId",
                    "deleteAccessPointForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", deleteAccessPointForObjectLambdaRequest.accountId());

            CompletableFuture<DeleteAccessPointForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessPointForObjectLambdaRequest, DeleteAccessPointForObjectLambdaResponse>()
                            .withOperationName("DeleteAccessPointForObjectLambda")
                            .withMarshaller(new DeleteAccessPointForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAccessPointForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAccessPointForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAccessPointForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the access point policy for the specified access point.
     * </p>
     * <p/>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html#API_control_DeleteAccessPointPolicy_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>DeleteAccessPointPolicy</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html">
     * PutAccessPointPolicy</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html">
     * GetAccessPointPolicy</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAccessPointPolicyRequest
     * @return A Java Future containing the result of the DeleteAccessPointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteAccessPointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessPointPolicyResponse> deleteAccessPointPolicy(
            DeleteAccessPointPolicyRequest deleteAccessPointPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccessPointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessPointPolicy");

            HttpResponseHandler<Response<DeleteAccessPointPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteAccessPointPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String name = deleteAccessPointPolicyRequest.name();
            Arn arn = null;
            if (name != null && name.startsWith("arn:")) {
                arn = Arn.fromString(name);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3AccessPointResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3AccessPointResource resource = (S3AccessPointResource) s3Resource;
                String accountId = deleteAccessPointPolicyRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                deleteAccessPointPolicyRequest = deleteAccessPointPolicyRequest.toBuilder().name(resource.accessPointName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteAccessPointPolicyRequest.accountId(), "AccountId",
                    "deleteAccessPointPolicyRequest");
            String resolvedHostExpression = String.format("%s.", deleteAccessPointPolicyRequest.accountId());

            CompletableFuture<DeleteAccessPointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessPointPolicyRequest, DeleteAccessPointPolicyResponse>()
                            .withOperationName("DeleteAccessPointPolicy")
                            .withMarshaller(new DeleteAccessPointPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(deleteAccessPointPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAccessPointPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteAccessPointPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the resource policy for an Object Lambda Access Point.
     * </p>
     * <p>
     * The following actions are related to <code>DeleteAccessPointPolicyForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyForObjectLambda.html">
     * GetAccessPointPolicyForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicyForObjectLambda.html">
     * PutAccessPointPolicyForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteAccessPointPolicyForObjectLambdaRequest
     * @return A Java Future containing the result of the DeleteAccessPointPolicyForObjectLambda operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteAccessPointPolicyForObjectLambda
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteAccessPointPolicyForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAccessPointPolicyForObjectLambdaResponse> deleteAccessPointPolicyForObjectLambda(
            DeleteAccessPointPolicyForObjectLambdaRequest deleteAccessPointPolicyForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteAccessPointPolicyForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccessPointPolicyForObjectLambda");

            HttpResponseHandler<Response<DeleteAccessPointPolicyForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteAccessPointPolicyForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteAccessPointPolicyForObjectLambdaRequest.accountId(), "AccountId",
                    "deleteAccessPointPolicyForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", deleteAccessPointPolicyForObjectLambdaRequest.accountId());

            CompletableFuture<DeleteAccessPointPolicyForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAccessPointPolicyForObjectLambdaRequest, DeleteAccessPointPolicyForObjectLambdaResponse>()
                            .withOperationName("DeleteAccessPointPolicyForObjectLambda")
                            .withMarshaller(new DeleteAccessPointPolicyForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAccessPointPolicyForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteAccessPointPolicyForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteAccessPointPolicyForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action deletes an Amazon S3 on Outposts bucket. To delete an S3 bucket, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html">DeleteBucket</a> in the <i>Amazon
     * Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Deletes the Amazon S3 on Outposts bucket. All objects (including all object versions and delete markers) in the
     * bucket must be deleted before the bucket itself can be deleted. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucket.html#API_control_DeleteBucket_Examples"
     * >Examples</a> section.
     * </p>
     * <p class="title">
     * <b>Related Resources</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateBucket.html">CreateBucket</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucket.html">GetBucket</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html">DeleteObject</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteBucketRequest
     * @return A Java Future containing the result of the DeleteBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteBucket" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBucketResponse> deleteBucket(DeleteBucketRequest deleteBucketRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBucket");

            HttpResponseHandler<Response<DeleteBucketResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    DeleteBucketResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = deleteBucketRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = deleteBucketRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                deleteBucketRequest = deleteBucketRequest.toBuilder().bucket(resource.bucketName()).accountId(accountIdInArn)
                        .build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteBucketRequest.accountId(), "AccountId", "deleteBucketRequest");
            String resolvedHostExpression = String.format("%s.", deleteBucketRequest.accountId());

            CompletableFuture<DeleteBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBucketRequest, DeleteBucketResponse>()
                            .withOperationName("DeleteBucket")
                            .withMarshaller(new DeleteBucketRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(deleteBucketRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBucketRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteBucketResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action deletes an Amazon S3 on Outposts bucket's lifecycle configuration. To delete an S3 bucket's lifecycle
     * configuration, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketLifecycle.html">DeleteBucketLifecycle</a>
     * in the <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Deletes the lifecycle configuration from the specified Outposts bucket. Amazon S3 on Outposts removes all the
     * lifecycle configuration rules in the lifecycle subresource associated with the bucket. Your objects never expire,
     * and Amazon S3 on Outposts no longer automatically deletes any objects on the basis of rules contained in the
     * deleted lifecycle configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * To use this action, you must have permission to perform the <code>s3-outposts:DeleteLifecycleConfiguration</code>
     * action. By default, the bucket owner has this permission and the Outposts bucket owner can grant this permission
     * to others.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketLifecycleConfiguration.html#API_control_DeleteBucketLifecycleConfiguration_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * For more information about object expiration, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/intro-lifecycle-rules.html#intro-lifecycle-rules-actions"
     * >Elements to Describe Lifecycle Actions</a>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketLifecycleConfiguration.html">
     * PutBucketLifecycleConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketLifecycleConfiguration.html">
     * GetBucketLifecycleConfiguration</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteBucketLifecycleConfigurationRequest
     * @return A Java Future containing the result of the DeleteBucketLifecycleConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteBucketLifecycleConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteBucketLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBucketLifecycleConfigurationResponse> deleteBucketLifecycleConfiguration(
            DeleteBucketLifecycleConfigurationRequest deleteBucketLifecycleConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteBucketLifecycleConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBucketLifecycleConfiguration");

            HttpResponseHandler<Response<DeleteBucketLifecycleConfigurationResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteBucketLifecycleConfigurationResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = deleteBucketLifecycleConfigurationRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = deleteBucketLifecycleConfigurationRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                deleteBucketLifecycleConfigurationRequest = deleteBucketLifecycleConfigurationRequest.toBuilder()
                        .bucket(resource.bucketName()).accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteBucketLifecycleConfigurationRequest.accountId(), "AccountId",
                    "deleteBucketLifecycleConfigurationRequest");
            String resolvedHostExpression = String.format("%s.", deleteBucketLifecycleConfigurationRequest.accountId());

            CompletableFuture<DeleteBucketLifecycleConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBucketLifecycleConfigurationRequest, DeleteBucketLifecycleConfigurationResponse>()
                            .withOperationName("DeleteBucketLifecycleConfiguration")
                            .withMarshaller(new DeleteBucketLifecycleConfigurationRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build())
                            .withInput(deleteBucketLifecycleConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBucketLifecycleConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteBucketLifecycleConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action deletes an Amazon S3 on Outposts bucket policy. To delete an S3 bucket policy, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html">DeleteBucketPolicy</a> in the
     * <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * This implementation of the DELETE action uses the policy subresource to delete the policy of a specified Amazon
     * S3 on Outposts bucket. If you are using an identity other than the root user of the AWS account that owns the
     * bucket, the calling identity must have the <code>s3-outposts:DeleteBucketPolicy</code> permissions on the
     * specified Outposts bucket and belong to the bucket owner's account to use this action. For more information, see
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a>
     * in <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * If you don't have <code>DeleteBucketPolicy</code> permissions, Amazon S3 returns a <code>403 Access Denied</code>
     * error. If you have the correct permissions, but you're not using an identity that belongs to the bucket owner's
     * account, Amazon S3 returns a <code>405 Method Not Allowed</code> error.
     * </p>
     * <important>
     * <p>
     * As a security precaution, the root user of the AWS account that owns a bucket can always use this action, even if
     * the policy explicitly denies the root user the ability to perform this action.
     * </p>
     * </important>
     * <p>
     * For more information about bucket policies, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/using-iam-policies.html">Using Bucket Policies and User
     * Policies</a>.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketPolicy.html#API_control_DeleteBucketPolicy_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>DeleteBucketPolicy</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketPolicy.html">GetBucketPolicy</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketPolicy.html">PutBucketPolicy</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteBucketPolicyRequest
     * @return A Java Future containing the result of the DeleteBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteBucketPolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBucketPolicyResponse> deleteBucketPolicy(DeleteBucketPolicyRequest deleteBucketPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBucketPolicy");

            HttpResponseHandler<Response<DeleteBucketPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteBucketPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = deleteBucketPolicyRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = deleteBucketPolicyRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                deleteBucketPolicyRequest = deleteBucketPolicyRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteBucketPolicyRequest.accountId(), "AccountId",
                    "deleteBucketPolicyRequest");
            String resolvedHostExpression = String.format("%s.", deleteBucketPolicyRequest.accountId());

            CompletableFuture<DeleteBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBucketPolicyRequest, DeleteBucketPolicyResponse>()
                            .withOperationName("DeleteBucketPolicy")
                            .withMarshaller(new DeleteBucketPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(deleteBucketPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBucketPolicyRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteBucketPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action deletes an Amazon S3 on Outposts bucket's tags. To delete an S3 bucket tags, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketTagging.html">DeleteBucketTagging</a> in
     * the <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Deletes the tags from the Outposts bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * To use this action, you must have permission to perform the <code>PutBucketTagging</code> action. By default, the
     * bucket owner has this permission and can grant this permission to others.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketTagging.html#API_control_DeleteBucketTagging_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>DeleteBucketTagging</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketTagging.html">GetBucketTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketTagging.html">PutBucketTagging</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteBucketTaggingRequest
     * @return A Java Future containing the result of the DeleteBucketTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteBucketTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteBucketTagging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBucketTaggingResponse> deleteBucketTagging(
            DeleteBucketTaggingRequest deleteBucketTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBucketTaggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBucketTagging");

            HttpResponseHandler<Response<DeleteBucketTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteBucketTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = deleteBucketTaggingRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = deleteBucketTaggingRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                deleteBucketTaggingRequest = deleteBucketTaggingRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteBucketTaggingRequest.accountId(), "AccountId",
                    "deleteBucketTaggingRequest");
            String resolvedHostExpression = String.format("%s.", deleteBucketTaggingRequest.accountId());

            CompletableFuture<DeleteBucketTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBucketTaggingRequest, DeleteBucketTaggingResponse>()
                            .withOperationName("DeleteBucketTagging")
                            .withMarshaller(new DeleteBucketTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(deleteBucketTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBucketTaggingRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteBucketTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the entire tag set from the specified S3 Batch Operations job. To use this operation, you must have
     * permission to perform the <code>s3:DeleteJobTagging</code> action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html#batch-ops-job-tags"
     * >Controlling access and labeling jobs using tags</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html">GetJobTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html">PutJobTagging</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteJobTaggingRequest
     * @return A Java Future containing the result of the DeleteJobTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteJobTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteJobTagging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobTaggingResponse> deleteJobTagging(DeleteJobTaggingRequest deleteJobTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobTaggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJobTagging");

            HttpResponseHandler<Response<DeleteJobTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteJobTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteJobTaggingRequest.accountId(), "AccountId",
                    "deleteJobTaggingRequest");
            String resolvedHostExpression = String.format("%s.", deleteJobTaggingRequest.accountId());

            CompletableFuture<DeleteJobTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobTaggingRequest, DeleteJobTaggingResponse>()
                            .withOperationName("DeleteJobTagging")
                            .withMarshaller(new DeleteJobTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteJobTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteJobTaggingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteJobTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the <code>PublicAccessBlock</code> configuration for an AWS account. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html"> Using Amazon S3
     * block public access</a>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html">
     * GetPublicAccessBlock</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html">
     * PutPublicAccessBlock</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deletePublicAccessBlockRequest
     * @return A Java Future containing the result of the DeletePublicAccessBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeletePublicAccessBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeletePublicAccessBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePublicAccessBlockResponse> deletePublicAccessBlock(
            DeletePublicAccessBlockRequest deletePublicAccessBlockRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePublicAccessBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePublicAccessBlock");

            HttpResponseHandler<Response<DeletePublicAccessBlockResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeletePublicAccessBlockResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deletePublicAccessBlockRequest.accountId(), "AccountId",
                    "deletePublicAccessBlockRequest");
            String resolvedHostExpression = String.format("%s.", deletePublicAccessBlockRequest.accountId());

            CompletableFuture<DeletePublicAccessBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePublicAccessBlockRequest, DeletePublicAccessBlockResponse>()
                            .withOperationName("DeletePublicAccessBlock")
                            .withMarshaller(new DeletePublicAccessBlockRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePublicAccessBlockRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePublicAccessBlockRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeletePublicAccessBlockResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Amazon S3 Storage Lens configuration. For more information about S3 Storage Lens, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Assessing your storage activity and
     * usage with Amazon S3 Storage Lens </a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:DeleteStorageLensConfiguration</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note>
     *
     * @param deleteStorageLensConfigurationRequest
     * @return A Java Future containing the result of the DeleteStorageLensConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteStorageLensConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteStorageLensConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStorageLensConfigurationResponse> deleteStorageLensConfiguration(
            DeleteStorageLensConfigurationRequest deleteStorageLensConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteStorageLensConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStorageLensConfiguration");

            HttpResponseHandler<Response<DeleteStorageLensConfigurationResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteStorageLensConfigurationResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteStorageLensConfigurationRequest.accountId(), "AccountId",
                    "deleteStorageLensConfigurationRequest");
            String resolvedHostExpression = String.format("%s.", deleteStorageLensConfigurationRequest.accountId());

            CompletableFuture<DeleteStorageLensConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStorageLensConfigurationRequest, DeleteStorageLensConfigurationResponse>()
                            .withOperationName("DeleteStorageLensConfiguration")
                            .withMarshaller(new DeleteStorageLensConfigurationRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteStorageLensConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteStorageLensConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteStorageLensConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Amazon S3 Storage Lens configuration tags. For more information about S3 Storage Lens, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Assessing your storage activity and
     * usage with Amazon S3 Storage Lens </a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:DeleteStorageLensConfigurationTagging</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note>
     *
     * @param deleteStorageLensConfigurationTaggingRequest
     * @return A Java Future containing the result of the DeleteStorageLensConfigurationTagging operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DeleteStorageLensConfigurationTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DeleteStorageLensConfigurationTagging"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStorageLensConfigurationTaggingResponse> deleteStorageLensConfigurationTagging(
            DeleteStorageLensConfigurationTaggingRequest deleteStorageLensConfigurationTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteStorageLensConfigurationTaggingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStorageLensConfigurationTagging");

            HttpResponseHandler<Response<DeleteStorageLensConfigurationTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(DeleteStorageLensConfigurationTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(deleteStorageLensConfigurationTaggingRequest.accountId(), "AccountId",
                    "deleteStorageLensConfigurationTaggingRequest");
            String resolvedHostExpression = String.format("%s.", deleteStorageLensConfigurationTaggingRequest.accountId());

            CompletableFuture<DeleteStorageLensConfigurationTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStorageLensConfigurationTaggingRequest, DeleteStorageLensConfigurationTaggingResponse>()
                            .withOperationName("DeleteStorageLensConfigurationTagging")
                            .withMarshaller(new DeleteStorageLensConfigurationTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteStorageLensConfigurationTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteStorageLensConfigurationTaggingRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DeleteStorageLensConfigurationTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the configuration parameters and status for a Batch Operations job. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">S3 Batch Operations</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html">ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html">UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html">UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeJobRequest
     * @return A Java Future containing the result of the DescribeJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.DescribeJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/DescribeJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobResponse> describeJob(DescribeJobRequest describeJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJob");

            HttpResponseHandler<Response<DescribeJobResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    DescribeJobResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(describeJobRequest.accountId(), "AccountId", "describeJobRequest");
            String resolvedHostExpression = String.format("%s.", describeJobRequest.accountId());

            CompletableFuture<DescribeJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobRequest, DescribeJobResponse>()
                            .withOperationName("DescribeJob").withMarshaller(new DescribeJobRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeJobRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeJobResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration information about the specified access point.
     * </p>
     * <p/>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html#API_control_GetAccessPoint_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>GetAccessPoint</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html">CreateAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html">DeleteAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html">ListAccessPoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAccessPointRequest
     * @return A Java Future containing the result of the GetAccessPoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointResponse> getAccessPoint(GetAccessPointRequest getAccessPointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPoint");

            HttpResponseHandler<Response<GetAccessPointResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String name = getAccessPointRequest.name();
            Arn arn = null;
            if (name != null && name.startsWith("arn:")) {
                arn = Arn.fromString(name);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3AccessPointResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3AccessPointResource resource = (S3AccessPointResource) s3Resource;
                String accountId = getAccessPointRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                getAccessPointRequest = getAccessPointRequest.toBuilder().name(resource.accessPointName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getAccessPointRequest.accountId(), "AccountId", "getAccessPointRequest");
            String resolvedHostExpression = String.format("%s.", getAccessPointRequest.accountId());

            CompletableFuture<GetAccessPointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointRequest, GetAccessPointResponse>()
                            .withOperationName("GetAccessPoint")
                            .withMarshaller(new GetAccessPointRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(getAccessPointRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPointRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetAccessPointResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration for an Object Lambda Access Point.
     * </p>
     * <p>
     * The following actions are related to <code>GetAccessPointConfigurationForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointConfigurationForObjectLambda.html"
     * >PutAccessPointConfigurationForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAccessPointConfigurationForObjectLambdaRequest
     * @return A Java Future containing the result of the GetAccessPointConfigurationForObjectLambda operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointConfigurationForObjectLambda
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointConfigurationForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointConfigurationForObjectLambdaResponse> getAccessPointConfigurationForObjectLambda(
            GetAccessPointConfigurationForObjectLambdaRequest getAccessPointConfigurationForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAccessPointConfigurationForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPointConfigurationForObjectLambda");

            HttpResponseHandler<Response<GetAccessPointConfigurationForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointConfigurationForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getAccessPointConfigurationForObjectLambdaRequest.accountId(),
                    "AccountId", "getAccessPointConfigurationForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", getAccessPointConfigurationForObjectLambdaRequest.accountId());

            CompletableFuture<GetAccessPointConfigurationForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointConfigurationForObjectLambdaRequest, GetAccessPointConfigurationForObjectLambdaResponse>()
                            .withOperationName("GetAccessPointConfigurationForObjectLambda")
                            .withMarshaller(new GetAccessPointConfigurationForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccessPointConfigurationForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPointConfigurationForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetAccessPointConfigurationForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns configuration information about the specified Object Lambda Access Point
     * </p>
     * <p>
     * The following actions are related to <code>GetAccessPointForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPointForObjectLambda.html">
     * CreateAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointForObjectLambda.html">
     * DeleteAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForObjectLambda.html">
     * ListAccessPointsForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAccessPointForObjectLambdaRequest
     * @return A Java Future containing the result of the GetAccessPointForObjectLambda operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointForObjectLambda
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointForObjectLambdaResponse> getAccessPointForObjectLambda(
            GetAccessPointForObjectLambdaRequest getAccessPointForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAccessPointForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPointForObjectLambda");

            HttpResponseHandler<Response<GetAccessPointForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getAccessPointForObjectLambdaRequest.accountId(), "AccountId",
                    "getAccessPointForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", getAccessPointForObjectLambdaRequest.accountId());

            CompletableFuture<GetAccessPointForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointForObjectLambdaRequest, GetAccessPointForObjectLambdaResponse>()
                            .withOperationName("GetAccessPointForObjectLambda")
                            .withMarshaller(new GetAccessPointForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccessPointForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPointForObjectLambdaRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetAccessPointForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the access point policy associated with the specified access point.
     * </p>
     * <p>
     * The following actions are related to <code>GetAccessPointPolicy</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html">
     * PutAccessPointPolicy</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html">
     * DeleteAccessPointPolicy</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAccessPointPolicyRequest
     * @return A Java Future containing the result of the GetAccessPointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointPolicyResponse> getAccessPointPolicy(
            GetAccessPointPolicyRequest getAccessPointPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessPointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPointPolicy");

            HttpResponseHandler<Response<GetAccessPointPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String name = getAccessPointPolicyRequest.name();
            Arn arn = null;
            if (name != null && name.startsWith("arn:")) {
                arn = Arn.fromString(name);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3AccessPointResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3AccessPointResource resource = (S3AccessPointResource) s3Resource;
                String accountId = getAccessPointPolicyRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                getAccessPointPolicyRequest = getAccessPointPolicyRequest.toBuilder().name(resource.accessPointName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getAccessPointPolicyRequest.accountId(), "AccountId",
                    "getAccessPointPolicyRequest");
            String resolvedHostExpression = String.format("%s.", getAccessPointPolicyRequest.accountId());

            CompletableFuture<GetAccessPointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointPolicyRequest, GetAccessPointPolicyResponse>()
                            .withOperationName("GetAccessPointPolicy")
                            .withMarshaller(new GetAccessPointPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(getAccessPointPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPointPolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetAccessPointPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the resource policy for an Object Lambda Access Point.
     * </p>
     * <p>
     * The following actions are related to <code>GetAccessPointPolicyForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicyForObjectLambda.html">
     * DeleteAccessPointPolicyForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicyForObjectLambda.html">
     * PutAccessPointPolicyForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getAccessPointPolicyForObjectLambdaRequest
     * @return A Java Future containing the result of the GetAccessPointPolicyForObjectLambda operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointPolicyForObjectLambda
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicyForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointPolicyForObjectLambdaResponse> getAccessPointPolicyForObjectLambda(
            GetAccessPointPolicyForObjectLambdaRequest getAccessPointPolicyForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAccessPointPolicyForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPointPolicyForObjectLambda");

            HttpResponseHandler<Response<GetAccessPointPolicyForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointPolicyForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getAccessPointPolicyForObjectLambdaRequest.accountId(), "AccountId",
                    "getAccessPointPolicyForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", getAccessPointPolicyForObjectLambdaRequest.accountId());

            CompletableFuture<GetAccessPointPolicyForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointPolicyForObjectLambdaRequest, GetAccessPointPolicyForObjectLambdaResponse>()
                            .withOperationName("GetAccessPointPolicyForObjectLambda")
                            .withMarshaller(new GetAccessPointPolicyForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccessPointPolicyForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPointPolicyForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetAccessPointPolicyForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Indicates whether the specified access point currently has a policy that allows public access. For more
     * information about public access through access points, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points.html">Managing Data Access with Amazon
     * S3 Access Points</a> in the <i>Amazon Simple Storage Service Developer Guide</i>.
     * </p>
     *
     * @param getAccessPointPolicyStatusRequest
     * @return A Java Future containing the result of the GetAccessPointPolicyStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointPolicyStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicyStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointPolicyStatusResponse> getAccessPointPolicyStatus(
            GetAccessPointPolicyStatusRequest getAccessPointPolicyStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessPointPolicyStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPointPolicyStatus");

            HttpResponseHandler<Response<GetAccessPointPolicyStatusResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointPolicyStatusResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getAccessPointPolicyStatusRequest.accountId(), "AccountId",
                    "getAccessPointPolicyStatusRequest");
            String resolvedHostExpression = String.format("%s.", getAccessPointPolicyStatusRequest.accountId());

            CompletableFuture<GetAccessPointPolicyStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointPolicyStatusRequest, GetAccessPointPolicyStatusResponse>()
                            .withOperationName("GetAccessPointPolicyStatus")
                            .withMarshaller(new GetAccessPointPolicyStatusRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAccessPointPolicyStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPointPolicyStatusRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetAccessPointPolicyStatusResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the status of the resource policy associated with an Object Lambda Access Point.
     * </p>
     *
     * @param getAccessPointPolicyStatusForObjectLambdaRequest
     * @return A Java Future containing the result of the GetAccessPointPolicyStatusForObjectLambda operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetAccessPointPolicyStatusForObjectLambda
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetAccessPointPolicyStatusForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessPointPolicyStatusForObjectLambdaResponse> getAccessPointPolicyStatusForObjectLambda(
            GetAccessPointPolicyStatusForObjectLambdaRequest getAccessPointPolicyStatusForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAccessPointPolicyStatusForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessPointPolicyStatusForObjectLambda");

            HttpResponseHandler<Response<GetAccessPointPolicyStatusForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetAccessPointPolicyStatusForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getAccessPointPolicyStatusForObjectLambdaRequest.accountId(),
                    "AccountId", "getAccessPointPolicyStatusForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", getAccessPointPolicyStatusForObjectLambdaRequest.accountId());

            CompletableFuture<GetAccessPointPolicyStatusForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessPointPolicyStatusForObjectLambdaRequest, GetAccessPointPolicyStatusForObjectLambdaResponse>()
                            .withOperationName("GetAccessPointPolicyStatusForObjectLambda")
                            .withMarshaller(new GetAccessPointPolicyStatusForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccessPointPolicyStatusForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAccessPointPolicyStatusForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetAccessPointPolicyStatusForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an Amazon S3 on Outposts bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html"> Using Amazon S3 on Outposts</a>
     * in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * If you are using an identity other than the root user of the AWS account that owns the Outposts bucket, the
     * calling identity must have the <code>s3-outposts:GetBucket</code> permissions on the specified Outposts bucket
     * and belong to the Outposts bucket owner's account in order to use this action. Only users from Outposts bucket
     * owner account with the right permissions can perform actions on an Outposts bucket.
     * </p>
     * <p>
     * If you don't have <code>s3-outposts:GetBucket</code> permissions or you're not using an identity that belongs to
     * the bucket owner's account, Amazon S3 returns a <code>403 Access Denied</code> error.
     * </p>
     * <p>
     * The following actions are related to <code>GetBucket</code> for Amazon S3 on Outposts:
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucket.html#API_control_GetBucket_Examples"
     * >Examples</a> section.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html">PutObject</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateBucket.html">CreateBucket</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucket.html">DeleteBucket</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getBucketRequest
     * @return A Java Future containing the result of the GetBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetBucket" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBucketResponse> getBucket(GetBucketRequest getBucketRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBucket");

            HttpResponseHandler<Response<GetBucketResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    GetBucketResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = getBucketRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = getBucketRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                getBucketRequest = getBucketRequest.toBuilder().bucket(resource.bucketName()).accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getBucketRequest.accountId(), "AccountId", "getBucketRequest");
            String resolvedHostExpression = String.format("%s.", getBucketRequest.accountId());

            CompletableFuture<GetBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBucketRequest, GetBucketResponse>()
                            .withOperationName("GetBucket")
                            .withMarshaller(new GetBucketRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(getBucketRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getBucketRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetBucketResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action gets an Amazon S3 on Outposts bucket's lifecycle configuration. To get an S3 bucket's lifecycle
     * configuration, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html"
     * >GetBucketLifecycleConfiguration</a> in the <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Returns the lifecycle configuration information set on the Outposts bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a>
     * and for information about lifecycle configuration, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lifecycle-mgmt.html"> Object Lifecycle
     * Management</a> in <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * To use this action, you must have permission to perform the <code>s3-outposts:GetLifecycleConfiguration</code>
     * action. The Outposts bucket owner has this permission, by default. The bucket owner can grant this permission to
     * others. For more information about permissions, see <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-with-s3-actions.html#using-with-s3-actions-related-to-bucket-subresources"
     * >Permissions Related to Bucket Subresource Operations</a> and <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-access-control.html">Managing Access Permissions
     * to Your Amazon S3 Resources</a>.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketLifecycleConfiguration.html#API_control_GetBucketLifecycleConfiguration_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * <code>GetBucketLifecycleConfiguration</code> has the following special error:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Error code: <code>NoSuchLifecycleConfiguration</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Description: The lifecycle configuration does not exist.
     * </p>
     * </li>
     * <li>
     * <p>
     * HTTP Status Code: 404 Not Found
     * </p>
     * </li>
     * <li>
     * <p>
     * SOAP Fault Code Prefix: Client
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * The following actions are related to <code>GetBucketLifecycleConfiguration</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketLifecycleConfiguration.html">
     * PutBucketLifecycleConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketLifecycleConfiguration.html">
     * DeleteBucketLifecycleConfiguration</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getBucketLifecycleConfigurationRequest
     * @return A Java Future containing the result of the GetBucketLifecycleConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetBucketLifecycleConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetBucketLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBucketLifecycleConfigurationResponse> getBucketLifecycleConfiguration(
            GetBucketLifecycleConfigurationRequest getBucketLifecycleConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getBucketLifecycleConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBucketLifecycleConfiguration");

            HttpResponseHandler<Response<GetBucketLifecycleConfigurationResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetBucketLifecycleConfigurationResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = getBucketLifecycleConfigurationRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = getBucketLifecycleConfigurationRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                getBucketLifecycleConfigurationRequest = getBucketLifecycleConfigurationRequest.toBuilder()
                        .bucket(resource.bucketName()).accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getBucketLifecycleConfigurationRequest.accountId(), "AccountId",
                    "getBucketLifecycleConfigurationRequest");
            String resolvedHostExpression = String.format("%s.", getBucketLifecycleConfigurationRequest.accountId());

            CompletableFuture<GetBucketLifecycleConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBucketLifecycleConfigurationRequest, GetBucketLifecycleConfigurationResponse>()
                            .withOperationName("GetBucketLifecycleConfiguration")
                            .withMarshaller(new GetBucketLifecycleConfigurationRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(getBucketLifecycleConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getBucketLifecycleConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetBucketLifecycleConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action gets a bucket policy for an Amazon S3 on Outposts bucket. To get a policy for an S3 bucket, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html">GetBucketPolicy</a> in the
     * <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Returns the policy of a specified Outposts bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * If you are using an identity other than the root user of the AWS account that owns the bucket, the calling
     * identity must have the <code>GetBucketPolicy</code> permissions on the specified bucket and belong to the bucket
     * owner's account in order to use this action.
     * </p>
     * <p>
     * Only users from Outposts bucket owner account with the right permissions can perform actions on an Outposts
     * bucket. If you don't have <code>s3-outposts:GetBucketPolicy</code> permissions or you're not using an identity
     * that belongs to the bucket owner's account, Amazon S3 returns a <code>403 Access Denied</code> error.
     * </p>
     * <important>
     * <p>
     * As a security precaution, the root user of the AWS account that owns a bucket can always use this action, even if
     * the policy explicitly denies the root user the ability to perform this action.
     * </p>
     * </important>
     * <p>
     * For more information about bucket policies, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/using-iam-policies.html">Using Bucket Policies and User
     * Policies</a>.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketPolicy.html#API_control_GetBucketPolicy_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>GetBucketPolicy</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html">GetObject</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketPolicy.html">PutBucketPolicy</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketPolicy.html">DeleteBucketPolicy</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getBucketPolicyRequest
     * @return A Java Future containing the result of the GetBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetBucketPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBucketPolicyResponse> getBucketPolicy(GetBucketPolicyRequest getBucketPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBucketPolicy");

            HttpResponseHandler<Response<GetBucketPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetBucketPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = getBucketPolicyRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = getBucketPolicyRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                getBucketPolicyRequest = getBucketPolicyRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator
                    .validateHostnameCompliant(getBucketPolicyRequest.accountId(), "AccountId", "getBucketPolicyRequest");
            String resolvedHostExpression = String.format("%s.", getBucketPolicyRequest.accountId());

            CompletableFuture<GetBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBucketPolicyRequest, GetBucketPolicyResponse>()
                            .withOperationName("GetBucketPolicy")
                            .withMarshaller(new GetBucketPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(getBucketPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getBucketPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetBucketPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action gets an Amazon S3 on Outposts bucket's tags. To get an S3 bucket tags, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html">GetBucketTagging</a> in the
     * <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Returns the tag set associated with the Outposts bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * To use this action, you must have permission to perform the <code>GetBucketTagging</code> action. By default, the
     * bucket owner has this permission and can grant this permission to others.
     * </p>
     * <p>
     * <code>GetBucketTagging</code> has the following special error:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Error code: <code>NoSuchTagSetError</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Description: There is no tag set associated with the bucket.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketTagging.html#API_control_GetBucketTagging_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>GetBucketTagging</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketTagging.html">PutBucketTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketTagging.html">DeleteBucketTagging
     * </a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getBucketTaggingRequest
     * @return A Java Future containing the result of the GetBucketTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetBucketTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetBucketTagging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBucketTaggingResponse> getBucketTagging(GetBucketTaggingRequest getBucketTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBucketTaggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBucketTagging");

            HttpResponseHandler<Response<GetBucketTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetBucketTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = getBucketTaggingRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = getBucketTaggingRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                getBucketTaggingRequest = getBucketTaggingRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getBucketTaggingRequest.accountId(), "AccountId",
                    "getBucketTaggingRequest");
            String resolvedHostExpression = String.format("%s.", getBucketTaggingRequest.accountId());

            CompletableFuture<GetBucketTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBucketTaggingRequest, GetBucketTaggingResponse>()
                            .withOperationName("GetBucketTagging")
                            .withMarshaller(new GetBucketTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(getBucketTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getBucketTaggingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetBucketTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the tags on an S3 Batch Operations job. To use this operation, you must have permission to perform the
     * <code>s3:GetJobTagging</code> action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html#batch-ops-job-tags"
     * >Controlling access and labeling jobs using tags</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html">PutJobTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html">DeleteJobTagging</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getJobTaggingRequest
     * @return A Java Future containing the result of the GetJobTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetJobTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobTaggingResponse> getJobTagging(GetJobTaggingRequest getJobTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobTaggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobTagging");

            HttpResponseHandler<Response<GetJobTaggingResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    GetJobTaggingResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getJobTaggingRequest.accountId(), "AccountId", "getJobTaggingRequest");
            String resolvedHostExpression = String.format("%s.", getJobTaggingRequest.accountId());

            CompletableFuture<GetJobTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobTaggingRequest, GetJobTaggingResponse>()
                            .withOperationName("GetJobTagging")
                            .withMarshaller(new GetJobTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(getJobTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getJobTaggingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetJobTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <code>PublicAccessBlock</code> configuration for an AWS account. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html"> Using Amazon S3
     * block public access</a>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeletePublicAccessBlock.html">
     * DeletePublicAccessBlock</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html">
     * PutPublicAccessBlock</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getPublicAccessBlockRequest
     * @return A Java Future containing the result of the GetPublicAccessBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NoSuchPublicAccessBlockConfigurationException Amazon S3 throws this exception if you make a
     *         <code>GetPublicAccessBlock</code> request against an account that doesn't have a
     *         <code>PublicAccessBlockConfiguration</code> set.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetPublicAccessBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetPublicAccessBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPublicAccessBlockResponse> getPublicAccessBlock(
            GetPublicAccessBlockRequest getPublicAccessBlockRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPublicAccessBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPublicAccessBlock");

            HttpResponseHandler<Response<GetPublicAccessBlockResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetPublicAccessBlockResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getPublicAccessBlockRequest.accountId(), "AccountId",
                    "getPublicAccessBlockRequest");
            String resolvedHostExpression = String.format("%s.", getPublicAccessBlockRequest.accountId());

            CompletableFuture<GetPublicAccessBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPublicAccessBlockRequest, GetPublicAccessBlockResponse>()
                            .withOperationName("GetPublicAccessBlock")
                            .withMarshaller(new GetPublicAccessBlockRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPublicAccessBlockRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getPublicAccessBlockRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetPublicAccessBlockResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the Amazon S3 Storage Lens configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Assessing your storage activity and
     * usage with Amazon S3 Storage Lens </a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:GetStorageLensConfiguration</code> action.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note>
     *
     * @param getStorageLensConfigurationRequest
     * @return A Java Future containing the result of the GetStorageLensConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetStorageLensConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetStorageLensConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStorageLensConfigurationResponse> getStorageLensConfiguration(
            GetStorageLensConfigurationRequest getStorageLensConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStorageLensConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStorageLensConfiguration");

            HttpResponseHandler<Response<GetStorageLensConfigurationResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetStorageLensConfigurationResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getStorageLensConfigurationRequest.accountId(), "AccountId",
                    "getStorageLensConfigurationRequest");
            String resolvedHostExpression = String.format("%s.", getStorageLensConfigurationRequest.accountId());

            CompletableFuture<GetStorageLensConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStorageLensConfigurationRequest, GetStorageLensConfigurationResponse>()
                            .withOperationName("GetStorageLensConfiguration")
                            .withMarshaller(new GetStorageLensConfigurationRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(getStorageLensConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getStorageLensConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetStorageLensConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the tags of Amazon S3 Storage Lens configuration. For more information about S3 Storage Lens, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Assessing your storage activity and
     * usage with Amazon S3 Storage Lens </a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:GetStorageLensConfigurationTagging</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note>
     *
     * @param getStorageLensConfigurationTaggingRequest
     * @return A Java Future containing the result of the GetStorageLensConfigurationTagging operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.GetStorageLensConfigurationTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/GetStorageLensConfigurationTagging"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStorageLensConfigurationTaggingResponse> getStorageLensConfigurationTagging(
            GetStorageLensConfigurationTaggingRequest getStorageLensConfigurationTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getStorageLensConfigurationTaggingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStorageLensConfigurationTagging");

            HttpResponseHandler<Response<GetStorageLensConfigurationTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(GetStorageLensConfigurationTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(getStorageLensConfigurationTaggingRequest.accountId(), "AccountId",
                    "getStorageLensConfigurationTaggingRequest");
            String resolvedHostExpression = String.format("%s.", getStorageLensConfigurationTaggingRequest.accountId());

            CompletableFuture<GetStorageLensConfigurationTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStorageLensConfigurationTaggingRequest, GetStorageLensConfigurationTaggingResponse>()
                            .withOperationName("GetStorageLensConfigurationTagging")
                            .withMarshaller(new GetStorageLensConfigurationTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(getStorageLensConfigurationTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getStorageLensConfigurationTaggingRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetStorageLensConfigurationTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1,000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), the response will include a continuation token that you can use to
     * list the additional access points.
     * </p>
     * <p/>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html#API_control_GetAccessPoint_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>ListAccessPoints</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html">CreateAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html">DeleteAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html">GetAccessPoint</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listAccessPointsRequest
     * @return A Java Future containing the result of the ListAccessPoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessPointsResponse> listAccessPoints(ListAccessPointsRequest listAccessPointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccessPointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPoints");

            HttpResponseHandler<Response<ListAccessPointsResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(ListAccessPointsResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = listAccessPointsRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = listAccessPointsRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                listAccessPointsRequest = listAccessPointsRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(listAccessPointsRequest.accountId(), "AccountId",
                    "listAccessPointsRequest");
            String resolvedHostExpression = String.format("%s.", listAccessPointsRequest.accountId());

            CompletableFuture<ListAccessPointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessPointsRequest, ListAccessPointsResponse>()
                            .withOperationName("ListAccessPoints")
                            .withMarshaller(new ListAccessPointsRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(listAccessPointsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAccessPointsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListAccessPointsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the access points associated with the Object Lambda Access Point. You can retrieve up to 1000
     * access points per call. If there are more than 1,000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), the response will include a continuation token that you can use to
     * list the additional access points.
     * </p>
     * <p>
     * The following actions are related to <code>ListAccessPointsForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPointForObjectLambda.html">
     * CreateAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointForObjectLambda.html">
     * DeleteAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointForObjectLambda.html">
     * GetAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listAccessPointsForObjectLambdaRequest
     * @return A Java Future containing the result of the ListAccessPointsForObjectLambda operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListAccessPointsForObjectLambda
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPointsForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccessPointsForObjectLambdaResponse> listAccessPointsForObjectLambda(
            ListAccessPointsForObjectLambdaRequest listAccessPointsForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAccessPointsForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccessPointsForObjectLambda");

            HttpResponseHandler<Response<ListAccessPointsForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(ListAccessPointsForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(listAccessPointsForObjectLambdaRequest.accountId(), "AccountId",
                    "listAccessPointsForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", listAccessPointsForObjectLambdaRequest.accountId());

            CompletableFuture<ListAccessPointsForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccessPointsForObjectLambdaRequest, ListAccessPointsForObjectLambdaResponse>()
                            .withOperationName("ListAccessPointsForObjectLambda")
                            .withMarshaller(new ListAccessPointsForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAccessPointsForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAccessPointsForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ListAccessPointsForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the access points associated with the Object Lambda Access Point. You can retrieve up to 1000
     * access points per call. If there are more than 1,000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), the response will include a continuation token that you can use to
     * list the additional access points.
     * </p>
     * <p>
     * The following actions are related to <code>ListAccessPointsForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPointForObjectLambda.html">
     * CreateAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointForObjectLambda.html">
     * DeleteAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointForObjectLambda.html">
     * GetAccessPointForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPointsForObjectLambda(software.amazon.awssdk.services.s3control.model.ListAccessPointsForObjectLambdaRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsForObjectLambdaPublisher publisher = client.listAccessPointsForObjectLambdaPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsForObjectLambdaPublisher publisher = client.listAccessPointsForObjectLambdaPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3control.model.ListAccessPointsForObjectLambdaResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3control.model.ListAccessPointsForObjectLambdaResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPointsForObjectLambda(software.amazon.awssdk.services.s3control.model.ListAccessPointsForObjectLambdaRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAccessPointsForObjectLambdaRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListAccessPointsForObjectLambda
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPointsForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAccessPointsForObjectLambdaPublisher listAccessPointsForObjectLambdaPaginator(
            ListAccessPointsForObjectLambdaRequest listAccessPointsForObjectLambdaRequest) {
        return new ListAccessPointsForObjectLambdaPublisher(this, applyPaginatorUserAgent(listAccessPointsForObjectLambdaRequest));
    }

    /**
     * <p>
     * Returns a list of the access points currently associated with the specified bucket. You can retrieve up to 1000
     * access points per call. If the specified bucket has more than 1,000 access points (or the number specified in
     * <code>maxResults</code>, whichever is less), the response will include a continuation token that you can use to
     * list the additional access points.
     * </p>
     * <p/>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html#API_control_GetAccessPoint_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>ListAccessPoints</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html">CreateAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html">DeleteAccessPoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html">GetAccessPoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsPublisher publisher = client.listAccessPointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListAccessPointsPublisher publisher = client.listAccessPointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3control.model.ListAccessPointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAccessPoints(software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest)} operation.</b>
     * </p>
     *
     * @param listAccessPointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListAccessPoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListAccessPoints" target="_top">AWS
     *      API Documentation</a>
     */
    public ListAccessPointsPublisher listAccessPointsPaginator(ListAccessPointsRequest listAccessPointsRequest) {
        return new ListAccessPointsPublisher(this, applyPaginatorUserAgent(listAccessPointsRequest));
    }

    /**
     * <p>
     * Lists current S3 Batch Operations jobs and jobs that have ended within the last 30 days for the AWS account
     * making the request. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">S3 Batch Operations</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <p/>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html">DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html">UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html">UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException </li> <li>InternalServiceException </li> <li>InvalidNextTokenException </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li> <li>SdkClientException If any client side error occurs such as
     *         an IO related failure, failure to get credentials, etc.</li> <li>S3ControlException Base class for all
     *         service exceptions. Unknown exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");

            HttpResponseHandler<Response<ListJobsResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    ListJobsResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(listJobsRequest.accountId(), "AccountId", "listJobsRequest");
            String resolvedHostExpression = String.format("%s.", listJobsRequest.accountId());

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJobsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listJobsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListJobsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists current S3 Batch Operations jobs and jobs that have ended within the last 30 days for the AWS account
     * making the request. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">S3 Batch Operations</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <p/>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html">DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html">UpdateJobPriority</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html">UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListJobsPublisher publisher = client.listJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3control.model.ListJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3control.model.ListJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listJobs(software.amazon.awssdk.services.s3control.model.ListJobsRequest)} operation.</b>
     * </p>
     *
     * @param listJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException</li>
     *         <li>InternalServiceException</li>
     *         <li>InvalidNextTokenException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    public ListJobsPublisher listJobsPaginator(ListJobsRequest listJobsRequest) {
        return new ListJobsPublisher(this, applyPaginatorUserAgent(listJobsRequest));
    }

    /**
     * <p>
     * Returns a list of all Outposts buckets in an Outpost that are owned by the authenticated sender of the request.
     * For more information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using
     * Amazon S3 on Outposts</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * For an example of the request syntax for Amazon S3 on Outposts that uses the S3 on Outposts endpoint hostname
     * prefix and <code>x-amz-outpost-id</code> in your request, see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListRegionalBuckets.html#API_control_ListRegionalBuckets_Examples"
     * >Examples</a> section.
     * </p>
     *
     * @param listRegionalBucketsRequest
     * @return A Java Future containing the result of the ListRegionalBuckets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListRegionalBuckets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListRegionalBuckets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRegionalBucketsResponse> listRegionalBuckets(
            ListRegionalBucketsRequest listRegionalBucketsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRegionalBucketsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRegionalBuckets");

            HttpResponseHandler<Response<ListRegionalBucketsResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(ListRegionalBucketsResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(listRegionalBucketsRequest.accountId(), "AccountId",
                    "listRegionalBucketsRequest");
            String resolvedHostExpression = String.format("%s.", listRegionalBucketsRequest.accountId());

            CompletableFuture<ListRegionalBucketsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRegionalBucketsRequest, ListRegionalBucketsResponse>()
                            .withOperationName("ListRegionalBuckets")
                            .withMarshaller(new ListRegionalBucketsRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRegionalBucketsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRegionalBucketsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListRegionalBucketsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all Outposts buckets in an Outpost that are owned by the authenticated sender of the request.
     * For more information, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using
     * Amazon S3 on Outposts</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * For an example of the request syntax for Amazon S3 on Outposts that uses the S3 on Outposts endpoint hostname
     * prefix and <code>x-amz-outpost-id</code> in your request, see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListRegionalBuckets.html#API_control_ListRegionalBuckets_Examples"
     * >Examples</a> section.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRegionalBuckets(software.amazon.awssdk.services.s3control.model.ListRegionalBucketsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListRegionalBucketsPublisher publisher = client.listRegionalBucketsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListRegionalBucketsPublisher publisher = client.listRegionalBucketsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3control.model.ListRegionalBucketsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3control.model.ListRegionalBucketsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRegionalBuckets(software.amazon.awssdk.services.s3control.model.ListRegionalBucketsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRegionalBucketsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListRegionalBuckets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListRegionalBuckets" target="_top">AWS
     *      API Documentation</a>
     */
    public ListRegionalBucketsPublisher listRegionalBucketsPaginator(ListRegionalBucketsRequest listRegionalBucketsRequest) {
        return new ListRegionalBucketsPublisher(this, applyPaginatorUserAgent(listRegionalBucketsRequest));
    }

    /**
     * <p>
     * Gets a list of Amazon S3 Storage Lens configurations. For more information about S3 Storage Lens, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Assessing your storage activity and
     * usage with Amazon S3 Storage Lens </a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:ListStorageLensConfigurations</code> action.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note>
     *
     * @param listStorageLensConfigurationsRequest
     * @return A Java Future containing the result of the ListStorageLensConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListStorageLensConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListStorageLensConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStorageLensConfigurationsResponse> listStorageLensConfigurations(
            ListStorageLensConfigurationsRequest listStorageLensConfigurationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listStorageLensConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStorageLensConfigurations");

            HttpResponseHandler<Response<ListStorageLensConfigurationsResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(ListStorageLensConfigurationsResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(listStorageLensConfigurationsRequest.accountId(), "AccountId",
                    "listStorageLensConfigurationsRequest");
            String resolvedHostExpression = String.format("%s.", listStorageLensConfigurationsRequest.accountId());

            CompletableFuture<ListStorageLensConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStorageLensConfigurationsRequest, ListStorageLensConfigurationsResponse>()
                            .withOperationName("ListStorageLensConfigurations")
                            .withMarshaller(new ListStorageLensConfigurationsRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(listStorageLensConfigurationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listStorageLensConfigurationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListStorageLensConfigurationsResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of Amazon S3 Storage Lens configurations. For more information about S3 Storage Lens, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Assessing your storage activity and
     * usage with Amazon S3 Storage Lens </a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:ListStorageLensConfigurations</code> action.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listStorageLensConfigurations(software.amazon.awssdk.services.s3control.model.ListStorageLensConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListStorageLensConfigurationsPublisher publisher = client.listStorageLensConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3control.paginators.ListStorageLensConfigurationsPublisher publisher = client.listStorageLensConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3control.model.ListStorageLensConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3control.model.ListStorageLensConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStorageLensConfigurations(software.amazon.awssdk.services.s3control.model.ListStorageLensConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listStorageLensConfigurationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.ListStorageLensConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/ListStorageLensConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListStorageLensConfigurationsPublisher listStorageLensConfigurationsPaginator(
            ListStorageLensConfigurationsRequest listStorageLensConfigurationsRequest) {
        return new ListStorageLensConfigurationsPublisher(this, applyPaginatorUserAgent(listStorageLensConfigurationsRequest));
    }

    /**
     * <p>
     * Replaces configuration for an Object Lambda Access Point.
     * </p>
     * <p>
     * The following actions are related to <code>PutAccessPointConfigurationForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointConfigurationForObjectLambda.html"
     * >GetAccessPointConfigurationForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putAccessPointConfigurationForObjectLambdaRequest
     * @return A Java Future containing the result of the PutAccessPointConfigurationForObjectLambda operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutAccessPointConfigurationForObjectLambda
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutAccessPointConfigurationForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccessPointConfigurationForObjectLambdaResponse> putAccessPointConfigurationForObjectLambda(
            PutAccessPointConfigurationForObjectLambdaRequest putAccessPointConfigurationForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putAccessPointConfigurationForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccessPointConfigurationForObjectLambda");

            HttpResponseHandler<Response<PutAccessPointConfigurationForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutAccessPointConfigurationForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putAccessPointConfigurationForObjectLambdaRequest.accountId(),
                    "AccountId", "putAccessPointConfigurationForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", putAccessPointConfigurationForObjectLambdaRequest.accountId());

            CompletableFuture<PutAccessPointConfigurationForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccessPointConfigurationForObjectLambdaRequest, PutAccessPointConfigurationForObjectLambdaResponse>()
                            .withOperationName("PutAccessPointConfigurationForObjectLambda")
                            .withMarshaller(new PutAccessPointConfigurationForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(putAccessPointConfigurationForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putAccessPointConfigurationForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<PutAccessPointConfigurationForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates an access policy with the specified access point. Each access point can have only one policy, so a
     * request made to this API replaces any existing policy associated with the specified access point.
     * </p>
     * <p/>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html#API_control_PutAccessPointPolicy_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>PutAccessPointPolicy</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html">
     * GetAccessPointPolicy</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html">
     * DeleteAccessPointPolicy</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putAccessPointPolicyRequest
     * @return A Java Future containing the result of the PutAccessPointPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutAccessPointPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutAccessPointPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccessPointPolicyResponse> putAccessPointPolicy(
            PutAccessPointPolicyRequest putAccessPointPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAccessPointPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccessPointPolicy");

            HttpResponseHandler<Response<PutAccessPointPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutAccessPointPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String name = putAccessPointPolicyRequest.name();
            Arn arn = null;
            if (name != null && name.startsWith("arn:")) {
                arn = Arn.fromString(name);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3AccessPointResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3AccessPointResource resource = (S3AccessPointResource) s3Resource;
                String accountId = putAccessPointPolicyRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                putAccessPointPolicyRequest = putAccessPointPolicyRequest.toBuilder().name(resource.accessPointName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putAccessPointPolicyRequest.accountId(), "AccountId",
                    "putAccessPointPolicyRequest");
            String resolvedHostExpression = String.format("%s.", putAccessPointPolicyRequest.accountId());

            CompletableFuture<PutAccessPointPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccessPointPolicyRequest, PutAccessPointPolicyResponse>()
                            .withOperationName("PutAccessPointPolicy")
                            .withMarshaller(new PutAccessPointPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(putAccessPointPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putAccessPointPolicyRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutAccessPointPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or replaces resource policy for an Object Lambda Access Point. For an example policy, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/olap-create.html#olap-create-cli">Creating Object
     * Lambda Access Points</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * The following actions are related to <code>PutAccessPointPolicyForObjectLambda</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicyForObjectLambda.html">
     * DeleteAccessPointPolicyForObjectLambda</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyForObjectLambda.html">
     * GetAccessPointPolicyForObjectLambda</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putAccessPointPolicyForObjectLambdaRequest
     * @return A Java Future containing the result of the PutAccessPointPolicyForObjectLambda operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutAccessPointPolicyForObjectLambda
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutAccessPointPolicyForObjectLambda"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAccessPointPolicyForObjectLambdaResponse> putAccessPointPolicyForObjectLambda(
            PutAccessPointPolicyForObjectLambdaRequest putAccessPointPolicyForObjectLambdaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putAccessPointPolicyForObjectLambdaRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAccessPointPolicyForObjectLambda");

            HttpResponseHandler<Response<PutAccessPointPolicyForObjectLambdaResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutAccessPointPolicyForObjectLambdaResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putAccessPointPolicyForObjectLambdaRequest.accountId(), "AccountId",
                    "putAccessPointPolicyForObjectLambdaRequest");
            String resolvedHostExpression = String.format("%s.", putAccessPointPolicyForObjectLambdaRequest.accountId());

            CompletableFuture<PutAccessPointPolicyForObjectLambdaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAccessPointPolicyForObjectLambdaRequest, PutAccessPointPolicyForObjectLambdaResponse>()
                            .withOperationName("PutAccessPointPolicyForObjectLambda")
                            .withMarshaller(new PutAccessPointPolicyForObjectLambdaRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(putAccessPointPolicyForObjectLambdaRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putAccessPointPolicyForObjectLambdaRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<PutAccessPointPolicyForObjectLambdaResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action puts a lifecycle configuration to an Amazon S3 on Outposts bucket. To put a lifecycle configuration
     * to an S3 bucket, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html"
     * >PutBucketLifecycleConfiguration</a> in the <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Creates a new lifecycle configuration for the S3 on Outposts bucket or replaces an existing lifecycle
     * configuration. Outposts buckets only support lifecycle configurations that delete/expire objects after a certain
     * period of time and abort incomplete multipart uploads.
     * </p>
     * <p/>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketLifecycleConfiguration.html#API_control_PutBucketLifecycleConfiguration_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>PutBucketLifecycleConfiguration</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketLifecycleConfiguration.html">
     * GetBucketLifecycleConfiguration</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketLifecycleConfiguration.html">
     * DeleteBucketLifecycleConfiguration</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putBucketLifecycleConfigurationRequest
     * @return A Java Future containing the result of the PutBucketLifecycleConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutBucketLifecycleConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutBucketLifecycleConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutBucketLifecycleConfigurationResponse> putBucketLifecycleConfiguration(
            PutBucketLifecycleConfigurationRequest putBucketLifecycleConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putBucketLifecycleConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBucketLifecycleConfiguration");

            HttpResponseHandler<Response<PutBucketLifecycleConfigurationResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutBucketLifecycleConfigurationResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = putBucketLifecycleConfigurationRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = putBucketLifecycleConfigurationRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                putBucketLifecycleConfigurationRequest = putBucketLifecycleConfigurationRequest.toBuilder()
                        .bucket(resource.bucketName()).accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putBucketLifecycleConfigurationRequest.accountId(), "AccountId",
                    "putBucketLifecycleConfigurationRequest");
            String resolvedHostExpression = String.format("%s.", putBucketLifecycleConfigurationRequest.accountId());

            CompletableFuture<PutBucketLifecycleConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutBucketLifecycleConfigurationRequest, PutBucketLifecycleConfigurationResponse>()
                            .withOperationName("PutBucketLifecycleConfiguration")
                            .withMarshaller(new PutBucketLifecycleConfigurationRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.HTTP_CHECKSUM_REQUIRED,
                                    HttpChecksumRequired.create())
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(putBucketLifecycleConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putBucketLifecycleConfigurationRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<PutBucketLifecycleConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action puts a bucket policy to an Amazon S3 on Outposts bucket. To put a policy on an S3 bucket, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html">PutBucketPolicy</a> in the
     * <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Applies an Amazon S3 bucket policy to an Outposts bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * If you are using an identity other than the root user of the AWS account that owns the Outposts bucket, the
     * calling identity must have the <code>PutBucketPolicy</code> permissions on the specified Outposts bucket and
     * belong to the bucket owner's account in order to use this action.
     * </p>
     * <p>
     * If you don't have <code>PutBucketPolicy</code> permissions, Amazon S3 returns a <code>403 Access Denied</code>
     * error. If you have the correct permissions, but you're not using an identity that belongs to the bucket owner's
     * account, Amazon S3 returns a <code>405 Method Not Allowed</code> error.
     * </p>
     * <important>
     * <p>
     * As a security precaution, the root user of the AWS account that owns a bucket can always use this action, even if
     * the policy explicitly denies the root user the ability to perform this action.
     * </p>
     * </important>
     * <p>
     * For more information about bucket policies, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/using-iam-policies.html">Using Bucket Policies and User
     * Policies</a>.
     * </p>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketPolicy.html#API_control_PutBucketPolicy_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>PutBucketPolicy</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketPolicy.html">GetBucketPolicy</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketPolicy.html">DeleteBucketPolicy</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putBucketPolicyRequest
     * @return A Java Future containing the result of the PutBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutBucketPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutBucketPolicyResponse> putBucketPolicy(PutBucketPolicyRequest putBucketPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBucketPolicy");

            HttpResponseHandler<Response<PutBucketPolicyResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutBucketPolicyResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = putBucketPolicyRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = putBucketPolicyRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                putBucketPolicyRequest = putBucketPolicyRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator
                    .validateHostnameCompliant(putBucketPolicyRequest.accountId(), "AccountId", "putBucketPolicyRequest");
            String resolvedHostExpression = String.format("%s.", putBucketPolicyRequest.accountId());

            CompletableFuture<PutBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutBucketPolicyRequest, PutBucketPolicyResponse>()
                            .withOperationName("PutBucketPolicy")
                            .withMarshaller(new PutBucketPolicyRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.HTTP_CHECKSUM_REQUIRED,
                                    HttpChecksumRequired.create())
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(putBucketPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putBucketPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutBucketPolicyResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * This action puts tags on an Amazon S3 on Outposts bucket. To put tags on an S3 bucket, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html">PutBucketTagging</a> in the
     * <i>Amazon Simple Storage Service API</i>.
     * </p>
     * </note>
     * <p>
     * Sets the tags for an S3 on Outposts bucket. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/S3onOutposts.html">Using Amazon S3 on Outposts</a> in
     * the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p>
     * Use tags to organize your AWS bill to reflect your own cost structure. To do this, sign up to get your AWS
     * account bill with tag key values included. Then, to see the cost of combined resources, organize your billing
     * information according to resources with the same tag key values. For example, you can tag several resources with
     * a specific application name, and then organize your billing information to see the total cost of that application
     * across several services. For more information, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Cost allocation and
     * tagging</a>.
     * </p>
     * <note>
     * <p>
     * Within a bucket, if you add a tag that has the same key as an existing tag, the new value overwrites the old
     * value. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/CostAllocTagging.html"> Using cost allocation in
     * Amazon S3 bucket tags</a>.
     * </p>
     * </note>
     * <p>
     * To use this action, you must have permissions to perform the <code>s3-outposts:PutBucketTagging</code> action.
     * The Outposts bucket owner has this permission by default and can grant this permission to others. For more
     * information about permissions, see <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-with-s3-actions.html#using-with-s3-actions-related-to-bucket-subresources"
     * > Permissions Related to Bucket Subresource Operations</a> and <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-access-control.html">Managing access permissions
     * to your Amazon S3 resources</a>.
     * </p>
     * <p>
     * <code>PutBucketTagging</code> has the following special errors:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Error code: <code>InvalidTagError</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Description: The tag provided was not a valid tag. This error can occur if the tag did not pass input validation.
     * For information about tag restrictions, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/allocation-tag-restrictions.html">
     * User-Defined Tag Restrictions</a> and <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/aws-tag-restrictions.html"> AWS-Generated Cost
     * Allocation Tag Restrictions</a>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Error code: <code>MalformedXMLError</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Description: The XML provided does not match the schema.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Error code: <code>OperationAbortedError </code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Description: A conflicting conditional action is currently in progress against this resource. Try again.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Error code: <code>InternalError</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Description: The service was unable to apply the provided tag to the bucket.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * All Amazon S3 on Outposts REST API requests for this action require an additional parameter of
     * <code>x-amz-outpost-id</code> to be passed with the request and an S3 on Outposts endpoint hostname prefix
     * instead of <code>s3-control</code>. For an example of the request syntax for Amazon S3 on Outposts that uses the
     * S3 on Outposts endpoint hostname prefix and the <code>x-amz-outpost-id</code> derived using the access point ARN,
     * see the <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutBucketTagging.html#API_control_PutBucketTagging_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * The following actions are related to <code>PutBucketTagging</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetBucketTagging.html">GetBucketTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteBucketTagging.html">DeleteBucketTagging
     * </a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putBucketTaggingRequest
     * @return A Java Future containing the result of the PutBucketTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutBucketTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutBucketTagging" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutBucketTaggingResponse> putBucketTagging(PutBucketTaggingRequest putBucketTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putBucketTaggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutBucketTagging");

            HttpResponseHandler<Response<PutBucketTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutBucketTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String bucket = putBucketTaggingRequest.bucket();
            Arn arn = null;
            if (bucket != null && bucket.startsWith("arn:")) {
                arn = Arn.fromString(bucket);
                S3Resource s3Resource = S3ControlArnConverter.getInstance().convertArn(arn);
                if (!(s3Resource instanceof S3ControlBucketResource)) {
                    throw new IllegalArgumentException(String.format("Unsupported ARN type: %s", s3Resource.type()));
                }
                S3ControlBucketResource resource = (S3ControlBucketResource) s3Resource;
                String accountId = putBucketTaggingRequest.accountId();
                String accountIdInArn = resource.accountId().orElseThrow(
                        () -> new IllegalArgumentException("accountId cannot be null"));
                if (accountId != null && !accountId.equals(accountIdInArn)) {
                    throw new IllegalArgumentException(String.format(
                            "%s field provided from the request (%s) is different from the one in the ARN (%s)", "accountId",
                            accountId, accountIdInArn));
                }
                putBucketTaggingRequest = putBucketTaggingRequest.toBuilder().bucket(resource.bucketName())
                        .accountId(accountIdInArn).build();
            }
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putBucketTaggingRequest.accountId(), "AccountId",
                    "putBucketTaggingRequest");
            String resolvedHostExpression = String.format("%s.", putBucketTaggingRequest.accountId());

            CompletableFuture<PutBucketTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutBucketTaggingRequest, PutBucketTaggingResponse>()
                            .withOperationName("PutBucketTagging")
                            .withMarshaller(new PutBucketTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler)
                            .hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector)
                            .putExecutionAttribute(SdkInternalExecutionAttribute.HTTP_CHECKSUM_REQUIRED,
                                    HttpChecksumRequired.create())
                            .putExecutionAttribute(S3ControlInternalExecutionAttribute.S3_ARNABLE_FIELD,
                                    S3ArnableField.builder().arn(arn).build()).withInput(putBucketTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putBucketTaggingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutBucketTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the supplied tag-set on an S3 Batch Operations job.
     * </p>
     * <p>
     * A tag is a key-value pair. You can associate S3 Batch Operations tags with any job by sending a PUT request
     * against the tagging subresource that is associated with the job. To modify the existing tag set, you can either
     * replace the existing tag set entirely, or make changes within the existing tag set by retrieving the existing tag
     * set using <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html">GetJobTagging</a>, modify
     * that tag set, and use this action to replace the tag set with the one you modified. For more information, see <a
     * href
     * ="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html#batch-ops-job-tags">Controlling
     * access and labeling jobs using tags</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p/>
     * <note>
     * <ul>
     * <li>
     * <p>
     * If you send this request with an empty tag set, Amazon S3 deletes the existing tag set on the Batch Operations
     * job. If you use this method, you are charged for a Tier 1 Request (PUT). For more information, see <a
     * href="http://aws.amazon.com/s3/pricing/">Amazon S3 pricing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For deleting existing tags for your Batch Operations job, a <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html">DeleteJobTagging</a>
     * request is preferred because it achieves the same result without incurring charges.
     * </p>
     * </li>
     * <li>
     * <p>
     * A few things to consider about using tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon S3 limits the maximum number of tags to 50 tags per job.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can associate up to 50 tags with a job as long as they have unique tag keys.
     * </p>
     * </li>
     * <li>
     * <p>
     * A tag key can be up to 128 Unicode characters in length, and tag values can be up to 256 Unicode characters in
     * length.
     * </p>
     * </li>
     * <li>
     * <p>
     * The key and values are case sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * For tagging-related restrictions related to characters and encodings, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/allocation-tag-restrictions.html">User-Defined
     * Tag Restrictions</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * </note>
     * <p/>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:PutJobTagging</code> action.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreatJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html">GetJobTagging</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html">DeleteJobTagging</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putJobTaggingRequest
     * @return A Java Future containing the result of the PutJobTagging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>TooManyTagsException Amazon S3 throws this exception if you have too many tags in your tag set.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutJobTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutJobTagging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutJobTaggingResponse> putJobTagging(PutJobTaggingRequest putJobTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putJobTaggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutJobTagging");

            HttpResponseHandler<Response<PutJobTaggingResponse>> responseHandler = protocolFactory.createCombinedResponseHandler(
                    PutJobTaggingResponse::builder, new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putJobTaggingRequest.accountId(), "AccountId", "putJobTaggingRequest");
            String resolvedHostExpression = String.format("%s.", putJobTaggingRequest.accountId());

            CompletableFuture<PutJobTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutJobTaggingRequest, PutJobTaggingResponse>()
                            .withOperationName("PutJobTagging")
                            .withMarshaller(new PutJobTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(putJobTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putJobTaggingRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutJobTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or modifies the <code>PublicAccessBlock</code> configuration for an AWS account. For more information,
     * see <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html"> Using
     * Amazon S3 block public access</a>.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html">
     * GetPublicAccessBlock</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeletePublicAccessBlock.html">
     * DeletePublicAccessBlock</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putPublicAccessBlockRequest
     * @return A Java Future containing the result of the PutPublicAccessBlock operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutPublicAccessBlock
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutPublicAccessBlock"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutPublicAccessBlockResponse> putPublicAccessBlock(
            PutPublicAccessBlockRequest putPublicAccessBlockRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPublicAccessBlockRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPublicAccessBlock");

            HttpResponseHandler<Response<PutPublicAccessBlockResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutPublicAccessBlockResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putPublicAccessBlockRequest.accountId(), "AccountId",
                    "putPublicAccessBlockRequest");
            String resolvedHostExpression = String.format("%s.", putPublicAccessBlockRequest.accountId());

            CompletableFuture<PutPublicAccessBlockResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPublicAccessBlockRequest, PutPublicAccessBlockResponse>()
                            .withOperationName("PutPublicAccessBlock")
                            .withMarshaller(new PutPublicAccessBlockRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPublicAccessBlockRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putPublicAccessBlockRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutPublicAccessBlockResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Puts an Amazon S3 Storage Lens configuration. For more information about S3 Storage Lens, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Working with Amazon S3 Storage Lens</a>
     * in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:PutStorageLensConfiguration</code> action.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note>
     *
     * @param putStorageLensConfigurationRequest
     * @return A Java Future containing the result of the PutStorageLensConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutStorageLensConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutStorageLensConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutStorageLensConfigurationResponse> putStorageLensConfiguration(
            PutStorageLensConfigurationRequest putStorageLensConfigurationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putStorageLensConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutStorageLensConfiguration");

            HttpResponseHandler<Response<PutStorageLensConfigurationResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutStorageLensConfigurationResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putStorageLensConfigurationRequest.accountId(), "AccountId",
                    "putStorageLensConfigurationRequest");
            String resolvedHostExpression = String.format("%s.", putStorageLensConfigurationRequest.accountId());

            CompletableFuture<PutStorageLensConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutStorageLensConfigurationRequest, PutStorageLensConfigurationResponse>()
                            .withOperationName("PutStorageLensConfiguration")
                            .withMarshaller(new PutStorageLensConfigurationRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(putStorageLensConfigurationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putStorageLensConfigurationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<PutStorageLensConfigurationResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Put or replace tags on an existing Amazon S3 Storage Lens configuration. For more information about S3 Storage
     * Lens, see <a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens.html">Assessing your storage
     * activity and usage with Amazon S3 Storage Lens </a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <note>
     * <p>
     * To use this action, you must have permission to perform the <code>s3:PutStorageLensConfigurationTagging</code>
     * action. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/storage_lens_iam_permissions.html">Setting permissions to
     * use Amazon S3 Storage Lens</a> in the <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * </note>
     *
     * @param putStorageLensConfigurationTaggingRequest
     * @return A Java Future containing the result of the PutStorageLensConfigurationTagging operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.PutStorageLensConfigurationTagging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/PutStorageLensConfigurationTagging"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutStorageLensConfigurationTaggingResponse> putStorageLensConfigurationTagging(
            PutStorageLensConfigurationTaggingRequest putStorageLensConfigurationTaggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putStorageLensConfigurationTaggingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutStorageLensConfigurationTagging");

            HttpResponseHandler<Response<PutStorageLensConfigurationTaggingResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(PutStorageLensConfigurationTaggingResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(putStorageLensConfigurationTaggingRequest.accountId(), "AccountId",
                    "putStorageLensConfigurationTaggingRequest");
            String resolvedHostExpression = String.format("%s.", putStorageLensConfigurationTaggingRequest.accountId());

            CompletableFuture<PutStorageLensConfigurationTaggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutStorageLensConfigurationTaggingRequest, PutStorageLensConfigurationTaggingResponse>()
                            .withOperationName("PutStorageLensConfigurationTagging")
                            .withMarshaller(new PutStorageLensConfigurationTaggingRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(putStorageLensConfigurationTaggingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putStorageLensConfigurationTaggingRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<PutStorageLensConfigurationTaggingResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing S3 Batch Operations job's priority. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">S3 Batch Operations</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html">ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html">DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html">UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateJobPriorityRequest
     * @return A Java Future containing the result of the UpdateJobPriority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.UpdateJobPriority
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobPriority" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobPriorityResponse> updateJobPriority(UpdateJobPriorityRequest updateJobPriorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobPriorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobPriority");

            HttpResponseHandler<Response<UpdateJobPriorityResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(UpdateJobPriorityResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator.validateHostnameCompliant(updateJobPriorityRequest.accountId(), "AccountId",
                    "updateJobPriorityRequest");
            String resolvedHostExpression = String.format("%s.", updateJobPriorityRequest.accountId());

            CompletableFuture<UpdateJobPriorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobPriorityRequest, UpdateJobPriorityResponse>()
                            .withOperationName("UpdateJobPriority")
                            .withMarshaller(new UpdateJobPriorityRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateJobPriorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateJobPriorityRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateJobPriorityResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status for the specified job. Use this action to confirm that you want to run a job or to cancel an
     * existing job. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-basics.html">S3 Batch Operations</a> in the
     * <i>Amazon Simple Storage Service User Guide</i>.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html">CreateJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html">ListJobs</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html">DescribeJob</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html">UpdateJobStatus</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateJobStatusRequest
     * @return A Java Future containing the result of the UpdateJobStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException</li>
     *         <li>TooManyRequestsException</li>
     *         <li>NotFoundException</li>
     *         <li>JobStatusException</li>
     *         <li>InternalServiceException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3ControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3ControlAsyncClient.UpdateJobStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3control-2018-08-20/UpdateJobStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobStatusResponse> updateJobStatus(UpdateJobStatusRequest updateJobStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3 Control");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobStatus");

            HttpResponseHandler<Response<UpdateJobStatusResponse>> responseHandler = protocolFactory
                    .createCombinedResponseHandler(UpdateJobStatusResponse::builder,
                            new XmlOperationMetadata().withHasStreamingSuccessResponse(false));
            String hostPrefix = "{AccountId}.";
            HostnameValidator
                    .validateHostnameCompliant(updateJobStatusRequest.accountId(), "AccountId", "updateJobStatusRequest");
            String resolvedHostExpression = String.format("%s.", updateJobStatusRequest.accountId());

            CompletableFuture<UpdateJobStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobStatusRequest, UpdateJobStatusResponse>()
                            .withOperationName("UpdateJobStatus")
                            .withMarshaller(new UpdateJobStatusRequestMarshaller(protocolFactory))
                            .withCombinedResponseHandler(responseHandler).hostPrefixExpression(resolvedHostExpression)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateJobStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateJobStatusRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateJobStatusResponse> whenCompleteFuture = null;
            whenCompleteFuture = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            return CompletableFutureUtils.forwardExceptionTo(whenCompleteFuture, executeFuture);
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private AwsXmlProtocolFactory init() {
        return AwsXmlProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("JobStatusException")
                                .exceptionBuilderSupplier(JobStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoSuchPublicAccessBlockConfiguration")
                                .exceptionBuilderSupplier(NoSuchPublicAccessBlockConfigurationException::builder)
                                .httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BucketAlreadyExists")
                                .exceptionBuilderSupplier(BucketAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotencyException")
                                .exceptionBuilderSupplier(IdempotencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BucketAlreadyOwnedByYou")
                                .exceptionBuilderSupplier(BucketAlreadyOwnedByYouException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(S3ControlException::builder).build();
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends S3ControlRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
