/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the configuration parameters for a PUT Copy object operation. S3 Batch Operations passes every object to the
 * underlying PUT Copy object API. For more information about the parameters for this operation, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/API/RESTObjectCOPY.html">PUT Object - Copy</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3CopyObjectOperation implements SdkPojo, Serializable,
        ToCopyableBuilder<S3CopyObjectOperation.Builder, S3CopyObjectOperation> {
    private static final SdkField<String> TARGET_RESOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetResource")
            .getter(getter(S3CopyObjectOperation::targetResource))
            .setter(setter(Builder::targetResource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetResource")
                    .unmarshallLocationName("TargetResource").build()).build();

    private static final SdkField<String> CANNED_ACCESS_CONTROL_LIST_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CannedAccessControlList")
            .getter(getter(S3CopyObjectOperation::cannedAccessControlListAsString))
            .setter(setter(Builder::cannedAccessControlList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CannedAccessControlList")
                    .unmarshallLocationName("CannedAccessControlList").build()).build();

    private static final SdkField<List<S3Grant>> ACCESS_CONTROL_GRANTS_FIELD = SdkField
            .<List<S3Grant>> builder(MarshallingType.LIST)
            .memberName("AccessControlGrants")
            .getter(getter(S3CopyObjectOperation::accessControlGrants))
            .setter(setter(Builder::accessControlGrants))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlGrants")
                    .unmarshallLocationName("AccessControlGrants").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Grant> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Grant::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<String> METADATA_DIRECTIVE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MetadataDirective")
            .getter(getter(S3CopyObjectOperation::metadataDirectiveAsString))
            .setter(setter(Builder::metadataDirective))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetadataDirective")
                    .unmarshallLocationName("MetadataDirective").build()).build();

    private static final SdkField<Instant> MODIFIED_SINCE_CONSTRAINT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ModifiedSinceConstraint")
            .getter(getter(S3CopyObjectOperation::modifiedSinceConstraint))
            .setter(setter(Builder::modifiedSinceConstraint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModifiedSinceConstraint")
                    .unmarshallLocationName("ModifiedSinceConstraint").build()).build();

    private static final SdkField<S3ObjectMetadata> NEW_OBJECT_METADATA_FIELD = SdkField
            .<S3ObjectMetadata> builder(MarshallingType.SDK_POJO)
            .memberName("NewObjectMetadata")
            .getter(getter(S3CopyObjectOperation::newObjectMetadata))
            .setter(setter(Builder::newObjectMetadata))
            .constructor(S3ObjectMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewObjectMetadata")
                    .unmarshallLocationName("NewObjectMetadata").build()).build();

    private static final SdkField<List<S3Tag>> NEW_OBJECT_TAGGING_FIELD = SdkField
            .<List<S3Tag>> builder(MarshallingType.LIST)
            .memberName("NewObjectTagging")
            .getter(getter(S3CopyObjectOperation::newObjectTagging))
            .setter(setter(Builder::newObjectTagging))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewObjectTagging")
                    .unmarshallLocationName("NewObjectTagging").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<String> REDIRECT_LOCATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RedirectLocation")
            .getter(getter(S3CopyObjectOperation::redirectLocation))
            .setter(setter(Builder::redirectLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RedirectLocation")
                    .unmarshallLocationName("RedirectLocation").build()).build();

    private static final SdkField<Boolean> REQUESTER_PAYS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RequesterPays")
            .getter(getter(S3CopyObjectOperation::requesterPays))
            .setter(setter(Builder::requesterPays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequesterPays")
                    .unmarshallLocationName("RequesterPays").build()).build();

    private static final SdkField<String> STORAGE_CLASS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StorageClass")
            .getter(getter(S3CopyObjectOperation::storageClassAsString))
            .setter(setter(Builder::storageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageClass")
                    .unmarshallLocationName("StorageClass").build()).build();

    private static final SdkField<Instant> UN_MODIFIED_SINCE_CONSTRAINT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UnModifiedSinceConstraint")
            .getter(getter(S3CopyObjectOperation::unModifiedSinceConstraint))
            .setter(setter(Builder::unModifiedSinceConstraint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnModifiedSinceConstraint")
                    .unmarshallLocationName("UnModifiedSinceConstraint").build()).build();

    private static final SdkField<String> SSE_AWS_KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SSEAwsKmsKeyId")
            .getter(getter(S3CopyObjectOperation::sseAwsKmsKeyId))
            .setter(setter(Builder::sseAwsKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SSEAwsKmsKeyId")
                    .unmarshallLocationName("SSEAwsKmsKeyId").build()).build();

    private static final SdkField<String> TARGET_KEY_PREFIX_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TargetKeyPrefix")
            .getter(getter(S3CopyObjectOperation::targetKeyPrefix))
            .setter(setter(Builder::targetKeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetKeyPrefix")
                    .unmarshallLocationName("TargetKeyPrefix").build()).build();

    private static final SdkField<String> OBJECT_LOCK_LEGAL_HOLD_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ObjectLockLegalHoldStatus")
            .getter(getter(S3CopyObjectOperation::objectLockLegalHoldStatusAsString))
            .setter(setter(Builder::objectLockLegalHoldStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectLockLegalHoldStatus")
                    .unmarshallLocationName("ObjectLockLegalHoldStatus").build()).build();

    private static final SdkField<String> OBJECT_LOCK_MODE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ObjectLockMode")
            .getter(getter(S3CopyObjectOperation::objectLockModeAsString))
            .setter(setter(Builder::objectLockMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectLockMode")
                    .unmarshallLocationName("ObjectLockMode").build()).build();

    private static final SdkField<Instant> OBJECT_LOCK_RETAIN_UNTIL_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ObjectLockRetainUntilDate")
            .getter(getter(S3CopyObjectOperation::objectLockRetainUntilDate))
            .setter(setter(Builder::objectLockRetainUntilDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectLockRetainUntilDate")
                    .unmarshallLocationName("ObjectLockRetainUntilDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TARGET_RESOURCE_FIELD,
            CANNED_ACCESS_CONTROL_LIST_FIELD, ACCESS_CONTROL_GRANTS_FIELD, METADATA_DIRECTIVE_FIELD,
            MODIFIED_SINCE_CONSTRAINT_FIELD, NEW_OBJECT_METADATA_FIELD, NEW_OBJECT_TAGGING_FIELD, REDIRECT_LOCATION_FIELD,
            REQUESTER_PAYS_FIELD, STORAGE_CLASS_FIELD, UN_MODIFIED_SINCE_CONSTRAINT_FIELD, SSE_AWS_KMS_KEY_ID_FIELD,
            TARGET_KEY_PREFIX_FIELD, OBJECT_LOCK_LEGAL_HOLD_STATUS_FIELD, OBJECT_LOCK_MODE_FIELD,
            OBJECT_LOCK_RETAIN_UNTIL_DATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String targetResource;

    private final String cannedAccessControlList;

    private final List<S3Grant> accessControlGrants;

    private final String metadataDirective;

    private final Instant modifiedSinceConstraint;

    private final S3ObjectMetadata newObjectMetadata;

    private final List<S3Tag> newObjectTagging;

    private final String redirectLocation;

    private final Boolean requesterPays;

    private final String storageClass;

    private final Instant unModifiedSinceConstraint;

    private final String sseAwsKmsKeyId;

    private final String targetKeyPrefix;

    private final String objectLockLegalHoldStatus;

    private final String objectLockMode;

    private final Instant objectLockRetainUntilDate;

    private S3CopyObjectOperation(BuilderImpl builder) {
        this.targetResource = builder.targetResource;
        this.cannedAccessControlList = builder.cannedAccessControlList;
        this.accessControlGrants = builder.accessControlGrants;
        this.metadataDirective = builder.metadataDirective;
        this.modifiedSinceConstraint = builder.modifiedSinceConstraint;
        this.newObjectMetadata = builder.newObjectMetadata;
        this.newObjectTagging = builder.newObjectTagging;
        this.redirectLocation = builder.redirectLocation;
        this.requesterPays = builder.requesterPays;
        this.storageClass = builder.storageClass;
        this.unModifiedSinceConstraint = builder.unModifiedSinceConstraint;
        this.sseAwsKmsKeyId = builder.sseAwsKmsKeyId;
        this.targetKeyPrefix = builder.targetKeyPrefix;
        this.objectLockLegalHoldStatus = builder.objectLockLegalHoldStatus;
        this.objectLockMode = builder.objectLockMode;
        this.objectLockRetainUntilDate = builder.objectLockRetainUntilDate;
    }

    /**
     * <p>
     * Specifies the destination bucket ARN for the batch copy operation. For example, to copy objects to a bucket named
     * "destinationBucket", set the TargetResource to "arn:aws:s3:::destinationBucket".
     * </p>
     * 
     * @return Specifies the destination bucket ARN for the batch copy operation. For example, to copy objects to a
     *         bucket named "destinationBucket", set the TargetResource to "arn:aws:s3:::destinationBucket".
     */
    public final String targetResource() {
        return targetResource;
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #cannedAccessControlList} will return {@link S3CannedAccessControlList#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #cannedAccessControlListAsString}.
     * </p>
     * 
     * @return
     * @see S3CannedAccessControlList
     */
    public final S3CannedAccessControlList cannedAccessControlList() {
        return S3CannedAccessControlList.fromValue(cannedAccessControlList);
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #cannedAccessControlList} will return {@link S3CannedAccessControlList#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #cannedAccessControlListAsString}.
     * </p>
     * 
     * @return
     * @see S3CannedAccessControlList
     */
    public final String cannedAccessControlListAsString() {
        return cannedAccessControlList;
    }

    /**
     * Returns true if the AccessControlGrants property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasAccessControlGrants() {
        return accessControlGrants != null && !(accessControlGrants instanceof SdkAutoConstructList);
    }

    /**
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAccessControlGrants()} to see if a value was sent in this field.
     * </p>
     * 
     * @return
     */
    public final List<S3Grant> accessControlGrants() {
        return accessControlGrants;
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metadataDirective}
     * will return {@link S3MetadataDirective#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #metadataDirectiveAsString}.
     * </p>
     * 
     * @return
     * @see S3MetadataDirective
     */
    public final S3MetadataDirective metadataDirective() {
        return S3MetadataDirective.fromValue(metadataDirective);
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metadataDirective}
     * will return {@link S3MetadataDirective#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #metadataDirectiveAsString}.
     * </p>
     * 
     * @return
     * @see S3MetadataDirective
     */
    public final String metadataDirectiveAsString() {
        return metadataDirective;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public final Instant modifiedSinceConstraint() {
        return modifiedSinceConstraint;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public final S3ObjectMetadata newObjectMetadata() {
        return newObjectMetadata;
    }

    /**
     * Returns true if the NewObjectTagging property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasNewObjectTagging() {
        return newObjectTagging != null && !(newObjectTagging instanceof SdkAutoConstructList);
    }

    /**
     * <p/>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNewObjectTagging()} to see if a value was sent in this field.
     * </p>
     * 
     * @return
     */
    public final List<S3Tag> newObjectTagging() {
        return newObjectTagging;
    }

    /**
     * <p>
     * Specifies an optional metadata property for website redirects, <code>x-amz-website-redirect-location</code>.
     * Allows webpage redirects if the object is accessed through a website endpoint.
     * </p>
     * 
     * @return Specifies an optional metadata property for website redirects,
     *         <code>x-amz-website-redirect-location</code>. Allows webpage redirects if the object is accessed through
     *         a website endpoint.
     */
    public final String redirectLocation() {
        return redirectLocation;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public final Boolean requesterPays() {
        return requesterPays;
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link S3StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return
     * @see S3StorageClass
     */
    public final S3StorageClass storageClass() {
        return S3StorageClass.fromValue(storageClass);
    }

    /**
     * <p/>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link S3StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return
     * @see S3StorageClass
     */
    public final String storageClassAsString() {
        return storageClass;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public final Instant unModifiedSinceConstraint() {
        return unModifiedSinceConstraint;
    }

    /**
     * <p/>
     * 
     * @return
     */
    public final String sseAwsKmsKeyId() {
        return sseAwsKmsKeyId;
    }

    /**
     * <p>
     * Specifies the folder prefix into which you would like the objects to be copied. For example, to copy objects into
     * a folder named "Folder1" in the destination bucket, set the TargetKeyPrefix to "Folder1/".
     * </p>
     * 
     * @return Specifies the folder prefix into which you would like the objects to be copied. For example, to copy
     *         objects into a folder named "Folder1" in the destination bucket, set the TargetKeyPrefix to "Folder1/".
     */
    public final String targetKeyPrefix() {
        return targetKeyPrefix;
    }

    /**
     * <p>
     * The legal hold status to be applied to all objects in the Batch Operations job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #objectLockLegalHoldStatus} will return {@link S3ObjectLockLegalHoldStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #objectLockLegalHoldStatusAsString}.
     * </p>
     * 
     * @return The legal hold status to be applied to all objects in the Batch Operations job.
     * @see S3ObjectLockLegalHoldStatus
     */
    public final S3ObjectLockLegalHoldStatus objectLockLegalHoldStatus() {
        return S3ObjectLockLegalHoldStatus.fromValue(objectLockLegalHoldStatus);
    }

    /**
     * <p>
     * The legal hold status to be applied to all objects in the Batch Operations job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #objectLockLegalHoldStatus} will return {@link S3ObjectLockLegalHoldStatus#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #objectLockLegalHoldStatusAsString}.
     * </p>
     * 
     * @return The legal hold status to be applied to all objects in the Batch Operations job.
     * @see S3ObjectLockLegalHoldStatus
     */
    public final String objectLockLegalHoldStatusAsString() {
        return objectLockLegalHoldStatus;
    }

    /**
     * <p>
     * The retention mode to be applied to all objects in the Batch Operations job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #objectLockMode}
     * will return {@link S3ObjectLockMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #objectLockModeAsString}.
     * </p>
     * 
     * @return The retention mode to be applied to all objects in the Batch Operations job.
     * @see S3ObjectLockMode
     */
    public final S3ObjectLockMode objectLockMode() {
        return S3ObjectLockMode.fromValue(objectLockMode);
    }

    /**
     * <p>
     * The retention mode to be applied to all objects in the Batch Operations job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #objectLockMode}
     * will return {@link S3ObjectLockMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #objectLockModeAsString}.
     * </p>
     * 
     * @return The retention mode to be applied to all objects in the Batch Operations job.
     * @see S3ObjectLockMode
     */
    public final String objectLockModeAsString() {
        return objectLockMode;
    }

    /**
     * <p>
     * The date when the applied object retention configuration expires on all objects in the Batch Operations job.
     * </p>
     * 
     * @return The date when the applied object retention configuration expires on all objects in the Batch Operations
     *         job.
     */
    public final Instant objectLockRetainUntilDate() {
        return objectLockRetainUntilDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(targetResource());
        hashCode = 31 * hashCode + Objects.hashCode(cannedAccessControlListAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessControlGrants() ? accessControlGrants() : null);
        hashCode = 31 * hashCode + Objects.hashCode(metadataDirectiveAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedSinceConstraint());
        hashCode = 31 * hashCode + Objects.hashCode(newObjectMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(hasNewObjectTagging() ? newObjectTagging() : null);
        hashCode = 31 * hashCode + Objects.hashCode(redirectLocation());
        hashCode = 31 * hashCode + Objects.hashCode(requesterPays());
        hashCode = 31 * hashCode + Objects.hashCode(storageClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(unModifiedSinceConstraint());
        hashCode = 31 * hashCode + Objects.hashCode(sseAwsKmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(targetKeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(objectLockLegalHoldStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(objectLockModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(objectLockRetainUntilDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3CopyObjectOperation)) {
            return false;
        }
        S3CopyObjectOperation other = (S3CopyObjectOperation) obj;
        return Objects.equals(targetResource(), other.targetResource())
                && Objects.equals(cannedAccessControlListAsString(), other.cannedAccessControlListAsString())
                && hasAccessControlGrants() == other.hasAccessControlGrants()
                && Objects.equals(accessControlGrants(), other.accessControlGrants())
                && Objects.equals(metadataDirectiveAsString(), other.metadataDirectiveAsString())
                && Objects.equals(modifiedSinceConstraint(), other.modifiedSinceConstraint())
                && Objects.equals(newObjectMetadata(), other.newObjectMetadata())
                && hasNewObjectTagging() == other.hasNewObjectTagging()
                && Objects.equals(newObjectTagging(), other.newObjectTagging())
                && Objects.equals(redirectLocation(), other.redirectLocation())
                && Objects.equals(requesterPays(), other.requesterPays())
                && Objects.equals(storageClassAsString(), other.storageClassAsString())
                && Objects.equals(unModifiedSinceConstraint(), other.unModifiedSinceConstraint())
                && Objects.equals(sseAwsKmsKeyId(), other.sseAwsKmsKeyId())
                && Objects.equals(targetKeyPrefix(), other.targetKeyPrefix())
                && Objects.equals(objectLockLegalHoldStatusAsString(), other.objectLockLegalHoldStatusAsString())
                && Objects.equals(objectLockModeAsString(), other.objectLockModeAsString())
                && Objects.equals(objectLockRetainUntilDate(), other.objectLockRetainUntilDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3CopyObjectOperation").add("TargetResource", targetResource())
                .add("CannedAccessControlList", cannedAccessControlListAsString())
                .add("AccessControlGrants", hasAccessControlGrants() ? accessControlGrants() : null)
                .add("MetadataDirective", metadataDirectiveAsString()).add("ModifiedSinceConstraint", modifiedSinceConstraint())
                .add("NewObjectMetadata", newObjectMetadata())
                .add("NewObjectTagging", hasNewObjectTagging() ? newObjectTagging() : null)
                .add("RedirectLocation", redirectLocation()).add("RequesterPays", requesterPays())
                .add("StorageClass", storageClassAsString()).add("UnModifiedSinceConstraint", unModifiedSinceConstraint())
                .add("SSEAwsKmsKeyId", sseAwsKmsKeyId()).add("TargetKeyPrefix", targetKeyPrefix())
                .add("ObjectLockLegalHoldStatus", objectLockLegalHoldStatusAsString())
                .add("ObjectLockMode", objectLockModeAsString()).add("ObjectLockRetainUntilDate", objectLockRetainUntilDate())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TargetResource":
            return Optional.ofNullable(clazz.cast(targetResource()));
        case "CannedAccessControlList":
            return Optional.ofNullable(clazz.cast(cannedAccessControlListAsString()));
        case "AccessControlGrants":
            return Optional.ofNullable(clazz.cast(accessControlGrants()));
        case "MetadataDirective":
            return Optional.ofNullable(clazz.cast(metadataDirectiveAsString()));
        case "ModifiedSinceConstraint":
            return Optional.ofNullable(clazz.cast(modifiedSinceConstraint()));
        case "NewObjectMetadata":
            return Optional.ofNullable(clazz.cast(newObjectMetadata()));
        case "NewObjectTagging":
            return Optional.ofNullable(clazz.cast(newObjectTagging()));
        case "RedirectLocation":
            return Optional.ofNullable(clazz.cast(redirectLocation()));
        case "RequesterPays":
            return Optional.ofNullable(clazz.cast(requesterPays()));
        case "StorageClass":
            return Optional.ofNullable(clazz.cast(storageClassAsString()));
        case "UnModifiedSinceConstraint":
            return Optional.ofNullable(clazz.cast(unModifiedSinceConstraint()));
        case "SSEAwsKmsKeyId":
            return Optional.ofNullable(clazz.cast(sseAwsKmsKeyId()));
        case "TargetKeyPrefix":
            return Optional.ofNullable(clazz.cast(targetKeyPrefix()));
        case "ObjectLockLegalHoldStatus":
            return Optional.ofNullable(clazz.cast(objectLockLegalHoldStatusAsString()));
        case "ObjectLockMode":
            return Optional.ofNullable(clazz.cast(objectLockModeAsString()));
        case "ObjectLockRetainUntilDate":
            return Optional.ofNullable(clazz.cast(objectLockRetainUntilDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3CopyObjectOperation, T> g) {
        return obj -> g.apply((S3CopyObjectOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3CopyObjectOperation> {
        /**
         * <p>
         * Specifies the destination bucket ARN for the batch copy operation. For example, to copy objects to a bucket
         * named "destinationBucket", set the TargetResource to "arn:aws:s3:::destinationBucket".
         * </p>
         * 
         * @param targetResource
         *        Specifies the destination bucket ARN for the batch copy operation. For example, to copy objects to a
         *        bucket named "destinationBucket", set the TargetResource to "arn:aws:s3:::destinationBucket".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetResource(String targetResource);

        /**
         * <p/>
         * 
         * @param cannedAccessControlList
         * @see S3CannedAccessControlList
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3CannedAccessControlList
         */
        Builder cannedAccessControlList(String cannedAccessControlList);

        /**
         * <p/>
         * 
         * @param cannedAccessControlList
         * @see S3CannedAccessControlList
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3CannedAccessControlList
         */
        Builder cannedAccessControlList(S3CannedAccessControlList cannedAccessControlList);

        /**
         * <p/>
         * 
         * @param accessControlGrants
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlGrants(Collection<S3Grant> accessControlGrants);

        /**
         * <p/>
         * 
         * @param accessControlGrants
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlGrants(S3Grant... accessControlGrants);

        /**
         * <p/>
         * This is a convenience that creates an instance of the {@link List<S3Grant>.Builder} avoiding the need to
         * create one manually via {@link List<S3Grant>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3Grant>.Builder#build()} is called immediately and its
         * result is passed to {@link #accessControlGrants(List<S3Grant>)}.
         * 
         * @param accessControlGrants
         *        a consumer that will call methods on {@link List<S3Grant>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlGrants(List<S3Grant>)
         */
        Builder accessControlGrants(Consumer<S3Grant.Builder>... accessControlGrants);

        /**
         * <p/>
         * 
         * @param metadataDirective
         * @see S3MetadataDirective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3MetadataDirective
         */
        Builder metadataDirective(String metadataDirective);

        /**
         * <p/>
         * 
         * @param metadataDirective
         * @see S3MetadataDirective
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3MetadataDirective
         */
        Builder metadataDirective(S3MetadataDirective metadataDirective);

        /**
         * <p/>
         * 
         * @param modifiedSinceConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedSinceConstraint(Instant modifiedSinceConstraint);

        /**
         * <p/>
         * 
         * @param newObjectMetadata
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newObjectMetadata(S3ObjectMetadata newObjectMetadata);

        /**
         * <p/>
         * This is a convenience that creates an instance of the {@link S3ObjectMetadata.Builder} avoiding the need to
         * create one manually via {@link S3ObjectMetadata#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3ObjectMetadata.Builder#build()} is called immediately and its
         * result is passed to {@link #newObjectMetadata(S3ObjectMetadata)}.
         * 
         * @param newObjectMetadata
         *        a consumer that will call methods on {@link S3ObjectMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #newObjectMetadata(S3ObjectMetadata)
         */
        default Builder newObjectMetadata(Consumer<S3ObjectMetadata.Builder> newObjectMetadata) {
            return newObjectMetadata(S3ObjectMetadata.builder().applyMutation(newObjectMetadata).build());
        }

        /**
         * <p/>
         * 
         * @param newObjectTagging
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newObjectTagging(Collection<S3Tag> newObjectTagging);

        /**
         * <p/>
         * 
         * @param newObjectTagging
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newObjectTagging(S3Tag... newObjectTagging);

        /**
         * <p/>
         * This is a convenience that creates an instance of the {@link List<S3Tag>.Builder} avoiding the need to create
         * one manually via {@link List<S3Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #newObjectTagging(List<S3Tag>)}.
         * 
         * @param newObjectTagging
         *        a consumer that will call methods on {@link List<S3Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #newObjectTagging(List<S3Tag>)
         */
        Builder newObjectTagging(Consumer<S3Tag.Builder>... newObjectTagging);

        /**
         * <p>
         * Specifies an optional metadata property for website redirects, <code>x-amz-website-redirect-location</code>.
         * Allows webpage redirects if the object is accessed through a website endpoint.
         * </p>
         * 
         * @param redirectLocation
         *        Specifies an optional metadata property for website redirects,
         *        <code>x-amz-website-redirect-location</code>. Allows webpage redirects if the object is accessed
         *        through a website endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redirectLocation(String redirectLocation);

        /**
         * <p/>
         * 
         * @param requesterPays
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requesterPays(Boolean requesterPays);

        /**
         * <p/>
         * 
         * @param storageClass
         * @see S3StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3StorageClass
         */
        Builder storageClass(String storageClass);

        /**
         * <p/>
         * 
         * @param storageClass
         * @see S3StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3StorageClass
         */
        Builder storageClass(S3StorageClass storageClass);

        /**
         * <p/>
         * 
         * @param unModifiedSinceConstraint
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unModifiedSinceConstraint(Instant unModifiedSinceConstraint);

        /**
         * <p/>
         * 
         * @param sseAwsKmsKeyId
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sseAwsKmsKeyId(String sseAwsKmsKeyId);

        /**
         * <p>
         * Specifies the folder prefix into which you would like the objects to be copied. For example, to copy objects
         * into a folder named "Folder1" in the destination bucket, set the TargetKeyPrefix to "Folder1/".
         * </p>
         * 
         * @param targetKeyPrefix
         *        Specifies the folder prefix into which you would like the objects to be copied. For example, to copy
         *        objects into a folder named "Folder1" in the destination bucket, set the TargetKeyPrefix to
         *        "Folder1/".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetKeyPrefix(String targetKeyPrefix);

        /**
         * <p>
         * The legal hold status to be applied to all objects in the Batch Operations job.
         * </p>
         * 
         * @param objectLockLegalHoldStatus
         *        The legal hold status to be applied to all objects in the Batch Operations job.
         * @see S3ObjectLockLegalHoldStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectLockLegalHoldStatus
         */
        Builder objectLockLegalHoldStatus(String objectLockLegalHoldStatus);

        /**
         * <p>
         * The legal hold status to be applied to all objects in the Batch Operations job.
         * </p>
         * 
         * @param objectLockLegalHoldStatus
         *        The legal hold status to be applied to all objects in the Batch Operations job.
         * @see S3ObjectLockLegalHoldStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectLockLegalHoldStatus
         */
        Builder objectLockLegalHoldStatus(S3ObjectLockLegalHoldStatus objectLockLegalHoldStatus);

        /**
         * <p>
         * The retention mode to be applied to all objects in the Batch Operations job.
         * </p>
         * 
         * @param objectLockMode
         *        The retention mode to be applied to all objects in the Batch Operations job.
         * @see S3ObjectLockMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectLockMode
         */
        Builder objectLockMode(String objectLockMode);

        /**
         * <p>
         * The retention mode to be applied to all objects in the Batch Operations job.
         * </p>
         * 
         * @param objectLockMode
         *        The retention mode to be applied to all objects in the Batch Operations job.
         * @see S3ObjectLockMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3ObjectLockMode
         */
        Builder objectLockMode(S3ObjectLockMode objectLockMode);

        /**
         * <p>
         * The date when the applied object retention configuration expires on all objects in the Batch Operations job.
         * </p>
         * 
         * @param objectLockRetainUntilDate
         *        The date when the applied object retention configuration expires on all objects in the Batch
         *        Operations job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectLockRetainUntilDate(Instant objectLockRetainUntilDate);
    }

    static final class BuilderImpl implements Builder {
        private String targetResource;

        private String cannedAccessControlList;

        private List<S3Grant> accessControlGrants = DefaultSdkAutoConstructList.getInstance();

        private String metadataDirective;

        private Instant modifiedSinceConstraint;

        private S3ObjectMetadata newObjectMetadata;

        private List<S3Tag> newObjectTagging = DefaultSdkAutoConstructList.getInstance();

        private String redirectLocation;

        private Boolean requesterPays;

        private String storageClass;

        private Instant unModifiedSinceConstraint;

        private String sseAwsKmsKeyId;

        private String targetKeyPrefix;

        private String objectLockLegalHoldStatus;

        private String objectLockMode;

        private Instant objectLockRetainUntilDate;

        private BuilderImpl() {
        }

        private BuilderImpl(S3CopyObjectOperation model) {
            targetResource(model.targetResource);
            cannedAccessControlList(model.cannedAccessControlList);
            accessControlGrants(model.accessControlGrants);
            metadataDirective(model.metadataDirective);
            modifiedSinceConstraint(model.modifiedSinceConstraint);
            newObjectMetadata(model.newObjectMetadata);
            newObjectTagging(model.newObjectTagging);
            redirectLocation(model.redirectLocation);
            requesterPays(model.requesterPays);
            storageClass(model.storageClass);
            unModifiedSinceConstraint(model.unModifiedSinceConstraint);
            sseAwsKmsKeyId(model.sseAwsKmsKeyId);
            targetKeyPrefix(model.targetKeyPrefix);
            objectLockLegalHoldStatus(model.objectLockLegalHoldStatus);
            objectLockMode(model.objectLockMode);
            objectLockRetainUntilDate(model.objectLockRetainUntilDate);
        }

        public final String getTargetResource() {
            return targetResource;
        }

        @Override
        public final Builder targetResource(String targetResource) {
            this.targetResource = targetResource;
            return this;
        }

        public final void setTargetResource(String targetResource) {
            this.targetResource = targetResource;
        }

        public final String getCannedAccessControlList() {
            return cannedAccessControlList;
        }

        @Override
        public final Builder cannedAccessControlList(String cannedAccessControlList) {
            this.cannedAccessControlList = cannedAccessControlList;
            return this;
        }

        @Override
        public final Builder cannedAccessControlList(S3CannedAccessControlList cannedAccessControlList) {
            this.cannedAccessControlList(cannedAccessControlList == null ? null : cannedAccessControlList.toString());
            return this;
        }

        public final void setCannedAccessControlList(String cannedAccessControlList) {
            this.cannedAccessControlList = cannedAccessControlList;
        }

        public final List<S3Grant.Builder> getAccessControlGrants() {
            List<S3Grant.Builder> result = S3GrantListCopier.copyToBuilder(this.accessControlGrants);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder accessControlGrants(Collection<S3Grant> accessControlGrants) {
            this.accessControlGrants = S3GrantListCopier.copy(accessControlGrants);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessControlGrants(S3Grant... accessControlGrants) {
            accessControlGrants(Arrays.asList(accessControlGrants));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessControlGrants(Consumer<S3Grant.Builder>... accessControlGrants) {
            accessControlGrants(Stream.of(accessControlGrants).map(c -> S3Grant.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAccessControlGrants(Collection<S3Grant.BuilderImpl> accessControlGrants) {
            this.accessControlGrants = S3GrantListCopier.copyFromBuilder(accessControlGrants);
        }

        public final String getMetadataDirective() {
            return metadataDirective;
        }

        @Override
        public final Builder metadataDirective(String metadataDirective) {
            this.metadataDirective = metadataDirective;
            return this;
        }

        @Override
        public final Builder metadataDirective(S3MetadataDirective metadataDirective) {
            this.metadataDirective(metadataDirective == null ? null : metadataDirective.toString());
            return this;
        }

        public final void setMetadataDirective(String metadataDirective) {
            this.metadataDirective = metadataDirective;
        }

        public final Instant getModifiedSinceConstraint() {
            return modifiedSinceConstraint;
        }

        @Override
        public final Builder modifiedSinceConstraint(Instant modifiedSinceConstraint) {
            this.modifiedSinceConstraint = modifiedSinceConstraint;
            return this;
        }

        public final void setModifiedSinceConstraint(Instant modifiedSinceConstraint) {
            this.modifiedSinceConstraint = modifiedSinceConstraint;
        }

        public final S3ObjectMetadata.Builder getNewObjectMetadata() {
            return newObjectMetadata != null ? newObjectMetadata.toBuilder() : null;
        }

        @Override
        public final Builder newObjectMetadata(S3ObjectMetadata newObjectMetadata) {
            this.newObjectMetadata = newObjectMetadata;
            return this;
        }

        public final void setNewObjectMetadata(S3ObjectMetadata.BuilderImpl newObjectMetadata) {
            this.newObjectMetadata = newObjectMetadata != null ? newObjectMetadata.build() : null;
        }

        public final List<S3Tag.Builder> getNewObjectTagging() {
            List<S3Tag.Builder> result = S3TagSetCopier.copyToBuilder(this.newObjectTagging);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder newObjectTagging(Collection<S3Tag> newObjectTagging) {
            this.newObjectTagging = S3TagSetCopier.copy(newObjectTagging);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder newObjectTagging(S3Tag... newObjectTagging) {
            newObjectTagging(Arrays.asList(newObjectTagging));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder newObjectTagging(Consumer<S3Tag.Builder>... newObjectTagging) {
            newObjectTagging(Stream.of(newObjectTagging).map(c -> S3Tag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNewObjectTagging(Collection<S3Tag.BuilderImpl> newObjectTagging) {
            this.newObjectTagging = S3TagSetCopier.copyFromBuilder(newObjectTagging);
        }

        public final String getRedirectLocation() {
            return redirectLocation;
        }

        @Override
        public final Builder redirectLocation(String redirectLocation) {
            this.redirectLocation = redirectLocation;
            return this;
        }

        public final void setRedirectLocation(String redirectLocation) {
            this.redirectLocation = redirectLocation;
        }

        public final Boolean getRequesterPays() {
            return requesterPays;
        }

        @Override
        public final Builder requesterPays(Boolean requesterPays) {
            this.requesterPays = requesterPays;
            return this;
        }

        public final void setRequesterPays(Boolean requesterPays) {
            this.requesterPays = requesterPays;
        }

        public final String getStorageClass() {
            return storageClass;
        }

        @Override
        public final Builder storageClass(String storageClass) {
            this.storageClass = storageClass;
            return this;
        }

        @Override
        public final Builder storageClass(S3StorageClass storageClass) {
            this.storageClass(storageClass == null ? null : storageClass.toString());
            return this;
        }

        public final void setStorageClass(String storageClass) {
            this.storageClass = storageClass;
        }

        public final Instant getUnModifiedSinceConstraint() {
            return unModifiedSinceConstraint;
        }

        @Override
        public final Builder unModifiedSinceConstraint(Instant unModifiedSinceConstraint) {
            this.unModifiedSinceConstraint = unModifiedSinceConstraint;
            return this;
        }

        public final void setUnModifiedSinceConstraint(Instant unModifiedSinceConstraint) {
            this.unModifiedSinceConstraint = unModifiedSinceConstraint;
        }

        public final String getSseAwsKmsKeyId() {
            return sseAwsKmsKeyId;
        }

        @Override
        public final Builder sseAwsKmsKeyId(String sseAwsKmsKeyId) {
            this.sseAwsKmsKeyId = sseAwsKmsKeyId;
            return this;
        }

        public final void setSseAwsKmsKeyId(String sseAwsKmsKeyId) {
            this.sseAwsKmsKeyId = sseAwsKmsKeyId;
        }

        public final String getTargetKeyPrefix() {
            return targetKeyPrefix;
        }

        @Override
        public final Builder targetKeyPrefix(String targetKeyPrefix) {
            this.targetKeyPrefix = targetKeyPrefix;
            return this;
        }

        public final void setTargetKeyPrefix(String targetKeyPrefix) {
            this.targetKeyPrefix = targetKeyPrefix;
        }

        public final String getObjectLockLegalHoldStatus() {
            return objectLockLegalHoldStatus;
        }

        @Override
        public final Builder objectLockLegalHoldStatus(String objectLockLegalHoldStatus) {
            this.objectLockLegalHoldStatus = objectLockLegalHoldStatus;
            return this;
        }

        @Override
        public final Builder objectLockLegalHoldStatus(S3ObjectLockLegalHoldStatus objectLockLegalHoldStatus) {
            this.objectLockLegalHoldStatus(objectLockLegalHoldStatus == null ? null : objectLockLegalHoldStatus.toString());
            return this;
        }

        public final void setObjectLockLegalHoldStatus(String objectLockLegalHoldStatus) {
            this.objectLockLegalHoldStatus = objectLockLegalHoldStatus;
        }

        public final String getObjectLockMode() {
            return objectLockMode;
        }

        @Override
        public final Builder objectLockMode(String objectLockMode) {
            this.objectLockMode = objectLockMode;
            return this;
        }

        @Override
        public final Builder objectLockMode(S3ObjectLockMode objectLockMode) {
            this.objectLockMode(objectLockMode == null ? null : objectLockMode.toString());
            return this;
        }

        public final void setObjectLockMode(String objectLockMode) {
            this.objectLockMode = objectLockMode;
        }

        public final Instant getObjectLockRetainUntilDate() {
            return objectLockRetainUntilDate;
        }

        @Override
        public final Builder objectLockRetainUntilDate(Instant objectLockRetainUntilDate) {
            this.objectLockRetainUntilDate = objectLockRetainUntilDate;
            return this;
        }

        public final void setObjectLockRetainUntilDate(Instant objectLockRetainUntilDate) {
            this.objectLockRetainUntilDate = objectLockRetainUntilDate;
        }

        @Override
        public S3CopyObjectOperation build() {
            return new S3CopyObjectOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
