/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container for the information about an asynchronous operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AsyncOperation implements SdkPojo, Serializable, ToCopyableBuilder<AsyncOperation.Builder, AsyncOperation> {
    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime")
            .getter(getter(AsyncOperation::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime")
                    .unmarshallLocationName("CreationTime").build()).build();

    private static final SdkField<String> OPERATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Operation")
            .getter(getter(AsyncOperation::operationAsString))
            .setter(setter(Builder::operation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation")
                    .unmarshallLocationName("Operation").build()).build();

    private static final SdkField<String> REQUEST_TOKEN_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RequestTokenARN")
            .getter(getter(AsyncOperation::requestTokenARN))
            .setter(setter(Builder::requestTokenARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestTokenARN")
                    .unmarshallLocationName("RequestTokenARN").build()).build();

    private static final SdkField<AsyncRequestParameters> REQUEST_PARAMETERS_FIELD = SdkField
            .<AsyncRequestParameters> builder(MarshallingType.SDK_POJO)
            .memberName("RequestParameters")
            .getter(getter(AsyncOperation::requestParameters))
            .setter(setter(Builder::requestParameters))
            .constructor(AsyncRequestParameters::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestParameters")
                    .unmarshallLocationName("RequestParameters").build()).build();

    private static final SdkField<String> REQUEST_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RequestStatus")
            .getter(getter(AsyncOperation::requestStatus))
            .setter(setter(Builder::requestStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestStatus")
                    .unmarshallLocationName("RequestStatus").build()).build();

    private static final SdkField<AsyncResponseDetails> RESPONSE_DETAILS_FIELD = SdkField
            .<AsyncResponseDetails> builder(MarshallingType.SDK_POJO)
            .memberName("ResponseDetails")
            .getter(getter(AsyncOperation::responseDetails))
            .setter(setter(Builder::responseDetails))
            .constructor(AsyncResponseDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseDetails")
                    .unmarshallLocationName("ResponseDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATION_TIME_FIELD,
            OPERATION_FIELD, REQUEST_TOKEN_ARN_FIELD, REQUEST_PARAMETERS_FIELD, REQUEST_STATUS_FIELD, RESPONSE_DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant creationTime;

    private final String operation;

    private final String requestTokenARN;

    private final AsyncRequestParameters requestParameters;

    private final String requestStatus;

    private final AsyncResponseDetails responseDetails;

    private AsyncOperation(BuilderImpl builder) {
        this.creationTime = builder.creationTime;
        this.operation = builder.operation;
        this.requestTokenARN = builder.requestTokenARN;
        this.requestParameters = builder.requestParameters;
        this.requestStatus = builder.requestStatus;
        this.responseDetails = builder.responseDetails;
    }

    /**
     * <p>
     * The time that the request was sent to the service.
     * </p>
     * 
     * @return The time that the request was sent to the service.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The specific operation for the asynchronous request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link AsyncOperationName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The specific operation for the asynchronous request.
     * @see AsyncOperationName
     */
    public final AsyncOperationName operation() {
        return AsyncOperationName.fromValue(operation);
    }

    /**
     * <p>
     * The specific operation for the asynchronous request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link AsyncOperationName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The specific operation for the asynchronous request.
     * @see AsyncOperationName
     */
    public final String operationAsString() {
        return operation;
    }

    /**
     * <p>
     * The request token associated with the request.
     * </p>
     * 
     * @return The request token associated with the request.
     */
    public final String requestTokenARN() {
        return requestTokenARN;
    }

    /**
     * <p>
     * The parameters associated with the request.
     * </p>
     * 
     * @return The parameters associated with the request.
     */
    public final AsyncRequestParameters requestParameters() {
        return requestParameters;
    }

    /**
     * <p>
     * The current status of the request.
     * </p>
     * 
     * @return The current status of the request.
     */
    public final String requestStatus() {
        return requestStatus;
    }

    /**
     * <p>
     * The details of the response.
     * </p>
     * 
     * @return The details of the response.
     */
    public final AsyncResponseDetails responseDetails() {
        return responseDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(operationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(requestTokenARN());
        hashCode = 31 * hashCode + Objects.hashCode(requestParameters());
        hashCode = 31 * hashCode + Objects.hashCode(requestStatus());
        hashCode = 31 * hashCode + Objects.hashCode(responseDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AsyncOperation)) {
            return false;
        }
        AsyncOperation other = (AsyncOperation) obj;
        return Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(operationAsString(), other.operationAsString())
                && Objects.equals(requestTokenARN(), other.requestTokenARN())
                && Objects.equals(requestParameters(), other.requestParameters())
                && Objects.equals(requestStatus(), other.requestStatus())
                && Objects.equals(responseDetails(), other.responseDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AsyncOperation").add("CreationTime", creationTime()).add("Operation", operationAsString())
                .add("RequestTokenARN", requestTokenARN()).add("RequestParameters", requestParameters())
                .add("RequestStatus", requestStatus()).add("ResponseDetails", responseDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "Operation":
            return Optional.ofNullable(clazz.cast(operationAsString()));
        case "RequestTokenARN":
            return Optional.ofNullable(clazz.cast(requestTokenARN()));
        case "RequestParameters":
            return Optional.ofNullable(clazz.cast(requestParameters()));
        case "RequestStatus":
            return Optional.ofNullable(clazz.cast(requestStatus()));
        case "ResponseDetails":
            return Optional.ofNullable(clazz.cast(responseDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AsyncOperation, T> g) {
        return obj -> g.apply((AsyncOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AsyncOperation> {
        /**
         * <p>
         * The time that the request was sent to the service.
         * </p>
         * 
         * @param creationTime
         *        The time that the request was sent to the service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The specific operation for the asynchronous request.
         * </p>
         * 
         * @param operation
         *        The specific operation for the asynchronous request.
         * @see AsyncOperationName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AsyncOperationName
         */
        Builder operation(String operation);

        /**
         * <p>
         * The specific operation for the asynchronous request.
         * </p>
         * 
         * @param operation
         *        The specific operation for the asynchronous request.
         * @see AsyncOperationName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AsyncOperationName
         */
        Builder operation(AsyncOperationName operation);

        /**
         * <p>
         * The request token associated with the request.
         * </p>
         * 
         * @param requestTokenARN
         *        The request token associated with the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestTokenARN(String requestTokenARN);

        /**
         * <p>
         * The parameters associated with the request.
         * </p>
         * 
         * @param requestParameters
         *        The parameters associated with the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestParameters(AsyncRequestParameters requestParameters);

        /**
         * <p>
         * The parameters associated with the request.
         * </p>
         * This is a convenience method that creates an instance of the {@link AsyncRequestParameters.Builder} avoiding
         * the need to create one manually via {@link AsyncRequestParameters#builder()}.
         *
         * When the {@link Consumer} completes, {@link AsyncRequestParameters.Builder#build()} is called immediately and
         * its result is passed to {@link #requestParameters(AsyncRequestParameters)}.
         * 
         * @param requestParameters
         *        a consumer that will call methods on {@link AsyncRequestParameters.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #requestParameters(AsyncRequestParameters)
         */
        default Builder requestParameters(Consumer<AsyncRequestParameters.Builder> requestParameters) {
            return requestParameters(AsyncRequestParameters.builder().applyMutation(requestParameters).build());
        }

        /**
         * <p>
         * The current status of the request.
         * </p>
         * 
         * @param requestStatus
         *        The current status of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestStatus(String requestStatus);

        /**
         * <p>
         * The details of the response.
         * </p>
         * 
         * @param responseDetails
         *        The details of the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseDetails(AsyncResponseDetails responseDetails);

        /**
         * <p>
         * The details of the response.
         * </p>
         * This is a convenience method that creates an instance of the {@link AsyncResponseDetails.Builder} avoiding
         * the need to create one manually via {@link AsyncResponseDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link AsyncResponseDetails.Builder#build()} is called immediately and
         * its result is passed to {@link #responseDetails(AsyncResponseDetails)}.
         * 
         * @param responseDetails
         *        a consumer that will call methods on {@link AsyncResponseDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #responseDetails(AsyncResponseDetails)
         */
        default Builder responseDetails(Consumer<AsyncResponseDetails.Builder> responseDetails) {
            return responseDetails(AsyncResponseDetails.builder().applyMutation(responseDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Instant creationTime;

        private String operation;

        private String requestTokenARN;

        private AsyncRequestParameters requestParameters;

        private String requestStatus;

        private AsyncResponseDetails responseDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(AsyncOperation model) {
            creationTime(model.creationTime);
            operation(model.operation);
            requestTokenARN(model.requestTokenARN);
            requestParameters(model.requestParameters);
            requestStatus(model.requestStatus);
            responseDetails(model.responseDetails);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getOperation() {
            return operation;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        @Override
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        @Override
        public final Builder operation(AsyncOperationName operation) {
            this.operation(operation == null ? null : operation.toString());
            return this;
        }

        public final String getRequestTokenARN() {
            return requestTokenARN;
        }

        public final void setRequestTokenARN(String requestTokenARN) {
            this.requestTokenARN = requestTokenARN;
        }

        @Override
        public final Builder requestTokenARN(String requestTokenARN) {
            this.requestTokenARN = requestTokenARN;
            return this;
        }

        public final AsyncRequestParameters.Builder getRequestParameters() {
            return requestParameters != null ? requestParameters.toBuilder() : null;
        }

        public final void setRequestParameters(AsyncRequestParameters.BuilderImpl requestParameters) {
            this.requestParameters = requestParameters != null ? requestParameters.build() : null;
        }

        @Override
        public final Builder requestParameters(AsyncRequestParameters requestParameters) {
            this.requestParameters = requestParameters;
            return this;
        }

        public final String getRequestStatus() {
            return requestStatus;
        }

        public final void setRequestStatus(String requestStatus) {
            this.requestStatus = requestStatus;
        }

        @Override
        public final Builder requestStatus(String requestStatus) {
            this.requestStatus = requestStatus;
            return this;
        }

        public final AsyncResponseDetails.Builder getResponseDetails() {
            return responseDetails != null ? responseDetails.toBuilder() : null;
        }

        public final void setResponseDetails(AsyncResponseDetails.BuilderImpl responseDetails) {
            this.responseDetails = responseDetails != null ? responseDetails.build() : null;
        }

        @Override
        public final Builder responseDetails(AsyncResponseDetails responseDetails) {
            this.responseDetails = responseDetails;
            return this;
        }

        @Override
        public AsyncOperation build() {
            return new AsyncOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
