/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetAccessPointResponse extends S3ControlResponse implements
        ToCopyableBuilder<GetAccessPointResponse.Builder, GetAccessPointResponse> {
    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(GetAccessPointResponse::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(GetAccessPointResponse::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<String> NETWORK_ORIGIN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkOrigin")
            .getter(getter(GetAccessPointResponse::networkOriginAsString))
            .setter(setter(Builder::networkOrigin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkOrigin")
                    .unmarshallLocationName("NetworkOrigin").build()).build();

    private static final SdkField<VpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<VpcConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("VpcConfiguration")
            .getter(getter(GetAccessPointResponse::vpcConfiguration))
            .setter(setter(Builder::vpcConfiguration))
            .constructor(VpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration")
                    .unmarshallLocationName("VpcConfiguration").build()).build();

    private static final SdkField<PublicAccessBlockConfiguration> PUBLIC_ACCESS_BLOCK_CONFIGURATION_FIELD = SdkField
            .<PublicAccessBlockConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("PublicAccessBlockConfiguration")
            .getter(getter(GetAccessPointResponse::publicAccessBlockConfiguration))
            .setter(setter(Builder::publicAccessBlockConfiguration))
            .constructor(PublicAccessBlockConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicAccessBlockConfiguration")
                    .unmarshallLocationName("PublicAccessBlockConfiguration").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate")
            .getter(getter(GetAccessPointResponse::creationDate))
            .setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate")
                    .unmarshallLocationName("CreationDate").build()).build();

    private static final SdkField<String> ALIAS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Alias")
            .getter(getter(GetAccessPointResponse::alias))
            .setter(setter(Builder::alias))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Alias")
                    .unmarshallLocationName("Alias").build()).build();

    private static final SdkField<String> ACCESS_POINT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccessPointArn")
            .getter(getter(GetAccessPointResponse::accessPointArn))
            .setter(setter(Builder::accessPointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessPointArn")
                    .unmarshallLocationName("AccessPointArn").build()).build();

    private static final SdkField<Map<String, String>> ENDPOINTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Endpoints")
            .getter(getter(GetAccessPointResponse::endpoints))
            .setter(setter(Builder::endpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoints")
                    .unmarshallLocationName("Endpoints").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").unmarshallLocationName("value").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, BUCKET_FIELD,
            NETWORK_ORIGIN_FIELD, VPC_CONFIGURATION_FIELD, PUBLIC_ACCESS_BLOCK_CONFIGURATION_FIELD, CREATION_DATE_FIELD,
            ALIAS_FIELD, ACCESS_POINT_ARN_FIELD, ENDPOINTS_FIELD));

    private final String name;

    private final String bucket;

    private final String networkOrigin;

    private final VpcConfiguration vpcConfiguration;

    private final PublicAccessBlockConfiguration publicAccessBlockConfiguration;

    private final Instant creationDate;

    private final String alias;

    private final String accessPointArn;

    private final Map<String, String> endpoints;

    private GetAccessPointResponse(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.bucket = builder.bucket;
        this.networkOrigin = builder.networkOrigin;
        this.vpcConfiguration = builder.vpcConfiguration;
        this.publicAccessBlockConfiguration = builder.publicAccessBlockConfiguration;
        this.creationDate = builder.creationDate;
        this.alias = builder.alias;
        this.accessPointArn = builder.accessPointArn;
        this.endpoints = builder.endpoints;
    }

    /**
     * <p>
     * The name of the specified access point.
     * </p>
     * 
     * @return The name of the specified access point.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The name of the bucket associated with the specified access point.
     * </p>
     * 
     * @return The name of the bucket associated with the specified access point.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * Indicates whether this access point allows access from the public internet. If <code>VpcConfiguration</code> is
     * specified for this access point, then <code>NetworkOrigin</code> is <code>VPC</code>, and the access point
     * doesn't allow access from the public internet. Otherwise, <code>NetworkOrigin</code> is <code>Internet</code>,
     * and the access point allows access from the public internet, subject to the access point and bucket access
     * policies.
     * </p>
     * <p>
     * This will always be true for an Amazon S3 on Outposts access point
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkOrigin}
     * will return {@link NetworkOrigin#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkOriginAsString}.
     * </p>
     * 
     * @return Indicates whether this access point allows access from the public internet. If
     *         <code>VpcConfiguration</code> is specified for this access point, then <code>NetworkOrigin</code> is
     *         <code>VPC</code>, and the access point doesn't allow access from the public internet. Otherwise,
     *         <code>NetworkOrigin</code> is <code>Internet</code>, and the access point allows access from the public
     *         internet, subject to the access point and bucket access policies.</p>
     *         <p>
     *         This will always be true for an Amazon S3 on Outposts access point
     * @see NetworkOrigin
     */
    public final NetworkOrigin networkOrigin() {
        return NetworkOrigin.fromValue(networkOrigin);
    }

    /**
     * <p>
     * Indicates whether this access point allows access from the public internet. If <code>VpcConfiguration</code> is
     * specified for this access point, then <code>NetworkOrigin</code> is <code>VPC</code>, and the access point
     * doesn't allow access from the public internet. Otherwise, <code>NetworkOrigin</code> is <code>Internet</code>,
     * and the access point allows access from the public internet, subject to the access point and bucket access
     * policies.
     * </p>
     * <p>
     * This will always be true for an Amazon S3 on Outposts access point
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #networkOrigin}
     * will return {@link NetworkOrigin#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #networkOriginAsString}.
     * </p>
     * 
     * @return Indicates whether this access point allows access from the public internet. If
     *         <code>VpcConfiguration</code> is specified for this access point, then <code>NetworkOrigin</code> is
     *         <code>VPC</code>, and the access point doesn't allow access from the public internet. Otherwise,
     *         <code>NetworkOrigin</code> is <code>Internet</code>, and the access point allows access from the public
     *         internet, subject to the access point and bucket access policies.</p>
     *         <p>
     *         This will always be true for an Amazon S3 on Outposts access point
     * @see NetworkOrigin
     */
    public final String networkOriginAsString() {
        return networkOrigin;
    }

    /**
     * <p>
     * Contains the virtual private cloud (VPC) configuration for the specified access point.
     * </p>
     * <note>
     * <p>
     * This element is empty if this access point is an Amazon S3 on Outposts access point that is used by other Amazon
     * Web Services.
     * </p>
     * </note>
     * 
     * @return Contains the virtual private cloud (VPC) configuration for the specified access point.</p> <note>
     *         <p>
     *         This element is empty if this access point is an Amazon S3 on Outposts access point that is used by other
     *         Amazon Web Services.
     *         </p>
     */
    public final VpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    /**
     * Returns the value of the PublicAccessBlockConfiguration property for this object.
     * 
     * @return The value of the PublicAccessBlockConfiguration property for this object.
     */
    public final PublicAccessBlockConfiguration publicAccessBlockConfiguration() {
        return publicAccessBlockConfiguration;
    }

    /**
     * <p>
     * The date and time when the specified access point was created.
     * </p>
     * 
     * @return The date and time when the specified access point was created.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The name or alias of the access point.
     * </p>
     * 
     * @return The name or alias of the access point.
     */
    public final String alias() {
        return alias;
    }

    /**
     * <p>
     * The ARN of the access point.
     * </p>
     * 
     * @return The ARN of the access point.
     */
    public final String accessPointArn() {
        return accessPointArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Endpoints property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEndpoints() {
        return endpoints != null && !(endpoints instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The VPC endpoint for the access point.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndpoints} method.
     * </p>
     * 
     * @return The VPC endpoint for the access point.
     */
    public final Map<String, String> endpoints() {
        return endpoints;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(networkOriginAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccessBlockConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(alias());
        hashCode = 31 * hashCode + Objects.hashCode(accessPointArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasEndpoints() ? endpoints() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetAccessPointResponse)) {
            return false;
        }
        GetAccessPointResponse other = (GetAccessPointResponse) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(bucket(), other.bucket())
                && Objects.equals(networkOriginAsString(), other.networkOriginAsString())
                && Objects.equals(vpcConfiguration(), other.vpcConfiguration())
                && Objects.equals(publicAccessBlockConfiguration(), other.publicAccessBlockConfiguration())
                && Objects.equals(creationDate(), other.creationDate()) && Objects.equals(alias(), other.alias())
                && Objects.equals(accessPointArn(), other.accessPointArn()) && hasEndpoints() == other.hasEndpoints()
                && Objects.equals(endpoints(), other.endpoints());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetAccessPointResponse").add("Name", name()).add("Bucket", bucket())
                .add("NetworkOrigin", networkOriginAsString()).add("VpcConfiguration", vpcConfiguration())
                .add("PublicAccessBlockConfiguration", publicAccessBlockConfiguration()).add("CreationDate", creationDate())
                .add("Alias", alias()).add("AccessPointArn", accessPointArn())
                .add("Endpoints", hasEndpoints() ? endpoints() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "NetworkOrigin":
            return Optional.ofNullable(clazz.cast(networkOriginAsString()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        case "PublicAccessBlockConfiguration":
            return Optional.ofNullable(clazz.cast(publicAccessBlockConfiguration()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "Alias":
            return Optional.ofNullable(clazz.cast(alias()));
        case "AccessPointArn":
            return Optional.ofNullable(clazz.cast(accessPointArn()));
        case "Endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetAccessPointResponse, T> g) {
        return obj -> g.apply((GetAccessPointResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3ControlResponse.Builder, SdkPojo, CopyableBuilder<Builder, GetAccessPointResponse> {
        /**
         * <p>
         * The name of the specified access point.
         * </p>
         * 
         * @param name
         *        The name of the specified access point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the bucket associated with the specified access point.
         * </p>
         * 
         * @param bucket
         *        The name of the bucket associated with the specified access point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * Indicates whether this access point allows access from the public internet. If <code>VpcConfiguration</code>
         * is specified for this access point, then <code>NetworkOrigin</code> is <code>VPC</code>, and the access point
         * doesn't allow access from the public internet. Otherwise, <code>NetworkOrigin</code> is <code>Internet</code>
         * , and the access point allows access from the public internet, subject to the access point and bucket access
         * policies.
         * </p>
         * <p>
         * This will always be true for an Amazon S3 on Outposts access point
         * </p>
         * 
         * @param networkOrigin
         *        Indicates whether this access point allows access from the public internet. If
         *        <code>VpcConfiguration</code> is specified for this access point, then <code>NetworkOrigin</code> is
         *        <code>VPC</code>, and the access point doesn't allow access from the public internet. Otherwise,
         *        <code>NetworkOrigin</code> is <code>Internet</code>, and the access point allows access from the
         *        public internet, subject to the access point and bucket access policies.</p>
         *        <p>
         *        This will always be true for an Amazon S3 on Outposts access point
         * @see NetworkOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkOrigin
         */
        Builder networkOrigin(String networkOrigin);

        /**
         * <p>
         * Indicates whether this access point allows access from the public internet. If <code>VpcConfiguration</code>
         * is specified for this access point, then <code>NetworkOrigin</code> is <code>VPC</code>, and the access point
         * doesn't allow access from the public internet. Otherwise, <code>NetworkOrigin</code> is <code>Internet</code>
         * , and the access point allows access from the public internet, subject to the access point and bucket access
         * policies.
         * </p>
         * <p>
         * This will always be true for an Amazon S3 on Outposts access point
         * </p>
         * 
         * @param networkOrigin
         *        Indicates whether this access point allows access from the public internet. If
         *        <code>VpcConfiguration</code> is specified for this access point, then <code>NetworkOrigin</code> is
         *        <code>VPC</code>, and the access point doesn't allow access from the public internet. Otherwise,
         *        <code>NetworkOrigin</code> is <code>Internet</code>, and the access point allows access from the
         *        public internet, subject to the access point and bucket access policies.</p>
         *        <p>
         *        This will always be true for an Amazon S3 on Outposts access point
         * @see NetworkOrigin
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkOrigin
         */
        Builder networkOrigin(NetworkOrigin networkOrigin);

        /**
         * <p>
         * Contains the virtual private cloud (VPC) configuration for the specified access point.
         * </p>
         * <note>
         * <p>
         * This element is empty if this access point is an Amazon S3 on Outposts access point that is used by other
         * Amazon Web Services.
         * </p>
         * </note>
         * 
         * @param vpcConfiguration
         *        Contains the virtual private cloud (VPC) configuration for the specified access point.</p> <note>
         *        <p>
         *        This element is empty if this access point is an Amazon S3 on Outposts access point that is used by
         *        other Amazon Web Services.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(VpcConfiguration vpcConfiguration);

        /**
         * <p>
         * Contains the virtual private cloud (VPC) configuration for the specified access point.
         * </p>
         * <note>
         * <p>
         * This element is empty if this access point is an Amazon S3 on Outposts access point that is used by other
         * Amazon Web Services.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link VpcConfiguration.Builder}
         * avoiding the need to create one manually via {@link VpcConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link VpcConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #vpcConfiguration(VpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link VpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(VpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<VpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(VpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }

        /**
         * Sets the value of the PublicAccessBlockConfiguration property for this object.
         *
         * @param publicAccessBlockConfiguration
         *        The new value for the PublicAccessBlockConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccessBlockConfiguration(PublicAccessBlockConfiguration publicAccessBlockConfiguration);

        /**
         * Sets the value of the PublicAccessBlockConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link PublicAccessBlockConfiguration.Builder}
         * avoiding the need to create one manually via {@link PublicAccessBlockConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link PublicAccessBlockConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #publicAccessBlockConfiguration(PublicAccessBlockConfiguration)}.
         * 
         * @param publicAccessBlockConfiguration
         *        a consumer that will call methods on {@link PublicAccessBlockConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccessBlockConfiguration(PublicAccessBlockConfiguration)
         */
        default Builder publicAccessBlockConfiguration(
                Consumer<PublicAccessBlockConfiguration.Builder> publicAccessBlockConfiguration) {
            return publicAccessBlockConfiguration(PublicAccessBlockConfiguration.builder()
                    .applyMutation(publicAccessBlockConfiguration).build());
        }

        /**
         * <p>
         * The date and time when the specified access point was created.
         * </p>
         * 
         * @param creationDate
         *        The date and time when the specified access point was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The name or alias of the access point.
         * </p>
         * 
         * @param alias
         *        The name or alias of the access point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alias(String alias);

        /**
         * <p>
         * The ARN of the access point.
         * </p>
         * 
         * @param accessPointArn
         *        The ARN of the access point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessPointArn(String accessPointArn);

        /**
         * <p>
         * The VPC endpoint for the access point.
         * </p>
         * 
         * @param endpoints
         *        The VPC endpoint for the access point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(Map<String, String> endpoints);
    }

    static final class BuilderImpl extends S3ControlResponse.BuilderImpl implements Builder {
        private String name;

        private String bucket;

        private String networkOrigin;

        private VpcConfiguration vpcConfiguration;

        private PublicAccessBlockConfiguration publicAccessBlockConfiguration;

        private Instant creationDate;

        private String alias;

        private String accessPointArn;

        private Map<String, String> endpoints = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetAccessPointResponse model) {
            super(model);
            name(model.name);
            bucket(model.bucket);
            networkOrigin(model.networkOrigin);
            vpcConfiguration(model.vpcConfiguration);
            publicAccessBlockConfiguration(model.publicAccessBlockConfiguration);
            creationDate(model.creationDate);
            alias(model.alias);
            accessPointArn(model.accessPointArn);
            endpoints(model.endpoints);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getNetworkOrigin() {
            return networkOrigin;
        }

        public final void setNetworkOrigin(String networkOrigin) {
            this.networkOrigin = networkOrigin;
        }

        @Override
        public final Builder networkOrigin(String networkOrigin) {
            this.networkOrigin = networkOrigin;
            return this;
        }

        @Override
        public final Builder networkOrigin(NetworkOrigin networkOrigin) {
            this.networkOrigin(networkOrigin == null ? null : networkOrigin.toString());
            return this;
        }

        public final VpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(VpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        public final Builder vpcConfiguration(VpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        public final PublicAccessBlockConfiguration.Builder getPublicAccessBlockConfiguration() {
            return publicAccessBlockConfiguration != null ? publicAccessBlockConfiguration.toBuilder() : null;
        }

        public final void setPublicAccessBlockConfiguration(
                PublicAccessBlockConfiguration.BuilderImpl publicAccessBlockConfiguration) {
            this.publicAccessBlockConfiguration = publicAccessBlockConfiguration != null ? publicAccessBlockConfiguration.build()
                    : null;
        }

        @Override
        public final Builder publicAccessBlockConfiguration(PublicAccessBlockConfiguration publicAccessBlockConfiguration) {
            this.publicAccessBlockConfiguration = publicAccessBlockConfiguration;
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final String getAlias() {
            return alias;
        }

        public final void setAlias(String alias) {
            this.alias = alias;
        }

        @Override
        public final Builder alias(String alias) {
            this.alias = alias;
            return this;
        }

        public final String getAccessPointArn() {
            return accessPointArn;
        }

        public final void setAccessPointArn(String accessPointArn) {
            this.accessPointArn = accessPointArn;
        }

        @Override
        public final Builder accessPointArn(String accessPointArn) {
            this.accessPointArn = accessPointArn;
            return this;
        }

        public final Map<String, String> getEndpoints() {
            if (endpoints instanceof SdkAutoConstructMap) {
                return null;
            }
            return endpoints;
        }

        public final void setEndpoints(Map<String, String> endpoints) {
            this.endpoints = EndpointsCopier.copy(endpoints);
        }

        @Override
        public final Builder endpoints(Map<String, String> endpoints) {
            this.endpoints = EndpointsCopier.copy(endpoints);
            return this;
        }

        @Override
        public GetAccessPointResponse build() {
            return new GetAccessPointResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
