/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Location details for where the generated manifest should be written.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3ManifestOutputLocation implements SdkPojo, Serializable,
        ToCopyableBuilder<S3ManifestOutputLocation.Builder, S3ManifestOutputLocation> {
    private static final SdkField<String> EXPECTED_MANIFEST_BUCKET_OWNER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ExpectedManifestBucketOwner")
            .getter(getter(S3ManifestOutputLocation::expectedManifestBucketOwner))
            .setter(setter(Builder::expectedManifestBucketOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedManifestBucketOwner")
                    .unmarshallLocationName("ExpectedManifestBucketOwner").build()).build();

    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(S3ManifestOutputLocation::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<String> MANIFEST_PREFIX_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ManifestPrefix")
            .getter(getter(S3ManifestOutputLocation::manifestPrefix))
            .setter(setter(Builder::manifestPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestPrefix")
                    .unmarshallLocationName("ManifestPrefix").build()).build();

    private static final SdkField<GeneratedManifestEncryption> MANIFEST_ENCRYPTION_FIELD = SdkField
            .<GeneratedManifestEncryption> builder(MarshallingType.SDK_POJO)
            .memberName("ManifestEncryption")
            .getter(getter(S3ManifestOutputLocation::manifestEncryption))
            .setter(setter(Builder::manifestEncryption))
            .constructor(GeneratedManifestEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestEncryption")
                    .unmarshallLocationName("ManifestEncryption").build()).build();

    private static final SdkField<String> MANIFEST_FORMAT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ManifestFormat")
            .getter(getter(S3ManifestOutputLocation::manifestFormatAsString))
            .setter(setter(Builder::manifestFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestFormat")
                    .unmarshallLocationName("ManifestFormat").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            EXPECTED_MANIFEST_BUCKET_OWNER_FIELD, BUCKET_FIELD, MANIFEST_PREFIX_FIELD, MANIFEST_ENCRYPTION_FIELD,
            MANIFEST_FORMAT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String expectedManifestBucketOwner;

    private final String bucket;

    private final String manifestPrefix;

    private final GeneratedManifestEncryption manifestEncryption;

    private final String manifestFormat;

    private S3ManifestOutputLocation(BuilderImpl builder) {
        this.expectedManifestBucketOwner = builder.expectedManifestBucketOwner;
        this.bucket = builder.bucket;
        this.manifestPrefix = builder.manifestPrefix;
        this.manifestEncryption = builder.manifestEncryption;
        this.manifestFormat = builder.manifestFormat;
    }

    /**
     * <p>
     * The Account ID that owns the bucket the generated manifest is written to.
     * </p>
     * 
     * @return The Account ID that owns the bucket the generated manifest is written to.
     */
    public final String expectedManifestBucketOwner() {
        return expectedManifestBucketOwner;
    }

    /**
     * <p>
     * The bucket ARN the generated manifest should be written to.
     * </p>
     * 
     * @return The bucket ARN the generated manifest should be written to.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * Prefix identifying one or more objects to which the manifest applies.
     * </p>
     * 
     * @return Prefix identifying one or more objects to which the manifest applies.
     */
    public final String manifestPrefix() {
        return manifestPrefix;
    }

    /**
     * <p>
     * Specifies what encryption should be used when the generated manifest objects are written.
     * </p>
     * 
     * @return Specifies what encryption should be used when the generated manifest objects are written.
     */
    public final GeneratedManifestEncryption manifestEncryption() {
        return manifestEncryption;
    }

    /**
     * <p>
     * The format of the generated manifest.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #manifestFormat}
     * will return {@link GeneratedManifestFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #manifestFormatAsString}.
     * </p>
     * 
     * @return The format of the generated manifest.
     * @see GeneratedManifestFormat
     */
    public final GeneratedManifestFormat manifestFormat() {
        return GeneratedManifestFormat.fromValue(manifestFormat);
    }

    /**
     * <p>
     * The format of the generated manifest.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #manifestFormat}
     * will return {@link GeneratedManifestFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #manifestFormatAsString}.
     * </p>
     * 
     * @return The format of the generated manifest.
     * @see GeneratedManifestFormat
     */
    public final String manifestFormatAsString() {
        return manifestFormat;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expectedManifestBucketOwner());
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(manifestPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(manifestEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(manifestFormatAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3ManifestOutputLocation)) {
            return false;
        }
        S3ManifestOutputLocation other = (S3ManifestOutputLocation) obj;
        return Objects.equals(expectedManifestBucketOwner(), other.expectedManifestBucketOwner())
                && Objects.equals(bucket(), other.bucket()) && Objects.equals(manifestPrefix(), other.manifestPrefix())
                && Objects.equals(manifestEncryption(), other.manifestEncryption())
                && Objects.equals(manifestFormatAsString(), other.manifestFormatAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3ManifestOutputLocation").add("ExpectedManifestBucketOwner", expectedManifestBucketOwner())
                .add("Bucket", bucket()).add("ManifestPrefix", manifestPrefix()).add("ManifestEncryption", manifestEncryption())
                .add("ManifestFormat", manifestFormatAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExpectedManifestBucketOwner":
            return Optional.ofNullable(clazz.cast(expectedManifestBucketOwner()));
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "ManifestPrefix":
            return Optional.ofNullable(clazz.cast(manifestPrefix()));
        case "ManifestEncryption":
            return Optional.ofNullable(clazz.cast(manifestEncryption()));
        case "ManifestFormat":
            return Optional.ofNullable(clazz.cast(manifestFormatAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3ManifestOutputLocation, T> g) {
        return obj -> g.apply((S3ManifestOutputLocation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3ManifestOutputLocation> {
        /**
         * <p>
         * The Account ID that owns the bucket the generated manifest is written to.
         * </p>
         * 
         * @param expectedManifestBucketOwner
         *        The Account ID that owns the bucket the generated manifest is written to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedManifestBucketOwner(String expectedManifestBucketOwner);

        /**
         * <p>
         * The bucket ARN the generated manifest should be written to.
         * </p>
         * 
         * @param bucket
         *        The bucket ARN the generated manifest should be written to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * Prefix identifying one or more objects to which the manifest applies.
         * </p>
         * 
         * @param manifestPrefix
         *        Prefix identifying one or more objects to which the manifest applies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestPrefix(String manifestPrefix);

        /**
         * <p>
         * Specifies what encryption should be used when the generated manifest objects are written.
         * </p>
         * 
         * @param manifestEncryption
         *        Specifies what encryption should be used when the generated manifest objects are written.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestEncryption(GeneratedManifestEncryption manifestEncryption);

        /**
         * <p>
         * Specifies what encryption should be used when the generated manifest objects are written.
         * </p>
         * This is a convenience method that creates an instance of the {@link GeneratedManifestEncryption.Builder}
         * avoiding the need to create one manually via {@link GeneratedManifestEncryption#builder()}.
         *
         * When the {@link Consumer} completes, {@link GeneratedManifestEncryption.Builder#build()} is called
         * immediately and its result is passed to {@link #manifestEncryption(GeneratedManifestEncryption)}.
         * 
         * @param manifestEncryption
         *        a consumer that will call methods on {@link GeneratedManifestEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #manifestEncryption(GeneratedManifestEncryption)
         */
        default Builder manifestEncryption(Consumer<GeneratedManifestEncryption.Builder> manifestEncryption) {
            return manifestEncryption(GeneratedManifestEncryption.builder().applyMutation(manifestEncryption).build());
        }

        /**
         * <p>
         * The format of the generated manifest.
         * </p>
         * 
         * @param manifestFormat
         *        The format of the generated manifest.
         * @see GeneratedManifestFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeneratedManifestFormat
         */
        Builder manifestFormat(String manifestFormat);

        /**
         * <p>
         * The format of the generated manifest.
         * </p>
         * 
         * @param manifestFormat
         *        The format of the generated manifest.
         * @see GeneratedManifestFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeneratedManifestFormat
         */
        Builder manifestFormat(GeneratedManifestFormat manifestFormat);
    }

    static final class BuilderImpl implements Builder {
        private String expectedManifestBucketOwner;

        private String bucket;

        private String manifestPrefix;

        private GeneratedManifestEncryption manifestEncryption;

        private String manifestFormat;

        private BuilderImpl() {
        }

        private BuilderImpl(S3ManifestOutputLocation model) {
            expectedManifestBucketOwner(model.expectedManifestBucketOwner);
            bucket(model.bucket);
            manifestPrefix(model.manifestPrefix);
            manifestEncryption(model.manifestEncryption);
            manifestFormat(model.manifestFormat);
        }

        public final String getExpectedManifestBucketOwner() {
            return expectedManifestBucketOwner;
        }

        public final void setExpectedManifestBucketOwner(String expectedManifestBucketOwner) {
            this.expectedManifestBucketOwner = expectedManifestBucketOwner;
        }

        @Override
        public final Builder expectedManifestBucketOwner(String expectedManifestBucketOwner) {
            this.expectedManifestBucketOwner = expectedManifestBucketOwner;
            return this;
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getManifestPrefix() {
            return manifestPrefix;
        }

        public final void setManifestPrefix(String manifestPrefix) {
            this.manifestPrefix = manifestPrefix;
        }

        @Override
        public final Builder manifestPrefix(String manifestPrefix) {
            this.manifestPrefix = manifestPrefix;
            return this;
        }

        public final GeneratedManifestEncryption.Builder getManifestEncryption() {
            return manifestEncryption != null ? manifestEncryption.toBuilder() : null;
        }

        public final void setManifestEncryption(GeneratedManifestEncryption.BuilderImpl manifestEncryption) {
            this.manifestEncryption = manifestEncryption != null ? manifestEncryption.build() : null;
        }

        @Override
        public final Builder manifestEncryption(GeneratedManifestEncryption manifestEncryption) {
            this.manifestEncryption = manifestEncryption;
            return this;
        }

        public final String getManifestFormat() {
            return manifestFormat;
        }

        public final void setManifestFormat(String manifestFormat) {
            this.manifestFormat = manifestFormat;
        }

        @Override
        public final Builder manifestFormat(String manifestFormat) {
            this.manifestFormat = manifestFormat;
            return this;
        }

        @Override
        public final Builder manifestFormat(GeneratedManifestFormat manifestFormat) {
            this.manifestFormat(manifestFormat == null ? null : manifestFormat.toString());
            return this;
        }

        @Override
        public S3ManifestOutputLocation build() {
            return new S3ManifestOutputLocation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
