/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A collection of statuses for a Multi-Region Access Point in the various Regions it supports.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MultiRegionAccessPointReport implements SdkPojo, Serializable,
        ToCopyableBuilder<MultiRegionAccessPointReport.Builder, MultiRegionAccessPointReport> {
    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(MultiRegionAccessPointReport::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<String> ALIAS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Alias")
            .getter(getter(MultiRegionAccessPointReport::alias))
            .setter(setter(Builder::alias))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Alias")
                    .unmarshallLocationName("Alias").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt")
            .getter(getter(MultiRegionAccessPointReport::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt")
                    .unmarshallLocationName("CreatedAt").build()).build();

    private static final SdkField<PublicAccessBlockConfiguration> PUBLIC_ACCESS_BLOCK_FIELD = SdkField
            .<PublicAccessBlockConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("PublicAccessBlock")
            .getter(getter(MultiRegionAccessPointReport::publicAccessBlock))
            .setter(setter(Builder::publicAccessBlock))
            .constructor(PublicAccessBlockConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicAccessBlock")
                    .unmarshallLocationName("PublicAccessBlock").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Status")
            .getter(getter(MultiRegionAccessPointReport::statusAsString))
            .setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("Status").build()).build();

    private static final SdkField<List<RegionReport>> REGIONS_FIELD = SdkField
            .<List<RegionReport>> builder(MarshallingType.LIST)
            .memberName("Regions")
            .getter(getter(MultiRegionAccessPointReport::regions))
            .setter(setter(Builder::regions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Regions")
                    .unmarshallLocationName("Regions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Region")
                            .memberFieldInfo(
                                    SdkField.<RegionReport> builder(MarshallingType.SDK_POJO)
                                            .constructor(RegionReport::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Region").unmarshallLocationName("Region").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ALIAS_FIELD,
            CREATED_AT_FIELD, PUBLIC_ACCESS_BLOCK_FIELD, STATUS_FIELD, REGIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String alias;

    private final Instant createdAt;

    private final PublicAccessBlockConfiguration publicAccessBlock;

    private final String status;

    private final List<RegionReport> regions;

    private MultiRegionAccessPointReport(BuilderImpl builder) {
        this.name = builder.name;
        this.alias = builder.alias;
        this.createdAt = builder.createdAt;
        this.publicAccessBlock = builder.publicAccessBlock;
        this.status = builder.status;
        this.regions = builder.regions;
    }

    /**
     * <p>
     * The name of the Multi-Region Access Point.
     * </p>
     * 
     * @return The name of the Multi-Region Access Point.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The alias for the Multi-Region Access Point. For more information about the distinction between the name and the
     * alias of an Multi-Region Access Point, see <a href=
     * "https://docs.aws.amazon.com/AmazonS3/latest/userguide/CreatingMultiRegionAccessPoints.html#multi-region-access-point-naming"
     * >Managing Multi-Region Access Points</a>.
     * </p>
     * 
     * @return The alias for the Multi-Region Access Point. For more information about the distinction between the name
     *         and the alias of an Multi-Region Access Point, see <a href=
     *         "https://docs.aws.amazon.com/AmazonS3/latest/userguide/CreatingMultiRegionAccessPoints.html#multi-region-access-point-naming"
     *         >Managing Multi-Region Access Points</a>.
     */
    public final String alias() {
        return alias;
    }

    /**
     * <p>
     * When the Multi-Region Access Point create request was received.
     * </p>
     * 
     * @return When the Multi-Region Access Point create request was received.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * Returns the value of the PublicAccessBlock property for this object.
     * 
     * @return The value of the PublicAccessBlock property for this object.
     */
    public final PublicAccessBlockConfiguration publicAccessBlock() {
        return publicAccessBlock;
    }

    /**
     * <p>
     * The current status of the Multi-Region Access Point.
     * </p>
     * <p>
     * <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is propogating
     * and being completed. If a Multi-Region Access Point has a status of <code>PARTIALLY_CREATED</code>, you can retry
     * creation or send a request to delete the Multi-Region Access Point. If a Multi-Region Access Point has a status
     * of <code>PARTIALLY_DELETED</code>, you can retry a delete request to finish the deletion of the Multi-Region
     * Access Point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MultiRegionAccessPointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the Multi-Region Access Point.</p>
     *         <p>
     *         <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is
     *         propogating and being completed. If a Multi-Region Access Point has a status of
     *         <code>PARTIALLY_CREATED</code>, you can retry creation or send a request to delete the Multi-Region
     *         Access Point. If a Multi-Region Access Point has a status of <code>PARTIALLY_DELETED</code>, you can
     *         retry a delete request to finish the deletion of the Multi-Region Access Point.
     * @see MultiRegionAccessPointStatus
     */
    public final MultiRegionAccessPointStatus status() {
        return MultiRegionAccessPointStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the Multi-Region Access Point.
     * </p>
     * <p>
     * <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is propogating
     * and being completed. If a Multi-Region Access Point has a status of <code>PARTIALLY_CREATED</code>, you can retry
     * creation or send a request to delete the Multi-Region Access Point. If a Multi-Region Access Point has a status
     * of <code>PARTIALLY_DELETED</code>, you can retry a delete request to finish the deletion of the Multi-Region
     * Access Point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MultiRegionAccessPointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the Multi-Region Access Point.</p>
     *         <p>
     *         <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is
     *         propogating and being completed. If a Multi-Region Access Point has a status of
     *         <code>PARTIALLY_CREATED</code>, you can retry creation or send a request to delete the Multi-Region
     *         Access Point. If a Multi-Region Access Point has a status of <code>PARTIALLY_DELETED</code>, you can
     *         retry a delete request to finish the deletion of the Multi-Region Access Point.
     * @see MultiRegionAccessPointStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Regions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRegions() {
        return regions != null && !(regions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of the Regions and buckets associated with the Multi-Region Access Point.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRegions} method.
     * </p>
     * 
     * @return A collection of the Regions and buckets associated with the Multi-Region Access Point.
     */
    public final List<RegionReport> regions() {
        return regions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(alias());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccessBlock());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRegions() ? regions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MultiRegionAccessPointReport)) {
            return false;
        }
        MultiRegionAccessPointReport other = (MultiRegionAccessPointReport) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(alias(), other.alias())
                && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(publicAccessBlock(), other.publicAccessBlock())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasRegions() == other.hasRegions()
                && Objects.equals(regions(), other.regions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MultiRegionAccessPointReport").add("Name", name()).add("Alias", alias())
                .add("CreatedAt", createdAt()).add("PublicAccessBlock", publicAccessBlock()).add("Status", statusAsString())
                .add("Regions", hasRegions() ? regions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Alias":
            return Optional.ofNullable(clazz.cast(alias()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "PublicAccessBlock":
            return Optional.ofNullable(clazz.cast(publicAccessBlock()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Regions":
            return Optional.ofNullable(clazz.cast(regions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MultiRegionAccessPointReport, T> g) {
        return obj -> g.apply((MultiRegionAccessPointReport) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MultiRegionAccessPointReport> {
        /**
         * <p>
         * The name of the Multi-Region Access Point.
         * </p>
         * 
         * @param name
         *        The name of the Multi-Region Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The alias for the Multi-Region Access Point. For more information about the distinction between the name and
         * the alias of an Multi-Region Access Point, see <a href=
         * "https://docs.aws.amazon.com/AmazonS3/latest/userguide/CreatingMultiRegionAccessPoints.html#multi-region-access-point-naming"
         * >Managing Multi-Region Access Points</a>.
         * </p>
         * 
         * @param alias
         *        The alias for the Multi-Region Access Point. For more information about the distinction between the
         *        name and the alias of an Multi-Region Access Point, see <a href=
         *        "https://docs.aws.amazon.com/AmazonS3/latest/userguide/CreatingMultiRegionAccessPoints.html#multi-region-access-point-naming"
         *        >Managing Multi-Region Access Points</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alias(String alias);

        /**
         * <p>
         * When the Multi-Region Access Point create request was received.
         * </p>
         * 
         * @param createdAt
         *        When the Multi-Region Access Point create request was received.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * Sets the value of the PublicAccessBlock property for this object.
         *
         * @param publicAccessBlock
         *        The new value for the PublicAccessBlock property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccessBlock(PublicAccessBlockConfiguration publicAccessBlock);

        /**
         * Sets the value of the PublicAccessBlock property for this object.
         *
         * This is a convenience method that creates an instance of the {@link PublicAccessBlockConfiguration.Builder}
         * avoiding the need to create one manually via {@link PublicAccessBlockConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link PublicAccessBlockConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #publicAccessBlock(PublicAccessBlockConfiguration)}.
         * 
         * @param publicAccessBlock
         *        a consumer that will call methods on {@link PublicAccessBlockConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccessBlock(PublicAccessBlockConfiguration)
         */
        default Builder publicAccessBlock(Consumer<PublicAccessBlockConfiguration.Builder> publicAccessBlock) {
            return publicAccessBlock(PublicAccessBlockConfiguration.builder().applyMutation(publicAccessBlock).build());
        }

        /**
         * <p>
         * The current status of the Multi-Region Access Point.
         * </p>
         * <p>
         * <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is
         * propogating and being completed. If a Multi-Region Access Point has a status of
         * <code>PARTIALLY_CREATED</code>, you can retry creation or send a request to delete the Multi-Region Access
         * Point. If a Multi-Region Access Point has a status of <code>PARTIALLY_DELETED</code>, you can retry a delete
         * request to finish the deletion of the Multi-Region Access Point.
         * </p>
         * 
         * @param status
         *        The current status of the Multi-Region Access Point.</p>
         *        <p>
         *        <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is
         *        propogating and being completed. If a Multi-Region Access Point has a status of
         *        <code>PARTIALLY_CREATED</code>, you can retry creation or send a request to delete the Multi-Region
         *        Access Point. If a Multi-Region Access Point has a status of <code>PARTIALLY_DELETED</code>, you can
         *        retry a delete request to finish the deletion of the Multi-Region Access Point.
         * @see MultiRegionAccessPointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MultiRegionAccessPointStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the Multi-Region Access Point.
         * </p>
         * <p>
         * <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is
         * propogating and being completed. If a Multi-Region Access Point has a status of
         * <code>PARTIALLY_CREATED</code>, you can retry creation or send a request to delete the Multi-Region Access
         * Point. If a Multi-Region Access Point has a status of <code>PARTIALLY_DELETED</code>, you can retry a delete
         * request to finish the deletion of the Multi-Region Access Point.
         * </p>
         * 
         * @param status
         *        The current status of the Multi-Region Access Point.</p>
         *        <p>
         *        <code>CREATING</code> and <code>DELETING</code> are temporary states that exist while the request is
         *        propogating and being completed. If a Multi-Region Access Point has a status of
         *        <code>PARTIALLY_CREATED</code>, you can retry creation or send a request to delete the Multi-Region
         *        Access Point. If a Multi-Region Access Point has a status of <code>PARTIALLY_DELETED</code>, you can
         *        retry a delete request to finish the deletion of the Multi-Region Access Point.
         * @see MultiRegionAccessPointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MultiRegionAccessPointStatus
         */
        Builder status(MultiRegionAccessPointStatus status);

        /**
         * <p>
         * A collection of the Regions and buckets associated with the Multi-Region Access Point.
         * </p>
         * 
         * @param regions
         *        A collection of the Regions and buckets associated with the Multi-Region Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<RegionReport> regions);

        /**
         * <p>
         * A collection of the Regions and buckets associated with the Multi-Region Access Point.
         * </p>
         * 
         * @param regions
         *        A collection of the Regions and buckets associated with the Multi-Region Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(RegionReport... regions);

        /**
         * <p>
         * A collection of the Regions and buckets associated with the Multi-Region Access Point.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<RegionReport>.Builder} avoiding the
         * need to create one manually via {@link List<RegionReport>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RegionReport>.Builder#build()} is called immediately and its
         * result is passed to {@link #regions(List<RegionReport>)}.
         * 
         * @param regions
         *        a consumer that will call methods on {@link List<RegionReport>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #regions(List<RegionReport>)
         */
        Builder regions(Consumer<RegionReport.Builder>... regions);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String alias;

        private Instant createdAt;

        private PublicAccessBlockConfiguration publicAccessBlock;

        private String status;

        private List<RegionReport> regions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MultiRegionAccessPointReport model) {
            name(model.name);
            alias(model.alias);
            createdAt(model.createdAt);
            publicAccessBlock(model.publicAccessBlock);
            status(model.status);
            regions(model.regions);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getAlias() {
            return alias;
        }

        public final void setAlias(String alias) {
            this.alias = alias;
        }

        @Override
        public final Builder alias(String alias) {
            this.alias = alias;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final PublicAccessBlockConfiguration.Builder getPublicAccessBlock() {
            return publicAccessBlock != null ? publicAccessBlock.toBuilder() : null;
        }

        public final void setPublicAccessBlock(PublicAccessBlockConfiguration.BuilderImpl publicAccessBlock) {
            this.publicAccessBlock = publicAccessBlock != null ? publicAccessBlock.build() : null;
        }

        @Override
        public final Builder publicAccessBlock(PublicAccessBlockConfiguration publicAccessBlock) {
            this.publicAccessBlock = publicAccessBlock;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MultiRegionAccessPointStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<RegionReport.Builder> getRegions() {
            List<RegionReport.Builder> result = RegionReportListCopier.copyToBuilder(this.regions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRegions(Collection<RegionReport.BuilderImpl> regions) {
            this.regions = RegionReportListCopier.copyFromBuilder(regions);
        }

        @Override
        public final Builder regions(Collection<RegionReport> regions) {
            this.regions = RegionReportListCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(RegionReport... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(Consumer<RegionReport.Builder>... regions) {
            regions(Stream.of(regions).map(c -> RegionReport.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public MultiRegionAccessPointReport build() {
            return new MultiRegionAccessPointReport(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
