/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A configuration used when creating an Object Lambda Access Point transformation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ObjectLambdaTransformationConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ObjectLambdaTransformationConfiguration.Builder, ObjectLambdaTransformationConfiguration> {
    private static final SdkField<List<String>> ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Actions")
            .getter(getter(ObjectLambdaTransformationConfiguration::actionsAsStrings))
            .setter(setter(Builder::actionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Actions")
                    .unmarshallLocationName("Actions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Action")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Action").unmarshallLocationName("Action").build()).build())
                            .build()).build();

    private static final SdkField<ObjectLambdaContentTransformation> CONTENT_TRANSFORMATION_FIELD = SdkField
            .<ObjectLambdaContentTransformation> builder(MarshallingType.SDK_POJO)
            .memberName("ContentTransformation")
            .getter(getter(ObjectLambdaTransformationConfiguration::contentTransformation))
            .setter(setter(Builder::contentTransformation))
            .constructor(ObjectLambdaContentTransformation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentTransformation")
                    .unmarshallLocationName("ContentTransformation").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTIONS_FIELD,
            CONTENT_TRANSFORMATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> actions;

    private final ObjectLambdaContentTransformation contentTransformation;

    private ObjectLambdaTransformationConfiguration(BuilderImpl builder) {
        this.actions = builder.actions;
        this.contentTransformation = builder.contentTransformation;
    }

    /**
     * <p>
     * A container for the action of an Object Lambda Access Point configuration. Valid input is <code>GetObject</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return A container for the action of an Object Lambda Access Point configuration. Valid input is
     *         <code>GetObject</code>.
     */
    public final List<ObjectLambdaTransformationConfigurationAction> actions() {
        return ObjectLambdaTransformationConfigurationActionsListCopier.copyStringToEnum(actions);
    }

    /**
     * For responses, this returns true if the service returned a value for the Actions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasActions() {
        return actions != null && !(actions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A container for the action of an Object Lambda Access Point configuration. Valid input is <code>GetObject</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActions} method.
     * </p>
     * 
     * @return A container for the action of an Object Lambda Access Point configuration. Valid input is
     *         <code>GetObject</code>.
     */
    public final List<String> actionsAsStrings() {
        return actions;
    }

    /**
     * <p>
     * A container for the content transformation of an Object Lambda Access Point configuration.
     * </p>
     * 
     * @return A container for the content transformation of an Object Lambda Access Point configuration.
     */
    public final ObjectLambdaContentTransformation contentTransformation() {
        return contentTransformation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasActions() ? actionsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(contentTransformation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ObjectLambdaTransformationConfiguration)) {
            return false;
        }
        ObjectLambdaTransformationConfiguration other = (ObjectLambdaTransformationConfiguration) obj;
        return hasActions() == other.hasActions() && Objects.equals(actionsAsStrings(), other.actionsAsStrings())
                && Objects.equals(contentTransformation(), other.contentTransformation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ObjectLambdaTransformationConfiguration")
                .add("Actions", hasActions() ? actionsAsStrings() : null).add("ContentTransformation", contentTransformation())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Actions":
            return Optional.ofNullable(clazz.cast(actionsAsStrings()));
        case "ContentTransformation":
            return Optional.ofNullable(clazz.cast(contentTransformation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ObjectLambdaTransformationConfiguration, T> g) {
        return obj -> g.apply((ObjectLambdaTransformationConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ObjectLambdaTransformationConfiguration> {
        /**
         * <p>
         * A container for the action of an Object Lambda Access Point configuration. Valid input is
         * <code>GetObject</code>.
         * </p>
         * 
         * @param actions
         *        A container for the action of an Object Lambda Access Point configuration. Valid input is
         *        <code>GetObject</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsWithStrings(Collection<String> actions);

        /**
         * <p>
         * A container for the action of an Object Lambda Access Point configuration. Valid input is
         * <code>GetObject</code>.
         * </p>
         * 
         * @param actions
         *        A container for the action of an Object Lambda Access Point configuration. Valid input is
         *        <code>GetObject</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionsWithStrings(String... actions);

        /**
         * <p>
         * A container for the action of an Object Lambda Access Point configuration. Valid input is
         * <code>GetObject</code>.
         * </p>
         * 
         * @param actions
         *        A container for the action of an Object Lambda Access Point configuration. Valid input is
         *        <code>GetObject</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(Collection<ObjectLambdaTransformationConfigurationAction> actions);

        /**
         * <p>
         * A container for the action of an Object Lambda Access Point configuration. Valid input is
         * <code>GetObject</code>.
         * </p>
         * 
         * @param actions
         *        A container for the action of an Object Lambda Access Point configuration. Valid input is
         *        <code>GetObject</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actions(ObjectLambdaTransformationConfigurationAction... actions);

        /**
         * <p>
         * A container for the content transformation of an Object Lambda Access Point configuration.
         * </p>
         * 
         * @param contentTransformation
         *        A container for the content transformation of an Object Lambda Access Point configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentTransformation(ObjectLambdaContentTransformation contentTransformation);

        /**
         * <p>
         * A container for the content transformation of an Object Lambda Access Point configuration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ObjectLambdaContentTransformation.Builder} avoiding the need to create one manually via
         * {@link ObjectLambdaContentTransformation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ObjectLambdaContentTransformation.Builder#build()} is called
         * immediately and its result is passed to {@link #contentTransformation(ObjectLambdaContentTransformation)}.
         * 
         * @param contentTransformation
         *        a consumer that will call methods on {@link ObjectLambdaContentTransformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contentTransformation(ObjectLambdaContentTransformation)
         */
        default Builder contentTransformation(Consumer<ObjectLambdaContentTransformation.Builder> contentTransformation) {
            return contentTransformation(ObjectLambdaContentTransformation.builder().applyMutation(contentTransformation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<String> actions = DefaultSdkAutoConstructList.getInstance();

        private ObjectLambdaContentTransformation contentTransformation;

        private BuilderImpl() {
        }

        private BuilderImpl(ObjectLambdaTransformationConfiguration model) {
            actionsWithStrings(model.actions);
            contentTransformation(model.contentTransformation);
        }

        public final Collection<String> getActions() {
            if (actions instanceof SdkAutoConstructList) {
                return null;
            }
            return actions;
        }

        public final void setActions(Collection<String> actions) {
            this.actions = ObjectLambdaTransformationConfigurationActionsListCopier.copy(actions);
        }

        @Override
        public final Builder actionsWithStrings(Collection<String> actions) {
            this.actions = ObjectLambdaTransformationConfigurationActionsListCopier.copy(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actionsWithStrings(String... actions) {
            actionsWithStrings(Arrays.asList(actions));
            return this;
        }

        @Override
        public final Builder actions(Collection<ObjectLambdaTransformationConfigurationAction> actions) {
            this.actions = ObjectLambdaTransformationConfigurationActionsListCopier.copyEnumToString(actions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actions(ObjectLambdaTransformationConfigurationAction... actions) {
            actions(Arrays.asList(actions));
            return this;
        }

        public final ObjectLambdaContentTransformation.Builder getContentTransformation() {
            return contentTransformation != null ? contentTransformation.toBuilder() : null;
        }

        public final void setContentTransformation(ObjectLambdaContentTransformation.BuilderImpl contentTransformation) {
            this.contentTransformation = contentTransformation != null ? contentTransformation.build() : null;
        }

        @Override
        public final Builder contentTransformation(ObjectLambdaContentTransformation contentTransformation) {
            this.contentTransformation = contentTransformation;
            return this;
        }

        @Override
        public ObjectLambdaTransformationConfiguration build() {
            return new ObjectLambdaTransformationConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
