/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.endpoints.internal;

import java.util.Optional;
import java.util.concurrent.CompletionException;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlClientContextParams;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlEndpointParams;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlEndpointProvider;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.CreateAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.CreateBucketRequest;
import software.amazon.awssdk.services.s3control.model.CreateJobRequest;
import software.amazon.awssdk.services.s3control.model.CreateMultiRegionAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.DeleteAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketLifecycleConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketPolicyRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketReplicationRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketRequest;
import software.amazon.awssdk.services.s3control.model.DeleteBucketTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DeleteMultiRegionAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.DeletePublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.DeleteStorageLensConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.DeleteStorageLensConfigurationTaggingRequest;
import software.amazon.awssdk.services.s3control.model.DescribeJobRequest;
import software.amazon.awssdk.services.s3control.model.DescribeMultiRegionAccessPointOperationRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointConfigurationForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointPolicyStatusRequest;
import software.amazon.awssdk.services.s3control.model.GetAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketLifecycleConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketReplicationRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetBucketVersioningRequest;
import software.amazon.awssdk.services.s3control.model.GetJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.GetMultiRegionAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.GetMultiRegionAccessPointPolicyStatusRequest;
import software.amazon.awssdk.services.s3control.model.GetMultiRegionAccessPointRequest;
import software.amazon.awssdk.services.s3control.model.GetMultiRegionAccessPointRoutesRequest;
import software.amazon.awssdk.services.s3control.model.GetPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.GetStorageLensConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.GetStorageLensConfigurationTaggingRequest;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.ListAccessPointsRequest;
import software.amazon.awssdk.services.s3control.model.ListJobsRequest;
import software.amazon.awssdk.services.s3control.model.ListMultiRegionAccessPointsRequest;
import software.amazon.awssdk.services.s3control.model.ListRegionalBucketsRequest;
import software.amazon.awssdk.services.s3control.model.ListStorageLensConfigurationsRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointConfigurationForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyForObjectLambdaRequest;
import software.amazon.awssdk.services.s3control.model.PutAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketLifecycleConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketReplicationRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutBucketVersioningRequest;
import software.amazon.awssdk.services.s3control.model.PutJobTaggingRequest;
import software.amazon.awssdk.services.s3control.model.PutMultiRegionAccessPointPolicyRequest;
import software.amazon.awssdk.services.s3control.model.PutPublicAccessBlockRequest;
import software.amazon.awssdk.services.s3control.model.PutStorageLensConfigurationRequest;
import software.amazon.awssdk.services.s3control.model.PutStorageLensConfigurationTaggingRequest;
import software.amazon.awssdk.services.s3control.model.SubmitMultiRegionAccessPointRoutesRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobPriorityRequest;
import software.amazon.awssdk.services.s3control.model.UpdateJobStatusRequest;
import software.amazon.awssdk.utils.AttributeMap;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class S3ControlResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return context.request();
        }
        S3ControlEndpointProvider provider = (S3ControlEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            Endpoint result = provider.resolveEndpoint(ruleParams(context, executionAttributes)).join();
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        context.request());
                if (hostPrefix.isPresent()) {
                    result = AwsEndpointProviderUtils.addHostPrefix(result, hostPrefix.get());
                }
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, result);
            return context.request();
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    private static S3ControlEndpointParams ruleParams(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        S3ControlEndpointParams.Builder builder = S3ControlEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setClientContextParams(builder, executionAttributes);
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), context.request());
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        return builder.build();
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, String operationName, SdkRequest request) {
        switch (operationName) {
        case "CreateAccessPoint":
            setContextParams(params, (CreateAccessPointRequest) request);
            break;
        case "CreateAccessPointForObjectLambda":
            setContextParams(params, (CreateAccessPointForObjectLambdaRequest) request);
            break;
        case "CreateBucket":
            setContextParams(params, (CreateBucketRequest) request);
            break;
        case "CreateJob":
            setContextParams(params, (CreateJobRequest) request);
            break;
        case "CreateMultiRegionAccessPoint":
            setContextParams(params, (CreateMultiRegionAccessPointRequest) request);
            break;
        case "DeleteAccessPoint":
            setContextParams(params, (DeleteAccessPointRequest) request);
            break;
        case "DeleteAccessPointForObjectLambda":
            setContextParams(params, (DeleteAccessPointForObjectLambdaRequest) request);
            break;
        case "DeleteAccessPointPolicy":
            setContextParams(params, (DeleteAccessPointPolicyRequest) request);
            break;
        case "DeleteAccessPointPolicyForObjectLambda":
            setContextParams(params, (DeleteAccessPointPolicyForObjectLambdaRequest) request);
            break;
        case "DeleteBucket":
            setContextParams(params, (DeleteBucketRequest) request);
            break;
        case "DeleteBucketLifecycleConfiguration":
            setContextParams(params, (DeleteBucketLifecycleConfigurationRequest) request);
            break;
        case "DeleteBucketPolicy":
            setContextParams(params, (DeleteBucketPolicyRequest) request);
            break;
        case "DeleteBucketReplication":
            setContextParams(params, (DeleteBucketReplicationRequest) request);
            break;
        case "DeleteBucketTagging":
            setContextParams(params, (DeleteBucketTaggingRequest) request);
            break;
        case "DeleteJobTagging":
            setContextParams(params, (DeleteJobTaggingRequest) request);
            break;
        case "DeleteMultiRegionAccessPoint":
            setContextParams(params, (DeleteMultiRegionAccessPointRequest) request);
            break;
        case "DeletePublicAccessBlock":
            setContextParams(params, (DeletePublicAccessBlockRequest) request);
            break;
        case "DeleteStorageLensConfiguration":
            setContextParams(params, (DeleteStorageLensConfigurationRequest) request);
            break;
        case "DeleteStorageLensConfigurationTagging":
            setContextParams(params, (DeleteStorageLensConfigurationTaggingRequest) request);
            break;
        case "DescribeJob":
            setContextParams(params, (DescribeJobRequest) request);
            break;
        case "DescribeMultiRegionAccessPointOperation":
            setContextParams(params, (DescribeMultiRegionAccessPointOperationRequest) request);
            break;
        case "GetAccessPoint":
            setContextParams(params, (GetAccessPointRequest) request);
            break;
        case "GetAccessPointConfigurationForObjectLambda":
            setContextParams(params, (GetAccessPointConfigurationForObjectLambdaRequest) request);
            break;
        case "GetAccessPointForObjectLambda":
            setContextParams(params, (GetAccessPointForObjectLambdaRequest) request);
            break;
        case "GetAccessPointPolicy":
            setContextParams(params, (GetAccessPointPolicyRequest) request);
            break;
        case "GetAccessPointPolicyForObjectLambda":
            setContextParams(params, (GetAccessPointPolicyForObjectLambdaRequest) request);
            break;
        case "GetAccessPointPolicyStatus":
            setContextParams(params, (GetAccessPointPolicyStatusRequest) request);
            break;
        case "GetAccessPointPolicyStatusForObjectLambda":
            setContextParams(params, (GetAccessPointPolicyStatusForObjectLambdaRequest) request);
            break;
        case "GetBucket":
            setContextParams(params, (GetBucketRequest) request);
            break;
        case "GetBucketLifecycleConfiguration":
            setContextParams(params, (GetBucketLifecycleConfigurationRequest) request);
            break;
        case "GetBucketPolicy":
            setContextParams(params, (GetBucketPolicyRequest) request);
            break;
        case "GetBucketReplication":
            setContextParams(params, (GetBucketReplicationRequest) request);
            break;
        case "GetBucketTagging":
            setContextParams(params, (GetBucketTaggingRequest) request);
            break;
        case "GetBucketVersioning":
            setContextParams(params, (GetBucketVersioningRequest) request);
            break;
        case "GetJobTagging":
            setContextParams(params, (GetJobTaggingRequest) request);
            break;
        case "GetMultiRegionAccessPoint":
            setContextParams(params, (GetMultiRegionAccessPointRequest) request);
            break;
        case "GetMultiRegionAccessPointPolicy":
            setContextParams(params, (GetMultiRegionAccessPointPolicyRequest) request);
            break;
        case "GetMultiRegionAccessPointPolicyStatus":
            setContextParams(params, (GetMultiRegionAccessPointPolicyStatusRequest) request);
            break;
        case "GetMultiRegionAccessPointRoutes":
            setContextParams(params, (GetMultiRegionAccessPointRoutesRequest) request);
            break;
        case "GetPublicAccessBlock":
            setContextParams(params, (GetPublicAccessBlockRequest) request);
            break;
        case "GetStorageLensConfiguration":
            setContextParams(params, (GetStorageLensConfigurationRequest) request);
            break;
        case "GetStorageLensConfigurationTagging":
            setContextParams(params, (GetStorageLensConfigurationTaggingRequest) request);
            break;
        case "ListAccessPoints":
            setContextParams(params, (ListAccessPointsRequest) request);
            break;
        case "ListAccessPointsForObjectLambda":
            setContextParams(params, (ListAccessPointsForObjectLambdaRequest) request);
            break;
        case "ListJobs":
            setContextParams(params, (ListJobsRequest) request);
            break;
        case "ListMultiRegionAccessPoints":
            setContextParams(params, (ListMultiRegionAccessPointsRequest) request);
            break;
        case "ListRegionalBuckets":
            setContextParams(params, (ListRegionalBucketsRequest) request);
            break;
        case "ListStorageLensConfigurations":
            setContextParams(params, (ListStorageLensConfigurationsRequest) request);
            break;
        case "PutAccessPointConfigurationForObjectLambda":
            setContextParams(params, (PutAccessPointConfigurationForObjectLambdaRequest) request);
            break;
        case "PutAccessPointPolicy":
            setContextParams(params, (PutAccessPointPolicyRequest) request);
            break;
        case "PutAccessPointPolicyForObjectLambda":
            setContextParams(params, (PutAccessPointPolicyForObjectLambdaRequest) request);
            break;
        case "PutBucketLifecycleConfiguration":
            setContextParams(params, (PutBucketLifecycleConfigurationRequest) request);
            break;
        case "PutBucketPolicy":
            setContextParams(params, (PutBucketPolicyRequest) request);
            break;
        case "PutBucketReplication":
            setContextParams(params, (PutBucketReplicationRequest) request);
            break;
        case "PutBucketTagging":
            setContextParams(params, (PutBucketTaggingRequest) request);
            break;
        case "PutBucketVersioning":
            setContextParams(params, (PutBucketVersioningRequest) request);
            break;
        case "PutJobTagging":
            setContextParams(params, (PutJobTaggingRequest) request);
            break;
        case "PutMultiRegionAccessPointPolicy":
            setContextParams(params, (PutMultiRegionAccessPointPolicyRequest) request);
            break;
        case "PutPublicAccessBlock":
            setContextParams(params, (PutPublicAccessBlockRequest) request);
            break;
        case "PutStorageLensConfiguration":
            setContextParams(params, (PutStorageLensConfigurationRequest) request);
            break;
        case "PutStorageLensConfigurationTagging":
            setContextParams(params, (PutStorageLensConfigurationTaggingRequest) request);
            break;
        case "SubmitMultiRegionAccessPointRoutes":
            setContextParams(params, (SubmitMultiRegionAccessPointRoutesRequest) request);
            break;
        case "UpdateJobPriority":
            setContextParams(params, (UpdateJobPriorityRequest) request);
            break;
        case "UpdateJobStatus":
            setContextParams(params, (UpdateJobStatusRequest) request);
            break;
        default:
            break;
        }
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, CreateAccessPointRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, CreateAccessPointForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, CreateBucketRequest request) {
        params.bucket(request.bucket());
        params.outpostId(request.outpostId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, CreateJobRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, CreateMultiRegionAccessPointRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteAccessPointRequest request) {
        params.accountId(request.accountId());
        params.accessPointName(request.name());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteAccessPointForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteAccessPointPolicyRequest request) {
        params.accountId(request.accountId());
        params.accessPointName(request.name());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            DeleteAccessPointPolicyForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteBucketRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteBucketLifecycleConfigurationRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteBucketPolicyRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteBucketReplicationRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteBucketTaggingRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteJobTaggingRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteMultiRegionAccessPointRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeletePublicAccessBlockRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DeleteStorageLensConfigurationRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            DeleteStorageLensConfigurationTaggingRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, DescribeJobRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            DescribeMultiRegionAccessPointOperationRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetAccessPointRequest request) {
        params.accountId(request.accountId());
        params.accessPointName(request.name());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            GetAccessPointConfigurationForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetAccessPointForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetAccessPointPolicyRequest request) {
        params.accountId(request.accountId());
        params.accessPointName(request.name());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            GetAccessPointPolicyForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetAccessPointPolicyStatusRequest request) {
        params.accountId(request.accountId());
        params.accessPointName(request.name());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            GetAccessPointPolicyStatusForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetBucketRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetBucketLifecycleConfigurationRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetBucketPolicyRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetBucketReplicationRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetBucketTaggingRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetBucketVersioningRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetJobTaggingRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetMultiRegionAccessPointRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetMultiRegionAccessPointPolicyRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            GetMultiRegionAccessPointPolicyStatusRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetMultiRegionAccessPointRoutesRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetPublicAccessBlockRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetStorageLensConfigurationRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, GetStorageLensConfigurationTaggingRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, ListAccessPointsRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, ListAccessPointsForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, ListJobsRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, ListMultiRegionAccessPointsRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, ListRegionalBucketsRequest request) {
        params.accountId(request.accountId());
        params.outpostId(request.outpostId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, ListStorageLensConfigurationsRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            PutAccessPointConfigurationForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutAccessPointPolicyRequest request) {
        params.accountId(request.accountId());
        params.accessPointName(request.name());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params,
            PutAccessPointPolicyForObjectLambdaRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutBucketLifecycleConfigurationRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutBucketPolicyRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutBucketReplicationRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutBucketTaggingRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutBucketVersioningRequest request) {
        params.accountId(request.accountId());
        params.bucket(request.bucket());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutJobTaggingRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutMultiRegionAccessPointPolicyRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutPublicAccessBlockRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutStorageLensConfigurationRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, PutStorageLensConfigurationTaggingRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, SubmitMultiRegionAccessPointRoutesRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, UpdateJobPriorityRequest request) {
        params.accountId(request.accountId());
    }

    private static void setContextParams(S3ControlEndpointParams.Builder params, UpdateJobStatusRequest request) {
        params.accountId(request.accountId());
    }

    private static void setStaticContextParams(S3ControlEndpointParams.Builder params, String operationName) {
        switch (operationName) {
        case "CreateAccessPoint":
            createAccessPointStaticContextParams(params);
            break;
        case "CreateAccessPointForObjectLambda":
            createAccessPointForObjectLambdaStaticContextParams(params);
            break;
        case "CreateJob":
            createJobStaticContextParams(params);
            break;
        case "CreateMultiRegionAccessPoint":
            createMultiRegionAccessPointStaticContextParams(params);
            break;
        case "DeleteAccessPoint":
            deleteAccessPointStaticContextParams(params);
            break;
        case "DeleteAccessPointForObjectLambda":
            deleteAccessPointForObjectLambdaStaticContextParams(params);
            break;
        case "DeleteAccessPointPolicy":
            deleteAccessPointPolicyStaticContextParams(params);
            break;
        case "DeleteAccessPointPolicyForObjectLambda":
            deleteAccessPointPolicyForObjectLambdaStaticContextParams(params);
            break;
        case "DeleteBucket":
            deleteBucketStaticContextParams(params);
            break;
        case "DeleteBucketLifecycleConfiguration":
            deleteBucketLifecycleConfigurationStaticContextParams(params);
            break;
        case "DeleteBucketPolicy":
            deleteBucketPolicyStaticContextParams(params);
            break;
        case "DeleteBucketReplication":
            deleteBucketReplicationStaticContextParams(params);
            break;
        case "DeleteBucketTagging":
            deleteBucketTaggingStaticContextParams(params);
            break;
        case "DeleteJobTagging":
            deleteJobTaggingStaticContextParams(params);
            break;
        case "DeleteMultiRegionAccessPoint":
            deleteMultiRegionAccessPointStaticContextParams(params);
            break;
        case "DeletePublicAccessBlock":
            deletePublicAccessBlockStaticContextParams(params);
            break;
        case "DeleteStorageLensConfiguration":
            deleteStorageLensConfigurationStaticContextParams(params);
            break;
        case "DeleteStorageLensConfigurationTagging":
            deleteStorageLensConfigurationTaggingStaticContextParams(params);
            break;
        case "DescribeJob":
            describeJobStaticContextParams(params);
            break;
        case "DescribeMultiRegionAccessPointOperation":
            describeMultiRegionAccessPointOperationStaticContextParams(params);
            break;
        case "GetAccessPoint":
            getAccessPointStaticContextParams(params);
            break;
        case "GetAccessPointConfigurationForObjectLambda":
            getAccessPointConfigurationForObjectLambdaStaticContextParams(params);
            break;
        case "GetAccessPointForObjectLambda":
            getAccessPointForObjectLambdaStaticContextParams(params);
            break;
        case "GetAccessPointPolicy":
            getAccessPointPolicyStaticContextParams(params);
            break;
        case "GetAccessPointPolicyForObjectLambda":
            getAccessPointPolicyForObjectLambdaStaticContextParams(params);
            break;
        case "GetAccessPointPolicyStatus":
            getAccessPointPolicyStatusStaticContextParams(params);
            break;
        case "GetAccessPointPolicyStatusForObjectLambda":
            getAccessPointPolicyStatusForObjectLambdaStaticContextParams(params);
            break;
        case "GetBucket":
            getBucketStaticContextParams(params);
            break;
        case "GetBucketLifecycleConfiguration":
            getBucketLifecycleConfigurationStaticContextParams(params);
            break;
        case "GetBucketPolicy":
            getBucketPolicyStaticContextParams(params);
            break;
        case "GetBucketReplication":
            getBucketReplicationStaticContextParams(params);
            break;
        case "GetBucketTagging":
            getBucketTaggingStaticContextParams(params);
            break;
        case "GetBucketVersioning":
            getBucketVersioningStaticContextParams(params);
            break;
        case "GetJobTagging":
            getJobTaggingStaticContextParams(params);
            break;
        case "GetMultiRegionAccessPoint":
            getMultiRegionAccessPointStaticContextParams(params);
            break;
        case "GetMultiRegionAccessPointPolicy":
            getMultiRegionAccessPointPolicyStaticContextParams(params);
            break;
        case "GetMultiRegionAccessPointPolicyStatus":
            getMultiRegionAccessPointPolicyStatusStaticContextParams(params);
            break;
        case "GetMultiRegionAccessPointRoutes":
            getMultiRegionAccessPointRoutesStaticContextParams(params);
            break;
        case "GetPublicAccessBlock":
            getPublicAccessBlockStaticContextParams(params);
            break;
        case "GetStorageLensConfiguration":
            getStorageLensConfigurationStaticContextParams(params);
            break;
        case "GetStorageLensConfigurationTagging":
            getStorageLensConfigurationTaggingStaticContextParams(params);
            break;
        case "ListAccessPoints":
            listAccessPointsStaticContextParams(params);
            break;
        case "ListAccessPointsForObjectLambda":
            listAccessPointsForObjectLambdaStaticContextParams(params);
            break;
        case "ListJobs":
            listJobsStaticContextParams(params);
            break;
        case "ListMultiRegionAccessPoints":
            listMultiRegionAccessPointsStaticContextParams(params);
            break;
        case "ListRegionalBuckets":
            listRegionalBucketsStaticContextParams(params);
            break;
        case "ListStorageLensConfigurations":
            listStorageLensConfigurationsStaticContextParams(params);
            break;
        case "PutAccessPointConfigurationForObjectLambda":
            putAccessPointConfigurationForObjectLambdaStaticContextParams(params);
            break;
        case "PutAccessPointPolicy":
            putAccessPointPolicyStaticContextParams(params);
            break;
        case "PutAccessPointPolicyForObjectLambda":
            putAccessPointPolicyForObjectLambdaStaticContextParams(params);
            break;
        case "PutBucketLifecycleConfiguration":
            putBucketLifecycleConfigurationStaticContextParams(params);
            break;
        case "PutBucketPolicy":
            putBucketPolicyStaticContextParams(params);
            break;
        case "PutBucketReplication":
            putBucketReplicationStaticContextParams(params);
            break;
        case "PutBucketTagging":
            putBucketTaggingStaticContextParams(params);
            break;
        case "PutBucketVersioning":
            putBucketVersioningStaticContextParams(params);
            break;
        case "PutJobTagging":
            putJobTaggingStaticContextParams(params);
            break;
        case "PutMultiRegionAccessPointPolicy":
            putMultiRegionAccessPointPolicyStaticContextParams(params);
            break;
        case "PutPublicAccessBlock":
            putPublicAccessBlockStaticContextParams(params);
            break;
        case "PutStorageLensConfiguration":
            putStorageLensConfigurationStaticContextParams(params);
            break;
        case "PutStorageLensConfigurationTagging":
            putStorageLensConfigurationTaggingStaticContextParams(params);
            break;
        case "SubmitMultiRegionAccessPointRoutes":
            submitMultiRegionAccessPointRoutesStaticContextParams(params);
            break;
        case "UpdateJobPriority":
            updateJobPriorityStaticContextParams(params);
            break;
        case "UpdateJobStatus":
            updateJobStatusStaticContextParams(params);
            break;
        default:
            break;
        }
    }

    private static void createAccessPointStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void createAccessPointForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void createJobStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void createMultiRegionAccessPointStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteAccessPointStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteAccessPointForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteAccessPointPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteAccessPointPolicyForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteBucketStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteBucketLifecycleConfigurationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteBucketPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteBucketReplicationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteBucketTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteJobTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteMultiRegionAccessPointStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deletePublicAccessBlockStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteStorageLensConfigurationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void deleteStorageLensConfigurationTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void describeJobStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void describeMultiRegionAccessPointOperationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getAccessPointStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getAccessPointConfigurationForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getAccessPointForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getAccessPointPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getAccessPointPolicyForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getAccessPointPolicyStatusStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getAccessPointPolicyStatusForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getBucketStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getBucketLifecycleConfigurationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getBucketPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getBucketReplicationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getBucketTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getBucketVersioningStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getJobTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getMultiRegionAccessPointStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getMultiRegionAccessPointPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getMultiRegionAccessPointPolicyStatusStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getMultiRegionAccessPointRoutesStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getPublicAccessBlockStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getStorageLensConfigurationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void getStorageLensConfigurationTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void listAccessPointsStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void listAccessPointsForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void listJobsStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void listMultiRegionAccessPointsStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void listRegionalBucketsStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void listStorageLensConfigurationsStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putAccessPointConfigurationForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putAccessPointPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putAccessPointPolicyForObjectLambdaStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putBucketLifecycleConfigurationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putBucketPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putBucketReplicationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putBucketTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putBucketVersioningStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putJobTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putMultiRegionAccessPointPolicyStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putPublicAccessBlockStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putStorageLensConfigurationStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void putStorageLensConfigurationTaggingStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void submitMultiRegionAccessPointRoutesStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void updateJobPriorityStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void updateJobStatusStaticContextParams(S3ControlEndpointParams.Builder params) {
        params.requiresAccountId(true);
    }

    private static void setClientContextParams(S3ControlEndpointParams.Builder params, ExecutionAttributes executionAttributes) {
        AttributeMap clientContextParams = executionAttributes.getAttribute(SdkInternalExecutionAttribute.CLIENT_CONTEXT_PARAMS);
        Optional.ofNullable(clientContextParams.get(S3ControlClientContextParams.USE_ARN_REGION)).ifPresent(params::useArnRegion);
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        return Optional.empty();
    }
}
