/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.builder.AwsDefaultClientBuilder;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.interceptor.ClasspathInterceptorChainFactory;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.http.auth.aws.scheme.AwsV4AuthScheme;
import software.amazon.awssdk.http.auth.scheme.NoAuthAuthScheme;
import software.amazon.awssdk.http.auth.spi.scheme.AuthScheme;
import software.amazon.awssdk.identity.spi.IdentityProvider;
import software.amazon.awssdk.identity.spi.IdentityProviders;
import software.amazon.awssdk.services.s3control.auth.scheme.S3ControlAuthSchemeProvider;
import software.amazon.awssdk.services.s3control.auth.scheme.internal.S3ControlAuthSchemeInterceptor;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlClientContextParams;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlEndpointProvider;
import software.amazon.awssdk.services.s3control.endpoints.internal.S3ControlRequestSetEndpointInterceptor;
import software.amazon.awssdk.services.s3control.endpoints.internal.S3ControlResolveEndpointInterceptor;
import software.amazon.awssdk.services.s3control.internal.S3ControlServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.s3control.internal.interceptors.ConfigureSignerInterceptor;
import software.amazon.awssdk.services.s3control.internal.interceptors.PayloadSigningInterceptor;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.Validate;

/**
 * Internal base class for {@link DefaultS3ControlClientBuilder} and {@link DefaultS3ControlAsyncClientBuilder}.
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
abstract class DefaultS3ControlBaseClientBuilder<B extends S3ControlBaseClientBuilder<B, C>, C> extends
        AwsDefaultClientBuilder<B, C> {
    private final Map<String, AuthScheme<?>> additionalAuthSchemes = new HashMap<>();

    @Override
    protected final String serviceEndpointPrefix() {
        return "s3-control";
    }

    @Override
    protected final String serviceName() {
        return "S3Control";
    }

    @Override
    protected final SdkClientConfiguration mergeServiceDefaults(SdkClientConfiguration config) {
        return config.merge(c -> c.option(SdkClientOption.ENDPOINT_PROVIDER, defaultEndpointProvider())
                .option(SdkClientOption.AUTH_SCHEME_PROVIDER, defaultAuthSchemeProvider())
                .option(SdkClientOption.AUTH_SCHEMES, authSchemes())
                .option(SdkClientOption.CRC32_FROM_COMPRESSED_DATA_ENABLED, false)
                .option(SdkClientOption.SERVICE_CONFIGURATION, S3ControlConfiguration.builder().build()));
    }

    @Override
    protected final SdkClientConfiguration finalizeServiceConfiguration(SdkClientConfiguration config) {
        List<ExecutionInterceptor> endpointInterceptors = new ArrayList<>();
        endpointInterceptors.add(new S3ControlAuthSchemeInterceptor());
        endpointInterceptors.add(new S3ControlResolveEndpointInterceptor());
        endpointInterceptors.add(new S3ControlRequestSetEndpointInterceptor());
        endpointInterceptors.add(new ConfigureSignerInterceptor());
        endpointInterceptors.add(new PayloadSigningInterceptor());
        ClasspathInterceptorChainFactory interceptorFactory = new ClasspathInterceptorChainFactory();
        List<ExecutionInterceptor> interceptors = interceptorFactory
                .getInterceptors("software/amazon/awssdk/services/s3control/execution.interceptors");
        List<ExecutionInterceptor> additionalInterceptors = new ArrayList<>();
        interceptors = CollectionUtils.mergeLists(endpointInterceptors, interceptors);
        interceptors = CollectionUtils.mergeLists(interceptors, additionalInterceptors);
        interceptors = CollectionUtils.mergeLists(interceptors, config.option(SdkClientOption.EXECUTION_INTERCEPTORS));
        S3ControlConfiguration.Builder serviceConfigBuilder = ((S3ControlConfiguration) config
                .option(SdkClientOption.SERVICE_CONFIGURATION)).toBuilder();
        serviceConfigBuilder.profileFile(serviceConfigBuilder.profileFileSupplier() != null ? serviceConfigBuilder
                .profileFileSupplier() : config.option(SdkClientOption.PROFILE_FILE_SUPPLIER));
        serviceConfigBuilder.profileName(serviceConfigBuilder.profileName() != null ? serviceConfigBuilder.profileName() : config
                .option(SdkClientOption.PROFILE_NAME));
        if (serviceConfigBuilder.dualstackEnabled() != null) {
            Validate.validState(
                    config.option(AwsClientOption.DUALSTACK_ENDPOINT_ENABLED) == null,
                    "Dualstack has been configured on both S3ControlConfiguration and the client/global level. Please limit dualstack configuration to one location.");
        } else {
            serviceConfigBuilder.dualstackEnabled(config.option(AwsClientOption.DUALSTACK_ENDPOINT_ENABLED));
        }
        if (serviceConfigBuilder.fipsModeEnabled() != null) {
            Validate.validState(
                    config.option(AwsClientOption.FIPS_ENDPOINT_ENABLED) == null,
                    "Fips has been configured on both S3ControlConfiguration and the client/global level. Please limit fips configuration to one location.");
        } else {
            serviceConfigBuilder.fipsModeEnabled(config.option(AwsClientOption.FIPS_ENDPOINT_ENABLED));
        }
        if (serviceConfigBuilder.useArnRegionEnabled() != null) {
            Validate.validState(
                    clientContextParams.get(S3ControlClientContextParams.USE_ARN_REGION) == null,
                    "UseArnRegion has been configured on both S3ControlConfiguration and the client/global level. Please limit UseArnRegion configuration to one location.");
        } else {
            serviceConfigBuilder.useArnRegionEnabled(clientContextParams.get(S3ControlClientContextParams.USE_ARN_REGION));
        }
        S3ControlConfiguration finalServiceConfig = serviceConfigBuilder.build();
        clientContextParams.put(S3ControlClientContextParams.USE_ARN_REGION, finalServiceConfig.useArnRegionEnabled());
        SdkClientConfiguration.Builder builder = config.toBuilder();
        builder.lazyOption(SdkClientOption.IDENTITY_PROVIDERS, c -> {
            IdentityProviders.Builder result = IdentityProviders.builder();
            IdentityProvider<?> credentialsIdentityProvider = c.get(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER);
            if (credentialsIdentityProvider != null) {
                result.putIdentityProvider(credentialsIdentityProvider);
            }
            return result.build();
        });
        builder.option(SdkClientOption.EXECUTION_INTERCEPTORS, interceptors)
                .option(AwsClientOption.DUALSTACK_ENDPOINT_ENABLED, finalServiceConfig.dualstackEnabled())
                .option(AwsClientOption.FIPS_ENDPOINT_ENABLED, finalServiceConfig.fipsModeEnabled())
                .option(SdkClientOption.SERVICE_CONFIGURATION, finalServiceConfig)
                .option(SdkClientOption.CLIENT_CONTEXT_PARAMS, clientContextParams.build());
        return builder.build();
    }

    @Override
    protected final String signingName() {
        return "s3";
    }

    private S3ControlEndpointProvider defaultEndpointProvider() {
        return S3ControlEndpointProvider.defaultProvider();
    }

    public B authSchemeProvider(S3ControlAuthSchemeProvider authSchemeProvider) {
        clientConfiguration.option(SdkClientOption.AUTH_SCHEME_PROVIDER, authSchemeProvider);
        return thisBuilder();
    }

    private S3ControlAuthSchemeProvider defaultAuthSchemeProvider() {
        return S3ControlAuthSchemeProvider.defaultProvider();
    }

    @Override
    public B putAuthScheme(AuthScheme<?> authScheme) {
        additionalAuthSchemes.put(authScheme.schemeId(), authScheme);
        return thisBuilder();
    }

    private Map<String, AuthScheme<?>> authSchemes() {
        Map<String, AuthScheme<?>> schemes = new HashMap<>(2 + this.additionalAuthSchemes.size());
        AwsV4AuthScheme awsV4AuthScheme = AwsV4AuthScheme.create();
        schemes.put(awsV4AuthScheme.schemeId(), awsV4AuthScheme);
        NoAuthAuthScheme noAuthAuthScheme = NoAuthAuthScheme.create();
        schemes.put(noAuthAuthScheme.schemeId(), noAuthAuthScheme);
        schemes.putAll(this.additionalAuthSchemes);
        return schemes;
    }

    public B useArnRegion(Boolean useArnRegion) {
        clientContextParams.put(S3ControlClientContextParams.USE_ARN_REGION, useArnRegion);
        return thisBuilder();
    }

    public B serviceConfiguration(S3ControlConfiguration serviceConfiguration) {
        clientConfiguration.option(SdkClientOption.SERVICE_CONFIGURATION, serviceConfiguration);
        return thisBuilder();
    }

    public void setServiceConfiguration(S3ControlConfiguration serviceConfiguration) {
        serviceConfiguration(serviceConfiguration);
    }

    @Override
    protected SdkClientConfiguration invokePlugins(SdkClientConfiguration config) {
        List<SdkPlugin> internalPlugins = internalPlugins();
        List<SdkPlugin> externalPlugins = plugins();
        if (internalPlugins.isEmpty() && externalPlugins.isEmpty()) {
            return config;
        }
        List<SdkPlugin> plugins = CollectionUtils.mergeLists(internalPlugins, externalPlugins);
        SdkClientConfiguration.Builder configuration = config.toBuilder();
        S3ControlServiceClientConfigurationBuilder serviceConfigBuilder = new S3ControlServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private List<SdkPlugin> internalPlugins() {
        return Collections.emptyList();
    }

    protected static void validateClientOptions(SdkClientConfiguration c) {
    }
}
