/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the configuration parameters for a <code>Lambda Invoke</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LambdaInvokeOperation implements SdkPojo, Serializable,
        ToCopyableBuilder<LambdaInvokeOperation.Builder, LambdaInvokeOperation> {
    private static final SdkField<String> FUNCTION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FunctionArn")
            .getter(getter(LambdaInvokeOperation::functionArn))
            .setter(setter(Builder::functionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FunctionArn")
                    .unmarshallLocationName("FunctionArn").build()).build();

    private static final SdkField<String> INVOCATION_SCHEMA_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InvocationSchemaVersion")
            .getter(getter(LambdaInvokeOperation::invocationSchemaVersion))
            .setter(setter(Builder::invocationSchemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvocationSchemaVersion")
                    .unmarshallLocationName("InvocationSchemaVersion").build()).build();

    private static final SdkField<Map<String, String>> USER_ARGUMENTS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("UserArguments")
            .getter(getter(LambdaInvokeOperation::userArguments))
            .setter(setter(Builder::userArguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserArguments")
                    .unmarshallLocationName("UserArguments").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").unmarshallLocationName("value").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FUNCTION_ARN_FIELD,
            INVOCATION_SCHEMA_VERSION_FIELD, USER_ARGUMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String functionArn;

    private final String invocationSchemaVersion;

    private final Map<String, String> userArguments;

    private LambdaInvokeOperation(BuilderImpl builder) {
        this.functionArn = builder.functionArn;
        this.invocationSchemaVersion = builder.invocationSchemaVersion;
        this.userArguments = builder.userArguments;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the Lambda function that the specified job will invoke on every object in the
     * manifest.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the Lambda function that the specified job will invoke on every object
     *         in the manifest.
     */
    public final String functionArn() {
        return functionArn;
    }

    /**
     * <p>
     * Specifies the schema version for the payload that Batch Operations sends when invoking an Lambda function.
     * Version <code>1.0</code> is the default. Version <code>2.0</code> is required when you use Batch Operations to
     * invoke Lambda functions that act on directory buckets, or if you need to specify <code>UserArguments</code>. For
     * more information, see <a href=
     * "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
     * >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in the
     * <i>Amazon Web Services Storage Blog</i>.
     * </p>
     * <important>
     * <p>
     * Ensure that your Lambda function code expects <code>InvocationSchemaVersion</code> <b>2.0</b> and uses bucket
     * name rather than bucket ARN. If the <code>InvocationSchemaVersion</code> does not match what your Lambda function
     * expects, your function might not work as expected.
     * </p>
     * </important> <note>
     * <p>
     * <b>Directory buckets</b> - To initiate Amazon Web Services Lambda function to perform custom actions on objects
     * in directory buckets, you must specify <code>2.0</code>.
     * </p>
     * </note>
     * 
     * @return Specifies the schema version for the payload that Batch Operations sends when invoking an Lambda
     *         function. Version <code>1.0</code> is the default. Version <code>2.0</code> is required when you use
     *         Batch Operations to invoke Lambda functions that act on directory buckets, or if you need to specify
     *         <code>UserArguments</code>. For more information, see <a href=
     *         "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
     *         >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in the
     *         <i>Amazon Web Services Storage Blog</i>.</p> <important>
     *         <p>
     *         Ensure that your Lambda function code expects <code>InvocationSchemaVersion</code> <b>2.0</b> and uses
     *         bucket name rather than bucket ARN. If the <code>InvocationSchemaVersion</code> does not match what your
     *         Lambda function expects, your function might not work as expected.
     *         </p>
     *         </important> <note>
     *         <p>
     *         <b>Directory buckets</b> - To initiate Amazon Web Services Lambda function to perform custom actions on
     *         objects in directory buckets, you must specify <code>2.0</code>.
     *         </p>
     */
    public final String invocationSchemaVersion() {
        return invocationSchemaVersion;
    }

    /**
     * For responses, this returns true if the service returned a value for the UserArguments property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUserArguments() {
        return userArguments != null && !(userArguments instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Key-value pairs that are passed in the payload that Batch Operations sends when invoking an Lambda function. You
     * must specify <code>InvocationSchemaVersion</code> <b>2.0</b> for <code>LambdaInvoke</code> operations that
     * include <code>UserArguments</code>. For more information, see <a href=
     * "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
     * >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in the
     * <i>Amazon Web Services Storage Blog</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUserArguments} method.
     * </p>
     * 
     * @return Key-value pairs that are passed in the payload that Batch Operations sends when invoking an Lambda
     *         function. You must specify <code>InvocationSchemaVersion</code> <b>2.0</b> for <code>LambdaInvoke</code>
     *         operations that include <code>UserArguments</code>. For more information, see <a href=
     *         "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
     *         >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in the
     *         <i>Amazon Web Services Storage Blog</i>.
     */
    public final Map<String, String> userArguments() {
        return userArguments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(functionArn());
        hashCode = 31 * hashCode + Objects.hashCode(invocationSchemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasUserArguments() ? userArguments() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LambdaInvokeOperation)) {
            return false;
        }
        LambdaInvokeOperation other = (LambdaInvokeOperation) obj;
        return Objects.equals(functionArn(), other.functionArn())
                && Objects.equals(invocationSchemaVersion(), other.invocationSchemaVersion())
                && hasUserArguments() == other.hasUserArguments() && Objects.equals(userArguments(), other.userArguments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LambdaInvokeOperation").add("FunctionArn", functionArn())
                .add("InvocationSchemaVersion", invocationSchemaVersion())
                .add("UserArguments", hasUserArguments() ? userArguments() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FunctionArn":
            return Optional.ofNullable(clazz.cast(functionArn()));
        case "InvocationSchemaVersion":
            return Optional.ofNullable(clazz.cast(invocationSchemaVersion()));
        case "UserArguments":
            return Optional.ofNullable(clazz.cast(userArguments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LambdaInvokeOperation, T> g) {
        return obj -> g.apply((LambdaInvokeOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LambdaInvokeOperation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) for the Lambda function that the specified job will invoke on every object in
         * the manifest.
         * </p>
         * 
         * @param functionArn
         *        The Amazon Resource Name (ARN) for the Lambda function that the specified job will invoke on every
         *        object in the manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionArn(String functionArn);

        /**
         * <p>
         * Specifies the schema version for the payload that Batch Operations sends when invoking an Lambda function.
         * Version <code>1.0</code> is the default. Version <code>2.0</code> is required when you use Batch Operations
         * to invoke Lambda functions that act on directory buckets, or if you need to specify
         * <code>UserArguments</code>. For more information, see <a href=
         * "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
         * >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in the
         * <i>Amazon Web Services Storage Blog</i>.
         * </p>
         * <important>
         * <p>
         * Ensure that your Lambda function code expects <code>InvocationSchemaVersion</code> <b>2.0</b> and uses bucket
         * name rather than bucket ARN. If the <code>InvocationSchemaVersion</code> does not match what your Lambda
         * function expects, your function might not work as expected.
         * </p>
         * </important> <note>
         * <p>
         * <b>Directory buckets</b> - To initiate Amazon Web Services Lambda function to perform custom actions on
         * objects in directory buckets, you must specify <code>2.0</code>.
         * </p>
         * </note>
         * 
         * @param invocationSchemaVersion
         *        Specifies the schema version for the payload that Batch Operations sends when invoking an Lambda
         *        function. Version <code>1.0</code> is the default. Version <code>2.0</code> is required when you use
         *        Batch Operations to invoke Lambda functions that act on directory buckets, or if you need to specify
         *        <code>UserArguments</code>. For more information, see <a href=
         *        "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
         *        >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in
         *        the <i>Amazon Web Services Storage Blog</i>.</p> <important>
         *        <p>
         *        Ensure that your Lambda function code expects <code>InvocationSchemaVersion</code> <b>2.0</b> and uses
         *        bucket name rather than bucket ARN. If the <code>InvocationSchemaVersion</code> does not match what
         *        your Lambda function expects, your function might not work as expected.
         *        </p>
         *        </important> <note>
         *        <p>
         *        <b>Directory buckets</b> - To initiate Amazon Web Services Lambda function to perform custom actions
         *        on objects in directory buckets, you must specify <code>2.0</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invocationSchemaVersion(String invocationSchemaVersion);

        /**
         * <p>
         * Key-value pairs that are passed in the payload that Batch Operations sends when invoking an Lambda function.
         * You must specify <code>InvocationSchemaVersion</code> <b>2.0</b> for <code>LambdaInvoke</code> operations
         * that include <code>UserArguments</code>. For more information, see <a href=
         * "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
         * >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in the
         * <i>Amazon Web Services Storage Blog</i>.
         * </p>
         * 
         * @param userArguments
         *        Key-value pairs that are passed in the payload that Batch Operations sends when invoking an Lambda
         *        function. You must specify <code>InvocationSchemaVersion</code> <b>2.0</b> for
         *        <code>LambdaInvoke</code> operations that include <code>UserArguments</code>. For more information,
         *        see <a href=
         *        "https://aws.amazon.com/blogs/storage/automate-object-processing-in-amazon-s3-directory-buckets-with-s3-batch-operations-and-aws-lambda/"
         *        >Automate object processing in Amazon S3 directory buckets with S3 Batch Operations and Lambda</a> in
         *        the <i>Amazon Web Services Storage Blog</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userArguments(Map<String, String> userArguments);
    }

    static final class BuilderImpl implements Builder {
        private String functionArn;

        private String invocationSchemaVersion;

        private Map<String, String> userArguments = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LambdaInvokeOperation model) {
            functionArn(model.functionArn);
            invocationSchemaVersion(model.invocationSchemaVersion);
            userArguments(model.userArguments);
        }

        public final String getFunctionArn() {
            return functionArn;
        }

        public final void setFunctionArn(String functionArn) {
            this.functionArn = functionArn;
        }

        @Override
        public final Builder functionArn(String functionArn) {
            this.functionArn = functionArn;
            return this;
        }

        public final String getInvocationSchemaVersion() {
            return invocationSchemaVersion;
        }

        public final void setInvocationSchemaVersion(String invocationSchemaVersion) {
            this.invocationSchemaVersion = invocationSchemaVersion;
        }

        @Override
        public final Builder invocationSchemaVersion(String invocationSchemaVersion) {
            this.invocationSchemaVersion = invocationSchemaVersion;
            return this;
        }

        public final Map<String, String> getUserArguments() {
            if (userArguments instanceof SdkAutoConstructMap) {
                return null;
            }
            return userArguments;
        }

        public final void setUserArguments(Map<String, String> userArguments) {
            this.userArguments = UserArgumentsCopier.copy(userArguments);
        }

        @Override
        public final Builder userArguments(Map<String, String> userArguments) {
            this.userArguments = UserArgumentsCopier.copy(userArguments);
            return this;
        }

        @Override
        public LambdaInvokeOperation build() {
            return new LambdaInvokeOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
