/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateJobRequest extends S3ControlRequest implements
        ToCopyableBuilder<CreateJobRequest.Builder, CreateJobRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccountId")
            .getter(getter(CreateJobRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-account-id")
                    .unmarshallLocationName("x-amz-account-id").build()).build();

    private static final SdkField<Boolean> CONFIRMATION_REQUIRED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ConfirmationRequired")
            .getter(getter(CreateJobRequest::confirmationRequired))
            .setter(setter(Builder::confirmationRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfirmationRequired")
                    .unmarshallLocationName("ConfirmationRequired").build()).build();

    private static final SdkField<JobOperation> OPERATION_FIELD = SdkField
            .<JobOperation> builder(MarshallingType.SDK_POJO)
            .memberName("Operation")
            .getter(getter(CreateJobRequest::operation))
            .setter(setter(Builder::operation))
            .constructor(JobOperation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation")
                    .unmarshallLocationName("Operation").build()).build();

    private static final SdkField<JobReport> REPORT_FIELD = SdkField
            .<JobReport> builder(MarshallingType.SDK_POJO)
            .memberName("Report")
            .getter(getter(CreateJobRequest::report))
            .setter(setter(Builder::report))
            .constructor(JobReport::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Report")
                    .unmarshallLocationName("Report").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientRequestToken")
            .getter(getter(CreateJobRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken")
                    .unmarshallLocationName("ClientRequestToken").build(), DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<JobManifest> MANIFEST_FIELD = SdkField
            .<JobManifest> builder(MarshallingType.SDK_POJO)
            .memberName("Manifest")
            .getter(getter(CreateJobRequest::manifest))
            .setter(setter(Builder::manifest))
            .constructor(JobManifest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Manifest")
                    .unmarshallLocationName("Manifest").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(CreateJobRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority")
            .getter(getter(CreateJobRequest::priority))
            .setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority")
                    .unmarshallLocationName("Priority").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RoleArn")
            .getter(getter(CreateJobRequest::roleArn))
            .setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn")
                    .unmarshallLocationName("RoleArn").build()).build();

    private static final SdkField<List<S3Tag>> TAGS_FIELD = SdkField
            .<List<S3Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateJobRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags")
                    .unmarshallLocationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<JobManifestGenerator> MANIFEST_GENERATOR_FIELD = SdkField
            .<JobManifestGenerator> builder(MarshallingType.SDK_POJO)
            .memberName("ManifestGenerator")
            .getter(getter(CreateJobRequest::manifestGenerator))
            .setter(setter(Builder::manifestGenerator))
            .constructor(JobManifestGenerator::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestGenerator")
                    .unmarshallLocationName("ManifestGenerator").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            CONFIRMATION_REQUIRED_FIELD, OPERATION_FIELD, REPORT_FIELD, CLIENT_REQUEST_TOKEN_FIELD, MANIFEST_FIELD,
            DESCRIPTION_FIELD, PRIORITY_FIELD, ROLE_ARN_FIELD, TAGS_FIELD, MANIFEST_GENERATOR_FIELD));

    private final String accountId;

    private final Boolean confirmationRequired;

    private final JobOperation operation;

    private final JobReport report;

    private final String clientRequestToken;

    private final JobManifest manifest;

    private final String description;

    private final Integer priority;

    private final String roleArn;

    private final List<S3Tag> tags;

    private final JobManifestGenerator manifestGenerator;

    private CreateJobRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.confirmationRequired = builder.confirmationRequired;
        this.operation = builder.operation;
        this.report = builder.report;
        this.clientRequestToken = builder.clientRequestToken;
        this.manifest = builder.manifest;
        this.description = builder.description;
        this.priority = builder.priority;
        this.roleArn = builder.roleArn;
        this.tags = builder.tags;
        this.manifestGenerator = builder.manifestGenerator;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that creates the job.
     * </p>
     * 
     * @return The Amazon Web Services account ID that creates the job.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Indicates whether confirmation is required before Amazon S3 runs the job. Confirmation is only required for jobs
     * created through the Amazon S3 console.
     * </p>
     * 
     * @return Indicates whether confirmation is required before Amazon S3 runs the job. Confirmation is only required
     *         for jobs created through the Amazon S3 console.
     */
    public final Boolean confirmationRequired() {
        return confirmationRequired;
    }

    /**
     * <p>
     * The action that you want this job to perform on every object listed in the manifest. For more information about
     * the available actions, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-actions.html">Operations</a> in the <i>Amazon S3
     * User Guide</i>.
     * </p>
     * 
     * @return The action that you want this job to perform on every object listed in the manifest. For more information
     *         about the available actions, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-actions.html">Operations</a> in the
     *         <i>Amazon S3 User Guide</i>.
     */
    public final JobOperation operation() {
        return operation;
    }

    /**
     * <p>
     * Configuration parameters for the optional job-completion report.
     * </p>
     * 
     * @return Configuration parameters for the optional job-completion report.
     */
    public final JobReport report() {
        return report;
    }

    /**
     * <p>
     * An idempotency token to ensure that you don't accidentally submit the same request twice. You can use any string
     * up to the maximum length.
     * </p>
     * 
     * @return An idempotency token to ensure that you don't accidentally submit the same request twice. You can use any
     *         string up to the maximum length.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * Configuration parameters for the manifest.
     * </p>
     * 
     * @return Configuration parameters for the manifest.
     */
    public final JobManifest manifest() {
        return manifest;
    }

    /**
     * <p>
     * A description for this job. You can use any string within the permitted length. Descriptions don't need to be
     * unique and can be used for multiple jobs.
     * </p>
     * 
     * @return A description for this job. You can use any string within the permitted length. Descriptions don't need
     *         to be unique and can be used for multiple jobs.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The numerical priority for this job. Higher numbers indicate higher priority.
     * </p>
     * 
     * @return The numerical priority for this job. Higher numbers indicate higher priority.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role that Batch Operations will use
     * to run this job's action on every object in the manifest.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role that Batch Operations
     *         will use to run this job's action on every object in the manifest.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A set of tags to associate with the S3 Batch Operations job. This is an optional parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A set of tags to associate with the S3 Batch Operations job. This is an optional parameter.
     */
    public final List<S3Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The attribute container for the ManifestGenerator details. Jobs must be created with either a manifest file or a
     * ManifestGenerator, but not both.
     * </p>
     * 
     * @return The attribute container for the ManifestGenerator details. Jobs must be created with either a manifest
     *         file or a ManifestGenerator, but not both.
     */
    public final JobManifestGenerator manifestGenerator() {
        return manifestGenerator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationRequired());
        hashCode = 31 * hashCode + Objects.hashCode(operation());
        hashCode = 31 * hashCode + Objects.hashCode(report());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(manifest());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(manifestGenerator());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateJobRequest)) {
            return false;
        }
        CreateJobRequest other = (CreateJobRequest) obj;
        return Objects.equals(accountId(), other.accountId())
                && Objects.equals(confirmationRequired(), other.confirmationRequired())
                && Objects.equals(operation(), other.operation()) && Objects.equals(report(), other.report())
                && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(manifest(), other.manifest()) && Objects.equals(description(), other.description())
                && Objects.equals(priority(), other.priority()) && Objects.equals(roleArn(), other.roleArn())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(manifestGenerator(), other.manifestGenerator());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateJobRequest").add("AccountId", accountId())
                .add("ConfirmationRequired", confirmationRequired()).add("Operation", operation()).add("Report", report())
                .add("ClientRequestToken", clientRequestToken()).add("Manifest", manifest()).add("Description", description())
                .add("Priority", priority()).add("RoleArn", roleArn()).add("Tags", hasTags() ? tags() : null)
                .add("ManifestGenerator", manifestGenerator()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "ConfirmationRequired":
            return Optional.ofNullable(clazz.cast(confirmationRequired()));
        case "Operation":
            return Optional.ofNullable(clazz.cast(operation()));
        case "Report":
            return Optional.ofNullable(clazz.cast(report()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "Manifest":
            return Optional.ofNullable(clazz.cast(manifest()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "ManifestGenerator":
            return Optional.ofNullable(clazz.cast(manifestGenerator()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateJobRequest, T> g) {
        return obj -> g.apply((CreateJobRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3ControlRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateJobRequest> {
        /**
         * <p>
         * The Amazon Web Services account ID that creates the job.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID that creates the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Indicates whether confirmation is required before Amazon S3 runs the job. Confirmation is only required for
         * jobs created through the Amazon S3 console.
         * </p>
         * 
         * @param confirmationRequired
         *        Indicates whether confirmation is required before Amazon S3 runs the job. Confirmation is only
         *        required for jobs created through the Amazon S3 console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confirmationRequired(Boolean confirmationRequired);

        /**
         * <p>
         * The action that you want this job to perform on every object listed in the manifest. For more information
         * about the available actions, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-actions.html">Operations</a> in the <i>Amazon
         * S3 User Guide</i>.
         * </p>
         * 
         * @param operation
         *        The action that you want this job to perform on every object listed in the manifest. For more
         *        information about the available actions, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-actions.html">Operations</a> in the
         *        <i>Amazon S3 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operation(JobOperation operation);

        /**
         * <p>
         * The action that you want this job to perform on every object listed in the manifest. For more information
         * about the available actions, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-actions.html">Operations</a> in the <i>Amazon
         * S3 User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobOperation.Builder} avoiding the need
         * to create one manually via {@link JobOperation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobOperation.Builder#build()} is called immediately and its
         * result is passed to {@link #operation(JobOperation)}.
         * 
         * @param operation
         *        a consumer that will call methods on {@link JobOperation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operation(JobOperation)
         */
        default Builder operation(Consumer<JobOperation.Builder> operation) {
            return operation(JobOperation.builder().applyMutation(operation).build());
        }

        /**
         * <p>
         * Configuration parameters for the optional job-completion report.
         * </p>
         * 
         * @param report
         *        Configuration parameters for the optional job-completion report.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder report(JobReport report);

        /**
         * <p>
         * Configuration parameters for the optional job-completion report.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobReport.Builder} avoiding the need to
         * create one manually via {@link JobReport#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobReport.Builder#build()} is called immediately and its result
         * is passed to {@link #report(JobReport)}.
         * 
         * @param report
         *        a consumer that will call methods on {@link JobReport.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #report(JobReport)
         */
        default Builder report(Consumer<JobReport.Builder> report) {
            return report(JobReport.builder().applyMutation(report).build());
        }

        /**
         * <p>
         * An idempotency token to ensure that you don't accidentally submit the same request twice. You can use any
         * string up to the maximum length.
         * </p>
         * 
         * @param clientRequestToken
         *        An idempotency token to ensure that you don't accidentally submit the same request twice. You can use
         *        any string up to the maximum length.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * Configuration parameters for the manifest.
         * </p>
         * 
         * @param manifest
         *        Configuration parameters for the manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifest(JobManifest manifest);

        /**
         * <p>
         * Configuration parameters for the manifest.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobManifest.Builder} avoiding the need to
         * create one manually via {@link JobManifest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobManifest.Builder#build()} is called immediately and its result
         * is passed to {@link #manifest(JobManifest)}.
         * 
         * @param manifest
         *        a consumer that will call methods on {@link JobManifest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #manifest(JobManifest)
         */
        default Builder manifest(Consumer<JobManifest.Builder> manifest) {
            return manifest(JobManifest.builder().applyMutation(manifest).build());
        }

        /**
         * <p>
         * A description for this job. You can use any string within the permitted length. Descriptions don't need to be
         * unique and can be used for multiple jobs.
         * </p>
         * 
         * @param description
         *        A description for this job. You can use any string within the permitted length. Descriptions don't
         *        need to be unique and can be used for multiple jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The numerical priority for this job. Higher numbers indicate higher priority.
         * </p>
         * 
         * @param priority
         *        The numerical priority for this job. Higher numbers indicate higher priority.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role that Batch Operations will
         * use to run this job's action on every object in the manifest.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role that Batch Operations
         *        will use to run this job's action on every object in the manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * A set of tags to associate with the S3 Batch Operations job. This is an optional parameter.
         * </p>
         * 
         * @param tags
         *        A set of tags to associate with the S3 Batch Operations job. This is an optional parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<S3Tag> tags);

        /**
         * <p>
         * A set of tags to associate with the S3 Batch Operations job. This is an optional parameter.
         * </p>
         * 
         * @param tags
         *        A set of tags to associate with the S3 Batch Operations job. This is an optional parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(S3Tag... tags);

        /**
         * <p>
         * A set of tags to associate with the S3 Batch Operations job. This is an optional parameter.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.s3control.model.S3Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.s3control.model.S3Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.s3control.model.S3Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<S3Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.s3control.model.S3Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<S3Tag>)
         */
        Builder tags(Consumer<S3Tag.Builder>... tags);

        /**
         * <p>
         * The attribute container for the ManifestGenerator details. Jobs must be created with either a manifest file
         * or a ManifestGenerator, but not both.
         * </p>
         * 
         * @param manifestGenerator
         *        The attribute container for the ManifestGenerator details. Jobs must be created with either a manifest
         *        file or a ManifestGenerator, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestGenerator(JobManifestGenerator manifestGenerator);

        /**
         * <p>
         * The attribute container for the ManifestGenerator details. Jobs must be created with either a manifest file
         * or a ManifestGenerator, but not both.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobManifestGenerator.Builder} avoiding
         * the need to create one manually via {@link JobManifestGenerator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobManifestGenerator.Builder#build()} is called immediately and
         * its result is passed to {@link #manifestGenerator(JobManifestGenerator)}.
         * 
         * @param manifestGenerator
         *        a consumer that will call methods on {@link JobManifestGenerator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #manifestGenerator(JobManifestGenerator)
         */
        default Builder manifestGenerator(Consumer<JobManifestGenerator.Builder> manifestGenerator) {
            return manifestGenerator(JobManifestGenerator.builder().applyMutation(manifestGenerator).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3ControlRequest.BuilderImpl implements Builder {
        private String accountId;

        private Boolean confirmationRequired;

        private JobOperation operation;

        private JobReport report;

        private String clientRequestToken;

        private JobManifest manifest;

        private String description;

        private Integer priority;

        private String roleArn;

        private List<S3Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private JobManifestGenerator manifestGenerator;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateJobRequest model) {
            super(model);
            accountId(model.accountId);
            confirmationRequired(model.confirmationRequired);
            operation(model.operation);
            report(model.report);
            clientRequestToken(model.clientRequestToken);
            manifest(model.manifest);
            description(model.description);
            priority(model.priority);
            roleArn(model.roleArn);
            tags(model.tags);
            manifestGenerator(model.manifestGenerator);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final Boolean getConfirmationRequired() {
            return confirmationRequired;
        }

        public final void setConfirmationRequired(Boolean confirmationRequired) {
            this.confirmationRequired = confirmationRequired;
        }

        @Override
        public final Builder confirmationRequired(Boolean confirmationRequired) {
            this.confirmationRequired = confirmationRequired;
            return this;
        }

        public final JobOperation.Builder getOperation() {
            return operation != null ? operation.toBuilder() : null;
        }

        public final void setOperation(JobOperation.BuilderImpl operation) {
            this.operation = operation != null ? operation.build() : null;
        }

        @Override
        public final Builder operation(JobOperation operation) {
            this.operation = operation;
            return this;
        }

        public final JobReport.Builder getReport() {
            return report != null ? report.toBuilder() : null;
        }

        public final void setReport(JobReport.BuilderImpl report) {
            this.report = report != null ? report.build() : null;
        }

        @Override
        public final Builder report(JobReport report) {
            this.report = report;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final JobManifest.Builder getManifest() {
            return manifest != null ? manifest.toBuilder() : null;
        }

        public final void setManifest(JobManifest.BuilderImpl manifest) {
            this.manifest = manifest != null ? manifest.build() : null;
        }

        @Override
        public final Builder manifest(JobManifest manifest) {
            this.manifest = manifest;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final List<S3Tag.Builder> getTags() {
            List<S3Tag.Builder> result = S3TagSetCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<S3Tag.BuilderImpl> tags) {
            this.tags = S3TagSetCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<S3Tag> tags) {
            this.tags = S3TagSetCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(S3Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<S3Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> S3Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final JobManifestGenerator.Builder getManifestGenerator() {
            return manifestGenerator != null ? manifestGenerator.toBuilder() : null;
        }

        public final void setManifestGenerator(JobManifestGenerator.BuilderImpl manifestGenerator) {
            this.manifestGenerator = manifestGenerator != null ? manifestGenerator.build() : null;
        }

        @Override
        public final Builder manifestGenerator(JobManifestGenerator manifestGenerator) {
            this.manifestGenerator = manifestGenerator;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateJobRequest build() {
            return new CreateJobRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
