/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListAccessGrantsRequest extends S3ControlRequest implements
        ToCopyableBuilder<ListAccessGrantsRequest.Builder, ListAccessGrantsRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccountId")
            .getter(getter(ListAccessGrantsRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-account-id")
                    .unmarshallLocationName("x-amz-account-id").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NextToken")
            .getter(getter(ListAccessGrantsRequest::nextToken))
            .setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("nextToken")
                    .unmarshallLocationName("nextToken").build()).build();

    private static final SdkField<Integer> MAX_RESULTS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxResults")
            .getter(getter(ListAccessGrantsRequest::maxResults))
            .setter(setter(Builder::maxResults))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("maxResults")
                    .unmarshallLocationName("maxResults").build()).build();

    private static final SdkField<String> GRANTEE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GranteeType")
            .getter(getter(ListAccessGrantsRequest::granteeTypeAsString))
            .setter(setter(Builder::granteeType))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("granteetype")
                    .unmarshallLocationName("granteetype").build()).build();

    private static final SdkField<String> GRANTEE_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GranteeIdentifier")
            .getter(getter(ListAccessGrantsRequest::granteeIdentifier))
            .setter(setter(Builder::granteeIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("granteeidentifier")
                    .unmarshallLocationName("granteeidentifier").build()).build();

    private static final SdkField<String> PERMISSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Permission")
            .getter(getter(ListAccessGrantsRequest::permissionAsString))
            .setter(setter(Builder::permission))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("permission")
                    .unmarshallLocationName("permission").build()).build();

    private static final SdkField<String> GRANT_SCOPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GrantScope")
            .getter(getter(ListAccessGrantsRequest::grantScope))
            .setter(setter(Builder::grantScope))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("grantscope")
                    .unmarshallLocationName("grantscope").build()).build();

    private static final SdkField<String> APPLICATION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ApplicationArn")
            .getter(getter(ListAccessGrantsRequest::applicationArn))
            .setter(setter(Builder::applicationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("application_arn")
                    .unmarshallLocationName("application_arn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            NEXT_TOKEN_FIELD, MAX_RESULTS_FIELD, GRANTEE_TYPE_FIELD, GRANTEE_IDENTIFIER_FIELD, PERMISSION_FIELD,
            GRANT_SCOPE_FIELD, APPLICATION_ARN_FIELD));

    private final String accountId;

    private final String nextToken;

    private final Integer maxResults;

    private final String granteeType;

    private final String granteeIdentifier;

    private final String permission;

    private final String grantScope;

    private final String applicationArn;

    private ListAccessGrantsRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.nextToken = builder.nextToken;
        this.maxResults = builder.maxResults;
        this.granteeType = builder.granteeType;
        this.granteeIdentifier = builder.granteeIdentifier;
        this.permission = builder.permission;
        this.grantScope = builder.grantScope;
        this.applicationArn = builder.applicationArn;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the S3 Access Grants instance.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the S3 Access Grants instance.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * A pagination token to request the next page of results. Pass this value into a subsequent
     * <code>List Access Grants</code> request in order to retrieve the next page of results.
     * </p>
     * 
     * @return A pagination token to request the next page of results. Pass this value into a subsequent
     *         <code>List Access Grants</code> request in order to retrieve the next page of results.
     */
    public final String nextToken() {
        return nextToken;
    }

    /**
     * <p>
     * The maximum number of access grants that you would like returned in the <code>List Access Grants</code> response.
     * If the results include the pagination token <code>NextToken</code>, make another call using the
     * <code>NextToken</code> to determine if there are more results.
     * </p>
     * 
     * @return The maximum number of access grants that you would like returned in the <code>List Access Grants</code>
     *         response. If the results include the pagination token <code>NextToken</code>, make another call using the
     *         <code>NextToken</code> to determine if there are more results.
     */
    public final Integer maxResults() {
        return maxResults;
    }

    /**
     * <p>
     * The type of the grantee to which access has been granted. It can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IAM</code> - An IAM user or role.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added your
     * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your S3
     * Access Grants instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have added your
     * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your S3
     * Access Grants instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #granteeType} will
     * return {@link GranteeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #granteeTypeAsString}.
     * </p>
     * 
     * @return The type of the grantee to which access has been granted. It can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IAM</code> - An IAM user or role.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added
     *         your corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance
     *         with your S3 Access Grants instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have added
     *         your corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance
     *         with your S3 Access Grants instance.
     *         </p>
     *         </li>
     * @see GranteeType
     */
    public final GranteeType granteeType() {
        return GranteeType.fromValue(granteeType);
    }

    /**
     * <p>
     * The type of the grantee to which access has been granted. It can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>IAM</code> - An IAM user or role.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added your
     * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your S3
     * Access Grants instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have added your
     * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your S3
     * Access Grants instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #granteeType} will
     * return {@link GranteeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #granteeTypeAsString}.
     * </p>
     * 
     * @return The type of the grantee to which access has been granted. It can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>IAM</code> - An IAM user or role.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added
     *         your corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance
     *         with your S3 Access Grants instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have added
     *         your corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance
     *         with your S3 Access Grants instance.
     *         </p>
     *         </li>
     * @see GranteeType
     */
    public final String granteeTypeAsString() {
        return granteeType;
    }

    /**
     * <p>
     * The unique identifer of the <code>Grantee</code>. If the grantee type is <code>IAM</code>, the identifier is the
     * IAM Amazon Resource Name (ARN) of the user or role. If the grantee type is a directory user or group, the
     * identifier is 128-bit universally unique identifier (UUID) in the format
     * <code>a1b2c3d4-5678-90ab-cdef-EXAMPLE11111</code>. You can obtain this UUID from your Amazon Web Services IAM
     * Identity Center instance.
     * </p>
     * 
     * @return The unique identifer of the <code>Grantee</code>. If the grantee type is <code>IAM</code>, the identifier
     *         is the IAM Amazon Resource Name (ARN) of the user or role. If the grantee type is a directory user or
     *         group, the identifier is 128-bit universally unique identifier (UUID) in the format
     *         <code>a1b2c3d4-5678-90ab-cdef-EXAMPLE11111</code>. You can obtain this UUID from your Amazon Web Services
     *         IAM Identity Center instance.
     */
    public final String granteeIdentifier() {
        return granteeIdentifier;
    }

    /**
     * <p>
     * The type of permission granted to your S3 data, which can be set to one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>READ</code> – Grant read-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WRITE</code> – Grant write-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READWRITE</code> – Grant both read and write access to the S3 data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permission} will
     * return {@link Permission#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #permissionAsString}.
     * </p>
     * 
     * @return The type of permission granted to your S3 data, which can be set to one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>READ</code> – Grant read-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WRITE</code> – Grant write-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READWRITE</code> – Grant both read and write access to the S3 data.
     *         </p>
     *         </li>
     * @see Permission
     */
    public final Permission permission() {
        return Permission.fromValue(permission);
    }

    /**
     * <p>
     * The type of permission granted to your S3 data, which can be set to one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>READ</code> – Grant read-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WRITE</code> – Grant write-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READWRITE</code> – Grant both read and write access to the S3 data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permission} will
     * return {@link Permission#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #permissionAsString}.
     * </p>
     * 
     * @return The type of permission granted to your S3 data, which can be set to one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>READ</code> – Grant read-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WRITE</code> – Grant write-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READWRITE</code> – Grant both read and write access to the S3 data.
     *         </p>
     *         </li>
     * @see Permission
     */
    public final String permissionAsString() {
        return permission;
    }

    /**
     * <p>
     * The S3 path of the data to which you are granting access. It is the result of appending the
     * <code>Subprefix</code> to the location scope.
     * </p>
     * 
     * @return The S3 path of the data to which you are granting access. It is the result of appending the
     *         <code>Subprefix</code> to the location scope.
     */
    public final String grantScope() {
        return grantScope;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated with your
     * Identity Center instance. If the grant includes an application ARN, the grantee can only access the S3 data
     * through this application.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated with
     *         your Identity Center instance. If the grant includes an application ARN, the grantee can only access the
     *         S3 data through this application.
     */
    public final String applicationArn() {
        return applicationArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        hashCode = 31 * hashCode + Objects.hashCode(granteeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(granteeIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(permissionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(grantScope());
        hashCode = 31 * hashCode + Objects.hashCode(applicationArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ListAccessGrantsRequest)) {
            return false;
        }
        ListAccessGrantsRequest other = (ListAccessGrantsRequest) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(nextToken(), other.nextToken())
                && Objects.equals(maxResults(), other.maxResults())
                && Objects.equals(granteeTypeAsString(), other.granteeTypeAsString())
                && Objects.equals(granteeIdentifier(), other.granteeIdentifier())
                && Objects.equals(permissionAsString(), other.permissionAsString())
                && Objects.equals(grantScope(), other.grantScope()) && Objects.equals(applicationArn(), other.applicationArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ListAccessGrantsRequest").add("AccountId", accountId()).add("NextToken", nextToken())
                .add("MaxResults", maxResults()).add("GranteeType", granteeTypeAsString())
                .add("GranteeIdentifier", granteeIdentifier()).add("Permission", permissionAsString())
                .add("GrantScope", grantScope()).add("ApplicationArn", applicationArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        case "MaxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        case "GranteeType":
            return Optional.ofNullable(clazz.cast(granteeTypeAsString()));
        case "GranteeIdentifier":
            return Optional.ofNullable(clazz.cast(granteeIdentifier()));
        case "Permission":
            return Optional.ofNullable(clazz.cast(permissionAsString()));
        case "GrantScope":
            return Optional.ofNullable(clazz.cast(grantScope()));
        case "ApplicationArn":
            return Optional.ofNullable(clazz.cast(applicationArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ListAccessGrantsRequest, T> g) {
        return obj -> g.apply((ListAccessGrantsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3ControlRequest.Builder, SdkPojo, CopyableBuilder<Builder, ListAccessGrantsRequest> {
        /**
         * <p>
         * The Amazon Web Services account ID of the S3 Access Grants instance.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the S3 Access Grants instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * A pagination token to request the next page of results. Pass this value into a subsequent
         * <code>List Access Grants</code> request in order to retrieve the next page of results.
         * </p>
         * 
         * @param nextToken
         *        A pagination token to request the next page of results. Pass this value into a subsequent
         *        <code>List Access Grants</code> request in order to retrieve the next page of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        /**
         * <p>
         * The maximum number of access grants that you would like returned in the <code>List Access Grants</code>
         * response. If the results include the pagination token <code>NextToken</code>, make another call using the
         * <code>NextToken</code> to determine if there are more results.
         * </p>
         * 
         * @param maxResults
         *        The maximum number of access grants that you would like returned in the
         *        <code>List Access Grants</code> response. If the results include the pagination token
         *        <code>NextToken</code>, make another call using the <code>NextToken</code> to determine if there are
         *        more results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        /**
         * <p>
         * The type of the grantee to which access has been granted. It can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IAM</code> - An IAM user or role.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added your
         * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your
         * S3 Access Grants instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have added your
         * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your
         * S3 Access Grants instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param granteeType
         *        The type of the grantee to which access has been granted. It can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IAM</code> - An IAM user or role.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added
         *        your corporate identity directory to IAM Identity Center and associated the IAM Identity Center
         *        instance with your S3 Access Grants instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have
         *        added your corporate identity directory to IAM Identity Center and associated the IAM Identity Center
         *        instance with your S3 Access Grants instance.
         *        </p>
         *        </li>
         * @see GranteeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GranteeType
         */
        Builder granteeType(String granteeType);

        /**
         * <p>
         * The type of the grantee to which access has been granted. It can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>IAM</code> - An IAM user or role.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added your
         * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your
         * S3 Access Grants instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have added your
         * corporate identity directory to IAM Identity Center and associated the IAM Identity Center instance with your
         * S3 Access Grants instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param granteeType
         *        The type of the grantee to which access has been granted. It can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>IAM</code> - An IAM user or role.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DIRECTORY_USER</code> - Your corporate directory user. You can use this option if you have added
         *        your corporate identity directory to IAM Identity Center and associated the IAM Identity Center
         *        instance with your S3 Access Grants instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DIRECTORY_GROUP</code> - Your corporate directory group. You can use this option if you have
         *        added your corporate identity directory to IAM Identity Center and associated the IAM Identity Center
         *        instance with your S3 Access Grants instance.
         *        </p>
         *        </li>
         * @see GranteeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GranteeType
         */
        Builder granteeType(GranteeType granteeType);

        /**
         * <p>
         * The unique identifer of the <code>Grantee</code>. If the grantee type is <code>IAM</code>, the identifier is
         * the IAM Amazon Resource Name (ARN) of the user or role. If the grantee type is a directory user or group, the
         * identifier is 128-bit universally unique identifier (UUID) in the format
         * <code>a1b2c3d4-5678-90ab-cdef-EXAMPLE11111</code>. You can obtain this UUID from your Amazon Web Services IAM
         * Identity Center instance.
         * </p>
         * 
         * @param granteeIdentifier
         *        The unique identifer of the <code>Grantee</code>. If the grantee type is <code>IAM</code>, the
         *        identifier is the IAM Amazon Resource Name (ARN) of the user or role. If the grantee type is a
         *        directory user or group, the identifier is 128-bit universally unique identifier (UUID) in the format
         *        <code>a1b2c3d4-5678-90ab-cdef-EXAMPLE11111</code>. You can obtain this UUID from your Amazon Web
         *        Services IAM Identity Center instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder granteeIdentifier(String granteeIdentifier);

        /**
         * <p>
         * The type of permission granted to your S3 data, which can be set to one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>READ</code> – Grant read-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WRITE</code> – Grant write-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READWRITE</code> – Grant both read and write access to the S3 data.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permission
         *        The type of permission granted to your S3 data, which can be set to one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>READ</code> – Grant read-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WRITE</code> – Grant write-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READWRITE</code> – Grant both read and write access to the S3 data.
         *        </p>
         *        </li>
         * @see Permission
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Permission
         */
        Builder permission(String permission);

        /**
         * <p>
         * The type of permission granted to your S3 data, which can be set to one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>READ</code> – Grant read-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WRITE</code> – Grant write-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READWRITE</code> – Grant both read and write access to the S3 data.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permission
         *        The type of permission granted to your S3 data, which can be set to one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>READ</code> – Grant read-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WRITE</code> – Grant write-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READWRITE</code> – Grant both read and write access to the S3 data.
         *        </p>
         *        </li>
         * @see Permission
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Permission
         */
        Builder permission(Permission permission);

        /**
         * <p>
         * The S3 path of the data to which you are granting access. It is the result of appending the
         * <code>Subprefix</code> to the location scope.
         * </p>
         * 
         * @param grantScope
         *        The S3 path of the data to which you are granting access. It is the result of appending the
         *        <code>Subprefix</code> to the location scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantScope(String grantScope);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated with your
         * Identity Center instance. If the grant includes an application ARN, the grantee can only access the S3 data
         * through this application.
         * </p>
         * 
         * @param applicationArn
         *        The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated
         *        with your Identity Center instance. If the grant includes an application ARN, the grantee can only
         *        access the S3 data through this application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationArn(String applicationArn);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3ControlRequest.BuilderImpl implements Builder {
        private String accountId;

        private String nextToken;

        private Integer maxResults;

        private String granteeType;

        private String granteeIdentifier;

        private String permission;

        private String grantScope;

        private String applicationArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ListAccessGrantsRequest model) {
            super(model);
            accountId(model.accountId);
            nextToken(model.nextToken);
            maxResults(model.maxResults);
            granteeType(model.granteeType);
            granteeIdentifier(model.granteeIdentifier);
            permission(model.permission);
            grantScope(model.grantScope);
            applicationArn(model.applicationArn);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        public final String getGranteeType() {
            return granteeType;
        }

        public final void setGranteeType(String granteeType) {
            this.granteeType = granteeType;
        }

        @Override
        public final Builder granteeType(String granteeType) {
            this.granteeType = granteeType;
            return this;
        }

        @Override
        public final Builder granteeType(GranteeType granteeType) {
            this.granteeType(granteeType == null ? null : granteeType.toString());
            return this;
        }

        public final String getGranteeIdentifier() {
            return granteeIdentifier;
        }

        public final void setGranteeIdentifier(String granteeIdentifier) {
            this.granteeIdentifier = granteeIdentifier;
        }

        @Override
        public final Builder granteeIdentifier(String granteeIdentifier) {
            this.granteeIdentifier = granteeIdentifier;
            return this;
        }

        public final String getPermission() {
            return permission;
        }

        public final void setPermission(String permission) {
            this.permission = permission;
        }

        @Override
        public final Builder permission(String permission) {
            this.permission = permission;
            return this;
        }

        @Override
        public final Builder permission(Permission permission) {
            this.permission(permission == null ? null : permission.toString());
            return this;
        }

        public final String getGrantScope() {
            return grantScope;
        }

        public final void setGrantScope(String grantScope) {
            this.grantScope = grantScope;
        }

        @Override
        public final Builder grantScope(String grantScope) {
            this.grantScope = grantScope;
            return this;
        }

        public final String getApplicationArn() {
            return applicationArn;
        }

        public final void setApplicationArn(String applicationArn) {
            this.applicationArn = applicationArn;
        }

        @Override
        public final Builder applicationArn(String applicationArn) {
            this.applicationArn = applicationArn;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ListAccessGrantsRequest build() {
            return new ListAccessGrantsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
