/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAccessGrantRequest extends S3ControlRequest implements
        ToCopyableBuilder<CreateAccessGrantRequest.Builder, CreateAccessGrantRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccountId")
            .getter(getter(CreateAccessGrantRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-account-id")
                    .unmarshallLocationName("x-amz-account-id").build()).build();

    private static final SdkField<String> ACCESS_GRANTS_LOCATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccessGrantsLocationId")
            .getter(getter(CreateAccessGrantRequest::accessGrantsLocationId))
            .setter(setter(Builder::accessGrantsLocationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessGrantsLocationId")
                    .unmarshallLocationName("AccessGrantsLocationId").build()).build();

    private static final SdkField<AccessGrantsLocationConfiguration> ACCESS_GRANTS_LOCATION_CONFIGURATION_FIELD = SdkField
            .<AccessGrantsLocationConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("AccessGrantsLocationConfiguration")
            .getter(getter(CreateAccessGrantRequest::accessGrantsLocationConfiguration))
            .setter(setter(Builder::accessGrantsLocationConfiguration))
            .constructor(AccessGrantsLocationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessGrantsLocationConfiguration")
                    .unmarshallLocationName("AccessGrantsLocationConfiguration").build()).build();

    private static final SdkField<Grantee> GRANTEE_FIELD = SdkField
            .<Grantee> builder(MarshallingType.SDK_POJO)
            .memberName("Grantee")
            .getter(getter(CreateAccessGrantRequest::grantee))
            .setter(setter(Builder::grantee))
            .constructor(Grantee::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Grantee")
                    .unmarshallLocationName("Grantee").build()).build();

    private static final SdkField<String> PERMISSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Permission")
            .getter(getter(CreateAccessGrantRequest::permissionAsString))
            .setter(setter(Builder::permission))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Permission")
                    .unmarshallLocationName("Permission").build()).build();

    private static final SdkField<String> APPLICATION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ApplicationArn")
            .getter(getter(CreateAccessGrantRequest::applicationArn))
            .setter(setter(Builder::applicationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationArn")
                    .unmarshallLocationName("ApplicationArn").build()).build();

    private static final SdkField<String> S3_PREFIX_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("S3PrefixType")
            .getter(getter(CreateAccessGrantRequest::s3PrefixTypeAsString))
            .setter(setter(Builder::s3PrefixType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3PrefixType")
                    .unmarshallLocationName("S3PrefixType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateAccessGrantRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags")
                    .unmarshallLocationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").unmarshallLocationName("Tag").build()).build()).build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            ACCESS_GRANTS_LOCATION_ID_FIELD, ACCESS_GRANTS_LOCATION_CONFIGURATION_FIELD, GRANTEE_FIELD, PERMISSION_FIELD,
            APPLICATION_ARN_FIELD, S3_PREFIX_TYPE_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String accountId;

    private final String accessGrantsLocationId;

    private final AccessGrantsLocationConfiguration accessGrantsLocationConfiguration;

    private final Grantee grantee;

    private final String permission;

    private final String applicationArn;

    private final String s3PrefixType;

    private final List<Tag> tags;

    private CreateAccessGrantRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.accessGrantsLocationId = builder.accessGrantsLocationId;
        this.accessGrantsLocationConfiguration = builder.accessGrantsLocationConfiguration;
        this.grantee = builder.grantee;
        this.permission = builder.permission;
        this.applicationArn = builder.applicationArn;
        this.s3PrefixType = builder.s3PrefixType;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the S3 Access Grants instance.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the S3 Access Grants instance.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The ID of the registered location to which you are granting access. S3 Access Grants assigns this ID when you
     * register the location. S3 Access Grants assigns the ID <code>default</code> to the default location
     * <code>s3://</code> and assigns an auto-generated ID to other locations that you register.
     * </p>
     * <p>
     * If you are passing the <code>default</code> location, you cannot create an access grant for the entire default
     * location. You must also specify a bucket or a bucket and prefix in the <code>Subprefix</code> field.
     * </p>
     * 
     * @return The ID of the registered location to which you are granting access. S3 Access Grants assigns this ID when
     *         you register the location. S3 Access Grants assigns the ID <code>default</code> to the default location
     *         <code>s3://</code> and assigns an auto-generated ID to other locations that you register. </p>
     *         <p>
     *         If you are passing the <code>default</code> location, you cannot create an access grant for the entire
     *         default location. You must also specify a bucket or a bucket and prefix in the <code>Subprefix</code>
     *         field.
     */
    public final String accessGrantsLocationId() {
        return accessGrantsLocationId;
    }

    /**
     * <p>
     * The configuration options of the grant location. The grant location is the S3 path to the data to which you are
     * granting access. It contains the <code>S3SubPrefix</code> field. The grant scope is the result of appending the
     * subprefix to the location scope of the registered location.
     * </p>
     * 
     * @return The configuration options of the grant location. The grant location is the S3 path to the data to which
     *         you are granting access. It contains the <code>S3SubPrefix</code> field. The grant scope is the result of
     *         appending the subprefix to the location scope of the registered location.
     */
    public final AccessGrantsLocationConfiguration accessGrantsLocationConfiguration() {
        return accessGrantsLocationConfiguration;
    }

    /**
     * <p>
     * The user, group, or role to which you are granting access. You can grant access to an IAM user or role. If you
     * have added your corporate directory to Amazon Web Services IAM Identity Center and associated your Identity
     * Center instance with your S3 Access Grants instance, the grantee can also be a corporate directory user or group.
     * </p>
     * 
     * @return The user, group, or role to which you are granting access. You can grant access to an IAM user or role.
     *         If you have added your corporate directory to Amazon Web Services IAM Identity Center and associated your
     *         Identity Center instance with your S3 Access Grants instance, the grantee can also be a corporate
     *         directory user or group.
     */
    public final Grantee grantee() {
        return grantee;
    }

    /**
     * <p>
     * The type of access that you are granting to your S3 data, which can be set to one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>READ</code> – Grant read-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WRITE</code> – Grant write-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READWRITE</code> – Grant both read and write access to the S3 data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permission} will
     * return {@link Permission#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #permissionAsString}.
     * </p>
     * 
     * @return The type of access that you are granting to your S3 data, which can be set to one of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>READ</code> – Grant read-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WRITE</code> – Grant write-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READWRITE</code> – Grant both read and write access to the S3 data.
     *         </p>
     *         </li>
     * @see Permission
     */
    public final Permission permission() {
        return Permission.fromValue(permission);
    }

    /**
     * <p>
     * The type of access that you are granting to your S3 data, which can be set to one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>READ</code> – Grant read-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WRITE</code> – Grant write-only access to the S3 data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>READWRITE</code> – Grant both read and write access to the S3 data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permission} will
     * return {@link Permission#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #permissionAsString}.
     * </p>
     * 
     * @return The type of access that you are granting to your S3 data, which can be set to one of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>READ</code> – Grant read-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>WRITE</code> – Grant write-only access to the S3 data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>READWRITE</code> – Grant both read and write access to the S3 data.
     *         </p>
     *         </li>
     * @see Permission
     */
    public final String permissionAsString() {
        return permission;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated with your
     * Identity Center instance. If an application ARN is included in the request to create an access grant, the grantee
     * can only access the S3 data through this application.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated with
     *         your Identity Center instance. If an application ARN is included in the request to create an access
     *         grant, the grantee can only access the S3 data through this application.
     */
    public final String applicationArn() {
        return applicationArn;
    }

    /**
     * <p>
     * The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value if the
     * access grant scope is an object. Do not pass this value if the access grant scope is a bucket or a bucket and a
     * prefix.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3PrefixType} will
     * return {@link S3PrefixType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3PrefixTypeAsString}.
     * </p>
     * 
     * @return The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value if
     *         the access grant scope is an object. Do not pass this value if the access grant scope is a bucket or a
     *         bucket and a prefix.
     * @see S3PrefixType
     */
    public final S3PrefixType s3PrefixType() {
        return S3PrefixType.fromValue(s3PrefixType);
    }

    /**
     * <p>
     * The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value if the
     * access grant scope is an object. Do not pass this value if the access grant scope is a bucket or a bucket and a
     * prefix.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3PrefixType} will
     * return {@link S3PrefixType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #s3PrefixTypeAsString}.
     * </p>
     * 
     * @return The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value if
     *         the access grant scope is an object. Do not pass this value if the access grant scope is a bucket or a
     *         bucket and a prefix.
     * @see S3PrefixType
     */
    public final String s3PrefixTypeAsString() {
        return s3PrefixType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services resource tags that you are adding to the access grant. Each tag is a label consisting of
     * a user-defined key and value. Tags can help you manage, identify, organize, search for, and filter resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The Amazon Web Services resource tags that you are adding to the access grant. Each tag is a label
     *         consisting of a user-defined key and value. Tags can help you manage, identify, organize, search for, and
     *         filter resources.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(accessGrantsLocationId());
        hashCode = 31 * hashCode + Objects.hashCode(accessGrantsLocationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(grantee());
        hashCode = 31 * hashCode + Objects.hashCode(permissionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(applicationArn());
        hashCode = 31 * hashCode + Objects.hashCode(s3PrefixTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAccessGrantRequest)) {
            return false;
        }
        CreateAccessGrantRequest other = (CreateAccessGrantRequest) obj;
        return Objects.equals(accountId(), other.accountId())
                && Objects.equals(accessGrantsLocationId(), other.accessGrantsLocationId())
                && Objects.equals(accessGrantsLocationConfiguration(), other.accessGrantsLocationConfiguration())
                && Objects.equals(grantee(), other.grantee()) && Objects.equals(permissionAsString(), other.permissionAsString())
                && Objects.equals(applicationArn(), other.applicationArn())
                && Objects.equals(s3PrefixTypeAsString(), other.s3PrefixTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateAccessGrantRequest").add("AccountId", accountId())
                .add("AccessGrantsLocationId", accessGrantsLocationId())
                .add("AccessGrantsLocationConfiguration", accessGrantsLocationConfiguration()).add("Grantee", grantee())
                .add("Permission", permissionAsString()).add("ApplicationArn", applicationArn())
                .add("S3PrefixType", s3PrefixTypeAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "AccessGrantsLocationId":
            return Optional.ofNullable(clazz.cast(accessGrantsLocationId()));
        case "AccessGrantsLocationConfiguration":
            return Optional.ofNullable(clazz.cast(accessGrantsLocationConfiguration()));
        case "Grantee":
            return Optional.ofNullable(clazz.cast(grantee()));
        case "Permission":
            return Optional.ofNullable(clazz.cast(permissionAsString()));
        case "ApplicationArn":
            return Optional.ofNullable(clazz.cast(applicationArn()));
        case "S3PrefixType":
            return Optional.ofNullable(clazz.cast(s3PrefixTypeAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("x-amz-account-id", ACCOUNT_ID_FIELD);
        map.put("AccessGrantsLocationId", ACCESS_GRANTS_LOCATION_ID_FIELD);
        map.put("AccessGrantsLocationConfiguration", ACCESS_GRANTS_LOCATION_CONFIGURATION_FIELD);
        map.put("Grantee", GRANTEE_FIELD);
        map.put("Permission", PERMISSION_FIELD);
        map.put("ApplicationArn", APPLICATION_ARN_FIELD);
        map.put("S3PrefixType", S3_PREFIX_TYPE_FIELD);
        map.put("Tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateAccessGrantRequest, T> g) {
        return obj -> g.apply((CreateAccessGrantRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3ControlRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateAccessGrantRequest> {
        /**
         * <p>
         * The Amazon Web Services account ID of the S3 Access Grants instance.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the S3 Access Grants instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The ID of the registered location to which you are granting access. S3 Access Grants assigns this ID when you
         * register the location. S3 Access Grants assigns the ID <code>default</code> to the default location
         * <code>s3://</code> and assigns an auto-generated ID to other locations that you register.
         * </p>
         * <p>
         * If you are passing the <code>default</code> location, you cannot create an access grant for the entire
         * default location. You must also specify a bucket or a bucket and prefix in the <code>Subprefix</code> field.
         * </p>
         * 
         * @param accessGrantsLocationId
         *        The ID of the registered location to which you are granting access. S3 Access Grants assigns this ID
         *        when you register the location. S3 Access Grants assigns the ID <code>default</code> to the default
         *        location <code>s3://</code> and assigns an auto-generated ID to other locations that you register.
         *        </p>
         *        <p>
         *        If you are passing the <code>default</code> location, you cannot create an access grant for the entire
         *        default location. You must also specify a bucket or a bucket and prefix in the <code>Subprefix</code>
         *        field.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessGrantsLocationId(String accessGrantsLocationId);

        /**
         * <p>
         * The configuration options of the grant location. The grant location is the S3 path to the data to which you
         * are granting access. It contains the <code>S3SubPrefix</code> field. The grant scope is the result of
         * appending the subprefix to the location scope of the registered location.
         * </p>
         * 
         * @param accessGrantsLocationConfiguration
         *        The configuration options of the grant location. The grant location is the S3 path to the data to
         *        which you are granting access. It contains the <code>S3SubPrefix</code> field. The grant scope is the
         *        result of appending the subprefix to the location scope of the registered location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessGrantsLocationConfiguration(AccessGrantsLocationConfiguration accessGrantsLocationConfiguration);

        /**
         * <p>
         * The configuration options of the grant location. The grant location is the S3 path to the data to which you
         * are granting access. It contains the <code>S3SubPrefix</code> field. The grant scope is the result of
         * appending the subprefix to the location scope of the registered location.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link AccessGrantsLocationConfiguration.Builder} avoiding the need to create one manually via
         * {@link AccessGrantsLocationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AccessGrantsLocationConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #accessGrantsLocationConfiguration(AccessGrantsLocationConfiguration)}.
         * 
         * @param accessGrantsLocationConfiguration
         *        a consumer that will call methods on {@link AccessGrantsLocationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessGrantsLocationConfiguration(AccessGrantsLocationConfiguration)
         */
        default Builder accessGrantsLocationConfiguration(
                Consumer<AccessGrantsLocationConfiguration.Builder> accessGrantsLocationConfiguration) {
            return accessGrantsLocationConfiguration(AccessGrantsLocationConfiguration.builder()
                    .applyMutation(accessGrantsLocationConfiguration).build());
        }

        /**
         * <p>
         * The user, group, or role to which you are granting access. You can grant access to an IAM user or role. If
         * you have added your corporate directory to Amazon Web Services IAM Identity Center and associated your
         * Identity Center instance with your S3 Access Grants instance, the grantee can also be a corporate directory
         * user or group.
         * </p>
         * 
         * @param grantee
         *        The user, group, or role to which you are granting access. You can grant access to an IAM user or
         *        role. If you have added your corporate directory to Amazon Web Services IAM Identity Center and
         *        associated your Identity Center instance with your S3 Access Grants instance, the grantee can also be
         *        a corporate directory user or group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder grantee(Grantee grantee);

        /**
         * <p>
         * The user, group, or role to which you are granting access. You can grant access to an IAM user or role. If
         * you have added your corporate directory to Amazon Web Services IAM Identity Center and associated your
         * Identity Center instance with your S3 Access Grants instance, the grantee can also be a corporate directory
         * user or group.
         * </p>
         * This is a convenience method that creates an instance of the {@link Grantee.Builder} avoiding the need to
         * create one manually via {@link Grantee#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Grantee.Builder#build()} is called immediately and its result is
         * passed to {@link #grantee(Grantee)}.
         * 
         * @param grantee
         *        a consumer that will call methods on {@link Grantee.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #grantee(Grantee)
         */
        default Builder grantee(Consumer<Grantee.Builder> grantee) {
            return grantee(Grantee.builder().applyMutation(grantee).build());
        }

        /**
         * <p>
         * The type of access that you are granting to your S3 data, which can be set to one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>READ</code> – Grant read-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WRITE</code> – Grant write-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READWRITE</code> – Grant both read and write access to the S3 data.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permission
         *        The type of access that you are granting to your S3 data, which can be set to one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>READ</code> – Grant read-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WRITE</code> – Grant write-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READWRITE</code> – Grant both read and write access to the S3 data.
         *        </p>
         *        </li>
         * @see Permission
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Permission
         */
        Builder permission(String permission);

        /**
         * <p>
         * The type of access that you are granting to your S3 data, which can be set to one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>READ</code> – Grant read-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>WRITE</code> – Grant write-only access to the S3 data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>READWRITE</code> – Grant both read and write access to the S3 data.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permission
         *        The type of access that you are granting to your S3 data, which can be set to one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>READ</code> – Grant read-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>WRITE</code> – Grant write-only access to the S3 data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>READWRITE</code> – Grant both read and write access to the S3 data.
         *        </p>
         *        </li>
         * @see Permission
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Permission
         */
        Builder permission(Permission permission);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated with your
         * Identity Center instance. If an application ARN is included in the request to create an access grant, the
         * grantee can only access the S3 data through this application.
         * </p>
         * 
         * @param applicationArn
         *        The Amazon Resource Name (ARN) of an Amazon Web Services IAM Identity Center application associated
         *        with your Identity Center instance. If an application ARN is included in the request to create an
         *        access grant, the grantee can only access the S3 data through this application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationArn(String applicationArn);

        /**
         * <p>
         * The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value if the
         * access grant scope is an object. Do not pass this value if the access grant scope is a bucket or a bucket and
         * a prefix.
         * </p>
         * 
         * @param s3PrefixType
         *        The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value
         *        if the access grant scope is an object. Do not pass this value if the access grant scope is a bucket
         *        or a bucket and a prefix.
         * @see S3PrefixType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3PrefixType
         */
        Builder s3PrefixType(String s3PrefixType);

        /**
         * <p>
         * The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value if the
         * access grant scope is an object. Do not pass this value if the access grant scope is a bucket or a bucket and
         * a prefix.
         * </p>
         * 
         * @param s3PrefixType
         *        The type of <code>S3SubPrefix</code>. The only possible value is <code>Object</code>. Pass this value
         *        if the access grant scope is an object. Do not pass this value if the access grant scope is a bucket
         *        or a bucket and a prefix.
         * @see S3PrefixType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3PrefixType
         */
        Builder s3PrefixType(S3PrefixType s3PrefixType);

        /**
         * <p>
         * The Amazon Web Services resource tags that you are adding to the access grant. Each tag is a label consisting
         * of a user-defined key and value. Tags can help you manage, identify, organize, search for, and filter
         * resources.
         * </p>
         * 
         * @param tags
         *        The Amazon Web Services resource tags that you are adding to the access grant. Each tag is a label
         *        consisting of a user-defined key and value. Tags can help you manage, identify, organize, search for,
         *        and filter resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The Amazon Web Services resource tags that you are adding to the access grant. Each tag is a label consisting
         * of a user-defined key and value. Tags can help you manage, identify, organize, search for, and filter
         * resources.
         * </p>
         * 
         * @param tags
         *        The Amazon Web Services resource tags that you are adding to the access grant. Each tag is a label
         *        consisting of a user-defined key and value. Tags can help you manage, identify, organize, search for,
         *        and filter resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The Amazon Web Services resource tags that you are adding to the access grant. Each tag is a label consisting
         * of a user-defined key and value. Tags can help you manage, identify, organize, search for, and filter
         * resources.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.s3control.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.s3control.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.s3control.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.s3control.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3ControlRequest.BuilderImpl implements Builder {
        private String accountId;

        private String accessGrantsLocationId;

        private AccessGrantsLocationConfiguration accessGrantsLocationConfiguration;

        private Grantee grantee;

        private String permission;

        private String applicationArn;

        private String s3PrefixType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAccessGrantRequest model) {
            super(model);
            accountId(model.accountId);
            accessGrantsLocationId(model.accessGrantsLocationId);
            accessGrantsLocationConfiguration(model.accessGrantsLocationConfiguration);
            grantee(model.grantee);
            permission(model.permission);
            applicationArn(model.applicationArn);
            s3PrefixType(model.s3PrefixType);
            tags(model.tags);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getAccessGrantsLocationId() {
            return accessGrantsLocationId;
        }

        public final void setAccessGrantsLocationId(String accessGrantsLocationId) {
            this.accessGrantsLocationId = accessGrantsLocationId;
        }

        @Override
        public final Builder accessGrantsLocationId(String accessGrantsLocationId) {
            this.accessGrantsLocationId = accessGrantsLocationId;
            return this;
        }

        public final AccessGrantsLocationConfiguration.Builder getAccessGrantsLocationConfiguration() {
            return accessGrantsLocationConfiguration != null ? accessGrantsLocationConfiguration.toBuilder() : null;
        }

        public final void setAccessGrantsLocationConfiguration(
                AccessGrantsLocationConfiguration.BuilderImpl accessGrantsLocationConfiguration) {
            this.accessGrantsLocationConfiguration = accessGrantsLocationConfiguration != null ? accessGrantsLocationConfiguration
                    .build() : null;
        }

        @Override
        public final Builder accessGrantsLocationConfiguration(AccessGrantsLocationConfiguration accessGrantsLocationConfiguration) {
            this.accessGrantsLocationConfiguration = accessGrantsLocationConfiguration;
            return this;
        }

        public final Grantee.Builder getGrantee() {
            return grantee != null ? grantee.toBuilder() : null;
        }

        public final void setGrantee(Grantee.BuilderImpl grantee) {
            this.grantee = grantee != null ? grantee.build() : null;
        }

        @Override
        public final Builder grantee(Grantee grantee) {
            this.grantee = grantee;
            return this;
        }

        public final String getPermission() {
            return permission;
        }

        public final void setPermission(String permission) {
            this.permission = permission;
        }

        @Override
        public final Builder permission(String permission) {
            this.permission = permission;
            return this;
        }

        @Override
        public final Builder permission(Permission permission) {
            this.permission(permission == null ? null : permission.toString());
            return this;
        }

        public final String getApplicationArn() {
            return applicationArn;
        }

        public final void setApplicationArn(String applicationArn) {
            this.applicationArn = applicationArn;
        }

        @Override
        public final Builder applicationArn(String applicationArn) {
            this.applicationArn = applicationArn;
            return this;
        }

        public final String getS3PrefixType() {
            return s3PrefixType;
        }

        public final void setS3PrefixType(String s3PrefixType) {
            this.s3PrefixType = s3PrefixType;
        }

        @Override
        public final Builder s3PrefixType(String s3PrefixType) {
            this.s3PrefixType = s3PrefixType;
            return this;
        }

        @Override
        public final Builder s3PrefixType(S3PrefixType s3PrefixType) {
            this.s3PrefixType(s3PrefixType == null ? null : s3PrefixType.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateAccessGrantRequest build() {
            return new CreateAccessGrantRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
