/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The filter used to describe a set of objects for the job's manifest.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobManifestGeneratorFilter implements SdkPojo, Serializable,
        ToCopyableBuilder<JobManifestGeneratorFilter.Builder, JobManifestGeneratorFilter> {
    private static final SdkField<Boolean> ELIGIBLE_FOR_REPLICATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EligibleForReplication")
            .getter(getter(JobManifestGeneratorFilter::eligibleForReplication))
            .setter(setter(Builder::eligibleForReplication))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EligibleForReplication")
                    .unmarshallLocationName("EligibleForReplication").build()).build();

    private static final SdkField<Instant> CREATED_AFTER_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAfter")
            .getter(getter(JobManifestGeneratorFilter::createdAfter))
            .setter(setter(Builder::createdAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAfter")
                    .unmarshallLocationName("CreatedAfter").build()).build();

    private static final SdkField<Instant> CREATED_BEFORE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedBefore")
            .getter(getter(JobManifestGeneratorFilter::createdBefore))
            .setter(setter(Builder::createdBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBefore")
                    .unmarshallLocationName("CreatedBefore").build()).build();

    private static final SdkField<List<String>> OBJECT_REPLICATION_STATUSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ObjectReplicationStatuses")
            .getter(getter(JobManifestGeneratorFilter::objectReplicationStatusesAsStrings))
            .setter(setter(Builder::objectReplicationStatusesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectReplicationStatuses")
                    .unmarshallLocationName("ObjectReplicationStatuses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<KeyNameConstraint> KEY_NAME_CONSTRAINT_FIELD = SdkField
            .<KeyNameConstraint> builder(MarshallingType.SDK_POJO)
            .memberName("KeyNameConstraint")
            .getter(getter(JobManifestGeneratorFilter::keyNameConstraint))
            .setter(setter(Builder::keyNameConstraint))
            .constructor(KeyNameConstraint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyNameConstraint")
                    .unmarshallLocationName("KeyNameConstraint").build()).build();

    private static final SdkField<Long> OBJECT_SIZE_GREATER_THAN_BYTES_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("ObjectSizeGreaterThanBytes")
            .getter(getter(JobManifestGeneratorFilter::objectSizeGreaterThanBytes))
            .setter(setter(Builder::objectSizeGreaterThanBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectSizeGreaterThanBytes")
                    .unmarshallLocationName("ObjectSizeGreaterThanBytes").build()).build();

    private static final SdkField<Long> OBJECT_SIZE_LESS_THAN_BYTES_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("ObjectSizeLessThanBytes")
            .getter(getter(JobManifestGeneratorFilter::objectSizeLessThanBytes))
            .setter(setter(Builder::objectSizeLessThanBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectSizeLessThanBytes")
                    .unmarshallLocationName("ObjectSizeLessThanBytes").build()).build();

    private static final SdkField<List<String>> MATCH_ANY_STORAGE_CLASS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MatchAnyStorageClass")
            .getter(getter(JobManifestGeneratorFilter::matchAnyStorageClassAsStrings))
            .setter(setter(Builder::matchAnyStorageClassWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MatchAnyStorageClass")
                    .unmarshallLocationName("MatchAnyStorageClass").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ELIGIBLE_FOR_REPLICATION_FIELD, CREATED_AFTER_FIELD, CREATED_BEFORE_FIELD, OBJECT_REPLICATION_STATUSES_FIELD,
            KEY_NAME_CONSTRAINT_FIELD, OBJECT_SIZE_GREATER_THAN_BYTES_FIELD, OBJECT_SIZE_LESS_THAN_BYTES_FIELD,
            MATCH_ANY_STORAGE_CLASS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean eligibleForReplication;

    private final Instant createdAfter;

    private final Instant createdBefore;

    private final List<String> objectReplicationStatuses;

    private final KeyNameConstraint keyNameConstraint;

    private final Long objectSizeGreaterThanBytes;

    private final Long objectSizeLessThanBytes;

    private final List<String> matchAnyStorageClass;

    private JobManifestGeneratorFilter(BuilderImpl builder) {
        this.eligibleForReplication = builder.eligibleForReplication;
        this.createdAfter = builder.createdAfter;
        this.createdBefore = builder.createdBefore;
        this.objectReplicationStatuses = builder.objectReplicationStatuses;
        this.keyNameConstraint = builder.keyNameConstraint;
        this.objectSizeGreaterThanBytes = builder.objectSizeGreaterThanBytes;
        this.objectSizeLessThanBytes = builder.objectSizeLessThanBytes;
        this.matchAnyStorageClass = builder.matchAnyStorageClass;
    }

    /**
     * <p>
     * Include objects in the generated manifest only if they are eligible for replication according to the Replication
     * configuration on the source bucket.
     * </p>
     * 
     * @return Include objects in the generated manifest only if they are eligible for replication according to the
     *         Replication configuration on the source bucket.
     */
    public final Boolean eligibleForReplication() {
        return eligibleForReplication;
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects that were created after this time.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects that were created after this
     *         time.
     */
    public final Instant createdAfter() {
        return createdAfter;
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects that were created before this time.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects that were created before this
     *         time.
     */
    public final Instant createdBefore() {
        return createdBefore;
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects that have one of the specified
     * Replication statuses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasObjectReplicationStatuses} method.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects that have one of the specified
     *         Replication statuses.
     */
    public final List<ReplicationStatus> objectReplicationStatuses() {
        return ReplicationStatusFilterListCopier.copyStringToEnum(objectReplicationStatuses);
    }

    /**
     * For responses, this returns true if the service returned a value for the ObjectReplicationStatuses property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasObjectReplicationStatuses() {
        return objectReplicationStatuses != null && !(objectReplicationStatuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects that have one of the specified
     * Replication statuses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasObjectReplicationStatuses} method.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects that have one of the specified
     *         Replication statuses.
     */
    public final List<String> objectReplicationStatusesAsStrings() {
        return objectReplicationStatuses;
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects whose object keys match the string
     * constraints specified for <code>MatchAnyPrefix</code>, <code>MatchAnySuffix</code>, and
     * <code>MatchAnySubstring</code>.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects whose object keys match the
     *         string constraints specified for <code>MatchAnyPrefix</code>, <code>MatchAnySuffix</code>, and
     *         <code>MatchAnySubstring</code>.
     */
    public final KeyNameConstraint keyNameConstraint() {
        return keyNameConstraint;
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects whose file size is greater than the
     * specified number of bytes.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects whose file size is greater than
     *         the specified number of bytes.
     */
    public final Long objectSizeGreaterThanBytes() {
        return objectSizeGreaterThanBytes;
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects whose file size is less than the
     * specified number of bytes.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects whose file size is less than the
     *         specified number of bytes.
     */
    public final Long objectSizeLessThanBytes() {
        return objectSizeLessThanBytes;
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects that are stored with the specified
     * storage class.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMatchAnyStorageClass} method.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects that are stored with the
     *         specified storage class.
     */
    public final List<S3StorageClass> matchAnyStorageClass() {
        return StorageClassListCopier.copyStringToEnum(matchAnyStorageClass);
    }

    /**
     * For responses, this returns true if the service returned a value for the MatchAnyStorageClass property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMatchAnyStorageClass() {
        return matchAnyStorageClass != null && !(matchAnyStorageClass instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If provided, the generated manifest includes only source bucket objects that are stored with the specified
     * storage class.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMatchAnyStorageClass} method.
     * </p>
     * 
     * @return If provided, the generated manifest includes only source bucket objects that are stored with the
     *         specified storage class.
     */
    public final List<String> matchAnyStorageClassAsStrings() {
        return matchAnyStorageClass;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eligibleForReplication());
        hashCode = 31 * hashCode + Objects.hashCode(createdAfter());
        hashCode = 31 * hashCode + Objects.hashCode(createdBefore());
        hashCode = 31 * hashCode + Objects.hashCode(hasObjectReplicationStatuses() ? objectReplicationStatusesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(keyNameConstraint());
        hashCode = 31 * hashCode + Objects.hashCode(objectSizeGreaterThanBytes());
        hashCode = 31 * hashCode + Objects.hashCode(objectSizeLessThanBytes());
        hashCode = 31 * hashCode + Objects.hashCode(hasMatchAnyStorageClass() ? matchAnyStorageClassAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobManifestGeneratorFilter)) {
            return false;
        }
        JobManifestGeneratorFilter other = (JobManifestGeneratorFilter) obj;
        return Objects.equals(eligibleForReplication(), other.eligibleForReplication())
                && Objects.equals(createdAfter(), other.createdAfter()) && Objects.equals(createdBefore(), other.createdBefore())
                && hasObjectReplicationStatuses() == other.hasObjectReplicationStatuses()
                && Objects.equals(objectReplicationStatusesAsStrings(), other.objectReplicationStatusesAsStrings())
                && Objects.equals(keyNameConstraint(), other.keyNameConstraint())
                && Objects.equals(objectSizeGreaterThanBytes(), other.objectSizeGreaterThanBytes())
                && Objects.equals(objectSizeLessThanBytes(), other.objectSizeLessThanBytes())
                && hasMatchAnyStorageClass() == other.hasMatchAnyStorageClass()
                && Objects.equals(matchAnyStorageClassAsStrings(), other.matchAnyStorageClassAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobManifestGeneratorFilter").add("EligibleForReplication", eligibleForReplication())
                .add("CreatedAfter", createdAfter()).add("CreatedBefore", createdBefore())
                .add("ObjectReplicationStatuses", hasObjectReplicationStatuses() ? objectReplicationStatusesAsStrings() : null)
                .add("KeyNameConstraint", keyNameConstraint()).add("ObjectSizeGreaterThanBytes", objectSizeGreaterThanBytes())
                .add("ObjectSizeLessThanBytes", objectSizeLessThanBytes())
                .add("MatchAnyStorageClass", hasMatchAnyStorageClass() ? matchAnyStorageClassAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EligibleForReplication":
            return Optional.ofNullable(clazz.cast(eligibleForReplication()));
        case "CreatedAfter":
            return Optional.ofNullable(clazz.cast(createdAfter()));
        case "CreatedBefore":
            return Optional.ofNullable(clazz.cast(createdBefore()));
        case "ObjectReplicationStatuses":
            return Optional.ofNullable(clazz.cast(objectReplicationStatusesAsStrings()));
        case "KeyNameConstraint":
            return Optional.ofNullable(clazz.cast(keyNameConstraint()));
        case "ObjectSizeGreaterThanBytes":
            return Optional.ofNullable(clazz.cast(objectSizeGreaterThanBytes()));
        case "ObjectSizeLessThanBytes":
            return Optional.ofNullable(clazz.cast(objectSizeLessThanBytes()));
        case "MatchAnyStorageClass":
            return Optional.ofNullable(clazz.cast(matchAnyStorageClassAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EligibleForReplication", ELIGIBLE_FOR_REPLICATION_FIELD);
        map.put("CreatedAfter", CREATED_AFTER_FIELD);
        map.put("CreatedBefore", CREATED_BEFORE_FIELD);
        map.put("ObjectReplicationStatuses", OBJECT_REPLICATION_STATUSES_FIELD);
        map.put("KeyNameConstraint", KEY_NAME_CONSTRAINT_FIELD);
        map.put("ObjectSizeGreaterThanBytes", OBJECT_SIZE_GREATER_THAN_BYTES_FIELD);
        map.put("ObjectSizeLessThanBytes", OBJECT_SIZE_LESS_THAN_BYTES_FIELD);
        map.put("MatchAnyStorageClass", MATCH_ANY_STORAGE_CLASS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<JobManifestGeneratorFilter, T> g) {
        return obj -> g.apply((JobManifestGeneratorFilter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobManifestGeneratorFilter> {
        /**
         * <p>
         * Include objects in the generated manifest only if they are eligible for replication according to the
         * Replication configuration on the source bucket.
         * </p>
         * 
         * @param eligibleForReplication
         *        Include objects in the generated manifest only if they are eligible for replication according to the
         *        Replication configuration on the source bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eligibleForReplication(Boolean eligibleForReplication);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that were created after this time.
         * </p>
         * 
         * @param createdAfter
         *        If provided, the generated manifest includes only source bucket objects that were created after this
         *        time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAfter(Instant createdAfter);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that were created before this time.
         * </p>
         * 
         * @param createdBefore
         *        If provided, the generated manifest includes only source bucket objects that were created before this
         *        time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBefore(Instant createdBefore);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest includes only source bucket objects that have one of the specified
         *        Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatusesWithStrings(Collection<String> objectReplicationStatuses);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest includes only source bucket objects that have one of the specified
         *        Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatusesWithStrings(String... objectReplicationStatuses);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest includes only source bucket objects that have one of the specified
         *        Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatuses(Collection<ReplicationStatus> objectReplicationStatuses);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that have one of the specified
         * Replication statuses.
         * </p>
         * 
         * @param objectReplicationStatuses
         *        If provided, the generated manifest includes only source bucket objects that have one of the specified
         *        Replication statuses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectReplicationStatuses(ReplicationStatus... objectReplicationStatuses);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects whose object keys match the string
         * constraints specified for <code>MatchAnyPrefix</code>, <code>MatchAnySuffix</code>, and
         * <code>MatchAnySubstring</code>.
         * </p>
         * 
         * @param keyNameConstraint
         *        If provided, the generated manifest includes only source bucket objects whose object keys match the
         *        string constraints specified for <code>MatchAnyPrefix</code>, <code>MatchAnySuffix</code>, and
         *        <code>MatchAnySubstring</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyNameConstraint(KeyNameConstraint keyNameConstraint);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects whose object keys match the string
         * constraints specified for <code>MatchAnyPrefix</code>, <code>MatchAnySuffix</code>, and
         * <code>MatchAnySubstring</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link KeyNameConstraint.Builder} avoiding the
         * need to create one manually via {@link KeyNameConstraint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KeyNameConstraint.Builder#build()} is called immediately and its
         * result is passed to {@link #keyNameConstraint(KeyNameConstraint)}.
         * 
         * @param keyNameConstraint
         *        a consumer that will call methods on {@link KeyNameConstraint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keyNameConstraint(KeyNameConstraint)
         */
        default Builder keyNameConstraint(Consumer<KeyNameConstraint.Builder> keyNameConstraint) {
            return keyNameConstraint(KeyNameConstraint.builder().applyMutation(keyNameConstraint).build());
        }

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects whose file size is greater than the
         * specified number of bytes.
         * </p>
         * 
         * @param objectSizeGreaterThanBytes
         *        If provided, the generated manifest includes only source bucket objects whose file size is greater
         *        than the specified number of bytes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectSizeGreaterThanBytes(Long objectSizeGreaterThanBytes);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects whose file size is less than the
         * specified number of bytes.
         * </p>
         * 
         * @param objectSizeLessThanBytes
         *        If provided, the generated manifest includes only source bucket objects whose file size is less than
         *        the specified number of bytes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectSizeLessThanBytes(Long objectSizeLessThanBytes);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that are stored with the specified
         * storage class.
         * </p>
         * 
         * @param matchAnyStorageClass
         *        If provided, the generated manifest includes only source bucket objects that are stored with the
         *        specified storage class.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchAnyStorageClassWithStrings(Collection<String> matchAnyStorageClass);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that are stored with the specified
         * storage class.
         * </p>
         * 
         * @param matchAnyStorageClass
         *        If provided, the generated manifest includes only source bucket objects that are stored with the
         *        specified storage class.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchAnyStorageClassWithStrings(String... matchAnyStorageClass);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that are stored with the specified
         * storage class.
         * </p>
         * 
         * @param matchAnyStorageClass
         *        If provided, the generated manifest includes only source bucket objects that are stored with the
         *        specified storage class.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchAnyStorageClass(Collection<S3StorageClass> matchAnyStorageClass);

        /**
         * <p>
         * If provided, the generated manifest includes only source bucket objects that are stored with the specified
         * storage class.
         * </p>
         * 
         * @param matchAnyStorageClass
         *        If provided, the generated manifest includes only source bucket objects that are stored with the
         *        specified storage class.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder matchAnyStorageClass(S3StorageClass... matchAnyStorageClass);
    }

    static final class BuilderImpl implements Builder {
        private Boolean eligibleForReplication;

        private Instant createdAfter;

        private Instant createdBefore;

        private List<String> objectReplicationStatuses = DefaultSdkAutoConstructList.getInstance();

        private KeyNameConstraint keyNameConstraint;

        private Long objectSizeGreaterThanBytes;

        private Long objectSizeLessThanBytes;

        private List<String> matchAnyStorageClass = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(JobManifestGeneratorFilter model) {
            eligibleForReplication(model.eligibleForReplication);
            createdAfter(model.createdAfter);
            createdBefore(model.createdBefore);
            objectReplicationStatusesWithStrings(model.objectReplicationStatuses);
            keyNameConstraint(model.keyNameConstraint);
            objectSizeGreaterThanBytes(model.objectSizeGreaterThanBytes);
            objectSizeLessThanBytes(model.objectSizeLessThanBytes);
            matchAnyStorageClassWithStrings(model.matchAnyStorageClass);
        }

        public final Boolean getEligibleForReplication() {
            return eligibleForReplication;
        }

        public final void setEligibleForReplication(Boolean eligibleForReplication) {
            this.eligibleForReplication = eligibleForReplication;
        }

        @Override
        public final Builder eligibleForReplication(Boolean eligibleForReplication) {
            this.eligibleForReplication = eligibleForReplication;
            return this;
        }

        public final Instant getCreatedAfter() {
            return createdAfter;
        }

        public final void setCreatedAfter(Instant createdAfter) {
            this.createdAfter = createdAfter;
        }

        @Override
        public final Builder createdAfter(Instant createdAfter) {
            this.createdAfter = createdAfter;
            return this;
        }

        public final Instant getCreatedBefore() {
            return createdBefore;
        }

        public final void setCreatedBefore(Instant createdBefore) {
            this.createdBefore = createdBefore;
        }

        @Override
        public final Builder createdBefore(Instant createdBefore) {
            this.createdBefore = createdBefore;
            return this;
        }

        public final Collection<String> getObjectReplicationStatuses() {
            if (objectReplicationStatuses instanceof SdkAutoConstructList) {
                return null;
            }
            return objectReplicationStatuses;
        }

        public final void setObjectReplicationStatuses(Collection<String> objectReplicationStatuses) {
            this.objectReplicationStatuses = ReplicationStatusFilterListCopier.copy(objectReplicationStatuses);
        }

        @Override
        public final Builder objectReplicationStatusesWithStrings(Collection<String> objectReplicationStatuses) {
            this.objectReplicationStatuses = ReplicationStatusFilterListCopier.copy(objectReplicationStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder objectReplicationStatusesWithStrings(String... objectReplicationStatuses) {
            objectReplicationStatusesWithStrings(Arrays.asList(objectReplicationStatuses));
            return this;
        }

        @Override
        public final Builder objectReplicationStatuses(Collection<ReplicationStatus> objectReplicationStatuses) {
            this.objectReplicationStatuses = ReplicationStatusFilterListCopier.copyEnumToString(objectReplicationStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder objectReplicationStatuses(ReplicationStatus... objectReplicationStatuses) {
            objectReplicationStatuses(Arrays.asList(objectReplicationStatuses));
            return this;
        }

        public final KeyNameConstraint.Builder getKeyNameConstraint() {
            return keyNameConstraint != null ? keyNameConstraint.toBuilder() : null;
        }

        public final void setKeyNameConstraint(KeyNameConstraint.BuilderImpl keyNameConstraint) {
            this.keyNameConstraint = keyNameConstraint != null ? keyNameConstraint.build() : null;
        }

        @Override
        public final Builder keyNameConstraint(KeyNameConstraint keyNameConstraint) {
            this.keyNameConstraint = keyNameConstraint;
            return this;
        }

        public final Long getObjectSizeGreaterThanBytes() {
            return objectSizeGreaterThanBytes;
        }

        public final void setObjectSizeGreaterThanBytes(Long objectSizeGreaterThanBytes) {
            this.objectSizeGreaterThanBytes = objectSizeGreaterThanBytes;
        }

        @Override
        public final Builder objectSizeGreaterThanBytes(Long objectSizeGreaterThanBytes) {
            this.objectSizeGreaterThanBytes = objectSizeGreaterThanBytes;
            return this;
        }

        public final Long getObjectSizeLessThanBytes() {
            return objectSizeLessThanBytes;
        }

        public final void setObjectSizeLessThanBytes(Long objectSizeLessThanBytes) {
            this.objectSizeLessThanBytes = objectSizeLessThanBytes;
        }

        @Override
        public final Builder objectSizeLessThanBytes(Long objectSizeLessThanBytes) {
            this.objectSizeLessThanBytes = objectSizeLessThanBytes;
            return this;
        }

        public final Collection<String> getMatchAnyStorageClass() {
            if (matchAnyStorageClass instanceof SdkAutoConstructList) {
                return null;
            }
            return matchAnyStorageClass;
        }

        public final void setMatchAnyStorageClass(Collection<String> matchAnyStorageClass) {
            this.matchAnyStorageClass = StorageClassListCopier.copy(matchAnyStorageClass);
        }

        @Override
        public final Builder matchAnyStorageClassWithStrings(Collection<String> matchAnyStorageClass) {
            this.matchAnyStorageClass = StorageClassListCopier.copy(matchAnyStorageClass);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchAnyStorageClassWithStrings(String... matchAnyStorageClass) {
            matchAnyStorageClassWithStrings(Arrays.asList(matchAnyStorageClass));
            return this;
        }

        @Override
        public final Builder matchAnyStorageClass(Collection<S3StorageClass> matchAnyStorageClass) {
            this.matchAnyStorageClass = StorageClassListCopier.copyEnumToString(matchAnyStorageClass);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder matchAnyStorageClass(S3StorageClass... matchAnyStorageClass) {
            matchAnyStorageClass(Arrays.asList(matchAnyStorageClass));
            return this;
        }

        @Override
        public JobManifestGeneratorFilter build() {
            return new JobManifestGeneratorFilter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
