/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A configuration used when creating an Object Lambda Access Point.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ObjectLambdaConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ObjectLambdaConfiguration.Builder, ObjectLambdaConfiguration> {
    private static final SdkField<String> SUPPORTING_ACCESS_POINT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SupportingAccessPoint")
            .getter(getter(ObjectLambdaConfiguration::supportingAccessPoint))
            .setter(setter(Builder::supportingAccessPoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportingAccessPoint")
                    .unmarshallLocationName("SupportingAccessPoint").build()).build();

    private static final SdkField<Boolean> CLOUD_WATCH_METRICS_ENABLED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CloudWatchMetricsEnabled")
            .getter(getter(ObjectLambdaConfiguration::cloudWatchMetricsEnabled))
            .setter(setter(Builder::cloudWatchMetricsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchMetricsEnabled")
                    .unmarshallLocationName("CloudWatchMetricsEnabled").build()).build();

    private static final SdkField<List<String>> ALLOWED_FEATURES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllowedFeatures")
            .getter(getter(ObjectLambdaConfiguration::allowedFeaturesAsStrings))
            .setter(setter(Builder::allowedFeaturesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedFeatures")
                    .unmarshallLocationName("AllowedFeatures").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AllowedFeature")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AllowedFeature").unmarshallLocationName("AllowedFeature")
                                                    .build()).build()).build()).build();

    private static final SdkField<List<ObjectLambdaTransformationConfiguration>> TRANSFORMATION_CONFIGURATIONS_FIELD = SdkField
            .<List<ObjectLambdaTransformationConfiguration>> builder(MarshallingType.LIST)
            .memberName("TransformationConfigurations")
            .getter(getter(ObjectLambdaConfiguration::transformationConfigurations))
            .setter(setter(Builder::transformationConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransformationConfigurations")
                    .unmarshallLocationName("TransformationConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("TransformationConfiguration")
                            .memberFieldInfo(
                                    SdkField.<ObjectLambdaTransformationConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(ObjectLambdaTransformationConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("TransformationConfiguration")
                                                    .unmarshallLocationName("TransformationConfiguration").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUPPORTING_ACCESS_POINT_FIELD,
            CLOUD_WATCH_METRICS_ENABLED_FIELD, ALLOWED_FEATURES_FIELD, TRANSFORMATION_CONFIGURATIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String supportingAccessPoint;

    private final Boolean cloudWatchMetricsEnabled;

    private final List<String> allowedFeatures;

    private final List<ObjectLambdaTransformationConfiguration> transformationConfigurations;

    private ObjectLambdaConfiguration(BuilderImpl builder) {
        this.supportingAccessPoint = builder.supportingAccessPoint;
        this.cloudWatchMetricsEnabled = builder.cloudWatchMetricsEnabled;
        this.allowedFeatures = builder.allowedFeatures;
        this.transformationConfigurations = builder.transformationConfigurations;
    }

    /**
     * <p>
     * Standard access point associated with the Object Lambda Access Point.
     * </p>
     * 
     * @return Standard access point associated with the Object Lambda Access Point.
     */
    public final String supportingAccessPoint() {
        return supportingAccessPoint;
    }

    /**
     * <p>
     * A container for whether the CloudWatch metrics configuration is enabled.
     * </p>
     * 
     * @return A container for whether the CloudWatch metrics configuration is enabled.
     */
    public final Boolean cloudWatchMetricsEnabled() {
        return cloudWatchMetricsEnabled;
    }

    /**
     * <p>
     * A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
     * <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedFeatures} method.
     * </p>
     * 
     * @return A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
     *         <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
     */
    public final List<ObjectLambdaAllowedFeature> allowedFeatures() {
        return ObjectLambdaAllowedFeaturesListCopier.copyStringToEnum(allowedFeatures);
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowedFeatures property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowedFeatures() {
        return allowedFeatures != null && !(allowedFeatures instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
     * <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedFeatures} method.
     * </p>
     * 
     * @return A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
     *         <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
     */
    public final List<String> allowedFeaturesAsStrings() {
        return allowedFeatures;
    }

    /**
     * For responses, this returns true if the service returned a value for the TransformationConfigurations property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasTransformationConfigurations() {
        return transformationConfigurations != null && !(transformationConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A container for transformation configurations for an Object Lambda Access Point.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTransformationConfigurations} method.
     * </p>
     * 
     * @return A container for transformation configurations for an Object Lambda Access Point.
     */
    public final List<ObjectLambdaTransformationConfiguration> transformationConfigurations() {
        return transformationConfigurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(supportingAccessPoint());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchMetricsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowedFeatures() ? allowedFeaturesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTransformationConfigurations() ? transformationConfigurations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ObjectLambdaConfiguration)) {
            return false;
        }
        ObjectLambdaConfiguration other = (ObjectLambdaConfiguration) obj;
        return Objects.equals(supportingAccessPoint(), other.supportingAccessPoint())
                && Objects.equals(cloudWatchMetricsEnabled(), other.cloudWatchMetricsEnabled())
                && hasAllowedFeatures() == other.hasAllowedFeatures()
                && Objects.equals(allowedFeaturesAsStrings(), other.allowedFeaturesAsStrings())
                && hasTransformationConfigurations() == other.hasTransformationConfigurations()
                && Objects.equals(transformationConfigurations(), other.transformationConfigurations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ObjectLambdaConfiguration").add("SupportingAccessPoint", supportingAccessPoint())
                .add("CloudWatchMetricsEnabled", cloudWatchMetricsEnabled())
                .add("AllowedFeatures", hasAllowedFeatures() ? allowedFeaturesAsStrings() : null)
                .add("TransformationConfigurations", hasTransformationConfigurations() ? transformationConfigurations() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SupportingAccessPoint":
            return Optional.ofNullable(clazz.cast(supportingAccessPoint()));
        case "CloudWatchMetricsEnabled":
            return Optional.ofNullable(clazz.cast(cloudWatchMetricsEnabled()));
        case "AllowedFeatures":
            return Optional.ofNullable(clazz.cast(allowedFeaturesAsStrings()));
        case "TransformationConfigurations":
            return Optional.ofNullable(clazz.cast(transformationConfigurations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SupportingAccessPoint", SUPPORTING_ACCESS_POINT_FIELD);
        map.put("CloudWatchMetricsEnabled", CLOUD_WATCH_METRICS_ENABLED_FIELD);
        map.put("AllowedFeatures", ALLOWED_FEATURES_FIELD);
        map.put("TransformationConfigurations", TRANSFORMATION_CONFIGURATIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ObjectLambdaConfiguration, T> g) {
        return obj -> g.apply((ObjectLambdaConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ObjectLambdaConfiguration> {
        /**
         * <p>
         * Standard access point associated with the Object Lambda Access Point.
         * </p>
         * 
         * @param supportingAccessPoint
         *        Standard access point associated with the Object Lambda Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportingAccessPoint(String supportingAccessPoint);

        /**
         * <p>
         * A container for whether the CloudWatch metrics configuration is enabled.
         * </p>
         * 
         * @param cloudWatchMetricsEnabled
         *        A container for whether the CloudWatch metrics configuration is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchMetricsEnabled(Boolean cloudWatchMetricsEnabled);

        /**
         * <p>
         * A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         * <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
         * </p>
         * 
         * @param allowedFeatures
         *        A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         *        <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and
         *        <code>HeadObject-PartNumber</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedFeaturesWithStrings(Collection<String> allowedFeatures);

        /**
         * <p>
         * A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         * <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
         * </p>
         * 
         * @param allowedFeatures
         *        A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         *        <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and
         *        <code>HeadObject-PartNumber</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedFeaturesWithStrings(String... allowedFeatures);

        /**
         * <p>
         * A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         * <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
         * </p>
         * 
         * @param allowedFeatures
         *        A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         *        <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and
         *        <code>HeadObject-PartNumber</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedFeatures(Collection<ObjectLambdaAllowedFeature> allowedFeatures);

        /**
         * <p>
         * A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         * <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and <code>HeadObject-PartNumber</code>.
         * </p>
         * 
         * @param allowedFeatures
         *        A container for allowed features. Valid inputs are <code>GetObject-Range</code>,
         *        <code>GetObject-PartNumber</code>, <code>HeadObject-Range</code>, and
         *        <code>HeadObject-PartNumber</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedFeatures(ObjectLambdaAllowedFeature... allowedFeatures);

        /**
         * <p>
         * A container for transformation configurations for an Object Lambda Access Point.
         * </p>
         * 
         * @param transformationConfigurations
         *        A container for transformation configurations for an Object Lambda Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformationConfigurations(Collection<ObjectLambdaTransformationConfiguration> transformationConfigurations);

        /**
         * <p>
         * A container for transformation configurations for an Object Lambda Access Point.
         * </p>
         * 
         * @param transformationConfigurations
         *        A container for transformation configurations for an Object Lambda Access Point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformationConfigurations(ObjectLambdaTransformationConfiguration... transformationConfigurations);

        /**
         * <p>
         * A container for transformation configurations for an Object Lambda Access Point.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.s3control.model.ObjectLambdaTransformationConfiguration.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.s3control.model.ObjectLambdaTransformationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.s3control.model.ObjectLambdaTransformationConfiguration.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #transformationConfigurations(List<ObjectLambdaTransformationConfiguration>)}.
         * 
         * @param transformationConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.s3control.model.ObjectLambdaTransformationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformationConfigurations(java.util.Collection<ObjectLambdaTransformationConfiguration>)
         */
        Builder transformationConfigurations(
                Consumer<ObjectLambdaTransformationConfiguration.Builder>... transformationConfigurations);
    }

    static final class BuilderImpl implements Builder {
        private String supportingAccessPoint;

        private Boolean cloudWatchMetricsEnabled;

        private List<String> allowedFeatures = DefaultSdkAutoConstructList.getInstance();

        private List<ObjectLambdaTransformationConfiguration> transformationConfigurations = DefaultSdkAutoConstructList
                .getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ObjectLambdaConfiguration model) {
            supportingAccessPoint(model.supportingAccessPoint);
            cloudWatchMetricsEnabled(model.cloudWatchMetricsEnabled);
            allowedFeaturesWithStrings(model.allowedFeatures);
            transformationConfigurations(model.transformationConfigurations);
        }

        public final String getSupportingAccessPoint() {
            return supportingAccessPoint;
        }

        public final void setSupportingAccessPoint(String supportingAccessPoint) {
            this.supportingAccessPoint = supportingAccessPoint;
        }

        @Override
        public final Builder supportingAccessPoint(String supportingAccessPoint) {
            this.supportingAccessPoint = supportingAccessPoint;
            return this;
        }

        public final Boolean getCloudWatchMetricsEnabled() {
            return cloudWatchMetricsEnabled;
        }

        public final void setCloudWatchMetricsEnabled(Boolean cloudWatchMetricsEnabled) {
            this.cloudWatchMetricsEnabled = cloudWatchMetricsEnabled;
        }

        @Override
        public final Builder cloudWatchMetricsEnabled(Boolean cloudWatchMetricsEnabled) {
            this.cloudWatchMetricsEnabled = cloudWatchMetricsEnabled;
            return this;
        }

        public final Collection<String> getAllowedFeatures() {
            if (allowedFeatures instanceof SdkAutoConstructList) {
                return null;
            }
            return allowedFeatures;
        }

        public final void setAllowedFeatures(Collection<String> allowedFeatures) {
            this.allowedFeatures = ObjectLambdaAllowedFeaturesListCopier.copy(allowedFeatures);
        }

        @Override
        public final Builder allowedFeaturesWithStrings(Collection<String> allowedFeatures) {
            this.allowedFeatures = ObjectLambdaAllowedFeaturesListCopier.copy(allowedFeatures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedFeaturesWithStrings(String... allowedFeatures) {
            allowedFeaturesWithStrings(Arrays.asList(allowedFeatures));
            return this;
        }

        @Override
        public final Builder allowedFeatures(Collection<ObjectLambdaAllowedFeature> allowedFeatures) {
            this.allowedFeatures = ObjectLambdaAllowedFeaturesListCopier.copyEnumToString(allowedFeatures);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedFeatures(ObjectLambdaAllowedFeature... allowedFeatures) {
            allowedFeatures(Arrays.asList(allowedFeatures));
            return this;
        }

        public final List<ObjectLambdaTransformationConfiguration.Builder> getTransformationConfigurations() {
            List<ObjectLambdaTransformationConfiguration.Builder> result = ObjectLambdaTransformationConfigurationsListCopier
                    .copyToBuilder(this.transformationConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTransformationConfigurations(
                Collection<ObjectLambdaTransformationConfiguration.BuilderImpl> transformationConfigurations) {
            this.transformationConfigurations = ObjectLambdaTransformationConfigurationsListCopier
                    .copyFromBuilder(transformationConfigurations);
        }

        @Override
        public final Builder transformationConfigurations(
                Collection<ObjectLambdaTransformationConfiguration> transformationConfigurations) {
            this.transformationConfigurations = ObjectLambdaTransformationConfigurationsListCopier
                    .copy(transformationConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder transformationConfigurations(ObjectLambdaTransformationConfiguration... transformationConfigurations) {
            transformationConfigurations(Arrays.asList(transformationConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder transformationConfigurations(
                Consumer<ObjectLambdaTransformationConfiguration.Builder>... transformationConfigurations) {
            transformationConfigurations(Stream.of(transformationConfigurations)
                    .map(c -> ObjectLambdaTransformationConfiguration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ObjectLambdaConfiguration build() {
            return new ObjectLambdaConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
