/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies information about the replication destination bucket and its settings for an S3 on Outposts replication
 * configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Destination implements SdkPojo, Serializable, ToCopyableBuilder<Destination.Builder, Destination> {
    private static final SdkField<String> ACCOUNT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Account")
            .getter(getter(Destination::account))
            .setter(setter(Builder::account))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Account")
                    .unmarshallLocationName("Account").build()).build();

    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(Destination::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<ReplicationTime> REPLICATION_TIME_FIELD = SdkField
            .<ReplicationTime> builder(MarshallingType.SDK_POJO)
            .memberName("ReplicationTime")
            .getter(getter(Destination::replicationTime))
            .setter(setter(Builder::replicationTime))
            .constructor(ReplicationTime::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationTime")
                    .unmarshallLocationName("ReplicationTime").build()).build();

    private static final SdkField<AccessControlTranslation> ACCESS_CONTROL_TRANSLATION_FIELD = SdkField
            .<AccessControlTranslation> builder(MarshallingType.SDK_POJO)
            .memberName("AccessControlTranslation")
            .getter(getter(Destination::accessControlTranslation))
            .setter(setter(Builder::accessControlTranslation))
            .constructor(AccessControlTranslation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlTranslation")
                    .unmarshallLocationName("AccessControlTranslation").build()).build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("EncryptionConfiguration")
            .getter(getter(Destination::encryptionConfiguration))
            .setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionConfiguration")
                    .unmarshallLocationName("EncryptionConfiguration").build()).build();

    private static final SdkField<Metrics> METRICS_FIELD = SdkField
            .<Metrics> builder(MarshallingType.SDK_POJO)
            .memberName("Metrics")
            .getter(getter(Destination::metrics))
            .setter(setter(Builder::metrics))
            .constructor(Metrics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metrics")
                    .unmarshallLocationName("Metrics").build()).build();

    private static final SdkField<String> STORAGE_CLASS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StorageClass")
            .getter(getter(Destination::storageClassAsString))
            .setter(setter(Builder::storageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageClass")
                    .unmarshallLocationName("StorageClass").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_FIELD, BUCKET_FIELD,
            REPLICATION_TIME_FIELD, ACCESS_CONTROL_TRANSLATION_FIELD, ENCRYPTION_CONFIGURATION_FIELD, METRICS_FIELD,
            STORAGE_CLASS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String account;

    private final String bucket;

    private final ReplicationTime replicationTime;

    private final AccessControlTranslation accessControlTranslation;

    private final EncryptionConfiguration encryptionConfiguration;

    private final Metrics metrics;

    private final String storageClass;

    private Destination(BuilderImpl builder) {
        this.account = builder.account;
        this.bucket = builder.bucket;
        this.replicationTime = builder.replicationTime;
        this.accessControlTranslation = builder.accessControlTranslation;
        this.encryptionConfiguration = builder.encryptionConfiguration;
        this.metrics = builder.metrics;
        this.storageClass = builder.storageClass;
    }

    /**
     * <p>
     * The destination bucket owner's account ID.
     * </p>
     * 
     * @return The destination bucket owner's account ID.
     */
    public final String account() {
        return account;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the access point for the destination bucket where you want S3 on Outposts to
     * store the replication results.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the access point for the destination bucket where you want S3 on
     *         Outposts to store the replication results.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * A container that specifies S3 Replication Time Control (S3 RTC) settings, including whether S3 RTC is enabled and
     * the time when all objects and operations on objects must be replicated. Must be specified together with a
     * <code>Metrics</code> block.
     * </p>
     * <note>
     * <p>
     * This is not supported by Amazon S3 on Outposts buckets.
     * </p>
     * </note>
     * 
     * @return A container that specifies S3 Replication Time Control (S3 RTC) settings, including whether S3 RTC is
     *         enabled and the time when all objects and operations on objects must be replicated. Must be specified
     *         together with a <code>Metrics</code> block. </p> <note>
     *         <p>
     *         This is not supported by Amazon S3 on Outposts buckets.
     *         </p>
     */
    public final ReplicationTime replicationTime() {
        return replicationTime;
    }

    /**
     * <p>
     * Specify this property only in a cross-account scenario (where the source and destination bucket owners are not
     * the same), and you want to change replica ownership to the Amazon Web Services account that owns the destination
     * bucket. If this property is not specified in the replication configuration, the replicas are owned by same Amazon
     * Web Services account that owns the source object.
     * </p>
     * <note>
     * <p>
     * This is not supported by Amazon S3 on Outposts buckets.
     * </p>
     * </note>
     * 
     * @return Specify this property only in a cross-account scenario (where the source and destination bucket owners
     *         are not the same), and you want to change replica ownership to the Amazon Web Services account that owns
     *         the destination bucket. If this property is not specified in the replication configuration, the replicas
     *         are owned by same Amazon Web Services account that owns the source object.</p> <note>
     *         <p>
     *         This is not supported by Amazon S3 on Outposts buckets.
     *         </p>
     */
    public final AccessControlTranslation accessControlTranslation() {
        return accessControlTranslation;
    }

    /**
     * <p>
     * A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is specified, you
     * must specify this element.
     * </p>
     * <note>
     * <p>
     * This is not supported by Amazon S3 on Outposts buckets.
     * </p>
     * </note>
     * 
     * @return A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is
     *         specified, you must specify this element.</p> <note>
     *         <p>
     *         This is not supported by Amazon S3 on Outposts buckets.
     *         </p>
     */
    public final EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    /**
     * <p>
     * A container that specifies replication metrics-related settings.
     * </p>
     * 
     * @return A container that specifies replication metrics-related settings.
     */
    public final Metrics metrics() {
        return metrics;
    }

    /**
     * <p>
     * The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in the
     * <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class to create the
     * object replicas.
     * </p>
     * <note>
     * <p>
     * Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link ReplicationStorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #storageClassAsString}.
     * </p>
     * 
     * @return The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in the
     *         <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class to
     *         create the object replicas. </p> <note>
     *         <p>
     *         Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
     *         </p>
     * @see ReplicationStorageClass
     */
    public final ReplicationStorageClass storageClass() {
        return ReplicationStorageClass.fromValue(storageClass);
    }

    /**
     * <p>
     * The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in the
     * <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class to create the
     * object replicas.
     * </p>
     * <note>
     * <p>
     * Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link ReplicationStorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #storageClassAsString}.
     * </p>
     * 
     * @return The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in the
     *         <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class to
     *         create the object replicas. </p> <note>
     *         <p>
     *         Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
     *         </p>
     * @see ReplicationStorageClass
     */
    public final String storageClassAsString() {
        return storageClass;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(account());
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(replicationTime());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlTranslation());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(metrics());
        hashCode = 31 * hashCode + Objects.hashCode(storageClassAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Destination)) {
            return false;
        }
        Destination other = (Destination) obj;
        return Objects.equals(account(), other.account()) && Objects.equals(bucket(), other.bucket())
                && Objects.equals(replicationTime(), other.replicationTime())
                && Objects.equals(accessControlTranslation(), other.accessControlTranslation())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration())
                && Objects.equals(metrics(), other.metrics())
                && Objects.equals(storageClassAsString(), other.storageClassAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Destination").add("Account", account()).add("Bucket", bucket())
                .add("ReplicationTime", replicationTime()).add("AccessControlTranslation", accessControlTranslation())
                .add("EncryptionConfiguration", encryptionConfiguration()).add("Metrics", metrics())
                .add("StorageClass", storageClassAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Account":
            return Optional.ofNullable(clazz.cast(account()));
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "ReplicationTime":
            return Optional.ofNullable(clazz.cast(replicationTime()));
        case "AccessControlTranslation":
            return Optional.ofNullable(clazz.cast(accessControlTranslation()));
        case "EncryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        case "Metrics":
            return Optional.ofNullable(clazz.cast(metrics()));
        case "StorageClass":
            return Optional.ofNullable(clazz.cast(storageClassAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Account", ACCOUNT_FIELD);
        map.put("Bucket", BUCKET_FIELD);
        map.put("ReplicationTime", REPLICATION_TIME_FIELD);
        map.put("AccessControlTranslation", ACCESS_CONTROL_TRANSLATION_FIELD);
        map.put("EncryptionConfiguration", ENCRYPTION_CONFIGURATION_FIELD);
        map.put("Metrics", METRICS_FIELD);
        map.put("StorageClass", STORAGE_CLASS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Destination, T> g) {
        return obj -> g.apply((Destination) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Destination> {
        /**
         * <p>
         * The destination bucket owner's account ID.
         * </p>
         * 
         * @param account
         *        The destination bucket owner's account ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder account(String account);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the access point for the destination bucket where you want S3 on Outposts
         * to store the replication results.
         * </p>
         * 
         * @param bucket
         *        The Amazon Resource Name (ARN) of the access point for the destination bucket where you want S3 on
         *        Outposts to store the replication results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * A container that specifies S3 Replication Time Control (S3 RTC) settings, including whether S3 RTC is enabled
         * and the time when all objects and operations on objects must be replicated. Must be specified together with a
         * <code>Metrics</code> block.
         * </p>
         * <note>
         * <p>
         * This is not supported by Amazon S3 on Outposts buckets.
         * </p>
         * </note>
         * 
         * @param replicationTime
         *        A container that specifies S3 Replication Time Control (S3 RTC) settings, including whether S3 RTC is
         *        enabled and the time when all objects and operations on objects must be replicated. Must be specified
         *        together with a <code>Metrics</code> block. </p> <note>
         *        <p>
         *        This is not supported by Amazon S3 on Outposts buckets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationTime(ReplicationTime replicationTime);

        /**
         * <p>
         * A container that specifies S3 Replication Time Control (S3 RTC) settings, including whether S3 RTC is enabled
         * and the time when all objects and operations on objects must be replicated. Must be specified together with a
         * <code>Metrics</code> block.
         * </p>
         * <note>
         * <p>
         * This is not supported by Amazon S3 on Outposts buckets.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ReplicationTime.Builder} avoiding
         * the need to create one manually via {@link ReplicationTime#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ReplicationTime.Builder#build()} is called immediately and its
         * result is passed to {@link #replicationTime(ReplicationTime)}.
         * 
         * @param replicationTime
         *        a consumer that will call methods on {@link ReplicationTime.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #replicationTime(ReplicationTime)
         */
        default Builder replicationTime(Consumer<ReplicationTime.Builder> replicationTime) {
            return replicationTime(ReplicationTime.builder().applyMutation(replicationTime).build());
        }

        /**
         * <p>
         * Specify this property only in a cross-account scenario (where the source and destination bucket owners are
         * not the same), and you want to change replica ownership to the Amazon Web Services account that owns the
         * destination bucket. If this property is not specified in the replication configuration, the replicas are
         * owned by same Amazon Web Services account that owns the source object.
         * </p>
         * <note>
         * <p>
         * This is not supported by Amazon S3 on Outposts buckets.
         * </p>
         * </note>
         * 
         * @param accessControlTranslation
         *        Specify this property only in a cross-account scenario (where the source and destination bucket owners
         *        are not the same), and you want to change replica ownership to the Amazon Web Services account that
         *        owns the destination bucket. If this property is not specified in the replication configuration, the
         *        replicas are owned by same Amazon Web Services account that owns the source object.</p> <note>
         *        <p>
         *        This is not supported by Amazon S3 on Outposts buckets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlTranslation(AccessControlTranslation accessControlTranslation);

        /**
         * <p>
         * Specify this property only in a cross-account scenario (where the source and destination bucket owners are
         * not the same), and you want to change replica ownership to the Amazon Web Services account that owns the
         * destination bucket. If this property is not specified in the replication configuration, the replicas are
         * owned by same Amazon Web Services account that owns the source object.
         * </p>
         * <note>
         * <p>
         * This is not supported by Amazon S3 on Outposts buckets.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link AccessControlTranslation.Builder}
         * avoiding the need to create one manually via {@link AccessControlTranslation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AccessControlTranslation.Builder#build()} is called immediately
         * and its result is passed to {@link #accessControlTranslation(AccessControlTranslation)}.
         * 
         * @param accessControlTranslation
         *        a consumer that will call methods on {@link AccessControlTranslation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlTranslation(AccessControlTranslation)
         */
        default Builder accessControlTranslation(Consumer<AccessControlTranslation.Builder> accessControlTranslation) {
            return accessControlTranslation(AccessControlTranslation.builder().applyMutation(accessControlTranslation).build());
        }

        /**
         * <p>
         * A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is specified,
         * you must specify this element.
         * </p>
         * <note>
         * <p>
         * This is not supported by Amazon S3 on Outposts buckets.
         * </p>
         * </note>
         * 
         * @param encryptionConfiguration
         *        A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is
         *        specified, you must specify this element.</p> <note>
         *        <p>
         *        This is not supported by Amazon S3 on Outposts buckets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * A container that provides information about encryption. If <code>SourceSelectionCriteria</code> is specified,
         * you must specify this element.
         * </p>
         * <note>
         * <p>
         * This is not supported by Amazon S3 on Outposts buckets.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link EncryptionConfiguration.Builder}
         * avoiding the need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }

        /**
         * <p>
         * A container that specifies replication metrics-related settings.
         * </p>
         * 
         * @param metrics
         *        A container that specifies replication metrics-related settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(Metrics metrics);

        /**
         * <p>
         * A container that specifies replication metrics-related settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link Metrics.Builder} avoiding the need to
         * create one manually via {@link Metrics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Metrics.Builder#build()} is called immediately and its result is
         * passed to {@link #metrics(Metrics)}.
         * 
         * @param metrics
         *        a consumer that will call methods on {@link Metrics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metrics(Metrics)
         */
        default Builder metrics(Consumer<Metrics.Builder> metrics) {
            return metrics(Metrics.builder().applyMutation(metrics).build());
        }

        /**
         * <p>
         * The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in the
         * <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class to create
         * the object replicas.
         * </p>
         * <note>
         * <p>
         * Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
         * </p>
         * </note>
         * 
         * @param storageClass
         *        The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in
         *        the <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class
         *        to create the object replicas. </p> <note>
         *        <p>
         *        Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
         *        </p>
         * @see ReplicationStorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicationStorageClass
         */
        Builder storageClass(String storageClass);

        /**
         * <p>
         * The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in the
         * <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class to create
         * the object replicas.
         * </p>
         * <note>
         * <p>
         * Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
         * </p>
         * </note>
         * 
         * @param storageClass
         *        The storage class to use when replicating objects. All objects stored on S3 on Outposts are stored in
         *        the <code>OUTPOSTS</code> storage class. S3 on Outposts uses the <code>OUTPOSTS</code> storage class
         *        to create the object replicas. </p> <note>
         *        <p>
         *        Values other than <code>OUTPOSTS</code> aren't supported by Amazon S3 on Outposts.
         *        </p>
         * @see ReplicationStorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicationStorageClass
         */
        Builder storageClass(ReplicationStorageClass storageClass);
    }

    static final class BuilderImpl implements Builder {
        private String account;

        private String bucket;

        private ReplicationTime replicationTime;

        private AccessControlTranslation accessControlTranslation;

        private EncryptionConfiguration encryptionConfiguration;

        private Metrics metrics;

        private String storageClass;

        private BuilderImpl() {
        }

        private BuilderImpl(Destination model) {
            account(model.account);
            bucket(model.bucket);
            replicationTime(model.replicationTime);
            accessControlTranslation(model.accessControlTranslation);
            encryptionConfiguration(model.encryptionConfiguration);
            metrics(model.metrics);
            storageClass(model.storageClass);
        }

        public final String getAccount() {
            return account;
        }

        public final void setAccount(String account) {
            this.account = account;
        }

        @Override
        public final Builder account(String account) {
            this.account = account;
            return this;
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final ReplicationTime.Builder getReplicationTime() {
            return replicationTime != null ? replicationTime.toBuilder() : null;
        }

        public final void setReplicationTime(ReplicationTime.BuilderImpl replicationTime) {
            this.replicationTime = replicationTime != null ? replicationTime.build() : null;
        }

        @Override
        public final Builder replicationTime(ReplicationTime replicationTime) {
            this.replicationTime = replicationTime;
            return this;
        }

        public final AccessControlTranslation.Builder getAccessControlTranslation() {
            return accessControlTranslation != null ? accessControlTranslation.toBuilder() : null;
        }

        public final void setAccessControlTranslation(AccessControlTranslation.BuilderImpl accessControlTranslation) {
            this.accessControlTranslation = accessControlTranslation != null ? accessControlTranslation.build() : null;
        }

        @Override
        public final Builder accessControlTranslation(AccessControlTranslation accessControlTranslation) {
            this.accessControlTranslation = accessControlTranslation;
            return this;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final Metrics.Builder getMetrics() {
            return metrics != null ? metrics.toBuilder() : null;
        }

        public final void setMetrics(Metrics.BuilderImpl metrics) {
            this.metrics = metrics != null ? metrics.build() : null;
        }

        @Override
        public final Builder metrics(Metrics metrics) {
            this.metrics = metrics;
            return this;
        }

        public final String getStorageClass() {
            return storageClass;
        }

        public final void setStorageClass(String storageClass) {
            this.storageClass = storageClass;
        }

        @Override
        public final Builder storageClass(String storageClass) {
            this.storageClass = storageClass;
            return this;
        }

        @Override
        public final Builder storageClass(ReplicationStorageClass storageClass) {
            this.storageClass(storageClass == null ? null : storageClass.toString());
            return this;
        }

        @Override
        public Destination build() {
            return new Destination(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
