/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.endpoints.internal;

import java.net.URI;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlEndpointParams;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultS3ControlEndpointProvider implements S3ControlEndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(S3ControlEndpointParams params) {
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        Validate.notNull(params.useDualStack(), "Parameter 'UseDualStack' must not be null");
        try {
            RuleResult result = endpointRule0(params, new LocalState(params.region()));
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(S3ControlEndpointParams params, LocalState locals) {
        RuleResult result = endpointRule1(params, locals);
        if (result.isResolved()) {
            return result;
        }
        return endpointRule130(params, locals);
    }

    private static RuleResult endpointRule1(S3ControlEndpointParams params, LocalState locals) {
        if (locals.region() != null) {
            RuleResult result = endpointRule2(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule3(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule16(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule37(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule44(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule49(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule81(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule113(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule2(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            RulePartition partitionResult = null;
            if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
                locals = locals.toBuilder().partitionResult(partitionResult).build();
                if ("aws-cn".equals(locals.partitionResult().name())) {
                    return RuleResult.error("Partition does not support FIPS");
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule3(S3ControlEndpointParams params, LocalState locals) {
        if (params.outpostId() != null) {
            return endpointRule4(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule4(S3ControlEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule5(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule6(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule7(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule8(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule9(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule15(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule5(S3ControlEndpointParams params, LocalState locals) {
        if (params.requiresAccountId() != null && params.requiresAccountId() && params.accountId() == null) {
            return RuleResult.error("AccountId is required but not set");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule6(S3ControlEndpointParams params, LocalState locals) {
        if (params.accountId() != null && !(RulesFunctions.isValidHostLabel(params.accountId(), false))) {
            return RuleResult.error("AccountId must only contain a-z, A-Z, 0-9 and `-`.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule7(S3ControlEndpointParams params, LocalState locals) {
        if (!(RulesFunctions.isValidHostLabel(params.outpostId(), false))) {
            return RuleResult.error("OutpostId must only contain a-z, A-Z, 0-9 and `-`.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule8(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null && params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule9(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.region(), true)) {
            RuleResult result = endpointRule10(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule11(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule12(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule13(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule14(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule10(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleUrl url = null;
            if ((url = RulesFunctions.parseURL(params.endpoint())) != null) {
                locals = locals.toBuilder().url(url).build();
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create(locals.url().scheme() + "://" + locals.url().authority() + locals.url().path()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(locals.region()).build())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule11(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips() && params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts-fips." + locals.region() + "."
                            + locals.partitionResult().dualStackDnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule12(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts-fips." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule13(S3ControlEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts." + locals.region() + "."
                            + locals.partitionResult().dualStackDnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule14(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3-outposts." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule15(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid region: region was not a valid DNS name.");
    }

    private static RuleResult endpointRule16(S3ControlEndpointParams params, LocalState locals) {
        if (params.accessPointName() != null) {
            String accessPointSuffix = null;
            if ((accessPointSuffix = RulesFunctions.substring(params.accessPointName(), 0, 7, true)) != null) {
                locals = locals.toBuilder().accessPointSuffix(accessPointSuffix).build();
                if ("--xa-s3".equals(locals.accessPointSuffix())) {
                    return endpointRule17(params, locals);
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule17(S3ControlEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule18(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule19(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule20(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule21(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule24(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule27(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule30(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule33(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule36(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule18(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null && params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule19(S3ControlEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.error("S3Express does not support Dual-stack.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule20(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleUrl url = null;
            if ((url = RulesFunctions.parseURL(params.endpoint())) != null) {
                locals = locals.toBuilder().url(url).build();
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create(locals.url().scheme() + "://" + locals.url().authority()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                        .signingRegion(locals.region()).build())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule21(S3ControlEndpointParams params, LocalState locals) {
        String s3ExpressAvailabilityZoneId = null;
        String s3ExpressAvailabilityZoneDelim = null;
        if ((s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 15, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneId(s3ExpressAvailabilityZoneId).build();
        } else {
            return RuleResult.carryOn();
        }
        if ((s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 15, 17, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneDelim(s3ExpressAvailabilityZoneDelim).build();
            if ("--".equals(locals.s3ExpressAvailabilityZoneDelim())) {
                RuleResult result = endpointRule22(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule23(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule22(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control-fips." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule23(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3express-control." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule24(S3ControlEndpointParams params, LocalState locals) {
        String s3ExpressAvailabilityZoneId = null;
        String s3ExpressAvailabilityZoneDelim = null;
        if ((s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 16, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneId(s3ExpressAvailabilityZoneId).build();
        } else {
            return RuleResult.carryOn();
        }
        if ((s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 16, 18, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneDelim(s3ExpressAvailabilityZoneDelim).build();
            if ("--".equals(locals.s3ExpressAvailabilityZoneDelim())) {
                RuleResult result = endpointRule25(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule26(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule25(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control-fips." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule26(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3express-control." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule27(S3ControlEndpointParams params, LocalState locals) {
        String s3ExpressAvailabilityZoneId = null;
        String s3ExpressAvailabilityZoneDelim = null;
        if ((s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 20, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneId(s3ExpressAvailabilityZoneId).build();
        } else {
            return RuleResult.carryOn();
        }
        if ((s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 20, 22, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneDelim(s3ExpressAvailabilityZoneDelim).build();
            if ("--".equals(locals.s3ExpressAvailabilityZoneDelim())) {
                RuleResult result = endpointRule28(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule29(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule28(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control-fips." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule29(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3express-control." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule30(S3ControlEndpointParams params, LocalState locals) {
        String s3ExpressAvailabilityZoneId = null;
        String s3ExpressAvailabilityZoneDelim = null;
        if ((s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 21, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneId(s3ExpressAvailabilityZoneId).build();
        } else {
            return RuleResult.carryOn();
        }
        if ((s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 21, 23, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneDelim(s3ExpressAvailabilityZoneDelim).build();
            if ("--".equals(locals.s3ExpressAvailabilityZoneDelim())) {
                RuleResult result = endpointRule31(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule32(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule31(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control-fips." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule32(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3express-control." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule33(S3ControlEndpointParams params, LocalState locals) {
        String s3ExpressAvailabilityZoneId = null;
        String s3ExpressAvailabilityZoneDelim = null;
        if ((s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 27, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneId(s3ExpressAvailabilityZoneId).build();
        } else {
            return RuleResult.carryOn();
        }
        if ((s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 27, 29, true)) != null) {
            locals = locals.toBuilder().s3ExpressAvailabilityZoneDelim(s3ExpressAvailabilityZoneDelim).build();
            if ("--".equals(locals.s3ExpressAvailabilityZoneDelim())) {
                RuleResult result = endpointRule34(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule35(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule34(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control-fips." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule35(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3express-control." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule36(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Unrecognized S3Express Access Point name format.");
    }

    private static RuleResult endpointRule37(S3ControlEndpointParams params, LocalState locals) {
        if (params.useS3ExpressControlEndpoint() != null && params.useS3ExpressControlEndpoint()) {
            return endpointRule38(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule38(S3ControlEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule39(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule40(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule41(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule42(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule43(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule39(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null && params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule40(S3ControlEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.error("S3Express does not support Dual-stack.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule41(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleUrl url = null;
            if ((url = RulesFunctions.parseURL(params.endpoint())) != null) {
                locals = locals.toBuilder().url(url).build();
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create(locals.url().scheme() + "://" + locals.url().authority()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                        .signingRegion(locals.region()).build())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule42(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control-fips." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule43(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3express-control." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule44(S3ControlEndpointParams params, LocalState locals) {
        if ("snow".equals(locals.region()) && params.endpoint() != null) {
            RuleUrl url = null;
            if ((url = RulesFunctions.parseURL(params.endpoint())) != null) {
                locals = locals.toBuilder().url(url).build();
                return endpointRule45(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule45(S3ControlEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule46(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule47(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule48(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule46(S3ControlEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.error("S3 Snow does not support DualStack");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule47(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.error("S3 Snow does not support FIPS");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule48(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create(locals.url().scheme() + "://" + locals.url().authority()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule49(S3ControlEndpointParams params, LocalState locals) {
        if (params.accessPointName() != null) {
            RuleArn accessPointArn = null;
            if ((accessPointArn = RulesFunctions.awsParseArn(params.accessPointName())) != null) {
                locals = locals.toBuilder().accessPointArn(accessPointArn).build();
                RuleResult result = endpointRule50(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule80(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule50(S3ControlEndpointParams params, LocalState locals) {
        String arnType = null;
        if ((arnType = RulesFunctions.listAccess(locals.accessPointArn().resourceId(), 0)) != null) {
            locals = locals.toBuilder().arnType(arnType).build();
            if (!("".equals(locals.arnType()))) {
                return endpointRule51(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule51(S3ControlEndpointParams params, LocalState locals) {
        if ("s3-outposts".equals(locals.accessPointArn().service())) {
            RuleResult result = endpointRule52(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule79(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule52(S3ControlEndpointParams params, LocalState locals) {
        String outpostId = null;
        if ((outpostId = RulesFunctions.listAccess(locals.accessPointArn().resourceId(), 1)) != null) {
            locals = locals.toBuilder().outpostId(outpostId).build();
            RuleResult result = endpointRule53(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule78(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule53(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.outpostId(), false)) {
            RuleResult result = endpointRule54(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule55(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule56(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule54(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null && params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule55(S3ControlEndpointParams params, LocalState locals) {
        if (params.useArnRegion() != null && !params.useArnRegion()
                && !(RulesFunctions.stringEquals(locals.accessPointArn().region(), locals.region()))) {
            return RuleResult.error("Invalid configuration: region from ARN `" + locals.accessPointArn().region()
                    + "` does not match client region `" + locals.region() + "` and UseArnRegion is `false`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule56(S3ControlEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            return endpointRule57(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule57(S3ControlEndpointParams params, LocalState locals) {
        RulePartition arnPartition = null;
        if ((arnPartition = RulesFunctions.awsPartition(locals.accessPointArn().region())) != null) {
            locals = locals.toBuilder().arnPartition(arnPartition).build();
            RuleResult result = endpointRule58(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule77(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule58(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.stringEquals(locals.arnPartition().name(), locals.partitionResult().name())) {
            RuleResult result = endpointRule59(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule76(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule59(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.accessPointArn().region(), true)) {
            RuleResult result = endpointRule60(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule75(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule60(S3ControlEndpointParams params, LocalState locals) {
        if (!("".equals(locals.accessPointArn().accountId()))) {
            RuleResult result = endpointRule61(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule74(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule61(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.accessPointArn().accountId(), false)) {
            RuleResult result = endpointRule62(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule63(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule73(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule62(S3ControlEndpointParams params, LocalState locals) {
        if (params.accountId() != null && !(RulesFunctions.stringEquals(params.accountId(), locals.accessPointArn().accountId()))) {
            return RuleResult.error("Invalid ARN: the accountId specified in the ARN (`" + locals.accessPointArn().accountId()
                    + "`) does not match the parameter (`" + params.accountId() + "`)");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule63(S3ControlEndpointParams params, LocalState locals) {
        String outpostType = null;
        if ((outpostType = RulesFunctions.listAccess(locals.accessPointArn().resourceId(), 2)) != null) {
            locals = locals.toBuilder().outpostType(outpostType).build();
            RuleResult result = endpointRule64(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule72(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule64(S3ControlEndpointParams params, LocalState locals) {
        String accessPointName = null;
        if ((accessPointName = RulesFunctions.listAccess(locals.accessPointArn().resourceId(), 3)) != null) {
            locals = locals.toBuilder().accessPointName(accessPointName).build();
            RuleResult result = endpointRule65(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule71(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule65(S3ControlEndpointParams params, LocalState locals) {
        if ("accesspoint".equals(locals.outpostType())) {
            RuleResult result = endpointRule66(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule67(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule68(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule69(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule70(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule66(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips() && params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts-fips." + locals.accessPointArn().region() + "."
                            + locals.arnPartition().dualStackDnsSuffix()))
                    .putHeader("x-amz-account-id", locals.accessPointArn().accountId())
                    .putHeader("x-amz-outpost-id", locals.outpostId())
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.accessPointArn().region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule67(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts-fips." + locals.accessPointArn().region() + "."
                            + locals.arnPartition().dnsSuffix()))
                    .putHeader("x-amz-account-id", locals.accessPointArn().accountId())
                    .putHeader("x-amz-outpost-id", locals.outpostId())
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.accessPointArn().region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule68(S3ControlEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts." + locals.accessPointArn().region() + "."
                            + locals.arnPartition().dualStackDnsSuffix()))
                    .putHeader("x-amz-account-id", locals.accessPointArn().accountId())
                    .putHeader("x-amz-outpost-id", locals.outpostId())
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.accessPointArn().region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule69(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleUrl url = null;
            if ((url = RulesFunctions.parseURL(params.endpoint())) != null) {
                locals = locals.toBuilder().url(url).build();
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create(locals.url().scheme() + "://" + locals.url().authority() + locals.url().path()))
                        .putHeader("x-amz-account-id", locals.accessPointArn().accountId())
                        .putHeader("x-amz-outpost-id", locals.outpostId())
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(locals.accessPointArn().region()).build())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule70(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3-outposts." + locals.accessPointArn().region() + "."
                        + locals.arnPartition().dnsSuffix()))
                .putHeader("x-amz-account-id", locals.accessPointArn().accountId())
                .putHeader("x-amz-outpost-id", locals.outpostId())
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                .signingRegion(locals.accessPointArn().region()).build())).build());
    }

    private static RuleResult endpointRule71(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Expected an outpost type `accesspoint`, found `" + locals.outpostType() + "`");
    }

    private static RuleResult endpointRule72(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: expected an access point name");
    }

    private static RuleResult endpointRule73(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: Expected a 4-component resource");
    }

    private static RuleResult endpointRule74(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: The account id may only contain a-z, A-Z, 0-9 and `-`. Found: `"
                + locals.accessPointArn().accountId() + "`");
    }

    private static RuleResult endpointRule75(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: missing account ID");
    }

    private static RuleResult endpointRule76(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid region in ARN: `" + locals.accessPointArn().region() + "` (invalid DNS name)");
    }

    private static RuleResult endpointRule77(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Client was configured for partition `" + locals.partitionResult().name() + "` but ARN has `"
                + locals.arnPartition().name() + "`");
    }

    private static RuleResult endpointRule78(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: The outpost Id must only contain a-z, A-Z, 0-9 and `-`., found: `"
                + locals.outpostId() + "`");
    }

    private static RuleResult endpointRule79(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: The Outpost Id was not set");
    }

    private static RuleResult endpointRule80(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: No ARN type specified");
    }

    private static RuleResult endpointRule81(S3ControlEndpointParams params, LocalState locals) {
        if (params.bucket() != null) {
            RuleArn bucketArn = null;
            if ((bucketArn = RulesFunctions.awsParseArn(params.bucket())) != null) {
                locals = locals.toBuilder().bucketArn(bucketArn).build();
                RuleResult result = endpointRule82(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule112(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule82(S3ControlEndpointParams params, LocalState locals) {
        String arnType = null;
        if ((arnType = RulesFunctions.listAccess(locals.bucketArn().resourceId(), 0)) != null) {
            locals = locals.toBuilder().arnType(arnType).build();
            if (!("".equals(locals.arnType()))) {
                return endpointRule83(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule83(S3ControlEndpointParams params, LocalState locals) {
        if ("s3-outposts".equals(locals.bucketArn().service())) {
            RuleResult result = endpointRule84(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule111(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule84(S3ControlEndpointParams params, LocalState locals) {
        String outpostId = null;
        if ((outpostId = RulesFunctions.listAccess(locals.bucketArn().resourceId(), 1)) != null) {
            locals = locals.toBuilder().outpostId(outpostId).build();
            RuleResult result = endpointRule85(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule110(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule85(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.outpostId(), false)) {
            RuleResult result = endpointRule86(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule87(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule88(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule86(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null && params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule87(S3ControlEndpointParams params, LocalState locals) {
        if (params.useArnRegion() != null && !params.useArnRegion()
                && !(RulesFunctions.stringEquals(locals.bucketArn().region(), locals.region()))) {
            return RuleResult.error("Invalid configuration: region from ARN `" + locals.bucketArn().region()
                    + "` does not match client region `" + locals.region() + "` and UseArnRegion is `false`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule88(S3ControlEndpointParams params, LocalState locals) {
        RulePartition arnPartition = null;
        if ((arnPartition = RulesFunctions.awsPartition(locals.bucketArn().region())) != null) {
            locals = locals.toBuilder().arnPartition(arnPartition).build();
            return endpointRule89(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule89(S3ControlEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule90(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule109(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule90(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.stringEquals(locals.arnPartition().name(), locals.partitionResult().name())) {
            RuleResult result = endpointRule91(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule108(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule91(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.bucketArn().region(), true)) {
            RuleResult result = endpointRule92(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule107(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule92(S3ControlEndpointParams params, LocalState locals) {
        if (!("".equals(locals.bucketArn().accountId()))) {
            RuleResult result = endpointRule93(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule106(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule93(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.bucketArn().accountId(), false)) {
            RuleResult result = endpointRule94(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule95(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule105(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule94(S3ControlEndpointParams params, LocalState locals) {
        if (params.accountId() != null && !(RulesFunctions.stringEquals(params.accountId(), locals.bucketArn().accountId()))) {
            return RuleResult.error("Invalid ARN: the accountId specified in the ARN (`" + locals.bucketArn().accountId()
                    + "`) does not match the parameter (`" + params.accountId() + "`)");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule95(S3ControlEndpointParams params, LocalState locals) {
        String outpostType = null;
        if ((outpostType = RulesFunctions.listAccess(locals.bucketArn().resourceId(), 2)) != null) {
            locals = locals.toBuilder().outpostType(outpostType).build();
            RuleResult result = endpointRule96(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule104(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule96(S3ControlEndpointParams params, LocalState locals) {
        String bucketName = null;
        if ((bucketName = RulesFunctions.listAccess(locals.bucketArn().resourceId(), 3)) != null) {
            locals = locals.toBuilder().bucketName(bucketName).build();
            RuleResult result = endpointRule97(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule103(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule97(S3ControlEndpointParams params, LocalState locals) {
        if ("bucket".equals(locals.outpostType())) {
            RuleResult result = endpointRule98(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule99(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule100(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule101(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule102(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule98(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips() && params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts-fips." + locals.bucketArn().region() + "."
                            + locals.arnPartition().dualStackDnsSuffix()))
                    .putHeader("x-amz-account-id", locals.bucketArn().accountId())
                    .putHeader("x-amz-outpost-id", locals.outpostId())
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.bucketArn().region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule99(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts-fips." + locals.bucketArn().region() + "."
                            + locals.arnPartition().dnsSuffix()))
                    .putHeader("x-amz-account-id", locals.bucketArn().accountId())
                    .putHeader("x-amz-outpost-id", locals.outpostId())
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.bucketArn().region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule100(S3ControlEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts." + locals.bucketArn().region() + "."
                            + locals.arnPartition().dualStackDnsSuffix()))
                    .putHeader("x-amz-account-id", locals.bucketArn().accountId())
                    .putHeader("x-amz-outpost-id", locals.outpostId())
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(locals.bucketArn().region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule101(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleUrl url = null;
            if ((url = RulesFunctions.parseURL(params.endpoint())) != null) {
                locals = locals.toBuilder().url(url).build();
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create(locals.url().scheme() + "://" + locals.url().authority() + locals.url().path()))
                        .putHeader("x-amz-account-id", locals.bucketArn().accountId())
                        .putHeader("x-amz-outpost-id", locals.outpostId())
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(locals.bucketArn().region()).build())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule102(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create("https://s3-outposts." + locals.bucketArn().region() + "." + locals.arnPartition().dnsSuffix()))
                .putHeader("x-amz-account-id", locals.bucketArn().accountId())
                .putHeader("x-amz-outpost-id", locals.outpostId())
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                .signingRegion(locals.bucketArn().region()).build())).build());
    }

    private static RuleResult endpointRule103(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: Expected an outpost type `bucket`, found `" + locals.outpostType() + "`");
    }

    private static RuleResult endpointRule104(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: expected a bucket name");
    }

    private static RuleResult endpointRule105(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: Expected a 4-component resource");
    }

    private static RuleResult endpointRule106(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: The account id may only contain a-z, A-Z, 0-9 and `-`. Found: `"
                + locals.bucketArn().accountId() + "`");
    }

    private static RuleResult endpointRule107(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: missing account ID");
    }

    private static RuleResult endpointRule108(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid region in ARN: `" + locals.bucketArn().region() + "` (invalid DNS name)");
    }

    private static RuleResult endpointRule109(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Client was configured for partition `" + locals.partitionResult().name() + "` but ARN has `"
                + locals.arnPartition().name() + "`");
    }

    private static RuleResult endpointRule110(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: The outpost Id must only contain a-z, A-Z, 0-9 and `-`., found: `"
                + locals.outpostId() + "`");
    }

    private static RuleResult endpointRule111(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: The Outpost Id was not set");
    }

    private static RuleResult endpointRule112(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid ARN: No ARN type specified");
    }

    private static RuleResult endpointRule113(S3ControlEndpointParams params, LocalState locals) {
        RulePartition partitionResult = null;
        if ((partitionResult = RulesFunctions.awsPartition(locals.region())) != null) {
            locals = locals.toBuilder().partitionResult(partitionResult).build();
            RuleResult result = endpointRule114(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule129(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule114(S3ControlEndpointParams params, LocalState locals) {
        if (RulesFunctions.isValidHostLabel(locals.region(), true)) {
            RuleResult result = endpointRule115(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule116(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule117(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule121(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule122(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule123(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule124(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule125(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule126(params, locals);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule127(params, locals);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule128(params, locals);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule115(S3ControlEndpointParams params, LocalState locals) {
        if (params.requiresAccountId() != null && params.requiresAccountId() && params.accountId() == null) {
            return RuleResult.error("AccountId is required but not set");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule116(S3ControlEndpointParams params, LocalState locals) {
        if (params.accountId() != null && !(RulesFunctions.isValidHostLabel(params.accountId(), false))) {
            return RuleResult.error("AccountId must only contain a-z, A-Z, 0-9 and `-`.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule117(S3ControlEndpointParams params, LocalState locals) {
        if (params.endpoint() != null) {
            RuleUrl url = null;
            if ((url = RulesFunctions.parseURL(params.endpoint())) != null) {
                locals = locals.toBuilder().url(url).build();
                RuleResult result = endpointRule118(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                result = endpointRule119(params, locals);
                if (result.isResolved()) {
                    return result;
                }
                return endpointRule120(params, locals);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule118(S3ControlEndpointParams params, LocalState locals) {
        if (params.useDualStack()) {
            return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule119(S3ControlEndpointParams params, LocalState locals) {
        if (params.requiresAccountId() != null && params.requiresAccountId() && params.accountId() != null) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create(locals.url().scheme() + "://" + params.accountId() + "." + locals.url().authority()
                            + locals.url().path()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule120(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.endpoint(Endpoint
                .builder()
                .url(URI.create(locals.url().scheme() + "://" + locals.url().authority() + locals.url().path()))
                .putAttribute(
                        AwsEndpointAttribute.AUTH_SCHEMES,
                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                .signingRegion(locals.region()).build())).build());
    }

    private static RuleResult endpointRule121(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips() && params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                && params.accountId() != null) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://" + params.accountId() + ".s3-control-fips.dualstack." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule122(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips() && params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-control-fips.dualstack." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule123(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips() && !params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                && params.accountId() != null) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://" + params.accountId() + ".s3-control-fips." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule124(S3ControlEndpointParams params, LocalState locals) {
        if (params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-control-fips." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule125(S3ControlEndpointParams params, LocalState locals) {
        if (!params.useFips() && params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                && params.accountId() != null) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://" + params.accountId() + ".s3-control.dualstack." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule126(S3ControlEndpointParams params, LocalState locals) {
        if (!params.useFips() && params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-control.dualstack." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule127(S3ControlEndpointParams params, LocalState locals) {
        if (!params.useFips() && !params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                && params.accountId() != null) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://" + params.accountId() + ".s3-control." + locals.region() + "."
                            + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule128(S3ControlEndpointParams params, LocalState locals) {
        if (!params.useFips() && !params.useDualStack()) {
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-control." + locals.region() + "." + locals.partitionResult().dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(locals.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule129(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Invalid region: region was not a valid DNS name.");
    }

    private static RuleResult endpointRule130(S3ControlEndpointParams params, LocalState locals) {
        return RuleResult.error("Region must be set");
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }

    private static final class LocalState {
        private final String region;

        private final RulePartition partitionResult;

        private final RuleUrl url;

        private final String accessPointSuffix;

        private final String s3ExpressAvailabilityZoneId;

        private final String s3ExpressAvailabilityZoneDelim;

        private final RuleArn accessPointArn;

        private final String arnType;

        private final String outpostId;

        private final RulePartition arnPartition;

        private final String outpostType;

        private final String accessPointName;

        private final RuleArn bucketArn;

        private final String bucketName;

        LocalState() {
            this.region = null;
            this.partitionResult = null;
            this.url = null;
            this.accessPointSuffix = null;
            this.s3ExpressAvailabilityZoneId = null;
            this.s3ExpressAvailabilityZoneDelim = null;
            this.accessPointArn = null;
            this.arnType = null;
            this.outpostId = null;
            this.arnPartition = null;
            this.outpostType = null;
            this.accessPointName = null;
            this.bucketArn = null;
            this.bucketName = null;
        }

        LocalState(Region region) {
            if (region != null) {
                this.region = region.id();
            } else {
                this.region = null;
            }
            this.partitionResult = null;
            this.url = null;
            this.accessPointSuffix = null;
            this.s3ExpressAvailabilityZoneId = null;
            this.s3ExpressAvailabilityZoneDelim = null;
            this.accessPointArn = null;
            this.arnType = null;
            this.outpostId = null;
            this.arnPartition = null;
            this.outpostType = null;
            this.accessPointName = null;
            this.bucketArn = null;
            this.bucketName = null;
        }

        LocalState(LocalStateBuilder builder) {
            this.region = builder.region;
            this.partitionResult = builder.partitionResult;
            this.url = builder.url;
            this.accessPointSuffix = builder.accessPointSuffix;
            this.s3ExpressAvailabilityZoneId = builder.s3ExpressAvailabilityZoneId;
            this.s3ExpressAvailabilityZoneDelim = builder.s3ExpressAvailabilityZoneDelim;
            this.accessPointArn = builder.accessPointArn;
            this.arnType = builder.arnType;
            this.outpostId = builder.outpostId;
            this.arnPartition = builder.arnPartition;
            this.outpostType = builder.outpostType;
            this.accessPointName = builder.accessPointName;
            this.bucketArn = builder.bucketArn;
            this.bucketName = builder.bucketName;
        }

        public String region() {
            return this.region;
        }

        public RulePartition partitionResult() {
            return this.partitionResult;
        }

        public RuleUrl url() {
            return this.url;
        }

        public String accessPointSuffix() {
            return this.accessPointSuffix;
        }

        public String s3ExpressAvailabilityZoneId() {
            return this.s3ExpressAvailabilityZoneId;
        }

        public String s3ExpressAvailabilityZoneDelim() {
            return this.s3ExpressAvailabilityZoneDelim;
        }

        public RuleArn accessPointArn() {
            return this.accessPointArn;
        }

        public String arnType() {
            return this.arnType;
        }

        public String outpostId() {
            return this.outpostId;
        }

        public RulePartition arnPartition() {
            return this.arnPartition;
        }

        public String outpostType() {
            return this.outpostType;
        }

        public String accessPointName() {
            return this.accessPointName;
        }

        public RuleArn bucketArn() {
            return this.bucketArn;
        }

        public String bucketName() {
            return this.bucketName;
        }

        public LocalStateBuilder toBuilder() {
            return new LocalStateBuilder(this);
        }
    }

    private static final class LocalStateBuilder {
        private String region;

        private RulePartition partitionResult;

        private RuleUrl url;

        private String accessPointSuffix;

        private String s3ExpressAvailabilityZoneId;

        private String s3ExpressAvailabilityZoneDelim;

        private RuleArn accessPointArn;

        private String arnType;

        private String outpostId;

        private RulePartition arnPartition;

        private String outpostType;

        private String accessPointName;

        private RuleArn bucketArn;

        private String bucketName;

        LocalStateBuilder() {
            this.region = null;
            this.partitionResult = null;
            this.url = null;
            this.accessPointSuffix = null;
            this.s3ExpressAvailabilityZoneId = null;
            this.s3ExpressAvailabilityZoneDelim = null;
            this.accessPointArn = null;
            this.arnType = null;
            this.outpostId = null;
            this.arnPartition = null;
            this.outpostType = null;
            this.accessPointName = null;
            this.bucketArn = null;
            this.bucketName = null;
        }

        LocalStateBuilder(LocalState locals) {
            this.region = locals.region;
            this.partitionResult = locals.partitionResult;
            this.url = locals.url;
            this.accessPointSuffix = locals.accessPointSuffix;
            this.s3ExpressAvailabilityZoneId = locals.s3ExpressAvailabilityZoneId;
            this.s3ExpressAvailabilityZoneDelim = locals.s3ExpressAvailabilityZoneDelim;
            this.accessPointArn = locals.accessPointArn;
            this.arnType = locals.arnType;
            this.outpostId = locals.outpostId;
            this.arnPartition = locals.arnPartition;
            this.outpostType = locals.outpostType;
            this.accessPointName = locals.accessPointName;
            this.bucketArn = locals.bucketArn;
            this.bucketName = locals.bucketName;
        }

        public LocalStateBuilder region(String value) {
            this.region = value;
            return this;
        }

        public LocalStateBuilder partitionResult(RulePartition value) {
            this.partitionResult = value;
            return this;
        }

        public LocalStateBuilder url(RuleUrl value) {
            this.url = value;
            return this;
        }

        public LocalStateBuilder accessPointSuffix(String value) {
            this.accessPointSuffix = value;
            return this;
        }

        public LocalStateBuilder s3ExpressAvailabilityZoneId(String value) {
            this.s3ExpressAvailabilityZoneId = value;
            return this;
        }

        public LocalStateBuilder s3ExpressAvailabilityZoneDelim(String value) {
            this.s3ExpressAvailabilityZoneDelim = value;
            return this;
        }

        public LocalStateBuilder accessPointArn(RuleArn value) {
            this.accessPointArn = value;
            return this;
        }

        public LocalStateBuilder arnType(String value) {
            this.arnType = value;
            return this;
        }

        public LocalStateBuilder outpostId(String value) {
            this.outpostId = value;
            return this;
        }

        public LocalStateBuilder arnPartition(RulePartition value) {
            this.arnPartition = value;
            return this;
        }

        public LocalStateBuilder outpostType(String value) {
            this.outpostType = value;
            return this;
        }

        public LocalStateBuilder accessPointName(String value) {
            this.accessPointName = value;
            return this;
        }

        public LocalStateBuilder bucketArn(RuleArn value) {
            this.bucketArn = value;
            return this;
        }

        public LocalStateBuilder bucketName(String value) {
            this.bucketName = value;
            return this;
        }

        LocalState build() {
            return new LocalState(this);
        }
    }
}
