/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container element for the job configuration and status information returned by a <code>Describe Job</code> request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobDescriptor implements SdkPojo, Serializable, ToCopyableBuilder<JobDescriptor.Builder, JobDescriptor> {
    private static final SdkField<String> JOB_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("JobId")
            .getter(getter(JobDescriptor::jobId))
            .setter(setter(Builder::jobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobId")
                    .unmarshallLocationName("JobId").build()).build();

    private static final SdkField<Boolean> CONFIRMATION_REQUIRED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ConfirmationRequired")
            .getter(getter(JobDescriptor::confirmationRequired))
            .setter(setter(Builder::confirmationRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfirmationRequired")
                    .unmarshallLocationName("ConfirmationRequired").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(JobDescriptor::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<String> JOB_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("JobArn")
            .getter(getter(JobDescriptor::jobArn))
            .setter(setter(Builder::jobArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobArn")
                    .unmarshallLocationName("JobArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Status")
            .getter(getter(JobDescriptor::statusAsString))
            .setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("Status").build()).build();

    private static final SdkField<JobManifest> MANIFEST_FIELD = SdkField
            .<JobManifest> builder(MarshallingType.SDK_POJO)
            .memberName("Manifest")
            .getter(getter(JobDescriptor::manifest))
            .setter(setter(Builder::manifest))
            .constructor(JobManifest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Manifest")
                    .unmarshallLocationName("Manifest").build()).build();

    private static final SdkField<JobOperation> OPERATION_FIELD = SdkField
            .<JobOperation> builder(MarshallingType.SDK_POJO)
            .memberName("Operation")
            .getter(getter(JobDescriptor::operation))
            .setter(setter(Builder::operation))
            .constructor(JobOperation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation")
                    .unmarshallLocationName("Operation").build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority")
            .getter(getter(JobDescriptor::priority))
            .setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority")
                    .unmarshallLocationName("Priority").build()).build();

    private static final SdkField<JobProgressSummary> PROGRESS_SUMMARY_FIELD = SdkField
            .<JobProgressSummary> builder(MarshallingType.SDK_POJO)
            .memberName("ProgressSummary")
            .getter(getter(JobDescriptor::progressSummary))
            .setter(setter(Builder::progressSummary))
            .constructor(JobProgressSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProgressSummary")
                    .unmarshallLocationName("ProgressSummary").build()).build();

    private static final SdkField<String> STATUS_UPDATE_REASON_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StatusUpdateReason")
            .getter(getter(JobDescriptor::statusUpdateReason))
            .setter(setter(Builder::statusUpdateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusUpdateReason")
                    .unmarshallLocationName("StatusUpdateReason").build()).build();

    private static final SdkField<List<JobFailure>> FAILURE_REASONS_FIELD = SdkField
            .<List<JobFailure>> builder(MarshallingType.LIST)
            .memberName("FailureReasons")
            .getter(getter(JobDescriptor::failureReasons))
            .setter(setter(Builder::failureReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReasons")
                    .unmarshallLocationName("FailureReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<JobFailure> builder(MarshallingType.SDK_POJO)
                                            .constructor(JobFailure::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<JobReport> REPORT_FIELD = SdkField
            .<JobReport> builder(MarshallingType.SDK_POJO)
            .memberName("Report")
            .getter(getter(JobDescriptor::report))
            .setter(setter(Builder::report))
            .constructor(JobReport::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Report")
                    .unmarshallLocationName("Report").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime")
            .getter(getter(JobDescriptor::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime")
                    .unmarshallLocationName("CreationTime").build()).build();

    private static final SdkField<Instant> TERMINATION_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("TerminationDate")
            .getter(getter(JobDescriptor::terminationDate))
            .setter(setter(Builder::terminationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationDate")
                    .unmarshallLocationName("TerminationDate").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RoleArn")
            .getter(getter(JobDescriptor::roleArn))
            .setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn")
                    .unmarshallLocationName("RoleArn").build()).build();

    private static final SdkField<Instant> SUSPENDED_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SuspendedDate")
            .getter(getter(JobDescriptor::suspendedDate))
            .setter(setter(Builder::suspendedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SuspendedDate")
                    .unmarshallLocationName("SuspendedDate").build()).build();

    private static final SdkField<String> SUSPENDED_CAUSE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SuspendedCause")
            .getter(getter(JobDescriptor::suspendedCause))
            .setter(setter(Builder::suspendedCause))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SuspendedCause")
                    .unmarshallLocationName("SuspendedCause").build()).build();

    private static final SdkField<JobManifestGenerator> MANIFEST_GENERATOR_FIELD = SdkField
            .<JobManifestGenerator> builder(MarshallingType.SDK_POJO)
            .memberName("ManifestGenerator")
            .getter(getter(JobDescriptor::manifestGenerator))
            .setter(setter(Builder::manifestGenerator))
            .constructor(JobManifestGenerator::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManifestGenerator")
                    .unmarshallLocationName("ManifestGenerator").build()).build();

    private static final SdkField<S3GeneratedManifestDescriptor> GENERATED_MANIFEST_DESCRIPTOR_FIELD = SdkField
            .<S3GeneratedManifestDescriptor> builder(MarshallingType.SDK_POJO)
            .memberName("GeneratedManifestDescriptor")
            .getter(getter(JobDescriptor::generatedManifestDescriptor))
            .setter(setter(Builder::generatedManifestDescriptor))
            .constructor(S3GeneratedManifestDescriptor::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeneratedManifestDescriptor")
                    .unmarshallLocationName("GeneratedManifestDescriptor").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_ID_FIELD,
            CONFIRMATION_REQUIRED_FIELD, DESCRIPTION_FIELD, JOB_ARN_FIELD, STATUS_FIELD, MANIFEST_FIELD, OPERATION_FIELD,
            PRIORITY_FIELD, PROGRESS_SUMMARY_FIELD, STATUS_UPDATE_REASON_FIELD, FAILURE_REASONS_FIELD, REPORT_FIELD,
            CREATION_TIME_FIELD, TERMINATION_DATE_FIELD, ROLE_ARN_FIELD, SUSPENDED_DATE_FIELD, SUSPENDED_CAUSE_FIELD,
            MANIFEST_GENERATOR_FIELD, GENERATED_MANIFEST_DESCRIPTOR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String jobId;

    private final Boolean confirmationRequired;

    private final String description;

    private final String jobArn;

    private final String status;

    private final JobManifest manifest;

    private final JobOperation operation;

    private final Integer priority;

    private final JobProgressSummary progressSummary;

    private final String statusUpdateReason;

    private final List<JobFailure> failureReasons;

    private final JobReport report;

    private final Instant creationTime;

    private final Instant terminationDate;

    private final String roleArn;

    private final Instant suspendedDate;

    private final String suspendedCause;

    private final JobManifestGenerator manifestGenerator;

    private final S3GeneratedManifestDescriptor generatedManifestDescriptor;

    private JobDescriptor(BuilderImpl builder) {
        this.jobId = builder.jobId;
        this.confirmationRequired = builder.confirmationRequired;
        this.description = builder.description;
        this.jobArn = builder.jobArn;
        this.status = builder.status;
        this.manifest = builder.manifest;
        this.operation = builder.operation;
        this.priority = builder.priority;
        this.progressSummary = builder.progressSummary;
        this.statusUpdateReason = builder.statusUpdateReason;
        this.failureReasons = builder.failureReasons;
        this.report = builder.report;
        this.creationTime = builder.creationTime;
        this.terminationDate = builder.terminationDate;
        this.roleArn = builder.roleArn;
        this.suspendedDate = builder.suspendedDate;
        this.suspendedCause = builder.suspendedCause;
        this.manifestGenerator = builder.manifestGenerator;
        this.generatedManifestDescriptor = builder.generatedManifestDescriptor;
    }

    /**
     * <p>
     * The ID for the specified job.
     * </p>
     * 
     * @return The ID for the specified job.
     */
    public final String jobId() {
        return jobId;
    }

    /**
     * <p>
     * Indicates whether confirmation is required before Amazon S3 begins running the specified job. Confirmation is
     * required only for jobs created through the Amazon S3 console.
     * </p>
     * 
     * @return Indicates whether confirmation is required before Amazon S3 begins running the specified job.
     *         Confirmation is required only for jobs created through the Amazon S3 console.
     */
    public final Boolean confirmationRequired() {
        return confirmationRequired;
    }

    /**
     * <p>
     * The description for this job, if one was provided in this job's <code>Create Job</code> request.
     * </p>
     * 
     * @return The description for this job, if one was provided in this job's <code>Create Job</code> request.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for this job.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for this job.
     */
    public final String jobArn() {
        return jobArn;
    }

    /**
     * <p>
     * The current status of the specified job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the specified job.
     * @see JobStatus
     */
    public final JobStatus status() {
        return JobStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the specified job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the specified job.
     * @see JobStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The configuration information for the specified job's manifest object.
     * </p>
     * 
     * @return The configuration information for the specified job's manifest object.
     */
    public final JobManifest manifest() {
        return manifest;
    }

    /**
     * <p>
     * The operation that the specified job is configured to run on the objects listed in the manifest.
     * </p>
     * 
     * @return The operation that the specified job is configured to run on the objects listed in the manifest.
     */
    public final JobOperation operation() {
        return operation;
    }

    /**
     * <p>
     * The priority of the specified job.
     * </p>
     * 
     * @return The priority of the specified job.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * <p>
     * Describes the total number of tasks that the specified job has run, the number of tasks that succeeded, and the
     * number of tasks that failed.
     * </p>
     * 
     * @return Describes the total number of tasks that the specified job has run, the number of tasks that succeeded,
     *         and the number of tasks that failed.
     */
    public final JobProgressSummary progressSummary() {
        return progressSummary;
    }

    /**
     * <p>
     * The reason for updating the job.
     * </p>
     * 
     * @return The reason for updating the job.
     */
    public final String statusUpdateReason() {
        return statusUpdateReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureReasons() {
        return failureReasons != null && !(failureReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If the specified job failed, this field contains information describing the failure.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureReasons} method.
     * </p>
     * 
     * @return If the specified job failed, this field contains information describing the failure.
     */
    public final List<JobFailure> failureReasons() {
        return failureReasons;
    }

    /**
     * <p>
     * Contains the configuration information for the job-completion report if you requested one in the
     * <code>Create Job</code> request.
     * </p>
     * 
     * @return Contains the configuration information for the job-completion report if you requested one in the
     *         <code>Create Job</code> request.
     */
    public final JobReport report() {
        return report;
    }

    /**
     * <p>
     * A timestamp indicating when this job was created.
     * </p>
     * 
     * @return A timestamp indicating when this job was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A timestamp indicating when this job terminated. A job's termination date is the date and time when it succeeded,
     * failed, or was canceled.
     * </p>
     * 
     * @return A timestamp indicating when this job terminated. A job's termination date is the date and time when it
     *         succeeded, failed, or was canceled.
     */
    public final Instant terminationDate() {
        return terminationDate;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role assigned to run the tasks for
     * this job.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role assigned to run the
     *         tasks for this job.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The timestamp when this job was suspended, if it has been suspended.
     * </p>
     * 
     * @return The timestamp when this job was suspended, if it has been suspended.
     */
    public final Instant suspendedDate() {
        return suspendedDate;
    }

    /**
     * <p>
     * The reason why the specified job was suspended. A job is only suspended if you create it through the Amazon S3
     * console. When you create the job, it enters the <code>Suspended</code> state to await confirmation before
     * running. After you confirm the job, it automatically exits the <code>Suspended</code> state.
     * </p>
     * 
     * @return The reason why the specified job was suspended. A job is only suspended if you create it through the
     *         Amazon S3 console. When you create the job, it enters the <code>Suspended</code> state to await
     *         confirmation before running. After you confirm the job, it automatically exits the <code>Suspended</code>
     *         state.
     */
    public final String suspendedCause() {
        return suspendedCause;
    }

    /**
     * <p>
     * The manifest generator that was used to generate a job manifest for this job.
     * </p>
     * 
     * @return The manifest generator that was used to generate a job manifest for this job.
     */
    public final JobManifestGenerator manifestGenerator() {
        return manifestGenerator;
    }

    /**
     * <p>
     * The attribute of the JobDescriptor containing details about the job's generated manifest.
     * </p>
     * 
     * @return The attribute of the JobDescriptor containing details about the job's generated manifest.
     */
    public final S3GeneratedManifestDescriptor generatedManifestDescriptor() {
        return generatedManifestDescriptor;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobId());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationRequired());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(jobArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(manifest());
        hashCode = 31 * hashCode + Objects.hashCode(operation());
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(progressSummary());
        hashCode = 31 * hashCode + Objects.hashCode(statusUpdateReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureReasons() ? failureReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(report());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(terminationDate());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(suspendedDate());
        hashCode = 31 * hashCode + Objects.hashCode(suspendedCause());
        hashCode = 31 * hashCode + Objects.hashCode(manifestGenerator());
        hashCode = 31 * hashCode + Objects.hashCode(generatedManifestDescriptor());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobDescriptor)) {
            return false;
        }
        JobDescriptor other = (JobDescriptor) obj;
        return Objects.equals(jobId(), other.jobId()) && Objects.equals(confirmationRequired(), other.confirmationRequired())
                && Objects.equals(description(), other.description()) && Objects.equals(jobArn(), other.jobArn())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(manifest(), other.manifest())
                && Objects.equals(operation(), other.operation()) && Objects.equals(priority(), other.priority())
                && Objects.equals(progressSummary(), other.progressSummary())
                && Objects.equals(statusUpdateReason(), other.statusUpdateReason())
                && hasFailureReasons() == other.hasFailureReasons() && Objects.equals(failureReasons(), other.failureReasons())
                && Objects.equals(report(), other.report()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(terminationDate(), other.terminationDate()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(suspendedDate(), other.suspendedDate())
                && Objects.equals(suspendedCause(), other.suspendedCause())
                && Objects.equals(manifestGenerator(), other.manifestGenerator())
                && Objects.equals(generatedManifestDescriptor(), other.generatedManifestDescriptor());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobDescriptor").add("JobId", jobId()).add("ConfirmationRequired", confirmationRequired())
                .add("Description", description()).add("JobArn", jobArn()).add("Status", statusAsString())
                .add("Manifest", manifest()).add("Operation", operation()).add("Priority", priority())
                .add("ProgressSummary", progressSummary()).add("StatusUpdateReason", statusUpdateReason())
                .add("FailureReasons", hasFailureReasons() ? failureReasons() : null).add("Report", report())
                .add("CreationTime", creationTime()).add("TerminationDate", terminationDate()).add("RoleArn", roleArn())
                .add("SuspendedDate", suspendedDate()).add("SuspendedCause", suspendedCause())
                .add("ManifestGenerator", manifestGenerator()).add("GeneratedManifestDescriptor", generatedManifestDescriptor())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "JobId":
            return Optional.ofNullable(clazz.cast(jobId()));
        case "ConfirmationRequired":
            return Optional.ofNullable(clazz.cast(confirmationRequired()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "JobArn":
            return Optional.ofNullable(clazz.cast(jobArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Manifest":
            return Optional.ofNullable(clazz.cast(manifest()));
        case "Operation":
            return Optional.ofNullable(clazz.cast(operation()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "ProgressSummary":
            return Optional.ofNullable(clazz.cast(progressSummary()));
        case "StatusUpdateReason":
            return Optional.ofNullable(clazz.cast(statusUpdateReason()));
        case "FailureReasons":
            return Optional.ofNullable(clazz.cast(failureReasons()));
        case "Report":
            return Optional.ofNullable(clazz.cast(report()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "TerminationDate":
            return Optional.ofNullable(clazz.cast(terminationDate()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "SuspendedDate":
            return Optional.ofNullable(clazz.cast(suspendedDate()));
        case "SuspendedCause":
            return Optional.ofNullable(clazz.cast(suspendedCause()));
        case "ManifestGenerator":
            return Optional.ofNullable(clazz.cast(manifestGenerator()));
        case "GeneratedManifestDescriptor":
            return Optional.ofNullable(clazz.cast(generatedManifestDescriptor()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("JobId", JOB_ID_FIELD);
        map.put("ConfirmationRequired", CONFIRMATION_REQUIRED_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("JobArn", JOB_ARN_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("Manifest", MANIFEST_FIELD);
        map.put("Operation", OPERATION_FIELD);
        map.put("Priority", PRIORITY_FIELD);
        map.put("ProgressSummary", PROGRESS_SUMMARY_FIELD);
        map.put("StatusUpdateReason", STATUS_UPDATE_REASON_FIELD);
        map.put("FailureReasons", FAILURE_REASONS_FIELD);
        map.put("Report", REPORT_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("TerminationDate", TERMINATION_DATE_FIELD);
        map.put("RoleArn", ROLE_ARN_FIELD);
        map.put("SuspendedDate", SUSPENDED_DATE_FIELD);
        map.put("SuspendedCause", SUSPENDED_CAUSE_FIELD);
        map.put("ManifestGenerator", MANIFEST_GENERATOR_FIELD);
        map.put("GeneratedManifestDescriptor", GENERATED_MANIFEST_DESCRIPTOR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<JobDescriptor, T> g) {
        return obj -> g.apply((JobDescriptor) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobDescriptor> {
        /**
         * <p>
         * The ID for the specified job.
         * </p>
         * 
         * @param jobId
         *        The ID for the specified job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobId(String jobId);

        /**
         * <p>
         * Indicates whether confirmation is required before Amazon S3 begins running the specified job. Confirmation is
         * required only for jobs created through the Amazon S3 console.
         * </p>
         * 
         * @param confirmationRequired
         *        Indicates whether confirmation is required before Amazon S3 begins running the specified job.
         *        Confirmation is required only for jobs created through the Amazon S3 console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confirmationRequired(Boolean confirmationRequired);

        /**
         * <p>
         * The description for this job, if one was provided in this job's <code>Create Job</code> request.
         * </p>
         * 
         * @param description
         *        The description for this job, if one was provided in this job's <code>Create Job</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for this job.
         * </p>
         * 
         * @param jobArn
         *        The Amazon Resource Name (ARN) for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobArn(String jobArn);

        /**
         * <p>
         * The current status of the specified job.
         * </p>
         * 
         * @param status
         *        The current status of the specified job.
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the specified job.
         * </p>
         * 
         * @param status
         *        The current status of the specified job.
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder status(JobStatus status);

        /**
         * <p>
         * The configuration information for the specified job's manifest object.
         * </p>
         * 
         * @param manifest
         *        The configuration information for the specified job's manifest object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifest(JobManifest manifest);

        /**
         * <p>
         * The configuration information for the specified job's manifest object.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobManifest.Builder} avoiding the need to
         * create one manually via {@link JobManifest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobManifest.Builder#build()} is called immediately and its result
         * is passed to {@link #manifest(JobManifest)}.
         * 
         * @param manifest
         *        a consumer that will call methods on {@link JobManifest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #manifest(JobManifest)
         */
        default Builder manifest(Consumer<JobManifest.Builder> manifest) {
            return manifest(JobManifest.builder().applyMutation(manifest).build());
        }

        /**
         * <p>
         * The operation that the specified job is configured to run on the objects listed in the manifest.
         * </p>
         * 
         * @param operation
         *        The operation that the specified job is configured to run on the objects listed in the manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operation(JobOperation operation);

        /**
         * <p>
         * The operation that the specified job is configured to run on the objects listed in the manifest.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobOperation.Builder} avoiding the need
         * to create one manually via {@link JobOperation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobOperation.Builder#build()} is called immediately and its
         * result is passed to {@link #operation(JobOperation)}.
         * 
         * @param operation
         *        a consumer that will call methods on {@link JobOperation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operation(JobOperation)
         */
        default Builder operation(Consumer<JobOperation.Builder> operation) {
            return operation(JobOperation.builder().applyMutation(operation).build());
        }

        /**
         * <p>
         * The priority of the specified job.
         * </p>
         * 
         * @param priority
         *        The priority of the specified job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * Describes the total number of tasks that the specified job has run, the number of tasks that succeeded, and
         * the number of tasks that failed.
         * </p>
         * 
         * @param progressSummary
         *        Describes the total number of tasks that the specified job has run, the number of tasks that
         *        succeeded, and the number of tasks that failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder progressSummary(JobProgressSummary progressSummary);

        /**
         * <p>
         * Describes the total number of tasks that the specified job has run, the number of tasks that succeeded, and
         * the number of tasks that failed.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobProgressSummary.Builder} avoiding the
         * need to create one manually via {@link JobProgressSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobProgressSummary.Builder#build()} is called immediately and its
         * result is passed to {@link #progressSummary(JobProgressSummary)}.
         * 
         * @param progressSummary
         *        a consumer that will call methods on {@link JobProgressSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #progressSummary(JobProgressSummary)
         */
        default Builder progressSummary(Consumer<JobProgressSummary.Builder> progressSummary) {
            return progressSummary(JobProgressSummary.builder().applyMutation(progressSummary).build());
        }

        /**
         * <p>
         * The reason for updating the job.
         * </p>
         * 
         * @param statusUpdateReason
         *        The reason for updating the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusUpdateReason(String statusUpdateReason);

        /**
         * <p>
         * If the specified job failed, this field contains information describing the failure.
         * </p>
         * 
         * @param failureReasons
         *        If the specified job failed, this field contains information describing the failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(Collection<JobFailure> failureReasons);

        /**
         * <p>
         * If the specified job failed, this field contains information describing the failure.
         * </p>
         * 
         * @param failureReasons
         *        If the specified job failed, this field contains information describing the failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(JobFailure... failureReasons);

        /**
         * <p>
         * If the specified job failed, this field contains information describing the failure.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.s3control.model.JobFailure.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.s3control.model.JobFailure#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.s3control.model.JobFailure.Builder#build()} is called immediately and
         * its result is passed to {@link #failureReasons(List<JobFailure>)}.
         * 
         * @param failureReasons
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.s3control.model.JobFailure.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureReasons(java.util.Collection<JobFailure>)
         */
        Builder failureReasons(Consumer<JobFailure.Builder>... failureReasons);

        /**
         * <p>
         * Contains the configuration information for the job-completion report if you requested one in the
         * <code>Create Job</code> request.
         * </p>
         * 
         * @param report
         *        Contains the configuration information for the job-completion report if you requested one in the
         *        <code>Create Job</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder report(JobReport report);

        /**
         * <p>
         * Contains the configuration information for the job-completion report if you requested one in the
         * <code>Create Job</code> request.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobReport.Builder} avoiding the need to
         * create one manually via {@link JobReport#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobReport.Builder#build()} is called immediately and its result
         * is passed to {@link #report(JobReport)}.
         * 
         * @param report
         *        a consumer that will call methods on {@link JobReport.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #report(JobReport)
         */
        default Builder report(Consumer<JobReport.Builder> report) {
            return report(JobReport.builder().applyMutation(report).build());
        }

        /**
         * <p>
         * A timestamp indicating when this job was created.
         * </p>
         * 
         * @param creationTime
         *        A timestamp indicating when this job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A timestamp indicating when this job terminated. A job's termination date is the date and time when it
         * succeeded, failed, or was canceled.
         * </p>
         * 
         * @param terminationDate
         *        A timestamp indicating when this job terminated. A job's termination date is the date and time when it
         *        succeeded, failed, or was canceled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationDate(Instant terminationDate);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role assigned to run the tasks
         * for this job.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) for the Identity and Access Management (IAM) role assigned to run the
         *        tasks for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The timestamp when this job was suspended, if it has been suspended.
         * </p>
         * 
         * @param suspendedDate
         *        The timestamp when this job was suspended, if it has been suspended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedDate(Instant suspendedDate);

        /**
         * <p>
         * The reason why the specified job was suspended. A job is only suspended if you create it through the Amazon
         * S3 console. When you create the job, it enters the <code>Suspended</code> state to await confirmation before
         * running. After you confirm the job, it automatically exits the <code>Suspended</code> state.
         * </p>
         * 
         * @param suspendedCause
         *        The reason why the specified job was suspended. A job is only suspended if you create it through the
         *        Amazon S3 console. When you create the job, it enters the <code>Suspended</code> state to await
         *        confirmation before running. After you confirm the job, it automatically exits the
         *        <code>Suspended</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedCause(String suspendedCause);

        /**
         * <p>
         * The manifest generator that was used to generate a job manifest for this job.
         * </p>
         * 
         * @param manifestGenerator
         *        The manifest generator that was used to generate a job manifest for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder manifestGenerator(JobManifestGenerator manifestGenerator);

        /**
         * <p>
         * The manifest generator that was used to generate a job manifest for this job.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobManifestGenerator.Builder} avoiding
         * the need to create one manually via {@link JobManifestGenerator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobManifestGenerator.Builder#build()} is called immediately and
         * its result is passed to {@link #manifestGenerator(JobManifestGenerator)}.
         * 
         * @param manifestGenerator
         *        a consumer that will call methods on {@link JobManifestGenerator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #manifestGenerator(JobManifestGenerator)
         */
        default Builder manifestGenerator(Consumer<JobManifestGenerator.Builder> manifestGenerator) {
            return manifestGenerator(JobManifestGenerator.builder().applyMutation(manifestGenerator).build());
        }

        /**
         * <p>
         * The attribute of the JobDescriptor containing details about the job's generated manifest.
         * </p>
         * 
         * @param generatedManifestDescriptor
         *        The attribute of the JobDescriptor containing details about the job's generated manifest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder generatedManifestDescriptor(S3GeneratedManifestDescriptor generatedManifestDescriptor);

        /**
         * <p>
         * The attribute of the JobDescriptor containing details about the job's generated manifest.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3GeneratedManifestDescriptor.Builder}
         * avoiding the need to create one manually via {@link S3GeneratedManifestDescriptor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3GeneratedManifestDescriptor.Builder#build()} is called
         * immediately and its result is passed to {@link #generatedManifestDescriptor(S3GeneratedManifestDescriptor)}.
         * 
         * @param generatedManifestDescriptor
         *        a consumer that will call methods on {@link S3GeneratedManifestDescriptor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #generatedManifestDescriptor(S3GeneratedManifestDescriptor)
         */
        default Builder generatedManifestDescriptor(Consumer<S3GeneratedManifestDescriptor.Builder> generatedManifestDescriptor) {
            return generatedManifestDescriptor(S3GeneratedManifestDescriptor.builder().applyMutation(generatedManifestDescriptor)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String jobId;

        private Boolean confirmationRequired;

        private String description;

        private String jobArn;

        private String status;

        private JobManifest manifest;

        private JobOperation operation;

        private Integer priority;

        private JobProgressSummary progressSummary;

        private String statusUpdateReason;

        private List<JobFailure> failureReasons = DefaultSdkAutoConstructList.getInstance();

        private JobReport report;

        private Instant creationTime;

        private Instant terminationDate;

        private String roleArn;

        private Instant suspendedDate;

        private String suspendedCause;

        private JobManifestGenerator manifestGenerator;

        private S3GeneratedManifestDescriptor generatedManifestDescriptor;

        private BuilderImpl() {
        }

        private BuilderImpl(JobDescriptor model) {
            jobId(model.jobId);
            confirmationRequired(model.confirmationRequired);
            description(model.description);
            jobArn(model.jobArn);
            status(model.status);
            manifest(model.manifest);
            operation(model.operation);
            priority(model.priority);
            progressSummary(model.progressSummary);
            statusUpdateReason(model.statusUpdateReason);
            failureReasons(model.failureReasons);
            report(model.report);
            creationTime(model.creationTime);
            terminationDate(model.terminationDate);
            roleArn(model.roleArn);
            suspendedDate(model.suspendedDate);
            suspendedCause(model.suspendedCause);
            manifestGenerator(model.manifestGenerator);
            generatedManifestDescriptor(model.generatedManifestDescriptor);
        }

        public final String getJobId() {
            return jobId;
        }

        public final void setJobId(String jobId) {
            this.jobId = jobId;
        }

        @Override
        public final Builder jobId(String jobId) {
            this.jobId = jobId;
            return this;
        }

        public final Boolean getConfirmationRequired() {
            return confirmationRequired;
        }

        public final void setConfirmationRequired(Boolean confirmationRequired) {
            this.confirmationRequired = confirmationRequired;
        }

        @Override
        public final Builder confirmationRequired(Boolean confirmationRequired) {
            this.confirmationRequired = confirmationRequired;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getJobArn() {
            return jobArn;
        }

        public final void setJobArn(String jobArn) {
            this.jobArn = jobArn;
        }

        @Override
        public final Builder jobArn(String jobArn) {
            this.jobArn = jobArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(JobStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final JobManifest.Builder getManifest() {
            return manifest != null ? manifest.toBuilder() : null;
        }

        public final void setManifest(JobManifest.BuilderImpl manifest) {
            this.manifest = manifest != null ? manifest.build() : null;
        }

        @Override
        public final Builder manifest(JobManifest manifest) {
            this.manifest = manifest;
            return this;
        }

        public final JobOperation.Builder getOperation() {
            return operation != null ? operation.toBuilder() : null;
        }

        public final void setOperation(JobOperation.BuilderImpl operation) {
            this.operation = operation != null ? operation.build() : null;
        }

        @Override
        public final Builder operation(JobOperation operation) {
            this.operation = operation;
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final JobProgressSummary.Builder getProgressSummary() {
            return progressSummary != null ? progressSummary.toBuilder() : null;
        }

        public final void setProgressSummary(JobProgressSummary.BuilderImpl progressSummary) {
            this.progressSummary = progressSummary != null ? progressSummary.build() : null;
        }

        @Override
        public final Builder progressSummary(JobProgressSummary progressSummary) {
            this.progressSummary = progressSummary;
            return this;
        }

        public final String getStatusUpdateReason() {
            return statusUpdateReason;
        }

        public final void setStatusUpdateReason(String statusUpdateReason) {
            this.statusUpdateReason = statusUpdateReason;
        }

        @Override
        public final Builder statusUpdateReason(String statusUpdateReason) {
            this.statusUpdateReason = statusUpdateReason;
            return this;
        }

        public final List<JobFailure.Builder> getFailureReasons() {
            List<JobFailure.Builder> result = JobFailureListCopier.copyToBuilder(this.failureReasons);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFailureReasons(Collection<JobFailure.BuilderImpl> failureReasons) {
            this.failureReasons = JobFailureListCopier.copyFromBuilder(failureReasons);
        }

        @Override
        public final Builder failureReasons(Collection<JobFailure> failureReasons) {
            this.failureReasons = JobFailureListCopier.copy(failureReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureReasons(JobFailure... failureReasons) {
            failureReasons(Arrays.asList(failureReasons));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureReasons(Consumer<JobFailure.Builder>... failureReasons) {
            failureReasons(Stream.of(failureReasons).map(c -> JobFailure.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final JobReport.Builder getReport() {
            return report != null ? report.toBuilder() : null;
        }

        public final void setReport(JobReport.BuilderImpl report) {
            this.report = report != null ? report.build() : null;
        }

        @Override
        public final Builder report(JobReport report) {
            this.report = report;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getTerminationDate() {
            return terminationDate;
        }

        public final void setTerminationDate(Instant terminationDate) {
            this.terminationDate = terminationDate;
        }

        @Override
        public final Builder terminationDate(Instant terminationDate) {
            this.terminationDate = terminationDate;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final Instant getSuspendedDate() {
            return suspendedDate;
        }

        public final void setSuspendedDate(Instant suspendedDate) {
            this.suspendedDate = suspendedDate;
        }

        @Override
        public final Builder suspendedDate(Instant suspendedDate) {
            this.suspendedDate = suspendedDate;
            return this;
        }

        public final String getSuspendedCause() {
            return suspendedCause;
        }

        public final void setSuspendedCause(String suspendedCause) {
            this.suspendedCause = suspendedCause;
        }

        @Override
        public final Builder suspendedCause(String suspendedCause) {
            this.suspendedCause = suspendedCause;
            return this;
        }

        public final JobManifestGenerator.Builder getManifestGenerator() {
            return manifestGenerator != null ? manifestGenerator.toBuilder() : null;
        }

        public final void setManifestGenerator(JobManifestGenerator.BuilderImpl manifestGenerator) {
            this.manifestGenerator = manifestGenerator != null ? manifestGenerator.build() : null;
        }

        @Override
        public final Builder manifestGenerator(JobManifestGenerator manifestGenerator) {
            this.manifestGenerator = manifestGenerator;
            return this;
        }

        public final S3GeneratedManifestDescriptor.Builder getGeneratedManifestDescriptor() {
            return generatedManifestDescriptor != null ? generatedManifestDescriptor.toBuilder() : null;
        }

        public final void setGeneratedManifestDescriptor(S3GeneratedManifestDescriptor.BuilderImpl generatedManifestDescriptor) {
            this.generatedManifestDescriptor = generatedManifestDescriptor != null ? generatedManifestDescriptor.build() : null;
        }

        @Override
        public final Builder generatedManifestDescriptor(S3GeneratedManifestDescriptor generatedManifestDescriptor) {
            this.generatedManifestDescriptor = generatedManifestDescriptor;
            return this;
        }

        @Override
        public JobDescriptor build() {
            return new JobDescriptor(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
