/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListJobsRequest extends S3ControlRequest implements
        ToCopyableBuilder<ListJobsRequest.Builder, ListJobsRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccountId")
            .getter(getter(ListJobsRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-account-id")
                    .unmarshallLocationName("x-amz-account-id").build()).build();

    private static final SdkField<List<String>> JOB_STATUSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("JobStatuses")
            .getter(getter(ListJobsRequest::jobStatusesAsStrings))
            .setter(setter(Builder::jobStatusesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("jobStatuses")
                    .unmarshallLocationName("jobStatuses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NextToken")
            .getter(getter(ListJobsRequest::nextToken))
            .setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("nextToken")
                    .unmarshallLocationName("nextToken").build()).build();

    private static final SdkField<Integer> MAX_RESULTS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxResults")
            .getter(getter(ListJobsRequest::maxResults))
            .setter(setter(Builder::maxResults))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("maxResults")
                    .unmarshallLocationName("maxResults").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            JOB_STATUSES_FIELD, NEXT_TOKEN_FIELD, MAX_RESULTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String accountId;

    private final List<String> jobStatuses;

    private final String nextToken;

    private final Integer maxResults;

    private ListJobsRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.jobStatuses = builder.jobStatuses;
        this.nextToken = builder.nextToken;
        this.maxResults = builder.maxResults;
    }

    /**
     * <p>
     * The Amazon Web Services account ID associated with the S3 Batch Operations job.
     * </p>
     * 
     * @return The Amazon Web Services account ID associated with the S3 Batch Operations job.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasJobStatuses} method.
     * </p>
     * 
     * @return The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
     */
    public final List<JobStatus> jobStatuses() {
        return JobStatusListCopier.copyStringToEnum(jobStatuses);
    }

    /**
     * For responses, this returns true if the service returned a value for the JobStatuses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasJobStatuses() {
        return jobStatuses != null && !(jobStatuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasJobStatuses} method.
     * </p>
     * 
     * @return The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
     */
    public final List<String> jobStatusesAsStrings() {
        return jobStatuses;
    }

    /**
     * <p>
     * A pagination token to request the next page of results. Use the token that Amazon S3 returned in the
     * <code>NextToken</code> element of the <code>ListJobsResult</code> from the previous <code>List Jobs</code>
     * request.
     * </p>
     * 
     * @return A pagination token to request the next page of results. Use the token that Amazon S3 returned in the
     *         <code>NextToken</code> element of the <code>ListJobsResult</code> from the previous
     *         <code>List Jobs</code> request.
     */
    public final String nextToken() {
        return nextToken;
    }

    /**
     * <p>
     * The maximum number of jobs that Amazon S3 will include in the <code>List Jobs</code> response. If there are more
     * jobs than this number, the response will include a pagination token in the <code>NextToken</code> field to enable
     * you to retrieve the next page of results.
     * </p>
     * 
     * @return The maximum number of jobs that Amazon S3 will include in the <code>List Jobs</code> response. If there
     *         are more jobs than this number, the response will include a pagination token in the
     *         <code>NextToken</code> field to enable you to retrieve the next page of results.
     */
    public final Integer maxResults() {
        return maxResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasJobStatuses() ? jobStatusesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ListJobsRequest)) {
            return false;
        }
        ListJobsRequest other = (ListJobsRequest) obj;
        return Objects.equals(accountId(), other.accountId()) && hasJobStatuses() == other.hasJobStatuses()
                && Objects.equals(jobStatusesAsStrings(), other.jobStatusesAsStrings())
                && Objects.equals(nextToken(), other.nextToken()) && Objects.equals(maxResults(), other.maxResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ListJobsRequest").add("AccountId", accountId())
                .add("JobStatuses", hasJobStatuses() ? jobStatusesAsStrings() : null).add("NextToken", nextToken())
                .add("MaxResults", maxResults()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "JobStatuses":
            return Optional.ofNullable(clazz.cast(jobStatusesAsStrings()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        case "MaxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("x-amz-account-id", ACCOUNT_ID_FIELD);
        map.put("jobStatuses", JOB_STATUSES_FIELD);
        map.put("nextToken", NEXT_TOKEN_FIELD);
        map.put("maxResults", MAX_RESULTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ListJobsRequest, T> g) {
        return obj -> g.apply((ListJobsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends S3ControlRequest.Builder, SdkPojo, CopyableBuilder<Builder, ListJobsRequest> {
        /**
         * <p>
         * The Amazon Web Services account ID associated with the S3 Batch Operations job.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID associated with the S3 Batch Operations job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * </p>
         * 
         * @param jobStatuses
         *        The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatusesWithStrings(Collection<String> jobStatuses);

        /**
         * <p>
         * The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * </p>
         * 
         * @param jobStatuses
         *        The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatusesWithStrings(String... jobStatuses);

        /**
         * <p>
         * The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * </p>
         * 
         * @param jobStatuses
         *        The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatuses(Collection<JobStatus> jobStatuses);

        /**
         * <p>
         * The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * </p>
         * 
         * @param jobStatuses
         *        The <code>List Jobs</code> request returns jobs that match the statuses listed in this element.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobStatuses(JobStatus... jobStatuses);

        /**
         * <p>
         * A pagination token to request the next page of results. Use the token that Amazon S3 returned in the
         * <code>NextToken</code> element of the <code>ListJobsResult</code> from the previous <code>List Jobs</code>
         * request.
         * </p>
         * 
         * @param nextToken
         *        A pagination token to request the next page of results. Use the token that Amazon S3 returned in the
         *        <code>NextToken</code> element of the <code>ListJobsResult</code> from the previous
         *        <code>List Jobs</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        /**
         * <p>
         * The maximum number of jobs that Amazon S3 will include in the <code>List Jobs</code> response. If there are
         * more jobs than this number, the response will include a pagination token in the <code>NextToken</code> field
         * to enable you to retrieve the next page of results.
         * </p>
         * 
         * @param maxResults
         *        The maximum number of jobs that Amazon S3 will include in the <code>List Jobs</code> response. If
         *        there are more jobs than this number, the response will include a pagination token in the
         *        <code>NextToken</code> field to enable you to retrieve the next page of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3ControlRequest.BuilderImpl implements Builder {
        private String accountId;

        private List<String> jobStatuses = DefaultSdkAutoConstructList.getInstance();

        private String nextToken;

        private Integer maxResults;

        private BuilderImpl() {
        }

        private BuilderImpl(ListJobsRequest model) {
            super(model);
            accountId(model.accountId);
            jobStatusesWithStrings(model.jobStatuses);
            nextToken(model.nextToken);
            maxResults(model.maxResults);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final Collection<String> getJobStatuses() {
            if (jobStatuses instanceof SdkAutoConstructList) {
                return null;
            }
            return jobStatuses;
        }

        public final void setJobStatuses(Collection<String> jobStatuses) {
            this.jobStatuses = JobStatusListCopier.copy(jobStatuses);
        }

        @Override
        public final Builder jobStatusesWithStrings(Collection<String> jobStatuses) {
            this.jobStatuses = JobStatusListCopier.copy(jobStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatusesWithStrings(String... jobStatuses) {
            jobStatusesWithStrings(Arrays.asList(jobStatuses));
            return this;
        }

        @Override
        public final Builder jobStatuses(Collection<JobStatus> jobStatuses) {
            this.jobStatuses = JobStatusListCopier.copyEnumToString(jobStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder jobStatuses(JobStatus... jobStatuses) {
            jobStatuses(Arrays.asList(jobStatuses));
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ListJobsRequest build() {
            return new ListJobsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
