/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.endpoints.internal;

import java.net.URI;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlEndpointParams;
import software.amazon.awssdk.services.s3control.endpoints.S3ControlEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultS3ControlEndpointProvider implements S3ControlEndpointProvider {
    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(S3ControlEndpointParams params) {
        Validate.notNull(params.useFips(), "Parameter 'UseFIPS' must not be null");
        Validate.notNull(params.useDualStack(), "Parameter 'UseDualStack' must not be null");
        try {
            Region region = params.region();
            String regionId = region == null ? null : region.id();
            RuleResult result = endpointRule0(params, regionId);
            if (result.canContinue()) {
                throw SdkClientException.create("Rule engine did not reach an error or endpoint result");
            }
            if (result.isError()) {
                String errorMsg = result.error();
                if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                    errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
                }
                throw SdkClientException.create(errorMsg);
            }
            return CompletableFuture.completedFuture(result.endpoint());
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static RuleResult endpointRule0(S3ControlEndpointParams params, String region) {
        RuleResult result = endpointRule1(params, region);
        if (result.isResolved()) {
            return result;
        }
        return RuleResult.error("Region must be set");
    }

    private static RuleResult endpointRule1(S3ControlEndpointParams params, String region) {
        if (region != null) {
            if (params.useFips()) {
                RulePartition partitionResult = RulesFunctions.awsPartition(region);
                if (partitionResult != null) {
                    if ("aws-cn".equals(partitionResult.name())) {
                        return RuleResult.error("Partition does not support FIPS");
                    }
                }
            }
            RuleResult result = endpointRule3(params, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule16(params, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule27(params, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule48(params, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule55(params, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule60(params, region);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule92(params, region);
            if (result.isResolved()) {
                return result;
            }
            return endpointRule124(params, region);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule3(S3ControlEndpointParams params, String region) {
        if (params.outpostId() != null) {
            return endpointRule4(params, region);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule4(S3ControlEndpointParams params, String region) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            if (params.requiresAccountId() != null && params.requiresAccountId() && params.accountId() == null) {
                return RuleResult.error("AccountId is required but not set");
            }
            if (params.accountId() != null && !(RulesFunctions.isValidHostLabel(params.accountId(), false))) {
                return RuleResult.error("AccountId must only contain a-z, A-Z, 0-9 and `-`.");
            }
            if (!(RulesFunctions.isValidHostLabel(params.outpostId(), false))) {
                return RuleResult.error("OutpostId must only contain a-z, A-Z, 0-9 and `-`.");
            }
            if (params.endpoint() != null && params.useDualStack()) {
                return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
            }
            RuleResult result = endpointRule9(params, region, partitionResult);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid region: region was not a valid DNS name.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule9(S3ControlEndpointParams params, String region, RulePartition partitionResult) {
        if (RulesFunctions.isValidHostLabel(region, true)) {
            if (params.endpoint() != null) {
                RuleUrl url = RulesFunctions.parseURL(params.endpoint());
                if (url != null) {
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create(url.scheme() + "://" + url.authority() + url.path()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                            .signingName("s3-outposts").signingRegion(region).build())).build());
                }
            }
            if (params.useFips() && params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts-fips." + region + "." + partitionResult.dualStackDnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(region).build())).build());
            }
            if (params.useFips()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts-fips." + region + "." + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(region).build())).build());
            }
            if (params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts." + region + "." + partitionResult.dualStackDnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(region).build())).build());
            }
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts." + region + "." + partitionResult.dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(region).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule16(S3ControlEndpointParams params, String region) {
        if (params.resourceArn() != null) {
            RuleArn resourceArn = RulesFunctions.awsParseArn(params.resourceArn());
            if (resourceArn != null) {
                if ("s3express".equals(resourceArn.service())) {
                    return endpointRule17(params, region, resourceArn);
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule17(S3ControlEndpointParams params, String region, RuleArn resourceArn) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            return endpointRule18(params, resourceArn, partitionResult, region);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule18(S3ControlEndpointParams params, RuleArn resourceArn, RulePartition partitionResult,
            String region) {
        RulePartition arnPartition = RulesFunctions.awsPartition(resourceArn.region());
        if (arnPartition != null) {
            RuleResult result = endpointRule19(params, arnPartition, partitionResult, resourceArn, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Client was configured for partition `" + partitionResult.name() + "` but ARN has `"
                    + arnPartition.name() + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule19(S3ControlEndpointParams params, RulePartition arnPartition,
            RulePartition partitionResult, RuleArn resourceArn, String region) {
        if (RulesFunctions.stringEquals(arnPartition.name(), partitionResult.name())) {
            if (params.useArnRegion() != null && !params.useArnRegion()
                    && !(RulesFunctions.stringEquals(resourceArn.region(), region))) {
                return RuleResult.error("Invalid configuration: region from ARN `" + resourceArn.region()
                        + "` does not match client region `" + region + "` and UseArnRegion is `false`");
            }
            if (params.endpoint() != null && params.useDualStack()) {
                return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
            }
            if (params.useDualStack()) {
                return RuleResult.error("S3Express does not support Dual-stack.");
            }
            if (params.endpoint() != null) {
                RuleUrl url = RulesFunctions.parseURL(params.endpoint());
                if (url != null) {
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create(url.scheme() + "://" + url.authority()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
            if (params.useFips()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3express-control-fips." + region + "." + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                        .signingRegion(region).build())).build());
            }
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control." + region + "." + partitionResult.dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(region).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule27(S3ControlEndpointParams params, String region) {
        if (params.accessPointName() != null) {
            String accessPointSuffix = RulesFunctions.substring(params.accessPointName(), 0, 7, true);
            if (accessPointSuffix != null) {
                if ("--xa-s3".equals(accessPointSuffix)) {
                    return endpointRule28(params, region);
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule28(S3ControlEndpointParams params, String region) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            if (params.endpoint() != null && params.useDualStack()) {
                return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
            }
            if (params.useDualStack()) {
                return RuleResult.error("S3Express does not support Dual-stack.");
            }
            if (params.endpoint() != null) {
                RuleUrl url = RulesFunctions.parseURL(params.endpoint());
                if (url != null) {
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create(url.scheme() + "://" + url.authority()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
            RuleResult result = endpointRule32(params, region, partitionResult);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule35(params, region, partitionResult);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule38(params, region, partitionResult);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule41(params, region, partitionResult);
            if (result.isResolved()) {
                return result;
            }
            result = endpointRule44(params, region, partitionResult);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Unrecognized S3Express Access Point name format.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule32(S3ControlEndpointParams params, String region, RulePartition partitionResult) {
        String s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 15, true);
        if (s3ExpressAvailabilityZoneId != null) {
            String s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 15, 17, true);
            if (s3ExpressAvailabilityZoneDelim != null) {
                if ("--".equals(s3ExpressAvailabilityZoneDelim)) {
                    if (params.useFips()) {
                        return RuleResult.endpoint(Endpoint
                                .builder()
                                .url(URI.create("https://s3express-control-fips." + region + "." + partitionResult.dnsSuffix()))
                                .putAttribute(
                                        AwsEndpointAttribute.AUTH_SCHEMES,
                                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                                .signingName("s3express").signingRegion(region).build())).build());
                    }
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create("https://s3express-control." + region + "." + partitionResult.dnsSuffix()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule35(S3ControlEndpointParams params, String region, RulePartition partitionResult) {
        String s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 16, true);
        if (s3ExpressAvailabilityZoneId != null) {
            String s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 16, 18, true);
            if (s3ExpressAvailabilityZoneDelim != null) {
                if ("--".equals(s3ExpressAvailabilityZoneDelim)) {
                    if (params.useFips()) {
                        return RuleResult.endpoint(Endpoint
                                .builder()
                                .url(URI.create("https://s3express-control-fips." + region + "." + partitionResult.dnsSuffix()))
                                .putAttribute(
                                        AwsEndpointAttribute.AUTH_SCHEMES,
                                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                                .signingName("s3express").signingRegion(region).build())).build());
                    }
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create("https://s3express-control." + region + "." + partitionResult.dnsSuffix()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule38(S3ControlEndpointParams params, String region, RulePartition partitionResult) {
        String s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 20, true);
        if (s3ExpressAvailabilityZoneId != null) {
            String s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 20, 22, true);
            if (s3ExpressAvailabilityZoneDelim != null) {
                if ("--".equals(s3ExpressAvailabilityZoneDelim)) {
                    if (params.useFips()) {
                        return RuleResult.endpoint(Endpoint
                                .builder()
                                .url(URI.create("https://s3express-control-fips." + region + "." + partitionResult.dnsSuffix()))
                                .putAttribute(
                                        AwsEndpointAttribute.AUTH_SCHEMES,
                                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                                .signingName("s3express").signingRegion(region).build())).build());
                    }
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create("https://s3express-control." + region + "." + partitionResult.dnsSuffix()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule41(S3ControlEndpointParams params, String region, RulePartition partitionResult) {
        String s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 21, true);
        if (s3ExpressAvailabilityZoneId != null) {
            String s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 21, 23, true);
            if (s3ExpressAvailabilityZoneDelim != null) {
                if ("--".equals(s3ExpressAvailabilityZoneDelim)) {
                    if (params.useFips()) {
                        return RuleResult.endpoint(Endpoint
                                .builder()
                                .url(URI.create("https://s3express-control-fips." + region + "." + partitionResult.dnsSuffix()))
                                .putAttribute(
                                        AwsEndpointAttribute.AUTH_SCHEMES,
                                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                                .signingName("s3express").signingRegion(region).build())).build());
                    }
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create("https://s3express-control." + region + "." + partitionResult.dnsSuffix()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule44(S3ControlEndpointParams params, String region, RulePartition partitionResult) {
        String s3ExpressAvailabilityZoneId = RulesFunctions.substring(params.accessPointName(), 7, 27, true);
        if (s3ExpressAvailabilityZoneId != null) {
            String s3ExpressAvailabilityZoneDelim = RulesFunctions.substring(params.accessPointName(), 27, 29, true);
            if (s3ExpressAvailabilityZoneDelim != null) {
                if ("--".equals(s3ExpressAvailabilityZoneDelim)) {
                    if (params.useFips()) {
                        return RuleResult.endpoint(Endpoint
                                .builder()
                                .url(URI.create("https://s3express-control-fips." + region + "." + partitionResult.dnsSuffix()))
                                .putAttribute(
                                        AwsEndpointAttribute.AUTH_SCHEMES,
                                        Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                                .signingName("s3express").signingRegion(region).build())).build());
                    }
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create("https://s3express-control." + region + "." + partitionResult.dnsSuffix()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule48(S3ControlEndpointParams params, String region) {
        if (params.useS3ExpressControlEndpoint() != null && params.useS3ExpressControlEndpoint()) {
            return endpointRule49(params, region);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule49(S3ControlEndpointParams params, String region) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            if (params.endpoint() != null && params.useDualStack()) {
                return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
            }
            if (params.useDualStack()) {
                return RuleResult.error("S3Express does not support Dual-stack.");
            }
            if (params.endpoint() != null) {
                RuleUrl url = RulesFunctions.parseURL(params.endpoint());
                if (url != null) {
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create(url.scheme() + "://" + url.authority()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                            .signingRegion(region).build())).build());
                }
            }
            if (params.useFips()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3express-control-fips." + region + "." + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                        .signingRegion(region).build())).build());
            }
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3express-control." + region + "." + partitionResult.dnsSuffix()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3express")
                                    .signingRegion(region).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule55(S3ControlEndpointParams params, String region) {
        if ("snow".equals(region) && params.endpoint() != null) {
            RuleUrl url = RulesFunctions.parseURL(params.endpoint());
            if (url != null) {
                return endpointRule56(params, region, url);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule56(S3ControlEndpointParams params, String region, RuleUrl url) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            if (params.useDualStack()) {
                return RuleResult.error("S3 Snow does not support DualStack");
            }
            if (params.useFips()) {
                return RuleResult.error("S3 Snow does not support FIPS");
            }
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create(url.scheme() + "://" + url.authority()))
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                    .signingRegion(region).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule60(S3ControlEndpointParams params, String region) {
        if (params.accessPointName() != null) {
            RuleArn accessPointArn = RulesFunctions.awsParseArn(params.accessPointName());
            if (accessPointArn != null) {
                RuleResult result = endpointRule61(params, accessPointArn, region);
                if (result.isResolved()) {
                    return result;
                }
                return RuleResult.error("Invalid ARN: No ARN type specified");
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule61(S3ControlEndpointParams params, RuleArn accessPointArn, String region) {
        String arnType = RulesFunctions.listAccess(accessPointArn.resourceId(), 0);
        if (arnType != null) {
            if (!("".equals(arnType))) {
                return endpointRule62(params, accessPointArn, region);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule62(S3ControlEndpointParams params, RuleArn accessPointArn, String region) {
        if ("s3-outposts".equals(accessPointArn.service())) {
            RuleResult result = endpointRule63(params, accessPointArn, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: The Outpost Id was not set");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule63(S3ControlEndpointParams params, RuleArn accessPointArn, String region) {
        String outpostId = RulesFunctions.listAccess(accessPointArn.resourceId(), 1);
        if (outpostId != null) {
            RuleResult result = endpointRule64(params, outpostId, accessPointArn, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: The outpost Id must only contain a-z, A-Z, 0-9 and `-`., found: `" + outpostId
                    + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule64(S3ControlEndpointParams params, String outpostId, RuleArn accessPointArn,
            String region) {
        if (RulesFunctions.isValidHostLabel(outpostId, false)) {
            if (params.endpoint() != null && params.useDualStack()) {
                return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
            }
            if (params.useArnRegion() != null && !params.useArnRegion()
                    && !(RulesFunctions.stringEquals(accessPointArn.region(), region))) {
                return RuleResult.error("Invalid configuration: region from ARN `" + accessPointArn.region()
                        + "` does not match client region `" + region + "` and UseArnRegion is `false`");
            }
            return endpointRule67(params, region, accessPointArn, outpostId);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule67(S3ControlEndpointParams params, String region, RuleArn accessPointArn,
            String outpostId) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            return endpointRule68(params, accessPointArn, partitionResult, outpostId);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule68(S3ControlEndpointParams params, RuleArn accessPointArn,
            RulePartition partitionResult, String outpostId) {
        RulePartition arnPartition = RulesFunctions.awsPartition(accessPointArn.region());
        if (arnPartition != null) {
            RuleResult result = endpointRule69(params, arnPartition, partitionResult, accessPointArn, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Client was configured for partition `" + partitionResult.name() + "` but ARN has `"
                    + arnPartition.name() + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule69(S3ControlEndpointParams params, RulePartition arnPartition,
            RulePartition partitionResult, RuleArn accessPointArn, String outpostId) {
        if (RulesFunctions.stringEquals(arnPartition.name(), partitionResult.name())) {
            RuleResult result = endpointRule70(params, accessPointArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid region in ARN: `" + accessPointArn.region() + "` (invalid DNS name)");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule70(S3ControlEndpointParams params, RuleArn accessPointArn, RulePartition arnPartition,
            String outpostId) {
        if (RulesFunctions.isValidHostLabel(accessPointArn.region(), true)) {
            RuleResult result = endpointRule71(params, accessPointArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: missing account ID");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule71(S3ControlEndpointParams params, RuleArn accessPointArn, RulePartition arnPartition,
            String outpostId) {
        if (!("".equals(accessPointArn.accountId()))) {
            RuleResult result = endpointRule72(params, accessPointArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: The account id may only contain a-z, A-Z, 0-9 and `-`. Found: `"
                    + accessPointArn.accountId() + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule72(S3ControlEndpointParams params, RuleArn accessPointArn, RulePartition arnPartition,
            String outpostId) {
        if (RulesFunctions.isValidHostLabel(accessPointArn.accountId(), false)) {
            if (params.accountId() != null && !(RulesFunctions.stringEquals(params.accountId(), accessPointArn.accountId()))) {
                return RuleResult.error("Invalid ARN: the accountId specified in the ARN (`" + accessPointArn.accountId()
                        + "`) does not match the parameter (`" + params.accountId() + "`)");
            }
            RuleResult result = endpointRule74(params, accessPointArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: Expected a 4-component resource");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule74(S3ControlEndpointParams params, RuleArn accessPointArn, RulePartition arnPartition,
            String outpostId) {
        String outpostType = RulesFunctions.listAccess(accessPointArn.resourceId(), 2);
        if (outpostType != null) {
            RuleResult result = endpointRule75(params, accessPointArn, outpostType, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: expected an access point name");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule75(S3ControlEndpointParams params, RuleArn accessPointArn, String outpostType,
            RulePartition arnPartition, String outpostId) {
        String accessPointName = RulesFunctions.listAccess(accessPointArn.resourceId(), 3);
        if (accessPointName != null) {
            RuleResult result = endpointRule76(params, outpostType, accessPointArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Expected an outpost type `accesspoint`, found `" + outpostType + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule76(S3ControlEndpointParams params, String outpostType, RuleArn accessPointArn,
            RulePartition arnPartition, String outpostId) {
        if ("accesspoint".equals(outpostType)) {
            if (params.useFips() && params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts-fips." + accessPointArn.region() + "."
                                + arnPartition.dualStackDnsSuffix()))
                        .putHeader("x-amz-account-id", accessPointArn.accountId())
                        .putHeader("x-amz-outpost-id", outpostId)
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(accessPointArn.region()).build())).build());
            }
            if (params.useFips()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts-fips." + accessPointArn.region() + "." + arnPartition.dnsSuffix()))
                        .putHeader("x-amz-account-id", accessPointArn.accountId())
                        .putHeader("x-amz-outpost-id", outpostId)
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(accessPointArn.region()).build())).build());
            }
            if (params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts." + accessPointArn.region() + "."
                                + arnPartition.dualStackDnsSuffix()))
                        .putHeader("x-amz-account-id", accessPointArn.accountId())
                        .putHeader("x-amz-outpost-id", outpostId)
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(accessPointArn.region()).build())).build());
            }
            if (params.endpoint() != null) {
                RuleUrl url = RulesFunctions.parseURL(params.endpoint());
                if (url != null) {
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create(url.scheme() + "://" + url.authority() + url.path()))
                            .putHeader("x-amz-account-id", accessPointArn.accountId())
                            .putHeader("x-amz-outpost-id", outpostId)
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                            .signingName("s3-outposts").signingRegion(accessPointArn.region()).build())).build());
                }
            }
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts." + accessPointArn.region() + "." + arnPartition.dnsSuffix()))
                    .putHeader("x-amz-account-id", accessPointArn.accountId())
                    .putHeader("x-amz-outpost-id", outpostId)
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(accessPointArn.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule92(S3ControlEndpointParams params, String region) {
        if (params.bucket() != null) {
            RuleArn bucketArn = RulesFunctions.awsParseArn(params.bucket());
            if (bucketArn != null) {
                RuleResult result = endpointRule93(params, bucketArn, region);
                if (result.isResolved()) {
                    return result;
                }
                return RuleResult.error("Invalid ARN: No ARN type specified");
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule93(S3ControlEndpointParams params, RuleArn bucketArn, String region) {
        String arnType = RulesFunctions.listAccess(bucketArn.resourceId(), 0);
        if (arnType != null) {
            if (!("".equals(arnType))) {
                return endpointRule94(params, bucketArn, region);
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule94(S3ControlEndpointParams params, RuleArn bucketArn, String region) {
        if ("s3-outposts".equals(bucketArn.service())) {
            RuleResult result = endpointRule95(params, bucketArn, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: The Outpost Id was not set");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule95(S3ControlEndpointParams params, RuleArn bucketArn, String region) {
        String outpostId = RulesFunctions.listAccess(bucketArn.resourceId(), 1);
        if (outpostId != null) {
            RuleResult result = endpointRule96(params, outpostId, bucketArn, region);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: The outpost Id must only contain a-z, A-Z, 0-9 and `-`., found: `" + outpostId
                    + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule96(S3ControlEndpointParams params, String outpostId, RuleArn bucketArn, String region) {
        if (RulesFunctions.isValidHostLabel(outpostId, false)) {
            if (params.endpoint() != null && params.useDualStack()) {
                return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
            }
            if (params.useArnRegion() != null && !params.useArnRegion()
                    && !(RulesFunctions.stringEquals(bucketArn.region(), region))) {
                return RuleResult.error("Invalid configuration: region from ARN `" + bucketArn.region()
                        + "` does not match client region `" + region + "` and UseArnRegion is `false`");
            }
            return endpointRule99(params, bucketArn, region, outpostId);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule99(S3ControlEndpointParams params, RuleArn bucketArn, String region, String outpostId) {
        RulePartition arnPartition = RulesFunctions.awsPartition(bucketArn.region());
        if (arnPartition != null) {
            return endpointRule100(params, region, arnPartition, bucketArn, outpostId);
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule100(S3ControlEndpointParams params, String region, RulePartition arnPartition,
            RuleArn bucketArn, String outpostId) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            RuleResult result = endpointRule101(params, arnPartition, partitionResult, bucketArn, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Client was configured for partition `" + partitionResult.name() + "` but ARN has `"
                    + arnPartition.name() + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule101(S3ControlEndpointParams params, RulePartition arnPartition,
            RulePartition partitionResult, RuleArn bucketArn, String outpostId) {
        if (RulesFunctions.stringEquals(arnPartition.name(), partitionResult.name())) {
            RuleResult result = endpointRule102(params, bucketArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid region in ARN: `" + bucketArn.region() + "` (invalid DNS name)");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule102(S3ControlEndpointParams params, RuleArn bucketArn, RulePartition arnPartition,
            String outpostId) {
        if (RulesFunctions.isValidHostLabel(bucketArn.region(), true)) {
            RuleResult result = endpointRule103(params, bucketArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: missing account ID");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule103(S3ControlEndpointParams params, RuleArn bucketArn, RulePartition arnPartition,
            String outpostId) {
        if (!("".equals(bucketArn.accountId()))) {
            RuleResult result = endpointRule104(params, bucketArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: The account id may only contain a-z, A-Z, 0-9 and `-`. Found: `"
                    + bucketArn.accountId() + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule104(S3ControlEndpointParams params, RuleArn bucketArn, RulePartition arnPartition,
            String outpostId) {
        if (RulesFunctions.isValidHostLabel(bucketArn.accountId(), false)) {
            if (params.accountId() != null && !(RulesFunctions.stringEquals(params.accountId(), bucketArn.accountId()))) {
                return RuleResult.error("Invalid ARN: the accountId specified in the ARN (`" + bucketArn.accountId()
                        + "`) does not match the parameter (`" + params.accountId() + "`)");
            }
            RuleResult result = endpointRule106(params, bucketArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: Expected a 4-component resource");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule106(S3ControlEndpointParams params, RuleArn bucketArn, RulePartition arnPartition,
            String outpostId) {
        String outpostType = RulesFunctions.listAccess(bucketArn.resourceId(), 2);
        if (outpostType != null) {
            RuleResult result = endpointRule107(params, bucketArn, outpostType, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: expected a bucket name");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule107(S3ControlEndpointParams params, RuleArn bucketArn, String outpostType,
            RulePartition arnPartition, String outpostId) {
        String bucketName = RulesFunctions.listAccess(bucketArn.resourceId(), 3);
        if (bucketName != null) {
            RuleResult result = endpointRule108(params, outpostType, bucketArn, arnPartition, outpostId);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid ARN: Expected an outpost type `bucket`, found `" + outpostType + "`");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule108(S3ControlEndpointParams params, String outpostType, RuleArn bucketArn,
            RulePartition arnPartition, String outpostId) {
        if ("bucket".equals(outpostType)) {
            if (params.useFips() && params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts-fips." + bucketArn.region() + "."
                                + arnPartition.dualStackDnsSuffix()))
                        .putHeader("x-amz-account-id", bucketArn.accountId())
                        .putHeader("x-amz-outpost-id", outpostId)
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(bucketArn.region()).build())).build());
            }
            if (params.useFips()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts-fips." + bucketArn.region() + "." + arnPartition.dnsSuffix()))
                        .putHeader("x-amz-account-id", bucketArn.accountId())
                        .putHeader("x-amz-outpost-id", outpostId)
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(bucketArn.region()).build())).build());
            }
            if (params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-outposts." + bucketArn.region() + "." + arnPartition.dualStackDnsSuffix()))
                        .putHeader("x-amz-account-id", bucketArn.accountId())
                        .putHeader("x-amz-outpost-id", outpostId)
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                        .signingRegion(bucketArn.region()).build())).build());
            }
            if (params.endpoint() != null) {
                RuleUrl url = RulesFunctions.parseURL(params.endpoint());
                if (url != null) {
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create(url.scheme() + "://" + url.authority() + url.path()))
                            .putHeader("x-amz-account-id", bucketArn.accountId())
                            .putHeader("x-amz-outpost-id", outpostId)
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true)
                                            .signingName("s3-outposts").signingRegion(bucketArn.region()).build())).build());
                }
            }
            return RuleResult.endpoint(Endpoint
                    .builder()
                    .url(URI.create("https://s3-outposts." + bucketArn.region() + "." + arnPartition.dnsSuffix()))
                    .putHeader("x-amz-account-id", bucketArn.accountId())
                    .putHeader("x-amz-outpost-id", outpostId)
                    .putAttribute(
                            AwsEndpointAttribute.AUTH_SCHEMES,
                            Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3-outposts")
                                    .signingRegion(bucketArn.region()).build())).build());
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule124(S3ControlEndpointParams params, String region) {
        RulePartition partitionResult = RulesFunctions.awsPartition(region);
        if (partitionResult != null) {
            RuleResult result = endpointRule125(params, region, partitionResult);
            if (result.isResolved()) {
                return result;
            }
            return RuleResult.error("Invalid region: region was not a valid DNS name.");
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule125(S3ControlEndpointParams params, String region, RulePartition partitionResult) {
        if (RulesFunctions.isValidHostLabel(region, true)) {
            if (params.requiresAccountId() != null && params.requiresAccountId() && params.accountId() == null) {
                return RuleResult.error("AccountId is required but not set");
            }
            if (params.accountId() != null && !(RulesFunctions.isValidHostLabel(params.accountId(), false))) {
                return RuleResult.error("AccountId must only contain a-z, A-Z, 0-9 and `-`.");
            }
            RuleResult result = endpointRule128(params, region);
            if (result.isResolved()) {
                return result;
            }
            if (params.useFips() && params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                    && params.accountId() != null) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://" + params.accountId() + ".s3-control-fips.dualstack." + region + "."
                                + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
            if (params.useFips() && params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-control-fips.dualstack." + region + "." + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
            if (params.useFips() && !params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                    && params.accountId() != null) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://" + params.accountId() + ".s3-control-fips." + region + "."
                                + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
            if (params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-control-fips." + region + "." + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
            if (!params.useFips() && params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                    && params.accountId() != null) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://" + params.accountId() + ".s3-control.dualstack." + region + "."
                                + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
            if (!params.useFips() && params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-control.dualstack." + region + "." + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
            if (!params.useFips() && !params.useDualStack() && params.requiresAccountId() != null && params.requiresAccountId()
                    && params.accountId() != null) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://" + params.accountId() + ".s3-control." + region + "."
                                + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
            if (!params.useFips() && !params.useDualStack()) {
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create("https://s3-control." + region + "." + partitionResult.dnsSuffix()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
        }
        return RuleResult.carryOn();
    }

    private static RuleResult endpointRule128(S3ControlEndpointParams params, String region) {
        if (params.endpoint() != null) {
            RuleUrl url = RulesFunctions.parseURL(params.endpoint());
            if (url != null) {
                if (params.useDualStack()) {
                    return RuleResult.error("Invalid Configuration: DualStack and custom endpoint are not supported");
                }
                if (params.requiresAccountId() != null && params.requiresAccountId() && params.accountId() != null) {
                    return RuleResult.endpoint(Endpoint
                            .builder()
                            .url(URI.create(url.scheme() + "://" + params.accountId() + "." + url.authority() + url.path()))
                            .putAttribute(
                                    AwsEndpointAttribute.AUTH_SCHEMES,
                                    Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                            .signingRegion(region).build())).build());
                }
                return RuleResult.endpoint(Endpoint
                        .builder()
                        .url(URI.create(url.scheme() + "://" + url.authority() + url.path()))
                        .putAttribute(
                                AwsEndpointAttribute.AUTH_SCHEMES,
                                Arrays.asList(SigV4AuthScheme.builder().disableDoubleEncoding(true).signingName("s3")
                                        .signingRegion(region).build())).build());
            }
        }
        return RuleResult.carryOn();
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }
}
