/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAccessPointRequest extends S3ControlRequest implements
        ToCopyableBuilder<CreateAccessPointRequest.Builder, CreateAccessPointRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccountId")
            .getter(getter(CreateAccessPointRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-account-id")
                    .unmarshallLocationName("x-amz-account-id").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(CreateAccessPointRequest::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("name").unmarshallLocationName("name")
                    .build()).build();

    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(CreateAccessPointRequest::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<VpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<VpcConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("VpcConfiguration")
            .getter(getter(CreateAccessPointRequest::vpcConfiguration))
            .setter(setter(Builder::vpcConfiguration))
            .constructor(VpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration")
                    .unmarshallLocationName("VpcConfiguration").build()).build();

    private static final SdkField<PublicAccessBlockConfiguration> PUBLIC_ACCESS_BLOCK_CONFIGURATION_FIELD = SdkField
            .<PublicAccessBlockConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("PublicAccessBlockConfiguration")
            .getter(getter(CreateAccessPointRequest::publicAccessBlockConfiguration))
            .setter(setter(Builder::publicAccessBlockConfiguration))
            .constructor(PublicAccessBlockConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicAccessBlockConfiguration")
                    .unmarshallLocationName("PublicAccessBlockConfiguration").build()).build();

    private static final SdkField<String> BUCKET_ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("BucketAccountId")
            .getter(getter(CreateAccessPointRequest::bucketAccountId))
            .setter(setter(Builder::bucketAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketAccountId")
                    .unmarshallLocationName("BucketAccountId").build()).build();

    private static final SdkField<Scope> SCOPE_FIELD = SdkField
            .<Scope> builder(MarshallingType.SDK_POJO)
            .memberName("Scope")
            .getter(getter(CreateAccessPointRequest::scope))
            .setter(setter(Builder::scope))
            .constructor(Scope::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scope")
                    .unmarshallLocationName("Scope").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateAccessPointRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags")
                    .unmarshallLocationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Tag")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Tag").unmarshallLocationName("Tag").build()).build()).build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD, NAME_FIELD,
            BUCKET_FIELD, VPC_CONFIGURATION_FIELD, PUBLIC_ACCESS_BLOCK_CONFIGURATION_FIELD, BUCKET_ACCOUNT_ID_FIELD, SCOPE_FIELD,
            TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String accountId;

    private final String name;

    private final String bucket;

    private final VpcConfiguration vpcConfiguration;

    private final PublicAccessBlockConfiguration publicAccessBlockConfiguration;

    private final String bucketAccountId;

    private final Scope scope;

    private final List<Tag> tags;

    private CreateAccessPointRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.name = builder.name;
        this.bucket = builder.bucket;
        this.vpcConfiguration = builder.vpcConfiguration;
        this.publicAccessBlockConfiguration = builder.publicAccessBlockConfiguration;
        this.bucketAccountId = builder.bucketAccountId;
        this.scope = builder.scope;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Web Services account ID for the account that owns the specified access point.
     * </p>
     * 
     * @return The Amazon Web Services account ID for the account that owns the specified access point.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The name you want to assign to this access point.
     * </p>
     * <p>
     * For directory buckets, the access point name must consist of a base name that you provide and suffix that
     * includes the <code>ZoneID</code> (Amazon Web Services Availability Zone or Local Zone) of your bucket location,
     * followed by <code>--xa-s3</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html">Managing access
     * to shared datasets in directory buckets with access points</a> in the <i>Amazon S3 User Guide</i>.
     * </p>
     * 
     * @return The name you want to assign to this access point.</p>
     *         <p>
     *         For directory buckets, the access point name must consist of a base name that you provide and suffix that
     *         includes the <code>ZoneID</code> (Amazon Web Services Availability Zone or Local Zone) of your bucket
     *         location, followed by <code>--xa-s3</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html"
     *         >Managing access to shared datasets in directory buckets with access points</a> in the <i>Amazon S3 User
     *         Guide</i>.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The name of the bucket that you want to associate this access point with.
     * </p>
     * <p>
     * For using this parameter with Amazon S3 on Outposts with the REST API, you must specify the name and the
     * x-amz-outpost-id as well.
     * </p>
     * <p>
     * For using this parameter with S3 on Outposts with the Amazon Web Services SDK and CLI, you must specify the ARN
     * of the bucket accessed in the format
     * <code>arn:aws:s3-outposts:&lt;Region&gt;:&lt;account-id&gt;:outpost/&lt;outpost-id&gt;/bucket/&lt;my-bucket-name&gt;</code>
     * . For example, to access the bucket <code>reports</code> through Outpost <code>my-outpost</code> owned by account
     * <code>123456789012</code> in Region <code>us-west-2</code>, use the URL encoding of
     * <code>arn:aws:s3-outposts:us-west-2:123456789012:outpost/my-outpost/bucket/reports</code>. The value must be URL
     * encoded.
     * </p>
     * 
     * @return The name of the bucket that you want to associate this access point with.</p>
     *         <p>
     *         For using this parameter with Amazon S3 on Outposts with the REST API, you must specify the name and the
     *         x-amz-outpost-id as well.
     *         </p>
     *         <p>
     *         For using this parameter with S3 on Outposts with the Amazon Web Services SDK and CLI, you must specify
     *         the ARN of the bucket accessed in the format
     *         <code>arn:aws:s3-outposts:&lt;Region&gt;:&lt;account-id&gt;:outpost/&lt;outpost-id&gt;/bucket/&lt;my-bucket-name&gt;</code>
     *         . For example, to access the bucket <code>reports</code> through Outpost <code>my-outpost</code> owned by
     *         account <code>123456789012</code> in Region <code>us-west-2</code>, use the URL encoding of
     *         <code>arn:aws:s3-outposts:us-west-2:123456789012:outpost/my-outpost/bucket/reports</code>. The value must
     *         be URL encoded.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * If you include this field, Amazon S3 restricts access to this access point to requests from the specified virtual
     * private cloud (VPC).
     * </p>
     * <note>
     * <p>
     * This is required for creating an access point for Amazon S3 on Outposts buckets.
     * </p>
     * </note>
     * 
     * @return If you include this field, Amazon S3 restricts access to this access point to requests from the specified
     *         virtual private cloud (VPC).</p> <note>
     *         <p>
     *         This is required for creating an access point for Amazon S3 on Outposts buckets.
     *         </p>
     */
    public final VpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    /**
     * <p>
     * The <code>PublicAccessBlock</code> configuration that you want to apply to the access point.
     * </p>
     * 
     * @return The <code>PublicAccessBlock</code> configuration that you want to apply to the access point.
     */
    public final PublicAccessBlockConfiguration publicAccessBlockConfiguration() {
        return publicAccessBlockConfiguration;
    }

    /**
     * <p>
     * The Amazon Web Services account ID associated with the S3 bucket associated with this access point.
     * </p>
     * <p>
     * For same account access point when your bucket and access point belong to the same account owner, the
     * <code>BucketAccountId</code> is not required. For cross-account access point when your bucket and access point
     * are not in the same account, the <code>BucketAccountId</code> is required.
     * </p>
     * 
     * @return The Amazon Web Services account ID associated with the S3 bucket associated with this access point.</p>
     *         <p>
     *         For same account access point when your bucket and access point belong to the same account owner, the
     *         <code>BucketAccountId</code> is not required. For cross-account access point when your bucket and access
     *         point are not in the same account, the <code>BucketAccountId</code> is required.
     */
    public final String bucketAccountId() {
        return bucketAccountId;
    }

    /**
     * <p>
     * For directory buckets, you can filter access control to specific prefixes, API operations, or a combination of
     * both. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html">Managing access
     * to shared datasets in directory buckets with access points</a> in the <i>Amazon S3 User Guide</i>.
     * </p>
     * <note>
     * <p>
     * Scope is only supported for access points attached to directory buckets.
     * </p>
     * </note>
     * 
     * @return For directory buckets, you can filter access control to specific prefixes, API operations, or a
     *         combination of both. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html"
     *         >Managing access to shared datasets in directory buckets with access points</a> in the <i>Amazon S3 User
     *         Guide</i>.</p> <note>
     *         <p>
     *         Scope is only supported for access points attached to directory buckets.
     *         </p>
     */
    public final Scope scope() {
        return scope;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of tags that you can apply to an access point. Tags are key-value pairs of metadata used to control
     * access to your access points. For more information about tags, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Using tags with Amazon S3</a>. For
     * information about tagging access points, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html#using-tags-for-abac">Using tags for
     * attribute-based access control (ABAC)</a>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * You must have the <code>s3:TagResource</code> permission to create an access point with tags for a general
     * purpose bucket.
     * </p>
     * </li>
     * <li>
     * <p>
     * You must have the <code>s3express:TagResource</code> permission to create an access point with tags for a
     * directory bucket.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return An array of tags that you can apply to an access point. Tags are key-value pairs of metadata used to
     *         control access to your access points. For more information about tags, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Using tags with Amazon S3</a>.
     *         For information about tagging access points, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html#using-tags-for-abac">Using tags
     *         for attribute-based access control (ABAC)</a>.</p> <note>
     *         <ul>
     *         <li>
     *         <p>
     *         You must have the <code>s3:TagResource</code> permission to create an access point with tags for a
     *         general purpose bucket.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You must have the <code>s3express:TagResource</code> permission to create an access point with tags for a
     *         directory bucket.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccessBlockConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(bucketAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(scope());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAccessPointRequest)) {
            return false;
        }
        CreateAccessPointRequest other = (CreateAccessPointRequest) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(name(), other.name())
                && Objects.equals(bucket(), other.bucket()) && Objects.equals(vpcConfiguration(), other.vpcConfiguration())
                && Objects.equals(publicAccessBlockConfiguration(), other.publicAccessBlockConfiguration())
                && Objects.equals(bucketAccountId(), other.bucketAccountId()) && Objects.equals(scope(), other.scope())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateAccessPointRequest").add("AccountId", accountId()).add("Name", name())
                .add("Bucket", bucket()).add("VpcConfiguration", vpcConfiguration())
                .add("PublicAccessBlockConfiguration", publicAccessBlockConfiguration())
                .add("BucketAccountId", bucketAccountId()).add("Scope", scope()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        case "PublicAccessBlockConfiguration":
            return Optional.ofNullable(clazz.cast(publicAccessBlockConfiguration()));
        case "BucketAccountId":
            return Optional.ofNullable(clazz.cast(bucketAccountId()));
        case "Scope":
            return Optional.ofNullable(clazz.cast(scope()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("x-amz-account-id", ACCOUNT_ID_FIELD);
        map.put("name", NAME_FIELD);
        map.put("Bucket", BUCKET_FIELD);
        map.put("VpcConfiguration", VPC_CONFIGURATION_FIELD);
        map.put("PublicAccessBlockConfiguration", PUBLIC_ACCESS_BLOCK_CONFIGURATION_FIELD);
        map.put("BucketAccountId", BUCKET_ACCOUNT_ID_FIELD);
        map.put("Scope", SCOPE_FIELD);
        map.put("Tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateAccessPointRequest, T> g) {
        return obj -> g.apply((CreateAccessPointRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends S3ControlRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateAccessPointRequest> {
        /**
         * <p>
         * The Amazon Web Services account ID for the account that owns the specified access point.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID for the account that owns the specified access point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The name you want to assign to this access point.
         * </p>
         * <p>
         * For directory buckets, the access point name must consist of a base name that you provide and suffix that
         * includes the <code>ZoneID</code> (Amazon Web Services Availability Zone or Local Zone) of your bucket
         * location, followed by <code>--xa-s3</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html">Managing
         * access to shared datasets in directory buckets with access points</a> in the <i>Amazon S3 User Guide</i>.
         * </p>
         * 
         * @param name
         *        The name you want to assign to this access point.</p>
         *        <p>
         *        For directory buckets, the access point name must consist of a base name that you provide and suffix
         *        that includes the <code>ZoneID</code> (Amazon Web Services Availability Zone or Local Zone) of your
         *        bucket location, followed by <code>--xa-s3</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html"
         *        >Managing access to shared datasets in directory buckets with access points</a> in the <i>Amazon S3
         *        User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the bucket that you want to associate this access point with.
         * </p>
         * <p>
         * For using this parameter with Amazon S3 on Outposts with the REST API, you must specify the name and the
         * x-amz-outpost-id as well.
         * </p>
         * <p>
         * For using this parameter with S3 on Outposts with the Amazon Web Services SDK and CLI, you must specify the
         * ARN of the bucket accessed in the format
         * <code>arn:aws:s3-outposts:&lt;Region&gt;:&lt;account-id&gt;:outpost/&lt;outpost-id&gt;/bucket/&lt;my-bucket-name&gt;</code>
         * . For example, to access the bucket <code>reports</code> through Outpost <code>my-outpost</code> owned by
         * account <code>123456789012</code> in Region <code>us-west-2</code>, use the URL encoding of
         * <code>arn:aws:s3-outposts:us-west-2:123456789012:outpost/my-outpost/bucket/reports</code>. The value must be
         * URL encoded.
         * </p>
         * 
         * @param bucket
         *        The name of the bucket that you want to associate this access point with.</p>
         *        <p>
         *        For using this parameter with Amazon S3 on Outposts with the REST API, you must specify the name and
         *        the x-amz-outpost-id as well.
         *        </p>
         *        <p>
         *        For using this parameter with S3 on Outposts with the Amazon Web Services SDK and CLI, you must
         *        specify the ARN of the bucket accessed in the format
         *        <code>arn:aws:s3-outposts:&lt;Region&gt;:&lt;account-id&gt;:outpost/&lt;outpost-id&gt;/bucket/&lt;my-bucket-name&gt;</code>
         *        . For example, to access the bucket <code>reports</code> through Outpost <code>my-outpost</code> owned
         *        by account <code>123456789012</code> in Region <code>us-west-2</code>, use the URL encoding of
         *        <code>arn:aws:s3-outposts:us-west-2:123456789012:outpost/my-outpost/bucket/reports</code>. The value
         *        must be URL encoded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * If you include this field, Amazon S3 restricts access to this access point to requests from the specified
         * virtual private cloud (VPC).
         * </p>
         * <note>
         * <p>
         * This is required for creating an access point for Amazon S3 on Outposts buckets.
         * </p>
         * </note>
         * 
         * @param vpcConfiguration
         *        If you include this field, Amazon S3 restricts access to this access point to requests from the
         *        specified virtual private cloud (VPC).</p> <note>
         *        <p>
         *        This is required for creating an access point for Amazon S3 on Outposts buckets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(VpcConfiguration vpcConfiguration);

        /**
         * <p>
         * If you include this field, Amazon S3 restricts access to this access point to requests from the specified
         * virtual private cloud (VPC).
         * </p>
         * <note>
         * <p>
         * This is required for creating an access point for Amazon S3 on Outposts buckets.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link VpcConfiguration.Builder}
         * avoiding the need to create one manually via {@link VpcConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #vpcConfiguration(VpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link VpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(VpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<VpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(VpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }

        /**
         * <p>
         * The <code>PublicAccessBlock</code> configuration that you want to apply to the access point.
         * </p>
         * 
         * @param publicAccessBlockConfiguration
         *        The <code>PublicAccessBlock</code> configuration that you want to apply to the access point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccessBlockConfiguration(PublicAccessBlockConfiguration publicAccessBlockConfiguration);

        /**
         * <p>
         * The <code>PublicAccessBlock</code> configuration that you want to apply to the access point.
         * </p>
         * This is a convenience method that creates an instance of the {@link PublicAccessBlockConfiguration.Builder}
         * avoiding the need to create one manually via {@link PublicAccessBlockConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PublicAccessBlockConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #publicAccessBlockConfiguration(PublicAccessBlockConfiguration)}.
         * 
         * @param publicAccessBlockConfiguration
         *        a consumer that will call methods on {@link PublicAccessBlockConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccessBlockConfiguration(PublicAccessBlockConfiguration)
         */
        default Builder publicAccessBlockConfiguration(
                Consumer<PublicAccessBlockConfiguration.Builder> publicAccessBlockConfiguration) {
            return publicAccessBlockConfiguration(PublicAccessBlockConfiguration.builder()
                    .applyMutation(publicAccessBlockConfiguration).build());
        }

        /**
         * <p>
         * The Amazon Web Services account ID associated with the S3 bucket associated with this access point.
         * </p>
         * <p>
         * For same account access point when your bucket and access point belong to the same account owner, the
         * <code>BucketAccountId</code> is not required. For cross-account access point when your bucket and access
         * point are not in the same account, the <code>BucketAccountId</code> is required.
         * </p>
         * 
         * @param bucketAccountId
         *        The Amazon Web Services account ID associated with the S3 bucket associated with this access
         *        point.</p>
         *        <p>
         *        For same account access point when your bucket and access point belong to the same account owner, the
         *        <code>BucketAccountId</code> is not required. For cross-account access point when your bucket and
         *        access point are not in the same account, the <code>BucketAccountId</code> is required.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketAccountId(String bucketAccountId);

        /**
         * <p>
         * For directory buckets, you can filter access control to specific prefixes, API operations, or a combination
         * of both. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html">Managing
         * access to shared datasets in directory buckets with access points</a> in the <i>Amazon S3 User Guide</i>.
         * </p>
         * <note>
         * <p>
         * Scope is only supported for access points attached to directory buckets.
         * </p>
         * </note>
         * 
         * @param scope
         *        For directory buckets, you can filter access control to specific prefixes, API operations, or a
         *        combination of both. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html"
         *        >Managing access to shared datasets in directory buckets with access points</a> in the <i>Amazon S3
         *        User Guide</i>.</p> <note>
         *        <p>
         *        Scope is only supported for access points attached to directory buckets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scope(Scope scope);

        /**
         * <p>
         * For directory buckets, you can filter access control to specific prefixes, API operations, or a combination
         * of both. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets.html">Managing
         * access to shared datasets in directory buckets with access points</a> in the <i>Amazon S3 User Guide</i>.
         * </p>
         * <note>
         * <p>
         * Scope is only supported for access points attached to directory buckets.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link Scope.Builder} avoiding the need
         * to create one manually via {@link Scope#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Scope.Builder#build()} is called immediately and its result is
         * passed to {@link #scope(Scope)}.
         * 
         * @param scope
         *        a consumer that will call methods on {@link Scope.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scope(Scope)
         */
        default Builder scope(Consumer<Scope.Builder> scope) {
            return scope(Scope.builder().applyMutation(scope).build());
        }

        /**
         * <p>
         * An array of tags that you can apply to an access point. Tags are key-value pairs of metadata used to control
         * access to your access points. For more information about tags, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Using tags with Amazon S3</a>. For
         * information about tagging access points, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html#using-tags-for-abac">Using tags for
         * attribute-based access control (ABAC)</a>.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You must have the <code>s3:TagResource</code> permission to create an access point with tags for a general
         * purpose bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * You must have the <code>s3express:TagResource</code> permission to create an access point with tags for a
         * directory bucket.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param tags
         *        An array of tags that you can apply to an access point. Tags are key-value pairs of metadata used to
         *        control access to your access points. For more information about tags, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Using tags with Amazon
         *        S3</a>. For information about tagging access points, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html#using-tags-for-abac">Using
         *        tags for attribute-based access control (ABAC)</a>.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You must have the <code>s3:TagResource</code> permission to create an access point with tags for a
         *        general purpose bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You must have the <code>s3express:TagResource</code> permission to create an access point with tags
         *        for a directory bucket.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * An array of tags that you can apply to an access point. Tags are key-value pairs of metadata used to control
         * access to your access points. For more information about tags, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Using tags with Amazon S3</a>. For
         * information about tagging access points, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html#using-tags-for-abac">Using tags for
         * attribute-based access control (ABAC)</a>.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You must have the <code>s3:TagResource</code> permission to create an access point with tags for a general
         * purpose bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * You must have the <code>s3express:TagResource</code> permission to create an access point with tags for a
         * directory bucket.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param tags
         *        An array of tags that you can apply to an access point. Tags are key-value pairs of metadata used to
         *        control access to your access points. For more information about tags, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Using tags with Amazon
         *        S3</a>. For information about tagging access points, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html#using-tags-for-abac">Using
         *        tags for attribute-based access control (ABAC)</a>.</p> <note>
         *        <ul>
         *        <li>
         *        <p>
         *        You must have the <code>s3:TagResource</code> permission to create an access point with tags for a
         *        general purpose bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You must have the <code>s3express:TagResource</code> permission to create an access point with tags
         *        for a directory bucket.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * An array of tags that you can apply to an access point. Tags are key-value pairs of metadata used to control
         * access to your access points. For more information about tags, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Using tags with Amazon S3</a>. For
         * information about tagging access points, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html#using-tags-for-abac">Using tags for
         * attribute-based access control (ABAC)</a>.
         * </p>
         * <note>
         * <ul>
         * <li>
         * <p>
         * You must have the <code>s3:TagResource</code> permission to create an access point with tags for a general
         * purpose bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * You must have the <code>s3express:TagResource</code> permission to create an access point with tags for a
         * directory bucket.
         * </p>
         * </li>
         * </ul>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.s3control.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.s3control.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.s3control.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.s3control.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3ControlRequest.BuilderImpl implements Builder {
        private String accountId;

        private String name;

        private String bucket;

        private VpcConfiguration vpcConfiguration;

        private PublicAccessBlockConfiguration publicAccessBlockConfiguration;

        private String bucketAccountId;

        private Scope scope;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAccessPointRequest model) {
            super(model);
            accountId(model.accountId);
            name(model.name);
            bucket(model.bucket);
            vpcConfiguration(model.vpcConfiguration);
            publicAccessBlockConfiguration(model.publicAccessBlockConfiguration);
            bucketAccountId(model.bucketAccountId);
            scope(model.scope);
            tags(model.tags);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final VpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(VpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        public final Builder vpcConfiguration(VpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        public final PublicAccessBlockConfiguration.Builder getPublicAccessBlockConfiguration() {
            return publicAccessBlockConfiguration != null ? publicAccessBlockConfiguration.toBuilder() : null;
        }

        public final void setPublicAccessBlockConfiguration(
                PublicAccessBlockConfiguration.BuilderImpl publicAccessBlockConfiguration) {
            this.publicAccessBlockConfiguration = publicAccessBlockConfiguration != null ? publicAccessBlockConfiguration.build()
                    : null;
        }

        @Override
        public final Builder publicAccessBlockConfiguration(PublicAccessBlockConfiguration publicAccessBlockConfiguration) {
            this.publicAccessBlockConfiguration = publicAccessBlockConfiguration;
            return this;
        }

        public final String getBucketAccountId() {
            return bucketAccountId;
        }

        public final void setBucketAccountId(String bucketAccountId) {
            this.bucketAccountId = bucketAccountId;
        }

        @Override
        public final Builder bucketAccountId(String bucketAccountId) {
            this.bucketAccountId = bucketAccountId;
            return this;
        }

        public final Scope.Builder getScope() {
            return scope != null ? scope.toBuilder() : null;
        }

        public final void setScope(Scope.BuilderImpl scope) {
            this.scope = scope != null ? scope.build() : null;
        }

        @Override
        public final Builder scope(Scope scope) {
            this.scope = scope;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateAccessPointRequest build() {
            return new CreateAccessPointRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
