/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure for a Multi-Region Access Point that indicates where Amazon S3 traffic can be routed. Routes can be
 * either active or passive. Active routes can process Amazon S3 requests through the Multi-Region Access Point, but
 * passive routes are not eligible to process Amazon S3 requests.
 * </p>
 * <p>
 * Each route contains the Amazon S3 bucket name and the Amazon Web Services Region that the bucket is located in. The
 * route also includes the <code>TrafficDialPercentage</code> value, which shows whether the bucket and Region are
 * active (indicated by a value of <code>100</code>) or passive (indicated by a value of <code>0</code>).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MultiRegionAccessPointRoute implements SdkPojo, Serializable,
        ToCopyableBuilder<MultiRegionAccessPointRoute.Builder, MultiRegionAccessPointRoute> {
    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(MultiRegionAccessPointRoute::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Region")
            .getter(getter(MultiRegionAccessPointRoute::region))
            .setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region")
                    .unmarshallLocationName("Region").build()).build();

    private static final SdkField<Integer> TRAFFIC_DIAL_PERCENTAGE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TrafficDialPercentage")
            .getter(getter(MultiRegionAccessPointRoute::trafficDialPercentage))
            .setter(setter(Builder::trafficDialPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrafficDialPercentage")
                    .unmarshallLocationName("TrafficDialPercentage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, REGION_FIELD,
            TRAFFIC_DIAL_PERCENTAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final String region;

    private final Integer trafficDialPercentage;

    private MultiRegionAccessPointRoute(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.region = builder.region;
        this.trafficDialPercentage = builder.trafficDialPercentage;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket for which you'll submit a routing configuration change. Either the
     * <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the bucket must be
     * in the specified Region.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket for which you'll submit a routing configuration change. Either the
     *         <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the bucket
     *         must be in the specified Region.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The Amazon Web Services Region to which you'll be submitting a routing configuration change. Either the
     * <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the bucket must be
     * in the specified Region.
     * </p>
     * 
     * @return The Amazon Web Services Region to which you'll be submitting a routing configuration change. Either the
     *         <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the bucket
     *         must be in the specified Region.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The traffic state for the specified bucket or Amazon Web Services Region.
     * </p>
     * <p>
     * A value of <code>0</code> indicates a passive state, which means that no new traffic will be routed to the
     * Region.
     * </p>
     * <p>
     * A value of <code>100</code> indicates an active state, which means that traffic will be routed to the specified
     * Region.
     * </p>
     * <p>
     * When the routing configuration for a Region is changed from active to passive, any in-progress operations
     * (uploads, copies, deletes, and so on) to the formerly active Region will continue to run to until a final success
     * or failure status is reached.
     * </p>
     * <p>
     * If all Regions in the routing configuration are designated as passive, you'll receive an
     * <code>InvalidRequest</code> error.
     * </p>
     * 
     * @return The traffic state for the specified bucket or Amazon Web Services Region. </p>
     *         <p>
     *         A value of <code>0</code> indicates a passive state, which means that no new traffic will be routed to
     *         the Region.
     *         </p>
     *         <p>
     *         A value of <code>100</code> indicates an active state, which means that traffic will be routed to the
     *         specified Region.
     *         </p>
     *         <p>
     *         When the routing configuration for a Region is changed from active to passive, any in-progress operations
     *         (uploads, copies, deletes, and so on) to the formerly active Region will continue to run to until a final
     *         success or failure status is reached.
     *         </p>
     *         <p>
     *         If all Regions in the routing configuration are designated as passive, you'll receive an
     *         <code>InvalidRequest</code> error.
     */
    public final Integer trafficDialPercentage() {
        return trafficDialPercentage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(trafficDialPercentage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MultiRegionAccessPointRoute)) {
            return false;
        }
        MultiRegionAccessPointRoute other = (MultiRegionAccessPointRoute) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(region(), other.region())
                && Objects.equals(trafficDialPercentage(), other.trafficDialPercentage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MultiRegionAccessPointRoute").add("Bucket", bucket()).add("Region", region())
                .add("TrafficDialPercentage", trafficDialPercentage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "TrafficDialPercentage":
            return Optional.ofNullable(clazz.cast(trafficDialPercentage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Bucket", BUCKET_FIELD);
        map.put("Region", REGION_FIELD);
        map.put("TrafficDialPercentage", TRAFFIC_DIAL_PERCENTAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MultiRegionAccessPointRoute, T> g) {
        return obj -> g.apply((MultiRegionAccessPointRoute) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MultiRegionAccessPointRoute> {
        /**
         * <p>
         * The name of the Amazon S3 bucket for which you'll submit a routing configuration change. Either the
         * <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the bucket must
         * be in the specified Region.
         * </p>
         * 
         * @param bucket
         *        The name of the Amazon S3 bucket for which you'll submit a routing configuration change. Either the
         *        <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the
         *        bucket must be in the specified Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The Amazon Web Services Region to which you'll be submitting a routing configuration change. Either the
         * <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the bucket must
         * be in the specified Region.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region to which you'll be submitting a routing configuration change. Either
         *        the <code>Bucket</code> or the <code>Region</code> value must be provided. If both are provided, the
         *        bucket must be in the specified Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The traffic state for the specified bucket or Amazon Web Services Region.
         * </p>
         * <p>
         * A value of <code>0</code> indicates a passive state, which means that no new traffic will be routed to the
         * Region.
         * </p>
         * <p>
         * A value of <code>100</code> indicates an active state, which means that traffic will be routed to the
         * specified Region.
         * </p>
         * <p>
         * When the routing configuration for a Region is changed from active to passive, any in-progress operations
         * (uploads, copies, deletes, and so on) to the formerly active Region will continue to run to until a final
         * success or failure status is reached.
         * </p>
         * <p>
         * If all Regions in the routing configuration are designated as passive, you'll receive an
         * <code>InvalidRequest</code> error.
         * </p>
         * 
         * @param trafficDialPercentage
         *        The traffic state for the specified bucket or Amazon Web Services Region. </p>
         *        <p>
         *        A value of <code>0</code> indicates a passive state, which means that no new traffic will be routed to
         *        the Region.
         *        </p>
         *        <p>
         *        A value of <code>100</code> indicates an active state, which means that traffic will be routed to the
         *        specified Region.
         *        </p>
         *        <p>
         *        When the routing configuration for a Region is changed from active to passive, any in-progress
         *        operations (uploads, copies, deletes, and so on) to the formerly active Region will continue to run to
         *        until a final success or failure status is reached.
         *        </p>
         *        <p>
         *        If all Regions in the routing configuration are designated as passive, you'll receive an
         *        <code>InvalidRequest</code> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trafficDialPercentage(Integer trafficDialPercentage);
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String region;

        private Integer trafficDialPercentage;

        private BuilderImpl() {
        }

        private BuilderImpl(MultiRegionAccessPointRoute model) {
            bucket(model.bucket);
            region(model.region);
            trafficDialPercentage(model.trafficDialPercentage);
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final Integer getTrafficDialPercentage() {
            return trafficDialPercentage;
        }

        public final void setTrafficDialPercentage(Integer trafficDialPercentage) {
            this.trafficDialPercentage = trafficDialPercentage;
        }

        @Override
        public final Builder trafficDialPercentage(Integer trafficDialPercentage) {
            this.trafficDialPercentage = trafficDialPercentage;
            return this;
        }

        @Override
        public MultiRegionAccessPointRoute build() {
            return new MultiRegionAccessPointRoute(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
