/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3control.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * You can use the access point scope to restrict access to specific prefixes, API operations, or a combination of both.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-directory-buckets-manage-scope.html">Manage
 * the scope of your access points for directory buckets</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Scope implements SdkPojo, Serializable, ToCopyableBuilder<Scope.Builder, Scope> {
    private static final SdkField<List<String>> PREFIXES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Prefixes")
            .getter(getter(Scope::prefixes))
            .setter(setter(Builder::prefixes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Prefixes")
                    .unmarshallLocationName("Prefixes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Prefix")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Prefix").unmarshallLocationName("Prefix").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> PERMISSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Permissions")
            .getter(getter(Scope::permissionsAsStrings))
            .setter(setter(Builder::permissionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Permissions")
                    .unmarshallLocationName("Permissions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Permission")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Permission").unmarshallLocationName("Permission").build())
                                            .build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PREFIXES_FIELD,
            PERMISSIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> prefixes;

    private final List<String> permissions;

    private Scope(BuilderImpl builder) {
        this.prefixes = builder.prefixes;
        this.permissions = builder.permissions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Prefixes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPrefixes() {
        return prefixes != null && !(prefixes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * You can specify any amount of prefixes, but the total length of characters of all prefixes must be less than 256
     * bytes in size.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPrefixes} method.
     * </p>
     * 
     * @return You can specify any amount of prefixes, but the total length of characters of all prefixes must be less
     *         than 256 bytes in size.
     */
    public final List<String> prefixes() {
        return prefixes;
    }

    /**
     * <p>
     * You can include one or more API operations as permissions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPermissions} method.
     * </p>
     * 
     * @return You can include one or more API operations as permissions.
     */
    public final List<ScopePermission> permissions() {
        return ScopePermissionListCopier.copyStringToEnum(permissions);
    }

    /**
     * For responses, this returns true if the service returned a value for the Permissions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPermissions() {
        return permissions != null && !(permissions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * You can include one or more API operations as permissions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPermissions} method.
     * </p>
     * 
     * @return You can include one or more API operations as permissions.
     */
    public final List<String> permissionsAsStrings() {
        return permissions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasPrefixes() ? prefixes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasPermissions() ? permissionsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Scope)) {
            return false;
        }
        Scope other = (Scope) obj;
        return hasPrefixes() == other.hasPrefixes() && Objects.equals(prefixes(), other.prefixes())
                && hasPermissions() == other.hasPermissions()
                && Objects.equals(permissionsAsStrings(), other.permissionsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Scope").add("Prefixes", hasPrefixes() ? prefixes() : null)
                .add("Permissions", hasPermissions() ? permissionsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Prefixes":
            return Optional.ofNullable(clazz.cast(prefixes()));
        case "Permissions":
            return Optional.ofNullable(clazz.cast(permissionsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Prefixes", PREFIXES_FIELD);
        map.put("Permissions", PERMISSIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Scope, T> g) {
        return obj -> g.apply((Scope) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Scope> {
        /**
         * <p>
         * You can specify any amount of prefixes, but the total length of characters of all prefixes must be less than
         * 256 bytes in size.
         * </p>
         * 
         * @param prefixes
         *        You can specify any amount of prefixes, but the total length of characters of all prefixes must be
         *        less than 256 bytes in size.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefixes(Collection<String> prefixes);

        /**
         * <p>
         * You can specify any amount of prefixes, but the total length of characters of all prefixes must be less than
         * 256 bytes in size.
         * </p>
         * 
         * @param prefixes
         *        You can specify any amount of prefixes, but the total length of characters of all prefixes must be
         *        less than 256 bytes in size.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefixes(String... prefixes);

        /**
         * <p>
         * You can include one or more API operations as permissions.
         * </p>
         * 
         * @param permissions
         *        You can include one or more API operations as permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsWithStrings(Collection<String> permissions);

        /**
         * <p>
         * You can include one or more API operations as permissions.
         * </p>
         * 
         * @param permissions
         *        You can include one or more API operations as permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissionsWithStrings(String... permissions);

        /**
         * <p>
         * You can include one or more API operations as permissions.
         * </p>
         * 
         * @param permissions
         *        You can include one or more API operations as permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissions(Collection<ScopePermission> permissions);

        /**
         * <p>
         * You can include one or more API operations as permissions.
         * </p>
         * 
         * @param permissions
         *        You can include one or more API operations as permissions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permissions(ScopePermission... permissions);
    }

    static final class BuilderImpl implements Builder {
        private List<String> prefixes = DefaultSdkAutoConstructList.getInstance();

        private List<String> permissions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Scope model) {
            prefixes(model.prefixes);
            permissionsWithStrings(model.permissions);
        }

        public final Collection<String> getPrefixes() {
            if (prefixes instanceof SdkAutoConstructList) {
                return null;
            }
            return prefixes;
        }

        public final void setPrefixes(Collection<String> prefixes) {
            this.prefixes = PrefixesListCopier.copy(prefixes);
        }

        @Override
        public final Builder prefixes(Collection<String> prefixes) {
            this.prefixes = PrefixesListCopier.copy(prefixes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder prefixes(String... prefixes) {
            prefixes(Arrays.asList(prefixes));
            return this;
        }

        public final Collection<String> getPermissions() {
            if (permissions instanceof SdkAutoConstructList) {
                return null;
            }
            return permissions;
        }

        public final void setPermissions(Collection<String> permissions) {
            this.permissions = ScopePermissionListCopier.copy(permissions);
        }

        @Override
        public final Builder permissionsWithStrings(Collection<String> permissions) {
            this.permissions = ScopePermissionListCopier.copy(permissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permissionsWithStrings(String... permissions) {
            permissionsWithStrings(Arrays.asList(permissions));
            return this;
        }

        @Override
        public final Builder permissions(Collection<ScopePermission> permissions) {
            this.permissions = ScopePermissionListCopier.copyEnumToString(permissions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permissions(ScopePermission... permissions) {
            permissions(Arrays.asList(permissions));
            return this;
        }

        @Override
        public Scope build() {
            return new Scope(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
