/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3outposts;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.s3outposts.model.AccessDeniedException;
import software.amazon.awssdk.services.s3outposts.model.ConflictException;
import software.amazon.awssdk.services.s3outposts.model.CreateEndpointRequest;
import software.amazon.awssdk.services.s3outposts.model.CreateEndpointResponse;
import software.amazon.awssdk.services.s3outposts.model.DeleteEndpointRequest;
import software.amazon.awssdk.services.s3outposts.model.DeleteEndpointResponse;
import software.amazon.awssdk.services.s3outposts.model.InternalServerException;
import software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest;
import software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse;
import software.amazon.awssdk.services.s3outposts.model.ResourceNotFoundException;
import software.amazon.awssdk.services.s3outposts.model.S3OutpostsException;
import software.amazon.awssdk.services.s3outposts.model.ValidationException;
import software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsIterable;

/**
 * Service client for accessing Amazon S3 Outposts. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Amazon S3 on Outposts provides access to S3 on Outposts operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface S3OutpostsClient extends SdkClient {
    String SERVICE_NAME = "s3-outposts";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "s3-outposts";

    /**
     * Create a {@link S3OutpostsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static S3OutpostsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link S3OutpostsClient}.
     */
    static S3OutpostsClientBuilder builder() {
        return new DefaultS3OutpostsClientBuilder();
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action creates an endpoint and associates it with the specified Outposts.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to complete.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createEndpointRequest
     * @return Result of the CreateEndpoint operation returned by the service.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.CreateEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/CreateEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEndpointResponse createEndpoint(CreateEndpointRequest createEndpointRequest) throws InternalServerException,
            ValidationException, AccessDeniedException, ResourceNotFoundException, ConflictException, AwsServiceException,
            SdkClientException, S3OutpostsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action creates an endpoint and associates it with the specified Outposts.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to complete.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateEndpointRequest.Builder} avoiding the need to
     * create one manually via {@link CreateEndpointRequest#builder()}
     * </p>
     *
     * @param createEndpointRequest
     *        A {@link Consumer} that will call methods on {@link CreateEndpointRequest.Builder} to create a request.
     * @return Result of the CreateEndpoint operation returned by the service.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ConflictException
     *         There was a conflict with this action, and it could not be completed.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.CreateEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/CreateEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateEndpointResponse createEndpoint(Consumer<CreateEndpointRequest.Builder> createEndpointRequest)
            throws InternalServerException, ValidationException, AccessDeniedException, ResourceNotFoundException,
            ConflictException, AwsServiceException, SdkClientException, S3OutpostsException {
        return createEndpoint(CreateEndpointRequest.builder().applyMutation(createEndpointRequest).build());
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action deletes an endpoint.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to complete.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteEndpointRequest
     * @return Result of the DeleteEndpoint operation returned by the service.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.DeleteEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEndpointResponse deleteEndpoint(DeleteEndpointRequest deleteEndpointRequest) throws InternalServerException,
            AccessDeniedException, ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            S3OutpostsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action deletes an endpoint.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to complete.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteEndpointRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteEndpointRequest#builder()}
     * </p>
     *
     * @param deleteEndpointRequest
     *        A {@link Consumer} that will call methods on {@link DeleteEndpointRequest.Builder} to create a request.
     * @return Result of the DeleteEndpoint operation returned by the service.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.DeleteEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteEndpointResponse deleteEndpoint(Consumer<DeleteEndpointRequest.Builder> deleteEndpointRequest)
            throws InternalServerException, AccessDeniedException, ResourceNotFoundException, ValidationException,
            AwsServiceException, SdkClientException, S3OutpostsException {
        return deleteEndpoint(DeleteEndpointRequest.builder().applyMutation(deleteEndpointRequest).build());
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action lists endpoints associated with the Outposts.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listEndpointsRequest
     * @return Result of the ListEndpoints operation returned by the service.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEndpointsResponse listEndpoints(ListEndpointsRequest listEndpointsRequest) throws InternalServerException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, AwsServiceException, SdkClientException,
            S3OutpostsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action lists endpoints associated with the Outposts.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEndpointsRequest.Builder} avoiding the need to
     * create one manually via {@link ListEndpointsRequest#builder()}
     * </p>
     *
     * @param listEndpointsRequest
     *        A {@link Consumer} that will call methods on {@link ListEndpointsRequest.Builder} to create a request.
     * @return Result of the ListEndpoints operation returned by the service.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEndpointsResponse listEndpoints(Consumer<ListEndpointsRequest.Builder> listEndpointsRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, S3OutpostsException {
        return listEndpoints(ListEndpointsRequest.builder().applyMutation(listEndpointsRequest).build());
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action lists endpoints associated with the Outposts.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsIterable responses = client.listEndpointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsIterable responses = client
     *             .listEndpointsPaginator(request);
     *     for (software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsIterable responses = client.listEndpointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation.</b>
     * </p>
     *
     * @param listEndpointsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEndpointsIterable listEndpointsPaginator(ListEndpointsRequest listEndpointsRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, S3OutpostsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts.
     * S3 on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual
     * private cloud (VPC). For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/AccessingS3Outposts.html"> Accessing S3 on Outposts
     * using VPC only access points</a>.
     * </p>
     * <p>
     * This action lists endpoints associated with the Outposts.
     * </p>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsIterable responses = client.listEndpointsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsIterable responses = client
     *             .listEndpointsPaginator(request);
     *     for (software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsIterable responses = client.listEndpointsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEndpointsRequest.Builder} avoiding the need to
     * create one manually via {@link ListEndpointsRequest#builder()}
     * </p>
     *
     * @param listEndpointsRequest
     *        A {@link Consumer} that will call methods on {@link ListEndpointsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an exception with the internal server.
     * @throws ResourceNotFoundException
     *         The requested resource was not found.
     * @throws AccessDeniedException
     *         Access was denied for this action.
     * @throws ValidationException
     *         There was an exception validating this data.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws S3OutpostsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample S3OutpostsClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEndpointsIterable listEndpointsPaginator(Consumer<ListEndpointsRequest.Builder> listEndpointsRequest)
            throws InternalServerException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, S3OutpostsException {
        return listEndpointsPaginator(ListEndpointsRequest.builder().applyMutation(listEndpointsRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }
}
