/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3outposts.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Amazon S3 on Outposts Access Points simplify managing data access at scale for shared datasets in S3 on Outposts. S3
 * on Outposts uses endpoints to connect to Outposts buckets so that you can perform actions within your virtual private
 * cloud (VPC). For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/WorkingWithS3Outposts.html"> Accessing S3 on Outposts
 * using VPC-only access points</a> in the <i>Amazon Simple Storage Service User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointArn").getter(getter(Endpoint::endpointArn)).setter(setter(Builder::endpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointArn").build()).build();

    private static final SdkField<String> OUTPOSTS_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OutpostsId").getter(getter(Endpoint::outpostsId)).setter(setter(Builder::outpostsId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostsId").build()).build();

    private static final SdkField<String> CIDR_BLOCK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CidrBlock").getter(getter(Endpoint::cidrBlock)).setter(setter(Builder::cidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CidrBlock").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Endpoint::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Endpoint::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<List<NetworkInterface>> NETWORK_INTERFACES_FIELD = SdkField
            .<List<NetworkInterface>> builder(MarshallingType.LIST)
            .memberName("NetworkInterfaces")
            .getter(getter(Endpoint::networkInterfaces))
            .setter(setter(Builder::networkInterfaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NetworkInterface> builder(MarshallingType.SDK_POJO)
                                            .constructor(NetworkInterface::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(Endpoint::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetId").getter(getter(Endpoint::subnetId)).setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId").build()).build();

    private static final SdkField<String> SECURITY_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecurityGroupId").getter(getter(Endpoint::securityGroupId)).setter(setter(Builder::securityGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupId").build()).build();

    private static final SdkField<String> ACCESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessType").getter(getter(Endpoint::accessTypeAsString)).setter(setter(Builder::accessType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessType").build()).build();

    private static final SdkField<String> CUSTOMER_OWNED_IPV4_POOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomerOwnedIpv4Pool").getter(getter(Endpoint::customerOwnedIpv4Pool))
            .setter(setter(Builder::customerOwnedIpv4Pool))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerOwnedIpv4Pool").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENDPOINT_ARN_FIELD,
            OUTPOSTS_ID_FIELD, CIDR_BLOCK_FIELD, STATUS_FIELD, CREATION_TIME_FIELD, NETWORK_INTERFACES_FIELD, VPC_ID_FIELD,
            SUBNET_ID_FIELD, SECURITY_GROUP_ID_FIELD, ACCESS_TYPE_FIELD, CUSTOMER_OWNED_IPV4_POOL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String endpointArn;

    private final String outpostsId;

    private final String cidrBlock;

    private final String status;

    private final Instant creationTime;

    private final List<NetworkInterface> networkInterfaces;

    private final String vpcId;

    private final String subnetId;

    private final String securityGroupId;

    private final String accessType;

    private final String customerOwnedIpv4Pool;

    private Endpoint(BuilderImpl builder) {
        this.endpointArn = builder.endpointArn;
        this.outpostsId = builder.outpostsId;
        this.cidrBlock = builder.cidrBlock;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
        this.networkInterfaces = builder.networkInterfaces;
        this.vpcId = builder.vpcId;
        this.subnetId = builder.subnetId;
        this.securityGroupId = builder.securityGroupId;
        this.accessType = builder.accessType;
        this.customerOwnedIpv4Pool = builder.customerOwnedIpv4Pool;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the endpoint.
     */
    public final String endpointArn() {
        return endpointArn;
    }

    /**
     * <p>
     * The ID of the Outposts.
     * </p>
     * 
     * @return The ID of the Outposts.
     */
    public final String outpostsId() {
        return outpostsId;
    }

    /**
     * <p>
     * The VPC CIDR committed by this endpoint.
     * </p>
     * 
     * @return The VPC CIDR committed by this endpoint.
     */
    public final String cidrBlock() {
        return cidrBlock;
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.
     * @see EndpointStatus
     */
    public final EndpointStatus status() {
        return EndpointStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.
     * @see EndpointStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time the endpoint was created.
     * </p>
     * 
     * @return The time the endpoint was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkInterfaces property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkInterfaces() {
        return networkInterfaces != null && !(networkInterfaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network interface of the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkInterfaces} method.
     * </p>
     * 
     * @return The network interface of the endpoint.
     */
    public final List<NetworkInterface> networkInterfaces() {
        return networkInterfaces;
    }

    /**
     * <p>
     * The ID of the VPC used for the endpoint.
     * </p>
     * 
     * @return The ID of the VPC used for the endpoint.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The ID of the subnet used for the endpoint.
     * </p>
     * 
     * @return The ID of the subnet used for the endpoint.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The ID of the security group used for the endpoint.
     * </p>
     * 
     * @return The ID of the security group used for the endpoint.
     */
    public final String securityGroupId() {
        return securityGroupId;
    }

    /**
     * <p>
     * The type of connectivity used to access the Amazon S3 on Outposts endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessType} will
     * return {@link EndpointAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTypeAsString}.
     * </p>
     * 
     * @return The type of connectivity used to access the Amazon S3 on Outposts endpoint.
     * @see EndpointAccessType
     */
    public final EndpointAccessType accessType() {
        return EndpointAccessType.fromValue(accessType);
    }

    /**
     * <p>
     * The type of connectivity used to access the Amazon S3 on Outposts endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessType} will
     * return {@link EndpointAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTypeAsString}.
     * </p>
     * 
     * @return The type of connectivity used to access the Amazon S3 on Outposts endpoint.
     * @see EndpointAccessType
     */
    public final String accessTypeAsString() {
        return accessType;
    }

    /**
     * <p>
     * The ID of the customer-owned IPv4 address pool used for the endpoint.
     * </p>
     * 
     * @return The ID of the customer-owned IPv4 address pool used for the endpoint.
     */
    public final String customerOwnedIpv4Pool() {
        return customerOwnedIpv4Pool;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(endpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(outpostsId());
        hashCode = 31 * hashCode + Objects.hashCode(cidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkInterfaces() ? networkInterfaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(accessTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customerOwnedIpv4Pool());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(endpointArn(), other.endpointArn()) && Objects.equals(outpostsId(), other.outpostsId())
                && Objects.equals(cidrBlock(), other.cidrBlock()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationTime(), other.creationTime()) && hasNetworkInterfaces() == other.hasNetworkInterfaces()
                && Objects.equals(networkInterfaces(), other.networkInterfaces()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(securityGroupId(), other.securityGroupId())
                && Objects.equals(accessTypeAsString(), other.accessTypeAsString())
                && Objects.equals(customerOwnedIpv4Pool(), other.customerOwnedIpv4Pool());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Endpoint").add("EndpointArn", endpointArn()).add("OutpostsId", outpostsId())
                .add("CidrBlock", cidrBlock()).add("Status", statusAsString()).add("CreationTime", creationTime())
                .add("NetworkInterfaces", hasNetworkInterfaces() ? networkInterfaces() : null).add("VpcId", vpcId())
                .add("SubnetId", subnetId()).add("SecurityGroupId", securityGroupId()).add("AccessType", accessTypeAsString())
                .add("CustomerOwnedIpv4Pool", customerOwnedIpv4Pool()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EndpointArn":
            return Optional.ofNullable(clazz.cast(endpointArn()));
        case "OutpostsId":
            return Optional.ofNullable(clazz.cast(outpostsId()));
        case "CidrBlock":
            return Optional.ofNullable(clazz.cast(cidrBlock()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "NetworkInterfaces":
            return Optional.ofNullable(clazz.cast(networkInterfaces()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "SecurityGroupId":
            return Optional.ofNullable(clazz.cast(securityGroupId()));
        case "AccessType":
            return Optional.ofNullable(clazz.cast(accessTypeAsString()));
        case "CustomerOwnedIpv4Pool":
            return Optional.ofNullable(clazz.cast(customerOwnedIpv4Pool()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the endpoint.
         * </p>
         * 
         * @param endpointArn
         *        The Amazon Resource Name (ARN) of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointArn(String endpointArn);

        /**
         * <p>
         * The ID of the Outposts.
         * </p>
         * 
         * @param outpostsId
         *        The ID of the Outposts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostsId(String outpostsId);

        /**
         * <p>
         * The VPC CIDR committed by this endpoint.
         * </p>
         * 
         * @param cidrBlock
         *        The VPC CIDR committed by this endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrBlock(String cidrBlock);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * 
         * @param status
         *        The status of the endpoint.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * 
         * @param status
         *        The status of the endpoint.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder status(EndpointStatus status);

        /**
         * <p>
         * The time the endpoint was created.
         * </p>
         * 
         * @param creationTime
         *        The time the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The network interface of the endpoint.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interface of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces);

        /**
         * <p>
         * The network interface of the endpoint.
         * </p>
         * 
         * @param networkInterfaces
         *        The network interface of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaces(NetworkInterface... networkInterfaces);

        /**
         * <p>
         * The network interface of the endpoint.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<NetworkInterface>.Builder} avoiding
         * the need to create one manually via {@link List<NetworkInterface>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NetworkInterface>.Builder#build()} is called immediately and
         * its result is passed to {@link #networkInterfaces(List<NetworkInterface>)}.
         * 
         * @param networkInterfaces
         *        a consumer that will call methods on {@link List<NetworkInterface>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaces(List<NetworkInterface>)
         */
        Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces);

        /**
         * <p>
         * The ID of the VPC used for the endpoint.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC used for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The ID of the subnet used for the endpoint.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet used for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The ID of the security group used for the endpoint.
         * </p>
         * 
         * @param securityGroupId
         *        The ID of the security group used for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupId(String securityGroupId);

        /**
         * <p>
         * The type of connectivity used to access the Amazon S3 on Outposts endpoint.
         * </p>
         * 
         * @param accessType
         *        The type of connectivity used to access the Amazon S3 on Outposts endpoint.
         * @see EndpointAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointAccessType
         */
        Builder accessType(String accessType);

        /**
         * <p>
         * The type of connectivity used to access the Amazon S3 on Outposts endpoint.
         * </p>
         * 
         * @param accessType
         *        The type of connectivity used to access the Amazon S3 on Outposts endpoint.
         * @see EndpointAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointAccessType
         */
        Builder accessType(EndpointAccessType accessType);

        /**
         * <p>
         * The ID of the customer-owned IPv4 address pool used for the endpoint.
         * </p>
         * 
         * @param customerOwnedIpv4Pool
         *        The ID of the customer-owned IPv4 address pool used for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerOwnedIpv4Pool(String customerOwnedIpv4Pool);
    }

    static final class BuilderImpl implements Builder {
        private String endpointArn;

        private String outpostsId;

        private String cidrBlock;

        private String status;

        private Instant creationTime;

        private List<NetworkInterface> networkInterfaces = DefaultSdkAutoConstructList.getInstance();

        private String vpcId;

        private String subnetId;

        private String securityGroupId;

        private String accessType;

        private String customerOwnedIpv4Pool;

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            endpointArn(model.endpointArn);
            outpostsId(model.outpostsId);
            cidrBlock(model.cidrBlock);
            status(model.status);
            creationTime(model.creationTime);
            networkInterfaces(model.networkInterfaces);
            vpcId(model.vpcId);
            subnetId(model.subnetId);
            securityGroupId(model.securityGroupId);
            accessType(model.accessType);
            customerOwnedIpv4Pool(model.customerOwnedIpv4Pool);
        }

        public final String getEndpointArn() {
            return endpointArn;
        }

        public final void setEndpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
        }

        @Override
        public final Builder endpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
            return this;
        }

        public final String getOutpostsId() {
            return outpostsId;
        }

        public final void setOutpostsId(String outpostsId) {
            this.outpostsId = outpostsId;
        }

        @Override
        public final Builder outpostsId(String outpostsId) {
            this.outpostsId = outpostsId;
            return this;
        }

        public final String getCidrBlock() {
            return cidrBlock;
        }

        public final void setCidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
        }

        @Override
        public final Builder cidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(EndpointStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final List<NetworkInterface.Builder> getNetworkInterfaces() {
            List<NetworkInterface.Builder> result = NetworkInterfacesCopier.copyToBuilder(this.networkInterfaces);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNetworkInterfaces(Collection<NetworkInterface.BuilderImpl> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copyFromBuilder(networkInterfaces);
        }

        @Override
        public final Builder networkInterfaces(Collection<NetworkInterface> networkInterfaces) {
            this.networkInterfaces = NetworkInterfacesCopier.copy(networkInterfaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(NetworkInterface... networkInterfaces) {
            networkInterfaces(Arrays.asList(networkInterfaces));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkInterfaces(Consumer<NetworkInterface.Builder>... networkInterfaces) {
            networkInterfaces(Stream.of(networkInterfaces).map(c -> NetworkInterface.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getSecurityGroupId() {
            return securityGroupId;
        }

        public final void setSecurityGroupId(String securityGroupId) {
            this.securityGroupId = securityGroupId;
        }

        @Override
        public final Builder securityGroupId(String securityGroupId) {
            this.securityGroupId = securityGroupId;
            return this;
        }

        public final String getAccessType() {
            return accessType;
        }

        public final void setAccessType(String accessType) {
            this.accessType = accessType;
        }

        @Override
        public final Builder accessType(String accessType) {
            this.accessType = accessType;
            return this;
        }

        @Override
        public final Builder accessType(EndpointAccessType accessType) {
            this.accessType(accessType == null ? null : accessType.toString());
            return this;
        }

        public final String getCustomerOwnedIpv4Pool() {
            return customerOwnedIpv4Pool;
        }

        public final void setCustomerOwnedIpv4Pool(String customerOwnedIpv4Pool) {
            this.customerOwnedIpv4Pool = customerOwnedIpv4Pool;
        }

        @Override
        public final Builder customerOwnedIpv4Pool(String customerOwnedIpv4Pool) {
            this.customerOwnedIpv4Pool = customerOwnedIpv4Pool;
            return this;
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
