/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3outposts;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.s3outposts.model.AccessDeniedException;
import software.amazon.awssdk.services.s3outposts.model.ConflictException;
import software.amazon.awssdk.services.s3outposts.model.CreateEndpointRequest;
import software.amazon.awssdk.services.s3outposts.model.CreateEndpointResponse;
import software.amazon.awssdk.services.s3outposts.model.DeleteEndpointRequest;
import software.amazon.awssdk.services.s3outposts.model.DeleteEndpointResponse;
import software.amazon.awssdk.services.s3outposts.model.InternalServerException;
import software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest;
import software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse;
import software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request;
import software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response;
import software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest;
import software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse;
import software.amazon.awssdk.services.s3outposts.model.ResourceNotFoundException;
import software.amazon.awssdk.services.s3outposts.model.S3OutpostsException;
import software.amazon.awssdk.services.s3outposts.model.S3OutpostsRequest;
import software.amazon.awssdk.services.s3outposts.model.ThrottlingException;
import software.amazon.awssdk.services.s3outposts.model.ValidationException;
import software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher;
import software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher;
import software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher;
import software.amazon.awssdk.services.s3outposts.transform.CreateEndpointRequestMarshaller;
import software.amazon.awssdk.services.s3outposts.transform.DeleteEndpointRequestMarshaller;
import software.amazon.awssdk.services.s3outposts.transform.ListEndpointsRequestMarshaller;
import software.amazon.awssdk.services.s3outposts.transform.ListOutpostsWithS3RequestMarshaller;
import software.amazon.awssdk.services.s3outposts.transform.ListSharedEndpointsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link S3OutpostsAsyncClient}.
 *
 * @see S3OutpostsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultS3OutpostsAsyncClient implements S3OutpostsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultS3OutpostsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultS3OutpostsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an endpoint and associates it with the specified Outpost.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to finish.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createEndpointRequest
     * @return A Java Future containing the result of the CreateEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.CreateEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/CreateEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEndpointResponse> createEndpoint(CreateEndpointRequest createEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEndpointRequest, CreateEndpointResponse>()
                            .withOperationName("CreateEndpoint")
                            .withMarshaller(new CreateEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createEndpointRequest));
            CompletableFuture<CreateEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an endpoint.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to finish.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteEndpointRequest
     * @return A Java Future containing the result of the DeleteEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.DeleteEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEndpointResponse> deleteEndpoint(DeleteEndpointRequest deleteEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEndpointRequest, DeleteEndpointResponse>()
                            .withOperationName("DeleteEndpoint")
                            .withMarshaller(new DeleteEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteEndpointRequest));
            CompletableFuture<DeleteEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists endpoints associated with the specified Outpost.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listEndpointsRequest
     * @return A Java Future containing the result of the ListEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEndpointsResponse> listEndpoints(ListEndpointsRequest listEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEndpointsRequest, ListEndpointsResponse>()
                            .withOperationName("ListEndpoints")
                            .withMarshaller(new ListEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEndpointsRequest));
            CompletableFuture<ListEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists endpoints associated with the specified Outpost.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher publisher = client.listEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher publisher = client.listEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation.</b>
     * </p>
     *
     * @param listEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    public ListEndpointsPublisher listEndpointsPaginator(ListEndpointsRequest listEndpointsRequest) {
        return new ListEndpointsPublisher(this, applyPaginatorUserAgent(listEndpointsRequest));
    }

    /**
     * <p>
     * Lists the Outposts with S3 on Outposts capacity for your Amazon Web Services account. Includes S3 on Outposts
     * that you have access to as the Outposts owner, or as a shared user from Resource Access Manager (RAM).
     * </p>
     *
     * @param listOutpostsWithS3Request
     * @return A Java Future containing the result of the ListOutpostsWithS3 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListOutpostsWithS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListOutpostsWithS3" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListOutpostsWithS3Response> listOutpostsWithS3(ListOutpostsWithS3Request listOutpostsWithS3Request) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listOutpostsWithS3Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListOutpostsWithS3");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListOutpostsWithS3Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListOutpostsWithS3Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListOutpostsWithS3Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListOutpostsWithS3Request, ListOutpostsWithS3Response>()
                            .withOperationName("ListOutpostsWithS3")
                            .withMarshaller(new ListOutpostsWithS3RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listOutpostsWithS3Request));
            CompletableFuture<ListOutpostsWithS3Response> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Outposts with S3 on Outposts capacity for your Amazon Web Services account. Includes S3 on Outposts
     * that you have access to as the Outposts owner, or as a shared user from Resource Access Manager (RAM).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listOutpostsWithS3(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher publisher = client.listOutpostsWithS3Paginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher publisher = client.listOutpostsWithS3Paginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOutpostsWithS3(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request)}
     * operation.</b>
     * </p>
     *
     * @param listOutpostsWithS3Request
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListOutpostsWithS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListOutpostsWithS3" target="_top">AWS
     *      API Documentation</a>
     */
    public ListOutpostsWithS3Publisher listOutpostsWithS3Paginator(ListOutpostsWithS3Request listOutpostsWithS3Request) {
        return new ListOutpostsWithS3Publisher(this, applyPaginatorUserAgent(listOutpostsWithS3Request));
    }

    /**
     * <p>
     * Lists all endpoints associated with an Outpost that has been shared by Amazon Web Services Resource Access
     * Manager (RAM).
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listSharedEndpointsRequest
     * @return A Java Future containing the result of the ListSharedEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListSharedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListSharedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSharedEndpointsResponse> listSharedEndpoints(
            ListSharedEndpointsRequest listSharedEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSharedEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Outposts");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSharedEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSharedEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSharedEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSharedEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSharedEndpointsRequest, ListSharedEndpointsResponse>()
                            .withOperationName("ListSharedEndpoints")
                            .withMarshaller(new ListSharedEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSharedEndpointsRequest));
            CompletableFuture<ListSharedEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all endpoints associated with an Outpost that has been shared by Amazon Web Services Resource Access
     * Manager (RAM).
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSharedEndpoints(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher publisher = client.listSharedEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher publisher = client.listSharedEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSharedEndpoints(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSharedEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListSharedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListSharedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    public ListSharedEndpointsPublisher listSharedEndpointsPaginator(ListSharedEndpointsRequest listSharedEndpointsRequest) {
        return new ListSharedEndpointsPublisher(this, applyPaginatorUserAgent(listSharedEndpointsRequest));
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(S3OutpostsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends S3OutpostsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
