/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3outposts;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.s3outposts.model.CreateEndpointRequest;
import software.amazon.awssdk.services.s3outposts.model.CreateEndpointResponse;
import software.amazon.awssdk.services.s3outposts.model.DeleteEndpointRequest;
import software.amazon.awssdk.services.s3outposts.model.DeleteEndpointResponse;
import software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest;
import software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse;
import software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request;
import software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response;
import software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest;
import software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse;
import software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher;
import software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher;
import software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher;

/**
 * Service client for accessing Amazon S3 Outposts asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * Amazon S3 on Outposts provides access to S3 on Outposts operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface S3OutpostsAsyncClient extends AwsClient {
    String SERVICE_NAME = "s3-outposts";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "s3-outposts";

    /**
     * <p>
     * Creates an endpoint and associates it with the specified Outpost.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to finish.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createEndpointRequest
     * @return A Java Future containing the result of the CreateEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>OutpostOfflineException The service link connection to your Outposts home Region is down. Check your
     *         connection and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.CreateEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/CreateEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateEndpointResponse> createEndpoint(CreateEndpointRequest createEndpointRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an endpoint and associates it with the specified Outpost.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to finish.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateEndpointRequest.Builder} avoiding the need to
     * create one manually via {@link CreateEndpointRequest#builder()}
     * </p>
     *
     * @param createEndpointRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.CreateEndpointRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CreateEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>OutpostOfflineException The service link connection to your Outposts home Region is down. Check your
     *         connection and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.CreateEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/CreateEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateEndpointResponse> createEndpoint(Consumer<CreateEndpointRequest.Builder> createEndpointRequest) {
        return createEndpoint(CreateEndpointRequest.builder().applyMutation(createEndpointRequest).build());
    }

    /**
     * <p>
     * Deletes an endpoint.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to finish.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param deleteEndpointRequest
     * @return A Java Future containing the result of the DeleteEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>OutpostOfflineException The service link connection to your Outposts home Region is down. Check your
     *         connection and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.DeleteEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteEndpointResponse> deleteEndpoint(DeleteEndpointRequest deleteEndpointRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes an endpoint.
     * </p>
     * <note>
     * <p>
     * It can take up to 5 minutes for this action to finish.
     * </p>
     * </note>
     * <p/>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_ListEndpoints.html">ListEndpoints</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteEndpointRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteEndpointRequest#builder()}
     * </p>
     *
     * @param deleteEndpointRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.DeleteEndpointRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>OutpostOfflineException The service link connection to your Outposts home Region is down. Check your
     *         connection and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.DeleteEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/DeleteEndpoint" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteEndpointResponse> deleteEndpoint(Consumer<DeleteEndpointRequest.Builder> deleteEndpointRequest) {
        return deleteEndpoint(DeleteEndpointRequest.builder().applyMutation(deleteEndpointRequest).build());
    }

    /**
     * <p>
     * Lists endpoints associated with the specified Outpost.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listEndpointsRequest
     * @return A Java Future containing the result of the ListEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEndpointsResponse> listEndpoints(ListEndpointsRequest listEndpointsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists endpoints associated with the specified Outpost.
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEndpointsRequest.Builder} avoiding the need to
     * create one manually via {@link ListEndpointsRequest#builder()}
     * </p>
     *
     * @param listEndpointsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListEndpointsResponse> listEndpoints(Consumer<ListEndpointsRequest.Builder> listEndpointsRequest) {
        return listEndpoints(ListEndpointsRequest.builder().applyMutation(listEndpointsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher publisher = client.listEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher publisher = client.listEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation.</b>
     * </p>
     *
     * @param listEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEndpointsPublisher listEndpointsPaginator(ListEndpointsRequest listEndpointsRequest) {
        return new ListEndpointsPublisher(this, listEndpointsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher publisher = client.listEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListEndpointsPublisher publisher = client.listEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEndpoints(software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListEndpointsRequest.Builder} avoiding the need to
     * create one manually via {@link ListEndpointsRequest#builder()}
     * </p>
     *
     * @param listEndpointsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.ListEndpointsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    default ListEndpointsPublisher listEndpointsPaginator(Consumer<ListEndpointsRequest.Builder> listEndpointsRequest) {
        return listEndpointsPaginator(ListEndpointsRequest.builder().applyMutation(listEndpointsRequest).build());
    }

    /**
     * <p>
     * Lists the Outposts with S3 on Outposts capacity for your Amazon Web Services account. Includes S3 on Outposts
     * that you have access to as the Outposts owner, or as a shared user from Resource Access Manager (RAM).
     * </p>
     *
     * @param listOutpostsWithS3Request
     * @return A Java Future containing the result of the ListOutpostsWithS3 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListOutpostsWithS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListOutpostsWithS3" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListOutpostsWithS3Response> listOutpostsWithS3(ListOutpostsWithS3Request listOutpostsWithS3Request) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the Outposts with S3 on Outposts capacity for your Amazon Web Services account. Includes S3 on Outposts
     * that you have access to as the Outposts owner, or as a shared user from Resource Access Manager (RAM).
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListOutpostsWithS3Request.Builder} avoiding the
     * need to create one manually via {@link ListOutpostsWithS3Request#builder()}
     * </p>
     *
     * @param listOutpostsWithS3Request
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListOutpostsWithS3 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListOutpostsWithS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListOutpostsWithS3" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListOutpostsWithS3Response> listOutpostsWithS3(
            Consumer<ListOutpostsWithS3Request.Builder> listOutpostsWithS3Request) {
        return listOutpostsWithS3(ListOutpostsWithS3Request.builder().applyMutation(listOutpostsWithS3Request).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listOutpostsWithS3(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher publisher = client.listOutpostsWithS3Paginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher publisher = client.listOutpostsWithS3Paginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOutpostsWithS3(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request)}
     * operation.</b>
     * </p>
     *
     * @param listOutpostsWithS3Request
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListOutpostsWithS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListOutpostsWithS3" target="_top">AWS
     *      API Documentation</a>
     */
    default ListOutpostsWithS3Publisher listOutpostsWithS3Paginator(ListOutpostsWithS3Request listOutpostsWithS3Request) {
        return new ListOutpostsWithS3Publisher(this, listOutpostsWithS3Request);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listOutpostsWithS3(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher publisher = client.listOutpostsWithS3Paginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListOutpostsWithS3Publisher publisher = client.listOutpostsWithS3Paginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Response response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listOutpostsWithS3(software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListOutpostsWithS3Request.Builder} avoiding the
     * need to create one manually via {@link ListOutpostsWithS3Request#builder()}
     * </p>
     *
     * @param listOutpostsWithS3Request
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.ListOutpostsWithS3Request.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListOutpostsWithS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListOutpostsWithS3" target="_top">AWS
     *      API Documentation</a>
     */
    default ListOutpostsWithS3Publisher listOutpostsWithS3Paginator(
            Consumer<ListOutpostsWithS3Request.Builder> listOutpostsWithS3Request) {
        return listOutpostsWithS3Paginator(ListOutpostsWithS3Request.builder().applyMutation(listOutpostsWithS3Request).build());
    }

    /**
     * <p>
     * Lists all endpoints associated with an Outpost that has been shared by Amazon Web Services Resource Access
     * Manager (RAM).
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param listSharedEndpointsRequest
     * @return A Java Future containing the result of the ListSharedEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListSharedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListSharedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListSharedEndpointsResponse> listSharedEndpoints(
            ListSharedEndpointsRequest listSharedEndpointsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all endpoints associated with an Outpost that has been shared by Amazon Web Services Resource Access
     * Manager (RAM).
     * </p>
     * <p>
     * Related actions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_CreateEndpoint.html">CreateEndpoint</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3outposts_DeleteEndpoint.html">DeleteEndpoint</a>
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSharedEndpointsRequest.Builder} avoiding the
     * need to create one manually via {@link ListSharedEndpointsRequest#builder()}
     * </p>
     *
     * @param listSharedEndpointsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListSharedEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListSharedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListSharedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListSharedEndpointsResponse> listSharedEndpoints(
            Consumer<ListSharedEndpointsRequest.Builder> listSharedEndpointsRequest) {
        return listSharedEndpoints(ListSharedEndpointsRequest.builder().applyMutation(listSharedEndpointsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSharedEndpoints(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher publisher = client.listSharedEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher publisher = client.listSharedEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSharedEndpoints(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSharedEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListSharedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListSharedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSharedEndpointsPublisher listSharedEndpointsPaginator(ListSharedEndpointsRequest listSharedEndpointsRequest) {
        return new ListSharedEndpointsPublisher(this, listSharedEndpointsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listSharedEndpoints(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher publisher = client.listSharedEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.s3outposts.paginators.ListSharedEndpointsPublisher publisher = client.listSharedEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSharedEndpoints(software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListSharedEndpointsRequest.Builder} avoiding the
     * need to create one manually via {@link ListSharedEndpointsRequest#builder()}
     * </p>
     *
     * @param listSharedEndpointsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.s3outposts.model.ListSharedEndpointsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalServerException There was an exception with the internal server.</li>
     *         <li>ResourceNotFoundException The requested resource was not found.</li>
     *         <li>AccessDeniedException Access was denied for this action.</li>
     *         <li>ValidationException There was an exception validating this data.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3OutpostsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3OutpostsAsyncClient.ListSharedEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3outposts-2017-07-25/ListSharedEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    default ListSharedEndpointsPublisher listSharedEndpointsPaginator(
            Consumer<ListSharedEndpointsRequest.Builder> listSharedEndpointsRequest) {
        return listSharedEndpointsPaginator(ListSharedEndpointsRequest.builder().applyMutation(listSharedEndpointsRequest)
                .build());
    }

    @Override
    default S3OutpostsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link S3OutpostsAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static S3OutpostsAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link S3OutpostsAsyncClient}.
     */
    static S3OutpostsAsyncClientBuilder builder() {
        return new DefaultS3OutpostsAsyncClientBuilder();
    }
}
