/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3tables.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration specifying how data should be encrypted. This structure defines the encryption algorithm and optional
 * KMS key to be used for server-side encryption.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EncryptionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<EncryptionConfiguration.Builder, EncryptionConfiguration> {
    private static final SdkField<String> SSE_ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sseAlgorithm").getter(getter(EncryptionConfiguration::sseAlgorithmAsString))
            .setter(setter(Builder::sseAlgorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sseAlgorithm").build()).build();

    private static final SdkField<String> KMS_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("kmsKeyArn").getter(getter(EncryptionConfiguration::kmsKeyArn)).setter(setter(Builder::kmsKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kmsKeyArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SSE_ALGORITHM_FIELD,
            KMS_KEY_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sseAlgorithm;

    private final String kmsKeyArn;

    private EncryptionConfiguration(BuilderImpl builder) {
        this.sseAlgorithm = builder.sseAlgorithm;
        this.kmsKeyArn = builder.kmsKeyArn;
    }

    /**
     * <p>
     * The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed encryption keys,
     * or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you choose SSE-KMS encryption you
     * must grant the S3 Tables maintenance principal access to your KMS key. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
     * requirements for S3 Tables SSE-KMS encryption</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseAlgorithm} will
     * return {@link SSEAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseAlgorithmAsString}.
     * </p>
     * 
     * @return The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed
     *         encryption keys, or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you
     *         choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS key. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
     *         requirements for S3 Tables SSE-KMS encryption</a>.
     * @see SSEAlgorithm
     */
    public final SSEAlgorithm sseAlgorithm() {
        return SSEAlgorithm.fromValue(sseAlgorithm);
    }

    /**
     * <p>
     * The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed encryption keys,
     * or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you choose SSE-KMS encryption you
     * must grant the S3 Tables maintenance principal access to your KMS key. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
     * requirements for S3 Tables SSE-KMS encryption</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sseAlgorithm} will
     * return {@link SSEAlgorithm#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sseAlgorithmAsString}.
     * </p>
     * 
     * @return The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed
     *         encryption keys, or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you
     *         choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS key. For
     *         more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
     *         requirements for S3 Tables SSE-KMS encryption</a>.
     * @see SSEAlgorithm
     */
    public final String sseAlgorithmAsString() {
        return sseAlgorithm;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the KMS key to use for encryption. This field is required only when
     * <code>sseAlgorithm</code> is set to <code>aws:kms</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the KMS key to use for encryption. This field is required only when
     *         <code>sseAlgorithm</code> is set to <code>aws:kms</code>.
     */
    public final String kmsKeyArn() {
        return kmsKeyArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sseAlgorithmAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EncryptionConfiguration)) {
            return false;
        }
        EncryptionConfiguration other = (EncryptionConfiguration) obj;
        return Objects.equals(sseAlgorithmAsString(), other.sseAlgorithmAsString())
                && Objects.equals(kmsKeyArn(), other.kmsKeyArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EncryptionConfiguration").add("SseAlgorithm", sseAlgorithmAsString())
                .add("KmsKeyArn", kmsKeyArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "sseAlgorithm":
            return Optional.ofNullable(clazz.cast(sseAlgorithmAsString()));
        case "kmsKeyArn":
            return Optional.ofNullable(clazz.cast(kmsKeyArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("sseAlgorithm", SSE_ALGORITHM_FIELD);
        map.put("kmsKeyArn", KMS_KEY_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EncryptionConfiguration, T> g) {
        return obj -> g.apply((EncryptionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EncryptionConfiguration> {
        /**
         * <p>
         * The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed encryption
         * keys, or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you choose SSE-KMS
         * encryption you must grant the S3 Tables maintenance principal access to your KMS key. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
         * requirements for S3 Tables SSE-KMS encryption</a>.
         * </p>
         * 
         * @param sseAlgorithm
         *        The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed
         *        encryption keys, or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you
         *        choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS key.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html"
         *        >Permissions requirements for S3 Tables SSE-KMS encryption</a>.
         * @see SSEAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSEAlgorithm
         */
        Builder sseAlgorithm(String sseAlgorithm);

        /**
         * <p>
         * The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed encryption
         * keys, or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you choose SSE-KMS
         * encryption you must grant the S3 Tables maintenance principal access to your KMS key. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
         * requirements for S3 Tables SSE-KMS encryption</a>.
         * </p>
         * 
         * @param sseAlgorithm
         *        The server-side encryption algorithm to use. Valid values are <code>AES256</code> for S3-managed
         *        encryption keys, or <code>aws:kms</code> for Amazon Web Services KMS-managed encryption keys. If you
         *        choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS key.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html"
         *        >Permissions requirements for S3 Tables SSE-KMS encryption</a>.
         * @see SSEAlgorithm
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSEAlgorithm
         */
        Builder sseAlgorithm(SSEAlgorithm sseAlgorithm);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the KMS key to use for encryption. This field is required only when
         * <code>sseAlgorithm</code> is set to <code>aws:kms</code>.
         * </p>
         * 
         * @param kmsKeyArn
         *        The Amazon Resource Name (ARN) of the KMS key to use for encryption. This field is required only when
         *        <code>sseAlgorithm</code> is set to <code>aws:kms</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyArn(String kmsKeyArn);
    }

    static final class BuilderImpl implements Builder {
        private String sseAlgorithm;

        private String kmsKeyArn;

        private BuilderImpl() {
        }

        private BuilderImpl(EncryptionConfiguration model) {
            sseAlgorithm(model.sseAlgorithm);
            kmsKeyArn(model.kmsKeyArn);
        }

        public final String getSseAlgorithm() {
            return sseAlgorithm;
        }

        public final void setSseAlgorithm(String sseAlgorithm) {
            this.sseAlgorithm = sseAlgorithm;
        }

        @Override
        public final Builder sseAlgorithm(String sseAlgorithm) {
            this.sseAlgorithm = sseAlgorithm;
            return this;
        }

        @Override
        public final Builder sseAlgorithm(SSEAlgorithm sseAlgorithm) {
            this.sseAlgorithm(sseAlgorithm == null ? null : sseAlgorithm.toString());
            return this;
        }

        public final String getKmsKeyArn() {
            return kmsKeyArn;
        }

        public final void setKmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
        }

        @Override
        public final Builder kmsKeyArn(String kmsKeyArn) {
            this.kmsKeyArn = kmsKeyArn;
            return this;
        }

        @Override
        public EncryptionConfiguration build() {
            return new EncryptionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
