/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3tables.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a namespace.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NamespaceSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<NamespaceSummary.Builder, NamespaceSummary> {
    private static final SdkField<List<String>> NAMESPACE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("namespace")
            .getter(getter(NamespaceSummary::namespace))
            .setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(NamespaceSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("createdBy").getter(getter(NamespaceSummary::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdBy").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ownerAccountId").getter(getter(NamespaceSummary::ownerAccountId))
            .setter(setter(Builder::ownerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ownerAccountId").build()).build();

    private static final SdkField<String> NAMESPACE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespaceId").getter(getter(NamespaceSummary::namespaceId)).setter(setter(Builder::namespaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespaceId").build()).build();

    private static final SdkField<String> TABLE_BUCKET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tableBucketId").getter(getter(NamespaceSummary::tableBucketId)).setter(setter(Builder::tableBucketId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tableBucketId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD,
            CREATED_AT_FIELD, CREATED_BY_FIELD, OWNER_ACCOUNT_ID_FIELD, NAMESPACE_ID_FIELD, TABLE_BUCKET_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> namespace;

    private final Instant createdAt;

    private final String createdBy;

    private final String ownerAccountId;

    private final String namespaceId;

    private final String tableBucketId;

    private NamespaceSummary(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.createdAt = builder.createdAt;
        this.createdBy = builder.createdBy;
        this.ownerAccountId = builder.ownerAccountId;
        this.namespaceId = builder.namespaceId;
        this.tableBucketId = builder.tableBucketId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Namespace property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNamespace() {
        return namespace != null && !(namespace instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the namespace.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNamespace} method.
     * </p>
     * 
     * @return The name of the namespace.
     */
    public final List<String> namespace() {
        return namespace;
    }

    /**
     * <p>
     * The date and time the namespace was created at.
     * </p>
     * 
     * @return The date and time the namespace was created at.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The ID of the account that created the namespace.
     * </p>
     * 
     * @return The ID of the account that created the namespace.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The ID of the account that owns the namespace.
     * </p>
     * 
     * @return The ID of the account that owns the namespace.
     */
    public final String ownerAccountId() {
        return ownerAccountId;
    }

    /**
     * <p>
     * The system-assigned unique identifier for the namespace.
     * </p>
     * 
     * @return The system-assigned unique identifier for the namespace.
     */
    public final String namespaceId() {
        return namespaceId;
    }

    /**
     * <p>
     * The system-assigned unique identifier for the table bucket that contains this namespace.
     * </p>
     * 
     * @return The system-assigned unique identifier for the table bucket that contains this namespace.
     */
    public final String tableBucketId() {
        return tableBucketId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasNamespace() ? namespace() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(namespaceId());
        hashCode = 31 * hashCode + Objects.hashCode(tableBucketId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NamespaceSummary)) {
            return false;
        }
        NamespaceSummary other = (NamespaceSummary) obj;
        return hasNamespace() == other.hasNamespace() && Objects.equals(namespace(), other.namespace())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(createdBy(), other.createdBy())
                && Objects.equals(ownerAccountId(), other.ownerAccountId()) && Objects.equals(namespaceId(), other.namespaceId())
                && Objects.equals(tableBucketId(), other.tableBucketId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NamespaceSummary").add("Namespace", hasNamespace() ? namespace() : null)
                .add("CreatedAt", createdAt()).add("CreatedBy", createdBy()).add("OwnerAccountId", ownerAccountId())
                .add("NamespaceId", namespaceId()).add("TableBucketId", tableBucketId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "createdBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "ownerAccountId":
            return Optional.ofNullable(clazz.cast(ownerAccountId()));
        case "namespaceId":
            return Optional.ofNullable(clazz.cast(namespaceId()));
        case "tableBucketId":
            return Optional.ofNullable(clazz.cast(tableBucketId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("namespace", NAMESPACE_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("createdBy", CREATED_BY_FIELD);
        map.put("ownerAccountId", OWNER_ACCOUNT_ID_FIELD);
        map.put("namespaceId", NAMESPACE_ID_FIELD);
        map.put("tableBucketId", TABLE_BUCKET_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NamespaceSummary, T> g) {
        return obj -> g.apply((NamespaceSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NamespaceSummary> {
        /**
         * <p>
         * The name of the namespace.
         * </p>
         * 
         * @param namespace
         *        The name of the namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(Collection<String> namespace);

        /**
         * <p>
         * The name of the namespace.
         * </p>
         * 
         * @param namespace
         *        The name of the namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String... namespace);

        /**
         * <p>
         * The date and time the namespace was created at.
         * </p>
         * 
         * @param createdAt
         *        The date and time the namespace was created at.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The ID of the account that created the namespace.
         * </p>
         * 
         * @param createdBy
         *        The ID of the account that created the namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The ID of the account that owns the namespace.
         * </p>
         * 
         * @param ownerAccountId
         *        The ID of the account that owns the namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccountId(String ownerAccountId);

        /**
         * <p>
         * The system-assigned unique identifier for the namespace.
         * </p>
         * 
         * @param namespaceId
         *        The system-assigned unique identifier for the namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespaceId(String namespaceId);

        /**
         * <p>
         * The system-assigned unique identifier for the table bucket that contains this namespace.
         * </p>
         * 
         * @param tableBucketId
         *        The system-assigned unique identifier for the table bucket that contains this namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableBucketId(String tableBucketId);
    }

    static final class BuilderImpl implements Builder {
        private List<String> namespace = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private String createdBy;

        private String ownerAccountId;

        private String namespaceId;

        private String tableBucketId;

        private BuilderImpl() {
        }

        private BuilderImpl(NamespaceSummary model) {
            namespace(model.namespace);
            createdAt(model.createdAt);
            createdBy(model.createdBy);
            ownerAccountId(model.ownerAccountId);
            namespaceId(model.namespaceId);
            tableBucketId(model.tableBucketId);
        }

        public final Collection<String> getNamespace() {
            if (namespace instanceof SdkAutoConstructList) {
                return null;
            }
            return namespace;
        }

        public final void setNamespace(Collection<String> namespace) {
            this.namespace = NamespaceListCopier.copy(namespace);
        }

        @Override
        public final Builder namespace(Collection<String> namespace) {
            this.namespace = NamespaceListCopier.copy(namespace);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder namespace(String... namespace) {
            namespace(Arrays.asList(namespace));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final String getOwnerAccountId() {
            return ownerAccountId;
        }

        public final void setOwnerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
        }

        @Override
        public final Builder ownerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
            return this;
        }

        public final String getNamespaceId() {
            return namespaceId;
        }

        public final void setNamespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
        }

        @Override
        public final Builder namespaceId(String namespaceId) {
            this.namespaceId = namespaceId;
            return this;
        }

        public final String getTableBucketId() {
            return tableBucketId;
        }

        public final void setTableBucketId(String tableBucketId) {
            this.tableBucketId = tableBucketId;
        }

        @Override
        public final Builder tableBucketId(String tableBucketId) {
            this.tableBucketId = tableBucketId;
            return this;
        }

        @Override
        public NamespaceSummary build() {
            return new NamespaceSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
