/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3tables.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateTableRequest extends S3TablesRequest implements
        ToCopyableBuilder<CreateTableRequest.Builder, CreateTableRequest> {
    private static final SdkField<String> TABLE_BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tableBucketARN").getter(getter(CreateTableRequest::tableBucketARN))
            .setter(setter(Builder::tableBucketARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("tableBucketARN").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(CreateTableRequest::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("namespace").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CreateTableRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(CreateTableRequest::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final SdkField<TableMetadata> METADATA_FIELD = SdkField.<TableMetadata> builder(MarshallingType.SDK_POJO)
            .memberName("metadata").getter(getter(CreateTableRequest::metadata)).setter(setter(Builder::metadata))
            .constructor(TableMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metadata").build()).build();

    private static final SdkField<EncryptionConfiguration> ENCRYPTION_CONFIGURATION_FIELD = SdkField
            .<EncryptionConfiguration> builder(MarshallingType.SDK_POJO).memberName("encryptionConfiguration")
            .getter(getter(CreateTableRequest::encryptionConfiguration)).setter(setter(Builder::encryptionConfiguration))
            .constructor(EncryptionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionConfiguration").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateTableRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TABLE_BUCKET_ARN_FIELD,
            NAMESPACE_FIELD, NAME_FIELD, FORMAT_FIELD, METADATA_FIELD, ENCRYPTION_CONFIGURATION_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String tableBucketARN;

    private final String namespace;

    private final String name;

    private final String format;

    private final TableMetadata metadata;

    private final EncryptionConfiguration encryptionConfiguration;

    private final Map<String, String> tags;

    private CreateTableRequest(BuilderImpl builder) {
        super(builder);
        this.tableBucketARN = builder.tableBucketARN;
        this.namespace = builder.namespace;
        this.name = builder.name;
        this.format = builder.format;
        this.metadata = builder.metadata;
        this.encryptionConfiguration = builder.encryptionConfiguration;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the table bucket to create the table in.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the table bucket to create the table in.
     */
    public final String tableBucketARN() {
        return tableBucketARN;
    }

    /**
     * <p>
     * The namespace to associated with the table.
     * </p>
     * 
     * @return The namespace to associated with the table.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name for the table.
     * </p>
     * 
     * @return The name for the table.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The format for the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link OpenTableFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format for the table.
     * @see OpenTableFormat
     */
    public final OpenTableFormat format() {
        return OpenTableFormat.fromValue(format);
    }

    /**
     * <p>
     * The format for the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link OpenTableFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format for the table.
     * @see OpenTableFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The metadata for the table.
     * </p>
     * 
     * @return The metadata for the table.
     */
    public final TableMetadata metadata() {
        return metadata;
    }

    /**
     * <p>
     * The encryption configuration to use for the table. This configuration specifies the encryption algorithm and, if
     * using SSE-KMS, the KMS key to use for encrypting the table.
     * </p>
     * <note>
     * <p>
     * If you choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS key. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
     * requirements for S3 Tables SSE-KMS encryption</a>.
     * </p>
     * </note>
     * 
     * @return The encryption configuration to use for the table. This configuration specifies the encryption algorithm
     *         and, if using SSE-KMS, the KMS key to use for encrypting the table. </p> <note>
     *         <p>
     *         If you choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS
     *         key. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
     *         requirements for S3 Tables SSE-KMS encryption</a>.
     *         </p>
     */
    public final EncryptionConfiguration encryptionConfiguration() {
        return encryptionConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of user-defined tags that you would like to apply to the table that you are creating. A tag is a key-value
     * pair that you apply to your resources. Tags can help you organize, track costs for, and control access to
     * resources. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Tagging for cost allocation or
     * attribute-based access control (ABAC)</a>.
     * </p>
     * <note>
     * <p>
     * You must have the <code>s3tables:TagResource</code> permission in addition to <code>s3tables:CreateTable</code>
     * permission to create a table with tags.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A map of user-defined tags that you would like to apply to the table that you are creating. A tag is a
     *         key-value pair that you apply to your resources. Tags can help you organize, track costs for, and control
     *         access to resources. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Tagging for cost allocation or
     *         attribute-based access control (ABAC)</a>.</p> <note>
     *         <p>
     *         You must have the <code>s3tables:TagResource</code> permission in addition to
     *         <code>s3tables:CreateTable</code> permission to create a table with tags.
     *         </p>
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(tableBucketARN());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(metadata());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateTableRequest)) {
            return false;
        }
        CreateTableRequest other = (CreateTableRequest) obj;
        return Objects.equals(tableBucketARN(), other.tableBucketARN()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(name(), other.name()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(metadata(), other.metadata())
                && Objects.equals(encryptionConfiguration(), other.encryptionConfiguration()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateTableRequest").add("TableBucketARN", tableBucketARN()).add("Namespace", namespace())
                .add("Name", name()).add("Format", formatAsString()).add("Metadata", metadata())
                .add("EncryptionConfiguration", encryptionConfiguration()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "tableBucketARN":
            return Optional.ofNullable(clazz.cast(tableBucketARN()));
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "metadata":
            return Optional.ofNullable(clazz.cast(metadata()));
        case "encryptionConfiguration":
            return Optional.ofNullable(clazz.cast(encryptionConfiguration()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("tableBucketARN", TABLE_BUCKET_ARN_FIELD);
        map.put("namespace", NAMESPACE_FIELD);
        map.put("name", NAME_FIELD);
        map.put("format", FORMAT_FIELD);
        map.put("metadata", METADATA_FIELD);
        map.put("encryptionConfiguration", ENCRYPTION_CONFIGURATION_FIELD);
        map.put("tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateTableRequest, T> g) {
        return obj -> g.apply((CreateTableRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends S3TablesRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateTableRequest> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the table bucket to create the table in.
         * </p>
         * 
         * @param tableBucketARN
         *        The Amazon Resource Name (ARN) of the table bucket to create the table in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableBucketARN(String tableBucketARN);

        /**
         * <p>
         * The namespace to associated with the table.
         * </p>
         * 
         * @param namespace
         *        The namespace to associated with the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The name for the table.
         * </p>
         * 
         * @param name
         *        The name for the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The format for the table.
         * </p>
         * 
         * @param format
         *        The format for the table.
         * @see OpenTableFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenTableFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format for the table.
         * </p>
         * 
         * @param format
         *        The format for the table.
         * @see OpenTableFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenTableFormat
         */
        Builder format(OpenTableFormat format);

        /**
         * <p>
         * The metadata for the table.
         * </p>
         * 
         * @param metadata
         *        The metadata for the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(TableMetadata metadata);

        /**
         * <p>
         * The metadata for the table.
         * </p>
         * This is a convenience method that creates an instance of the {@link TableMetadata.Builder} avoiding the need
         * to create one manually via {@link TableMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TableMetadata.Builder#build()} is called immediately and its
         * result is passed to {@link #metadata(TableMetadata)}.
         * 
         * @param metadata
         *        a consumer that will call methods on {@link TableMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metadata(TableMetadata)
         */
        default Builder metadata(Consumer<TableMetadata.Builder> metadata) {
            return metadata(TableMetadata.builder().applyMutation(metadata).build());
        }

        /**
         * <p>
         * The encryption configuration to use for the table. This configuration specifies the encryption algorithm and,
         * if using SSE-KMS, the KMS key to use for encrypting the table.
         * </p>
         * <note>
         * <p>
         * If you choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS key.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
         * requirements for S3 Tables SSE-KMS encryption</a>.
         * </p>
         * </note>
         * 
         * @param encryptionConfiguration
         *        The encryption configuration to use for the table. This configuration specifies the encryption
         *        algorithm and, if using SSE-KMS, the KMS key to use for encrypting the table. </p> <note>
         *        <p>
         *        If you choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS
         *        key. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html"
         *        >Permissions requirements for S3 Tables SSE-KMS encryption</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration);

        /**
         * <p>
         * The encryption configuration to use for the table. This configuration specifies the encryption algorithm and,
         * if using SSE-KMS, the KMS key to use for encrypting the table.
         * </p>
         * <note>
         * <p>
         * If you choose SSE-KMS encryption you must grant the S3 Tables maintenance principal access to your KMS key.
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-kms-permissions.html">Permissions
         * requirements for S3 Tables SSE-KMS encryption</a>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link EncryptionConfiguration.Builder}
         * avoiding the need to create one manually via {@link EncryptionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #encryptionConfiguration(EncryptionConfiguration)}.
         * 
         * @param encryptionConfiguration
         *        a consumer that will call methods on {@link EncryptionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfiguration(EncryptionConfiguration)
         */
        default Builder encryptionConfiguration(Consumer<EncryptionConfiguration.Builder> encryptionConfiguration) {
            return encryptionConfiguration(EncryptionConfiguration.builder().applyMutation(encryptionConfiguration).build());
        }

        /**
         * <p>
         * A map of user-defined tags that you would like to apply to the table that you are creating. A tag is a
         * key-value pair that you apply to your resources. Tags can help you organize, track costs for, and control
         * access to resources. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Tagging for cost allocation or
         * attribute-based access control (ABAC)</a>.
         * </p>
         * <note>
         * <p>
         * You must have the <code>s3tables:TagResource</code> permission in addition to
         * <code>s3tables:CreateTable</code> permission to create a table with tags.
         * </p>
         * </note>
         * 
         * @param tags
         *        A map of user-defined tags that you would like to apply to the table that you are creating. A tag is a
         *        key-value pair that you apply to your resources. Tags can help you organize, track costs for, and
         *        control access to resources. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/tagging.html">Tagging for cost allocation
         *        or attribute-based access control (ABAC)</a>.</p> <note>
         *        <p>
         *        You must have the <code>s3tables:TagResource</code> permission in addition to
         *        <code>s3tables:CreateTable</code> permission to create a table with tags.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends S3TablesRequest.BuilderImpl implements Builder {
        private String tableBucketARN;

        private String namespace;

        private String name;

        private String format;

        private TableMetadata metadata;

        private EncryptionConfiguration encryptionConfiguration;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateTableRequest model) {
            super(model);
            tableBucketARN(model.tableBucketARN);
            namespace(model.namespace);
            name(model.name);
            format(model.format);
            metadata(model.metadata);
            encryptionConfiguration(model.encryptionConfiguration);
            tags(model.tags);
        }

        public final String getTableBucketARN() {
            return tableBucketARN;
        }

        public final void setTableBucketARN(String tableBucketARN) {
            this.tableBucketARN = tableBucketARN;
        }

        @Override
        public final Builder tableBucketARN(String tableBucketARN) {
            this.tableBucketARN = tableBucketARN;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(OpenTableFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final TableMetadata.Builder getMetadata() {
            return metadata != null ? metadata.toBuilder() : null;
        }

        public final void setMetadata(TableMetadata.BuilderImpl metadata) {
            this.metadata = metadata != null ? metadata.build() : null;
        }

        @Override
        public final Builder metadata(TableMetadata metadata) {
            this.metadata = metadata;
            return this;
        }

        public final EncryptionConfiguration.Builder getEncryptionConfiguration() {
            return encryptionConfiguration != null ? encryptionConfiguration.toBuilder() : null;
        }

        public final void setEncryptionConfiguration(EncryptionConfiguration.BuilderImpl encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration != null ? encryptionConfiguration.build() : null;
        }

        @Override
        public final Builder encryptionConfiguration(EncryptionConfiguration encryptionConfiguration) {
            this.encryptionConfiguration = encryptionConfiguration;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateTableRequest build() {
            return new CreateTableRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
