/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3vectors;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.s3vectors.internal.S3VectorsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.s3vectors.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.s3vectors.model.AccessDeniedException;
import software.amazon.awssdk.services.s3vectors.model.ConflictException;
import software.amazon.awssdk.services.s3vectors.model.CreateIndexRequest;
import software.amazon.awssdk.services.s3vectors.model.CreateIndexResponse;
import software.amazon.awssdk.services.s3vectors.model.CreateVectorBucketRequest;
import software.amazon.awssdk.services.s3vectors.model.CreateVectorBucketResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteIndexRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteIndexResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketPolicyRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketPolicyResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorBucketResponse;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.DeleteVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.GetIndexRequest;
import software.amazon.awssdk.services.s3vectors.model.GetIndexResponse;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketPolicyRequest;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketPolicyResponse;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketRequest;
import software.amazon.awssdk.services.s3vectors.model.GetVectorBucketResponse;
import software.amazon.awssdk.services.s3vectors.model.GetVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.GetVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.InternalServerException;
import software.amazon.awssdk.services.s3vectors.model.KmsDisabledException;
import software.amazon.awssdk.services.s3vectors.model.KmsInvalidKeyUsageException;
import software.amazon.awssdk.services.s3vectors.model.KmsInvalidStateException;
import software.amazon.awssdk.services.s3vectors.model.KmsNotFoundException;
import software.amazon.awssdk.services.s3vectors.model.ListIndexesRequest;
import software.amazon.awssdk.services.s3vectors.model.ListIndexesResponse;
import software.amazon.awssdk.services.s3vectors.model.ListVectorBucketsRequest;
import software.amazon.awssdk.services.s3vectors.model.ListVectorBucketsResponse;
import software.amazon.awssdk.services.s3vectors.model.ListVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.ListVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.NotFoundException;
import software.amazon.awssdk.services.s3vectors.model.PutVectorBucketPolicyRequest;
import software.amazon.awssdk.services.s3vectors.model.PutVectorBucketPolicyResponse;
import software.amazon.awssdk.services.s3vectors.model.PutVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.PutVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.QueryVectorsRequest;
import software.amazon.awssdk.services.s3vectors.model.QueryVectorsResponse;
import software.amazon.awssdk.services.s3vectors.model.RequestTimeoutException;
import software.amazon.awssdk.services.s3vectors.model.S3VectorsException;
import software.amazon.awssdk.services.s3vectors.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.s3vectors.model.ServiceUnavailableException;
import software.amazon.awssdk.services.s3vectors.model.TooManyRequestsException;
import software.amazon.awssdk.services.s3vectors.model.ValidationException;
import software.amazon.awssdk.services.s3vectors.transform.CreateIndexRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.CreateVectorBucketRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteIndexRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteVectorBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteVectorBucketRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.DeleteVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetIndexRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetVectorBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetVectorBucketRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.GetVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.ListIndexesRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.ListVectorBucketsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.ListVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.PutVectorBucketPolicyRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.PutVectorsRequestMarshaller;
import software.amazon.awssdk.services.s3vectors.transform.QueryVectorsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link S3VectorsAsyncClient}.
 *
 * @see S3VectorsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultS3VectorsAsyncClient implements S3VectorsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultS3VectorsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultS3VectorsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "S3Vectors" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Creates a vector index within a vector bucket. To specify the vector bucket, you must use either the vector
     * bucket name or the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:CreateIndex</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param createIndexRequest
     * @return A Java Future containing the result of the CreateIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>ConflictException The request failed because a vector bucket name or a vector index name already
     *         exists. Vector bucket names must be unique within your Amazon Web Services account for each Amazon Web
     *         Services Region. Vector index names must be unique within your vector bucket. Choose a different vector
     *         bucket name or vector index name, and try again.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds a service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.CreateIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/CreateIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIndexResponse> createIndex(CreateIndexRequest createIndexRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateIndexResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIndexRequest, CreateIndexResponse>()
                            .withOperationName("CreateIndex").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createIndexRequest));
            CompletableFuture<CreateIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Creates a vector bucket in the Amazon Web Services Region that you want your bucket to be in.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:CreateVectorBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param createVectorBucketRequest
     * @return A Java Future containing the result of the CreateVectorBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>ConflictException The request failed because a vector bucket name or a vector index name already
     *         exists. Vector bucket names must be unique within your Amazon Web Services account for each Amazon Web
     *         Services Region. Vector index names must be unique within your vector bucket. Choose a different vector
     *         bucket name or vector index name, and try again.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds a service quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.CreateVectorBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/CreateVectorBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVectorBucketResponse> createVectorBucket(CreateVectorBucketRequest createVectorBucketRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVectorBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVectorBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVectorBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVectorBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVectorBucketResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateVectorBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVectorBucketRequest, CreateVectorBucketResponse>()
                            .withOperationName("CreateVectorBucket").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateVectorBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createVectorBucketRequest));
            CompletableFuture<CreateVectorBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes a vector index. To specify the vector index, you can either use both the vector bucket name and vector
     * index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteIndex</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteIndexRequest
     * @return A Java Future containing the result of the DeleteIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.DeleteIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIndexResponse> deleteIndex(DeleteIndexRequest deleteIndexRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteIndexResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIndexRequest, DeleteIndexResponse>()
                            .withOperationName("DeleteIndex").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteIndexRequest));
            CompletableFuture<DeleteIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes a vector bucket. All vector indexes in the vector bucket must be deleted before the vector bucket can be
     * deleted. To perform this operation, you must use either the vector bucket name or the vector bucket Amazon
     * Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteVectorBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteVectorBucketRequest
     * @return A Java Future containing the result of the DeleteVectorBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>ConflictException The request failed because a vector bucket name or a vector index name already
     *         exists. Vector bucket names must be unique within your Amazon Web Services account for each Amazon Web
     *         Services Region. Vector index names must be unique within your vector bucket. Choose a different vector
     *         bucket name or vector index name, and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.DeleteVectorBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteVectorBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVectorBucketResponse> deleteVectorBucket(DeleteVectorBucketRequest deleteVectorBucketRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVectorBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVectorBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVectorBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVectorBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVectorBucketResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteVectorBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVectorBucketRequest, DeleteVectorBucketResponse>()
                            .withOperationName("DeleteVectorBucket").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVectorBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVectorBucketRequest));
            CompletableFuture<DeleteVectorBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes a vector bucket policy. To specify the bucket, you must use either the vector bucket name or the vector
     * bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteVectorBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteVectorBucketPolicyRequest
     * @return A Java Future containing the result of the DeleteVectorBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.DeleteVectorBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteVectorBucketPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVectorBucketPolicyResponse> deleteVectorBucketPolicy(
            DeleteVectorBucketPolicyRequest deleteVectorBucketPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVectorBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVectorBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVectorBucketPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVectorBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteVectorBucketPolicyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteVectorBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVectorBucketPolicyRequest, DeleteVectorBucketPolicyResponse>()
                            .withOperationName("DeleteVectorBucketPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVectorBucketPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVectorBucketPolicyRequest));
            CompletableFuture<DeleteVectorBucketPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Deletes one or more vectors in a vector index. To specify the vector index, you can either use both the vector
     * bucket name and vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:DeleteVectors</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param deleteVectorsRequest
     * @return A Java Future containing the result of the DeleteVectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>KmsInvalidKeyUsageException The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>KmsInvalidStateException The key state of the KMS key isn't compatible with the operation.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>KmsNotFoundException The KMS key can't be found.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>KmsDisabledException The specified Amazon Web Services KMS key isn't enabled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.DeleteVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/DeleteVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteVectorsResponse> deleteVectors(DeleteVectorsRequest deleteVectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteVectorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteVectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteVectorsRequest, DeleteVectorsResponse>()
                            .withOperationName("DeleteVectors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteVectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteVectorsRequest));
            CompletableFuture<DeleteVectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns vector index attributes. To specify the vector index, you can either use both the vector bucket name and
     * the vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetIndex</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getIndexRequest
     * @return A Java Future containing the result of the GetIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.GetIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetIndexResponse> getIndex(GetIndexRequest getIndexRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getIndexRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetIndexResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIndexRequest, GetIndexResponse>().withOperationName("GetIndex")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetIndexRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getIndexRequest));
            CompletableFuture<GetIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns vector bucket attributes. To specify the bucket, you must use either the vector bucket name or the vector
     * bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetVectorBucket</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getVectorBucketRequest
     * @return A Java Future containing the result of the GetVectorBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.GetVectorBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetVectorBucket" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetVectorBucketResponse> getVectorBucket(GetVectorBucketRequest getVectorBucketRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVectorBucketRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVectorBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVectorBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVectorBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetVectorBucketResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetVectorBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVectorBucketRequest, GetVectorBucketResponse>()
                            .withOperationName("GetVectorBucket").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetVectorBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getVectorBucketRequest));
            CompletableFuture<GetVectorBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Gets details about a vector bucket policy. To specify the bucket, you must use either the vector bucket name or
     * the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetVectorBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getVectorBucketPolicyRequest
     * @return A Java Future containing the result of the GetVectorBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.GetVectorBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetVectorBucketPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetVectorBucketPolicyResponse> getVectorBucketPolicy(
            GetVectorBucketPolicyRequest getVectorBucketPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVectorBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVectorBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVectorBucketPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVectorBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetVectorBucketPolicyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetVectorBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVectorBucketPolicyRequest, GetVectorBucketPolicyResponse>()
                            .withOperationName("GetVectorBucketPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetVectorBucketPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getVectorBucketPolicyRequest));
            CompletableFuture<GetVectorBucketPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns vector attributes. To specify the vector index, you can either use both the vector bucket name and the
     * vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:GetVectors</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param getVectorsRequest
     * @return A Java Future containing the result of the GetVectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>KmsInvalidKeyUsageException The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>KmsInvalidStateException The key state of the KMS key isn't compatible with the operation.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>KmsNotFoundException The KMS key can't be found.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>KmsDisabledException The specified Amazon Web Services KMS key isn't enabled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.GetVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/GetVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetVectorsResponse> getVectors(GetVectorsRequest getVectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetVectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetVectorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetVectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVectorsRequest, GetVectorsResponse>().withOperationName("GetVectors")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetVectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getVectorsRequest));
            CompletableFuture<GetVectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns a list of all the vector indexes within the specified vector bucket. To specify the bucket, you must use
     * either the vector bucket name or the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:ListIndexes</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param listIndexesRequest
     * @return A Java Future containing the result of the ListIndexes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.ListIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/ListIndexes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListIndexesResponse> listIndexes(ListIndexesRequest listIndexesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listIndexesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIndexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIndexes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIndexesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListIndexesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListIndexesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIndexesRequest, ListIndexesResponse>()
                            .withOperationName("ListIndexes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListIndexesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listIndexesRequest));
            CompletableFuture<ListIndexesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Returns a list of all the vector buckets that are owned by the authenticated sender of the request.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:ListVectorBuckets</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param listVectorBucketsRequest
     * @return A Java Future containing the result of the ListVectorBuckets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.ListVectorBuckets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/ListVectorBuckets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListVectorBucketsResponse> listVectorBuckets(ListVectorBucketsRequest listVectorBucketsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVectorBucketsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVectorBucketsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVectorBuckets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVectorBucketsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListVectorBucketsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListVectorBucketsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVectorBucketsRequest, ListVectorBucketsResponse>()
                            .withOperationName("ListVectorBuckets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVectorBucketsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVectorBucketsRequest));
            CompletableFuture<ListVectorBucketsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * List vectors in the specified vector index. To specify the vector index, you can either use both the vector
     * bucket name and the vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <p>
     * <code>ListVectors</code> operations proceed sequentially; however, for faster performance on a large number of
     * vectors in a vector index, applications can request a parallel <code>ListVectors</code> operation by providing
     * the <code>segmentCount</code> and <code>segmentIndex</code> parameters.
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:ListVectors</code> permission to use this operation. Additional permissions are
     * required based on the request parameters you specify:
     * </p>
     * <ul>
     * <li>
     * <p>
     * With only <code>s3vectors:ListVectors</code> permission, you can list vector keys when <code>returnData</code>
     * and <code>returnMetadata</code> are both set to false or not specified..
     * </p>
     * </li>
     * <li>
     * <p>
     * If you set <code>returnData</code> or <code>returnMetadata</code> to true, you must have both
     * <code>s3vectors:ListVectors</code> and <code>s3vectors:GetVectors</code> permissions. The request fails with a
     * <code>403 Forbidden</code> error if you request vector data or metadata without the
     * <code>s3vectors:GetVectors</code> permission.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * </p>
     *
     * @param listVectorsRequest
     * @return A Java Future containing the result of the ListVectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.ListVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/ListVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListVectorsResponse> listVectors(ListVectorsRequest listVectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListVectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListVectorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListVectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListVectorsRequest, ListVectorsResponse>()
                            .withOperationName("ListVectors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListVectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listVectorsRequest));
            CompletableFuture<ListVectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Creates a bucket policy for a vector bucket. To specify the bucket, you must use either the vector bucket name or
     * the vector bucket Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:PutVectorBucketPolicy</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param putVectorBucketPolicyRequest
     * @return A Java Future containing the result of the PutVectorBucketPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.PutVectorBucketPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/PutVectorBucketPolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutVectorBucketPolicyResponse> putVectorBucketPolicy(
            PutVectorBucketPolicyRequest putVectorBucketPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putVectorBucketPolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putVectorBucketPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutVectorBucketPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutVectorBucketPolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutVectorBucketPolicyResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutVectorBucketPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutVectorBucketPolicyRequest, PutVectorBucketPolicyResponse>()
                            .withOperationName("PutVectorBucketPolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutVectorBucketPolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putVectorBucketPolicyRequest));
            CompletableFuture<PutVectorBucketPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Adds one or more vectors to a vector index. To specify the vector index, you can either use both the vector
     * bucket name and the vector index name, or use the vector index Amazon Resource Name (ARN).
     * </p>
     * <p>
     * For more information about limits, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-limitations.html">Limitations and
     * restrictions</a> in the <i>Amazon S3 User Guide</i>.
     * </p>
     * <note>
     * <p>
     * When inserting vector data into your vector index, you must provide the vector data as <code>float32</code>
     * (32-bit floating point) values. If you pass higher-precision values to an Amazon Web Services SDK, S3 Vectors
     * converts the values to 32-bit floating point before storing them, and <code>GetVectors</code>,
     * <code>ListVectors</code>, and <code>QueryVectors</code> operations return the float32 values. Different Amazon
     * Web Services SDKs may have different default numeric types, so ensure your vectors are properly formatted as
     * <code>float32</code> values regardless of which SDK you're using. For example, in Python, use
     * <code>numpy.float32</code> or explicitly cast your values.
     * </p>
     * </note>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:PutVectors</code> permission to use this operation.
     * </p>
     * </dd>
     * </dl>
     * </p>
     *
     * @param putVectorsRequest
     * @return A Java Future containing the result of the PutVectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>KmsInvalidKeyUsageException The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>KmsInvalidStateException The key state of the KMS key isn't compatible with the operation.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>KmsNotFoundException The KMS key can't be found.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>ServiceQuotaExceededException Your request exceeds a service quota.</li>
     *         <li>KmsDisabledException The specified Amazon Web Services KMS key isn't enabled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.PutVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/PutVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutVectorsResponse> putVectors(PutVectorsRequest putVectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutVectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutVectorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<PutVectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutVectorsRequest, PutVectorsResponse>().withOperationName("PutVectors")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutVectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putVectorsRequest));
            CompletableFuture<PutVectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <note>
     * <p>
     * Amazon S3 Vectors is in preview release for Amazon S3 and is subject to change.
     * </p>
     * </note>
     * <p>
     * Performs an approximate nearest neighbor search query in a vector index using a query vector. By default, it
     * returns the keys of approximate nearest neighbors. You can optionally include the computed distance (between the
     * query vector and each vector in the response), the vector data, and metadata of each vector in the response.
     * </p>
     * <p>
     * To specify the vector index, you can either use both the vector bucket name and the vector index name, or use the
     * vector index Amazon Resource Name (ARN).
     * </p>
     * <dl>
     * <dt>Permissions</dt>
     * <dd>
     * <p>
     * You must have the <code>s3vectors:QueryVectors</code> permission to use this operation. Additional permissions
     * are required based on the request parameters you specify:
     * </p>
     * <ul>
     * <li>
     * <p>
     * With only <code>s3vectors:QueryVectors</code> permission, you can retrieve vector keys of approximate nearest
     * neighbors and computed distances between these vectors. This permission is sufficient only when you don't set any
     * metadata filters and don't request vector data or metadata (by keeping the <code>returnMetadata</code> parameter
     * set to <code>false</code> or not specified).
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify a metadata filter or set <code>returnMetadata</code> to true, you must have both
     * <code>s3vectors:QueryVectors</code> and <code>s3vectors:GetVectors</code> permissions. The request fails with a
     * <code>403 Forbidden error</code> if you request metadata filtering, vector data, or metadata without the
     * <code>s3vectors:GetVectors</code> permission.
     * </p>
     * </li>
     * </ul>
     * </dd>
     * </dl>
     * </p>
     *
     * @param queryVectorsRequest
     * @return A Java Future containing the result of the QueryVectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The requested action isn't valid.</li>
     *         <li>ServiceUnavailableException The service is unavailable. Wait briefly and retry your request. If it
     *         continues to fail, increase your waiting time between retries.</li>
     *         <li>TooManyRequestsException The request was denied due to request throttling.</li>
     *         <li>KmsInvalidKeyUsageException The request was rejected for one of the following reasons: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The <code>KeyUsage</code> value of the KMS key is incompatible with the API operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The encryption algorithm or signing algorithm specified for the operation is incompatible with the type
     *         of key material in the KMS key (<code>KeySpec</code>).
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >InvalidKeyUsageException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>InternalServerException The request failed due to an internal server error.</li>
     *         <li>KmsInvalidStateException The key state of the KMS key isn't compatible with the operation.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html#API_Encrypt_Errors"
     *         >KMSInvalidStateException</a> in the <i>Amazon Web Services Key Management Service API Reference</i>.</li>
     *         <li>AccessDeniedException Access denied.</li>
     *         <li>KmsNotFoundException The KMS key can't be found.</li>
     *         <li>RequestTimeoutException The request timed out. Retry your request.</li>
     *         <li>NotFoundException The request was rejected because the specified resource can't be found.</li>
     *         <li>KmsDisabledException The specified Amazon Web Services KMS key isn't enabled.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>S3VectorsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample S3VectorsAsyncClient.QueryVectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/s3vectors-2025-07-15/QueryVectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<QueryVectorsResponse> queryVectors(QueryVectorsRequest queryVectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(queryVectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryVectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "S3Vectors");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "QueryVectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<QueryVectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    QueryVectorsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "AccessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "KmsInvalidKeyUsageException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidKeyUsageException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidKeyUsageException::builder).build());
                case "RequestTimeoutException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("RequestTimeoutException").httpStatusCode(408)
                            .exceptionBuilderSupplier(RequestTimeoutException::builder).build());
                case "KmsInvalidStateException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsInvalidStateException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsInvalidStateException::builder).build());
                case "KmsNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsNotFoundException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsNotFoundException::builder).build());
                case "ValidationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException")
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                case "NotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(NotFoundException::builder).build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "KmsDisabledException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("KmsDisabledException").httpStatusCode(400)
                            .exceptionBuilderSupplier(KmsDisabledException::builder).build());
                case "ServiceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "InternalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "TooManyRequestsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyRequestsException").httpStatusCode(429)
                            .exceptionBuilderSupplier(TooManyRequestsException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<QueryVectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<QueryVectorsRequest, QueryVectorsResponse>()
                            .withOperationName("QueryVectors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new QueryVectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(queryVectorsRequest));
            CompletableFuture<QueryVectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final S3VectorsServiceClientConfiguration serviceClientConfiguration() {
        return new S3VectorsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(S3VectorsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        S3VectorsServiceClientConfigurationBuilder serviceConfigBuilder = new S3VectorsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
