/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAlgorithmRequest extends SageMakerRequest implements
        ToCopyableBuilder<CreateAlgorithmRequest.Builder, CreateAlgorithmRequest> {
    private static final SdkField<String> ALGORITHM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAlgorithmRequest::algorithmName)).setter(setter(Builder::algorithmName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmName").build()).build();

    private static final SdkField<String> ALGORITHM_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAlgorithmRequest::algorithmDescription)).setter(setter(Builder::algorithmDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlgorithmDescription").build())
            .build();

    private static final SdkField<TrainingSpecification> TRAINING_SPECIFICATION_FIELD = SdkField
            .<TrainingSpecification> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateAlgorithmRequest::trainingSpecification)).setter(setter(Builder::trainingSpecification))
            .constructor(TrainingSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingSpecification").build())
            .build();

    private static final SdkField<InferenceSpecification> INFERENCE_SPECIFICATION_FIELD = SdkField
            .<InferenceSpecification> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateAlgorithmRequest::inferenceSpecification)).setter(setter(Builder::inferenceSpecification))
            .constructor(InferenceSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InferenceSpecification").build())
            .build();

    private static final SdkField<AlgorithmValidationSpecification> VALIDATION_SPECIFICATION_FIELD = SdkField
            .<AlgorithmValidationSpecification> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateAlgorithmRequest::validationSpecification)).setter(setter(Builder::validationSpecification))
            .constructor(AlgorithmValidationSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationSpecification").build())
            .build();

    private static final SdkField<Boolean> CERTIFY_FOR_MARKETPLACE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateAlgorithmRequest::certifyForMarketplace)).setter(setter(Builder::certifyForMarketplace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertifyForMarketplace").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_NAME_FIELD,
            ALGORITHM_DESCRIPTION_FIELD, TRAINING_SPECIFICATION_FIELD, INFERENCE_SPECIFICATION_FIELD,
            VALIDATION_SPECIFICATION_FIELD, CERTIFY_FOR_MARKETPLACE_FIELD));

    private final String algorithmName;

    private final String algorithmDescription;

    private final TrainingSpecification trainingSpecification;

    private final InferenceSpecification inferenceSpecification;

    private final AlgorithmValidationSpecification validationSpecification;

    private final Boolean certifyForMarketplace;

    private CreateAlgorithmRequest(BuilderImpl builder) {
        super(builder);
        this.algorithmName = builder.algorithmName;
        this.algorithmDescription = builder.algorithmDescription;
        this.trainingSpecification = builder.trainingSpecification;
        this.inferenceSpecification = builder.inferenceSpecification;
        this.validationSpecification = builder.validationSpecification;
        this.certifyForMarketplace = builder.certifyForMarketplace;
    }

    /**
     * <p>
     * The name of the algorithm.
     * </p>
     * 
     * @return The name of the algorithm.
     */
    public String algorithmName() {
        return algorithmName;
    }

    /**
     * <p>
     * A description of the algorithm.
     * </p>
     * 
     * @return A description of the algorithm.
     */
    public String algorithmDescription() {
        return algorithmDescription;
    }

    /**
     * <p>
     * Specifies details about training jobs run by this algorithm, including the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Amazon ECR path of the container and the version digest of the algorithm.
     * </p>
     * </li>
     * <li>
     * <p>
     * The hyperparameters that the algorithm supports.
     * </p>
     * </li>
     * <li>
     * <p>
     * The instance types that the algorithm supports for training.
     * </p>
     * </li>
     * <li>
     * <p>
     * Whether the algorithm supports distributed training.
     * </p>
     * </li>
     * <li>
     * <p>
     * The metrics that the algorithm emits to Amazon CloudWatch.
     * </p>
     * </li>
     * <li>
     * <p>
     * Which metrics that the algorithm emits can be used as the objective metric for hyperparameter tuning jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * The input channels that the algorithm supports for training data. For example, an algorithm might support
     * <code>train</code>, <code>validation</code>, and <code>test</code> channels.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies details about training jobs run by this algorithm, including the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The Amazon ECR path of the container and the version digest of the algorithm.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The hyperparameters that the algorithm supports.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The instance types that the algorithm supports for training.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Whether the algorithm supports distributed training.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The metrics that the algorithm emits to Amazon CloudWatch.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Which metrics that the algorithm emits can be used as the objective metric for hyperparameter tuning
     *         jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The input channels that the algorithm supports for training data. For example, an algorithm might support
     *         <code>train</code>, <code>validation</code>, and <code>test</code> channels.
     *         </p>
     *         </li>
     */
    public TrainingSpecification trainingSpecification() {
        return trainingSpecification;
    }

    /**
     * <p>
     * Specifies details about inference jobs that the algorithm runs, including the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The Amazon ECR paths of containers that contain the inference code and model artifacts.
     * </p>
     * </li>
     * <li>
     * <p>
     * The instance types that the algorithm supports for transform jobs and real-time endpoints used for inference.
     * </p>
     * </li>
     * <li>
     * <p>
     * The input and output content formats that the algorithm supports for inference.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies details about inference jobs that the algorithm runs, including the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The Amazon ECR paths of containers that contain the inference code and model artifacts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The instance types that the algorithm supports for transform jobs and real-time endpoints used for
     *         inference.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The input and output content formats that the algorithm supports for inference.
     *         </p>
     *         </li>
     */
    public InferenceSpecification inferenceSpecification() {
        return inferenceSpecification;
    }

    /**
     * <p>
     * Specifies configurations for one or more training jobs and that Amazon SageMaker runs to test the algorithm's
     * training code and, optionally, one or more batch transform jobs that Amazon SageMaker runs to test the
     * algorithm's inference code.
     * </p>
     * 
     * @return Specifies configurations for one or more training jobs and that Amazon SageMaker runs to test the
     *         algorithm's training code and, optionally, one or more batch transform jobs that Amazon SageMaker runs to
     *         test the algorithm's inference code.
     */
    public AlgorithmValidationSpecification validationSpecification() {
        return validationSpecification;
    }

    /**
     * <p>
     * Whether to certify the algorithm so that it can be listed in AWS Marketplace.
     * </p>
     * 
     * @return Whether to certify the algorithm so that it can be listed in AWS Marketplace.
     */
    public Boolean certifyForMarketplace() {
        return certifyForMarketplace;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(algorithmName());
        hashCode = 31 * hashCode + Objects.hashCode(algorithmDescription());
        hashCode = 31 * hashCode + Objects.hashCode(trainingSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(inferenceSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(validationSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(certifyForMarketplace());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAlgorithmRequest)) {
            return false;
        }
        CreateAlgorithmRequest other = (CreateAlgorithmRequest) obj;
        return Objects.equals(algorithmName(), other.algorithmName())
                && Objects.equals(algorithmDescription(), other.algorithmDescription())
                && Objects.equals(trainingSpecification(), other.trainingSpecification())
                && Objects.equals(inferenceSpecification(), other.inferenceSpecification())
                && Objects.equals(validationSpecification(), other.validationSpecification())
                && Objects.equals(certifyForMarketplace(), other.certifyForMarketplace());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateAlgorithmRequest").add("AlgorithmName", algorithmName())
                .add("AlgorithmDescription", algorithmDescription()).add("TrainingSpecification", trainingSpecification())
                .add("InferenceSpecification", inferenceSpecification())
                .add("ValidationSpecification", validationSpecification()).add("CertifyForMarketplace", certifyForMarketplace())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AlgorithmName":
            return Optional.ofNullable(clazz.cast(algorithmName()));
        case "AlgorithmDescription":
            return Optional.ofNullable(clazz.cast(algorithmDescription()));
        case "TrainingSpecification":
            return Optional.ofNullable(clazz.cast(trainingSpecification()));
        case "InferenceSpecification":
            return Optional.ofNullable(clazz.cast(inferenceSpecification()));
        case "ValidationSpecification":
            return Optional.ofNullable(clazz.cast(validationSpecification()));
        case "CertifyForMarketplace":
            return Optional.ofNullable(clazz.cast(certifyForMarketplace()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateAlgorithmRequest, T> g) {
        return obj -> g.apply((CreateAlgorithmRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SageMakerRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateAlgorithmRequest> {
        /**
         * <p>
         * The name of the algorithm.
         * </p>
         * 
         * @param algorithmName
         *        The name of the algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmName(String algorithmName);

        /**
         * <p>
         * A description of the algorithm.
         * </p>
         * 
         * @param algorithmDescription
         *        A description of the algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithmDescription(String algorithmDescription);

        /**
         * <p>
         * Specifies details about training jobs run by this algorithm, including the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The Amazon ECR path of the container and the version digest of the algorithm.
         * </p>
         * </li>
         * <li>
         * <p>
         * The hyperparameters that the algorithm supports.
         * </p>
         * </li>
         * <li>
         * <p>
         * The instance types that the algorithm supports for training.
         * </p>
         * </li>
         * <li>
         * <p>
         * Whether the algorithm supports distributed training.
         * </p>
         * </li>
         * <li>
         * <p>
         * The metrics that the algorithm emits to Amazon CloudWatch.
         * </p>
         * </li>
         * <li>
         * <p>
         * Which metrics that the algorithm emits can be used as the objective metric for hyperparameter tuning jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * The input channels that the algorithm supports for training data. For example, an algorithm might support
         * <code>train</code>, <code>validation</code>, and <code>test</code> channels.
         * </p>
         * </li>
         * </ul>
         * 
         * @param trainingSpecification
         *        Specifies details about training jobs run by this algorithm, including the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The Amazon ECR path of the container and the version digest of the algorithm.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The hyperparameters that the algorithm supports.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The instance types that the algorithm supports for training.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Whether the algorithm supports distributed training.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The metrics that the algorithm emits to Amazon CloudWatch.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Which metrics that the algorithm emits can be used as the objective metric for hyperparameter tuning
         *        jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The input channels that the algorithm supports for training data. For example, an algorithm might
         *        support <code>train</code>, <code>validation</code>, and <code>test</code> channels.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingSpecification(TrainingSpecification trainingSpecification);

        /**
         * <p>
         * Specifies details about training jobs run by this algorithm, including the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The Amazon ECR path of the container and the version digest of the algorithm.
         * </p>
         * </li>
         * <li>
         * <p>
         * The hyperparameters that the algorithm supports.
         * </p>
         * </li>
         * <li>
         * <p>
         * The instance types that the algorithm supports for training.
         * </p>
         * </li>
         * <li>
         * <p>
         * Whether the algorithm supports distributed training.
         * </p>
         * </li>
         * <li>
         * <p>
         * The metrics that the algorithm emits to Amazon CloudWatch.
         * </p>
         * </li>
         * <li>
         * <p>
         * Which metrics that the algorithm emits can be used as the objective metric for hyperparameter tuning jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * The input channels that the algorithm supports for training data. For example, an algorithm might support
         * <code>train</code>, <code>validation</code>, and <code>test</code> channels.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link TrainingSpecification.Builder} avoiding the need
         * to create one manually via {@link TrainingSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link TrainingSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #trainingSpecification(TrainingSpecification)}.
         * 
         * @param trainingSpecification
         *        a consumer that will call methods on {@link TrainingSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trainingSpecification(TrainingSpecification)
         */
        default Builder trainingSpecification(Consumer<TrainingSpecification.Builder> trainingSpecification) {
            return trainingSpecification(TrainingSpecification.builder().applyMutation(trainingSpecification).build());
        }

        /**
         * <p>
         * Specifies details about inference jobs that the algorithm runs, including the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The Amazon ECR paths of containers that contain the inference code and model artifacts.
         * </p>
         * </li>
         * <li>
         * <p>
         * The instance types that the algorithm supports for transform jobs and real-time endpoints used for inference.
         * </p>
         * </li>
         * <li>
         * <p>
         * The input and output content formats that the algorithm supports for inference.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferenceSpecification
         *        Specifies details about inference jobs that the algorithm runs, including the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The Amazon ECR paths of containers that contain the inference code and model artifacts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The instance types that the algorithm supports for transform jobs and real-time endpoints used for
         *        inference.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The input and output content formats that the algorithm supports for inference.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceSpecification(InferenceSpecification inferenceSpecification);

        /**
         * <p>
         * Specifies details about inference jobs that the algorithm runs, including the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The Amazon ECR paths of containers that contain the inference code and model artifacts.
         * </p>
         * </li>
         * <li>
         * <p>
         * The instance types that the algorithm supports for transform jobs and real-time endpoints used for inference.
         * </p>
         * </li>
         * <li>
         * <p>
         * The input and output content formats that the algorithm supports for inference.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link InferenceSpecification.Builder} avoiding the
         * need to create one manually via {@link InferenceSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link InferenceSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #inferenceSpecification(InferenceSpecification)}.
         * 
         * @param inferenceSpecification
         *        a consumer that will call methods on {@link InferenceSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inferenceSpecification(InferenceSpecification)
         */
        default Builder inferenceSpecification(Consumer<InferenceSpecification.Builder> inferenceSpecification) {
            return inferenceSpecification(InferenceSpecification.builder().applyMutation(inferenceSpecification).build());
        }

        /**
         * <p>
         * Specifies configurations for one or more training jobs and that Amazon SageMaker runs to test the algorithm's
         * training code and, optionally, one or more batch transform jobs that Amazon SageMaker runs to test the
         * algorithm's inference code.
         * </p>
         * 
         * @param validationSpecification
         *        Specifies configurations for one or more training jobs and that Amazon SageMaker runs to test the
         *        algorithm's training code and, optionally, one or more batch transform jobs that Amazon SageMaker runs
         *        to test the algorithm's inference code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationSpecification(AlgorithmValidationSpecification validationSpecification);

        /**
         * <p>
         * Specifies configurations for one or more training jobs and that Amazon SageMaker runs to test the algorithm's
         * training code and, optionally, one or more batch transform jobs that Amazon SageMaker runs to test the
         * algorithm's inference code.
         * </p>
         * This is a convenience that creates an instance of the {@link AlgorithmValidationSpecification.Builder}
         * avoiding the need to create one manually via {@link AlgorithmValidationSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link AlgorithmValidationSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #validationSpecification(AlgorithmValidationSpecification)}.
         * 
         * @param validationSpecification
         *        a consumer that will call methods on {@link AlgorithmValidationSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationSpecification(AlgorithmValidationSpecification)
         */
        default Builder validationSpecification(Consumer<AlgorithmValidationSpecification.Builder> validationSpecification) {
            return validationSpecification(AlgorithmValidationSpecification.builder().applyMutation(validationSpecification)
                    .build());
        }

        /**
         * <p>
         * Whether to certify the algorithm so that it can be listed in AWS Marketplace.
         * </p>
         * 
         * @param certifyForMarketplace
         *        Whether to certify the algorithm so that it can be listed in AWS Marketplace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certifyForMarketplace(Boolean certifyForMarketplace);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SageMakerRequest.BuilderImpl implements Builder {
        private String algorithmName;

        private String algorithmDescription;

        private TrainingSpecification trainingSpecification;

        private InferenceSpecification inferenceSpecification;

        private AlgorithmValidationSpecification validationSpecification;

        private Boolean certifyForMarketplace;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAlgorithmRequest model) {
            super(model);
            algorithmName(model.algorithmName);
            algorithmDescription(model.algorithmDescription);
            trainingSpecification(model.trainingSpecification);
            inferenceSpecification(model.inferenceSpecification);
            validationSpecification(model.validationSpecification);
            certifyForMarketplace(model.certifyForMarketplace);
        }

        public final String getAlgorithmName() {
            return algorithmName;
        }

        @Override
        public final Builder algorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
            return this;
        }

        public final void setAlgorithmName(String algorithmName) {
            this.algorithmName = algorithmName;
        }

        public final String getAlgorithmDescription() {
            return algorithmDescription;
        }

        @Override
        public final Builder algorithmDescription(String algorithmDescription) {
            this.algorithmDescription = algorithmDescription;
            return this;
        }

        public final void setAlgorithmDescription(String algorithmDescription) {
            this.algorithmDescription = algorithmDescription;
        }

        public final TrainingSpecification.Builder getTrainingSpecification() {
            return trainingSpecification != null ? trainingSpecification.toBuilder() : null;
        }

        @Override
        public final Builder trainingSpecification(TrainingSpecification trainingSpecification) {
            this.trainingSpecification = trainingSpecification;
            return this;
        }

        public final void setTrainingSpecification(TrainingSpecification.BuilderImpl trainingSpecification) {
            this.trainingSpecification = trainingSpecification != null ? trainingSpecification.build() : null;
        }

        public final InferenceSpecification.Builder getInferenceSpecification() {
            return inferenceSpecification != null ? inferenceSpecification.toBuilder() : null;
        }

        @Override
        public final Builder inferenceSpecification(InferenceSpecification inferenceSpecification) {
            this.inferenceSpecification = inferenceSpecification;
            return this;
        }

        public final void setInferenceSpecification(InferenceSpecification.BuilderImpl inferenceSpecification) {
            this.inferenceSpecification = inferenceSpecification != null ? inferenceSpecification.build() : null;
        }

        public final AlgorithmValidationSpecification.Builder getValidationSpecification() {
            return validationSpecification != null ? validationSpecification.toBuilder() : null;
        }

        @Override
        public final Builder validationSpecification(AlgorithmValidationSpecification validationSpecification) {
            this.validationSpecification = validationSpecification;
            return this;
        }

        public final void setValidationSpecification(AlgorithmValidationSpecification.BuilderImpl validationSpecification) {
            this.validationSpecification = validationSpecification != null ? validationSpecification.build() : null;
        }

        public final Boolean getCertifyForMarketplace() {
            return certifyForMarketplace;
        }

        @Override
        public final Builder certifyForMarketplace(Boolean certifyForMarketplace) {
            this.certifyForMarketplace = certifyForMarketplace;
            return this;
        }

        public final void setCertifyForMarketplace(Boolean certifyForMarketplace) {
            this.certifyForMarketplace = certifyForMarketplace;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateAlgorithmRequest build() {
            return new CreateAlgorithmRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
