/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines a list of <code>NestedFilters</code> objects. To satisfy the conditions specified in the
 * <code>NestedFilters</code> call, a resource must satisfy the conditions of all of the filters.
 * </p>
 * <p>
 * For example, you could define a <code>NestedFilters</code> using the training job's <code>InputDataConfig</code>
 * property to filter on <code>Channel</code> objects.
 * </p>
 * <p>
 * A <code>NestedFilters</code> object contains multiple filters. For example, to find all training jobs whose name
 * contains <code>train</code> and that have <code>cat/data</code> in their <code>S3Uri</code> (specified in
 * <code>InputDataConfig</code>), you need to create a <code>NestedFilters</code> object that specifies the
 * <code>InputDataConfig</code> property with the following <code>Filter</code> objects:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>'{Name:"InputDataConfig.ChannelName", "Operator":"EQUALS", "Value":"train"}',</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>'{Name:"InputDataConfig.DataSource.S3DataSource.S3Uri", "Operator":"CONTAINS", "Value":"cat/data"}'</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NestedFilters implements SdkPojo, Serializable, ToCopyableBuilder<NestedFilters.Builder, NestedFilters> {
    private static final SdkField<String> NESTED_PROPERTY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(NestedFilters::nestedPropertyName)).setter(setter(Builder::nestedPropertyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NestedPropertyName").build())
            .build();

    private static final SdkField<List<Filter>> FILTERS_FIELD = SdkField
            .<List<Filter>> builder(MarshallingType.LIST)
            .getter(getter(NestedFilters::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Filter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Filter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NESTED_PROPERTY_NAME_FIELD,
            FILTERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String nestedPropertyName;

    private final List<Filter> filters;

    private NestedFilters(BuilderImpl builder) {
        this.nestedPropertyName = builder.nestedPropertyName;
        this.filters = builder.filters;
    }

    /**
     * <p>
     * The name of the property to use in the nested filters. The value must match a listed property name, such as
     * <code>InputDataConfig</code> .
     * </p>
     * 
     * @return The name of the property to use in the nested filters. The value must match a listed property name, such
     *         as <code>InputDataConfig</code> .
     */
    public String nestedPropertyName() {
        return nestedPropertyName;
    }

    /**
     * <p>
     * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code> value.
     * For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code> parameter
     * of the <code>InputDataConfig</code> property: <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
     *         value. For example, a <code>NestedFilters</code> call might include a filter on the
     *         <code>PropertyName</code> parameter of the <code>InputDataConfig</code> property:
     *         <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
     */
    public List<Filter> filters() {
        return filters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nestedPropertyName());
        hashCode = 31 * hashCode + Objects.hashCode(filters());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NestedFilters)) {
            return false;
        }
        NestedFilters other = (NestedFilters) obj;
        return Objects.equals(nestedPropertyName(), other.nestedPropertyName()) && Objects.equals(filters(), other.filters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("NestedFilters").add("NestedPropertyName", nestedPropertyName()).add("Filters", filters())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NestedPropertyName":
            return Optional.ofNullable(clazz.cast(nestedPropertyName()));
        case "Filters":
            return Optional.ofNullable(clazz.cast(filters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NestedFilters, T> g) {
        return obj -> g.apply((NestedFilters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NestedFilters> {
        /**
         * <p>
         * The name of the property to use in the nested filters. The value must match a listed property name, such as
         * <code>InputDataConfig</code> .
         * </p>
         * 
         * @param nestedPropertyName
         *        The name of the property to use in the nested filters. The value must match a listed property name,
         *        such as <code>InputDataConfig</code> .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nestedPropertyName(String nestedPropertyName);

        /**
         * <p>
         * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
         * value. For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code>
         * parameter of the <code>InputDataConfig</code> property:
         * <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * </p>
         * 
         * @param filters
         *        A list of filters. Each filter acts on a property. Filters must contain at least one
         *        <code>Filters</code> value. For example, a <code>NestedFilters</code> call might include a filter on
         *        the <code>PropertyName</code> parameter of the <code>InputDataConfig</code> property:
         *        <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<Filter> filters);

        /**
         * <p>
         * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
         * value. For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code>
         * parameter of the <code>InputDataConfig</code> property:
         * <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * </p>
         * 
         * @param filters
         *        A list of filters. Each filter acts on a property. Filters must contain at least one
         *        <code>Filters</code> value. For example, a <code>NestedFilters</code> call might include a filter on
         *        the <code>PropertyName</code> parameter of the <code>InputDataConfig</code> property:
         *        <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Filter... filters);

        /**
         * <p>
         * A list of filters. Each filter acts on a property. Filters must contain at least one <code>Filters</code>
         * value. For example, a <code>NestedFilters</code> call might include a filter on the <code>PropertyName</code>
         * parameter of the <code>InputDataConfig</code> property:
         * <code>InputDataConfig.DataSource.S3DataSource.S3Uri</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Filter>.Builder} avoiding the need to
         * create one manually via {@link List<Filter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Filter>.Builder#build()} is called immediately and its
         * result is passed to {@link #filters(List<Filter>)}.
         * 
         * @param filters
         *        a consumer that will call methods on {@link List<Filter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(List<Filter>)
         */
        Builder filters(Consumer<Filter.Builder>... filters);
    }

    static final class BuilderImpl implements Builder {
        private String nestedPropertyName;

        private List<Filter> filters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NestedFilters model) {
            nestedPropertyName(model.nestedPropertyName);
            filters(model.filters);
        }

        public final String getNestedPropertyName() {
            return nestedPropertyName;
        }

        @Override
        public final Builder nestedPropertyName(String nestedPropertyName) {
            this.nestedPropertyName = nestedPropertyName;
            return this;
        }

        public final void setNestedPropertyName(String nestedPropertyName) {
            this.nestedPropertyName = nestedPropertyName;
        }

        public final Collection<Filter.Builder> getFilters() {
            return filters != null ? filters.stream().map(Filter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder filters(Collection<Filter> filters) {
            this.filters = FilterListCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Filter... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<Filter.Builder>... filters) {
            filters(Stream.of(filters).map(c -> Filter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setFilters(Collection<Filter.BuilderImpl> filters) {
            this.filters = FilterListCopier.copyFromBuilder(filters);
        }

        @Override
        public NestedFilters build() {
            return new NestedFilters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
