/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines the input needed to run a training job using the algorithm.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TrainingJobDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<TrainingJobDefinition.Builder, TrainingJobDefinition> {
    private static final SdkField<String> TRAINING_INPUT_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TrainingJobDefinition::trainingInputModeAsString)).setter(setter(Builder::trainingInputMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingInputMode").build()).build();

    private static final SdkField<Map<String, String>> HYPER_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(TrainingJobDefinition::hyperParameters))
            .setter(setter(Builder::hyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HyperParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Channel>> INPUT_DATA_CONFIG_FIELD = SdkField
            .<List<Channel>> builder(MarshallingType.LIST)
            .getter(getter(TrainingJobDefinition::inputDataConfig))
            .setter(setter(Builder::inputDataConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDataConfig").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Channel> builder(MarshallingType.SDK_POJO)
                                            .constructor(Channel::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<OutputDataConfig> OUTPUT_DATA_CONFIG_FIELD = SdkField
            .<OutputDataConfig> builder(MarshallingType.SDK_POJO).getter(getter(TrainingJobDefinition::outputDataConfig))
            .setter(setter(Builder::outputDataConfig)).constructor(OutputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputDataConfig").build()).build();

    private static final SdkField<ResourceConfig> RESOURCE_CONFIG_FIELD = SdkField
            .<ResourceConfig> builder(MarshallingType.SDK_POJO).getter(getter(TrainingJobDefinition::resourceConfig))
            .setter(setter(Builder::resourceConfig)).constructor(ResourceConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceConfig").build()).build();

    private static final SdkField<StoppingCondition> STOPPING_CONDITION_FIELD = SdkField
            .<StoppingCondition> builder(MarshallingType.SDK_POJO).getter(getter(TrainingJobDefinition::stoppingCondition))
            .setter(setter(Builder::stoppingCondition)).constructor(StoppingCondition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StoppingCondition").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRAINING_INPUT_MODE_FIELD,
            HYPER_PARAMETERS_FIELD, INPUT_DATA_CONFIG_FIELD, OUTPUT_DATA_CONFIG_FIELD, RESOURCE_CONFIG_FIELD,
            STOPPING_CONDITION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String trainingInputMode;

    private final Map<String, String> hyperParameters;

    private final List<Channel> inputDataConfig;

    private final OutputDataConfig outputDataConfig;

    private final ResourceConfig resourceConfig;

    private final StoppingCondition stoppingCondition;

    private TrainingJobDefinition(BuilderImpl builder) {
        this.trainingInputMode = builder.trainingInputMode;
        this.hyperParameters = builder.hyperParameters;
        this.inputDataConfig = builder.inputDataConfig;
        this.outputDataConfig = builder.outputDataConfig;
        this.resourceConfig = builder.resourceConfig;
        this.stoppingCondition = builder.stoppingCondition;
    }

    /**
     * <p>
     * The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker algorithms
     * support, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.
     * </p>
     * <p>
     * If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training data from S3
     * to the provisioned ML storage Volume, and mounts the directory to docker volume for training container. If an
     * algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker streams data directly from S3 to the
     * container.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trainingInputMode}
     * will return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trainingInputModeAsString}.
     * </p>
     * 
     * @return The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker
     *         algorithms support, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.</p>
     *         <p>
     *         If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training data
     *         from S3 to the provisioned ML storage Volume, and mounts the directory to docker volume for training
     *         container. If an algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker streams data
     *         directly from S3 to the container.
     * @see TrainingInputMode
     */
    public TrainingInputMode trainingInputMode() {
        return TrainingInputMode.fromValue(trainingInputMode);
    }

    /**
     * <p>
     * The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker algorithms
     * support, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.
     * </p>
     * <p>
     * If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training data from S3
     * to the provisioned ML storage Volume, and mounts the directory to docker volume for training container. If an
     * algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker streams data directly from S3 to the
     * container.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #trainingInputMode}
     * will return {@link TrainingInputMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #trainingInputModeAsString}.
     * </p>
     * 
     * @return The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker
     *         algorithms support, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.</p>
     *         <p>
     *         If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training data
     *         from S3 to the provisioned ML storage Volume, and mounts the directory to docker volume for training
     *         container. If an algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker streams data
     *         directly from S3 to the container.
     * @see TrainingInputMode
     */
    public String trainingInputModeAsString() {
        return trainingInputMode;
    }

    /**
     * Returns true if the HyperParameters property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasHyperParameters() {
        return hyperParameters != null && !(hyperParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The hyperparameters used for the training job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasHyperParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The hyperparameters used for the training job.
     */
    public Map<String, String> hyperParameters() {
        return hyperParameters;
    }

    /**
     * Returns true if the InputDataConfig property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasInputDataConfig() {
        return inputDataConfig != null && !(inputDataConfig instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of <code>Channel</code> objects, each of which specifies an input source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInputDataConfig()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of <code>Channel</code> objects, each of which specifies an input source.
     */
    public List<Channel> inputDataConfig() {
        return inputDataConfig;
    }

    /**
     * <p>
     * the path to the S3 bucket where you want to store model artifacts. Amazon SageMaker creates subfolders for the
     * artifacts.
     * </p>
     * 
     * @return the path to the S3 bucket where you want to store model artifacts. Amazon SageMaker creates subfolders
     *         for the artifacts.
     */
    public OutputDataConfig outputDataConfig() {
        return outputDataConfig;
    }

    /**
     * <p>
     * The resources, including the ML compute instances and ML storage volumes, to use for model training.
     * </p>
     * 
     * @return The resources, including the ML compute instances and ML storage volumes, to use for model training.
     */
    public ResourceConfig resourceConfig() {
        return resourceConfig;
    }

    /**
     * <p>
     * Specifies a limit to how long a model training job can run. When the job reaches the time limit, Amazon SageMaker
     * ends the training job. Use this API to cap model training costs.
     * </p>
     * <p>
     * To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120
     * seconds. Algorithms can use this 120-second window to save the model artifacts.
     * </p>
     * 
     * @return Specifies a limit to how long a model training job can run. When the job reaches the time limit, Amazon
     *         SageMaker ends the training job. Use this API to cap model training costs.</p>
     *         <p>
     *         To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for
     *         120 seconds. Algorithms can use this 120-second window to save the model artifacts.
     */
    public StoppingCondition stoppingCondition() {
        return stoppingCondition;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(trainingInputModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hyperParameters());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(outputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(resourceConfig());
        hashCode = 31 * hashCode + Objects.hashCode(stoppingCondition());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TrainingJobDefinition)) {
            return false;
        }
        TrainingJobDefinition other = (TrainingJobDefinition) obj;
        return Objects.equals(trainingInputModeAsString(), other.trainingInputModeAsString())
                && Objects.equals(hyperParameters(), other.hyperParameters())
                && Objects.equals(inputDataConfig(), other.inputDataConfig())
                && Objects.equals(outputDataConfig(), other.outputDataConfig())
                && Objects.equals(resourceConfig(), other.resourceConfig())
                && Objects.equals(stoppingCondition(), other.stoppingCondition());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TrainingJobDefinition").add("TrainingInputMode", trainingInputModeAsString())
                .add("HyperParameters", hyperParameters()).add("InputDataConfig", inputDataConfig())
                .add("OutputDataConfig", outputDataConfig()).add("ResourceConfig", resourceConfig())
                .add("StoppingCondition", stoppingCondition()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TrainingInputMode":
            return Optional.ofNullable(clazz.cast(trainingInputModeAsString()));
        case "HyperParameters":
            return Optional.ofNullable(clazz.cast(hyperParameters()));
        case "InputDataConfig":
            return Optional.ofNullable(clazz.cast(inputDataConfig()));
        case "OutputDataConfig":
            return Optional.ofNullable(clazz.cast(outputDataConfig()));
        case "ResourceConfig":
            return Optional.ofNullable(clazz.cast(resourceConfig()));
        case "StoppingCondition":
            return Optional.ofNullable(clazz.cast(stoppingCondition()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TrainingJobDefinition, T> g) {
        return obj -> g.apply((TrainingJobDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TrainingJobDefinition> {
        /**
         * <p>
         * The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker
         * algorithms support, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.
         * </p>
         * <p>
         * If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training data from
         * S3 to the provisioned ML storage Volume, and mounts the directory to docker volume for training container. If
         * an algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker streams data directly from S3 to the
         * container.
         * </p>
         * 
         * @param trainingInputMode
         *        The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker
         *        algorithms support, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.</p>
         *        <p>
         *        If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training
         *        data from S3 to the provisioned ML storage Volume, and mounts the directory to docker volume for
         *        training container. If an algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker
         *        streams data directly from S3 to the container.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder trainingInputMode(String trainingInputMode);

        /**
         * <p>
         * The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker
         * algorithms support, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.
         * </p>
         * <p>
         * If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training data from
         * S3 to the provisioned ML storage Volume, and mounts the directory to docker volume for training container. If
         * an algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker streams data directly from S3 to the
         * container.
         * </p>
         * 
         * @param trainingInputMode
         *        The input mode used by the algorithm for the training job. For the input modes that Amazon SageMaker
         *        algorithms support, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/algos.html">Algorithms</a>.</p>
         *        <p>
         *        If an algorithm supports the <code>File</code> input mode, Amazon SageMaker downloads the training
         *        data from S3 to the provisioned ML storage Volume, and mounts the directory to docker volume for
         *        training container. If an algorithm supports the <code>Pipe</code> input mode, Amazon SageMaker
         *        streams data directly from S3 to the container.
         * @see TrainingInputMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TrainingInputMode
         */
        Builder trainingInputMode(TrainingInputMode trainingInputMode);

        /**
         * <p>
         * The hyperparameters used for the training job.
         * </p>
         * 
         * @param hyperParameters
         *        The hyperparameters used for the training job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hyperParameters(Map<String, String> hyperParameters);

        /**
         * <p>
         * An array of <code>Channel</code> objects, each of which specifies an input source.
         * </p>
         * 
         * @param inputDataConfig
         *        An array of <code>Channel</code> objects, each of which specifies an input source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(Collection<Channel> inputDataConfig);

        /**
         * <p>
         * An array of <code>Channel</code> objects, each of which specifies an input source.
         * </p>
         * 
         * @param inputDataConfig
         *        An array of <code>Channel</code> objects, each of which specifies an input source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(Channel... inputDataConfig);

        /**
         * <p>
         * An array of <code>Channel</code> objects, each of which specifies an input source.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Channel>.Builder} avoiding the need to
         * create one manually via {@link List<Channel>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Channel>.Builder#build()} is called immediately and its
         * result is passed to {@link #inputDataConfig(List<Channel>)}.
         * 
         * @param inputDataConfig
         *        a consumer that will call methods on {@link List<Channel>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputDataConfig(List<Channel>)
         */
        Builder inputDataConfig(Consumer<Channel.Builder>... inputDataConfig);

        /**
         * <p>
         * the path to the S3 bucket where you want to store model artifacts. Amazon SageMaker creates subfolders for
         * the artifacts.
         * </p>
         * 
         * @param outputDataConfig
         *        the path to the S3 bucket where you want to store model artifacts. Amazon SageMaker creates subfolders
         *        for the artifacts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputDataConfig(OutputDataConfig outputDataConfig);

        /**
         * <p>
         * the path to the S3 bucket where you want to store model artifacts. Amazon SageMaker creates subfolders for
         * the artifacts.
         * </p>
         * This is a convenience that creates an instance of the {@link OutputDataConfig.Builder} avoiding the need to
         * create one manually via {@link OutputDataConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputDataConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #outputDataConfig(OutputDataConfig)}.
         * 
         * @param outputDataConfig
         *        a consumer that will call methods on {@link OutputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputDataConfig(OutputDataConfig)
         */
        default Builder outputDataConfig(Consumer<OutputDataConfig.Builder> outputDataConfig) {
            return outputDataConfig(OutputDataConfig.builder().applyMutation(outputDataConfig).build());
        }

        /**
         * <p>
         * The resources, including the ML compute instances and ML storage volumes, to use for model training.
         * </p>
         * 
         * @param resourceConfig
         *        The resources, including the ML compute instances and ML storage volumes, to use for model training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceConfig(ResourceConfig resourceConfig);

        /**
         * <p>
         * The resources, including the ML compute instances and ML storage volumes, to use for model training.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceConfig.Builder} avoiding the need to
         * create one manually via {@link ResourceConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceConfig(ResourceConfig)}.
         * 
         * @param resourceConfig
         *        a consumer that will call methods on {@link ResourceConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceConfig(ResourceConfig)
         */
        default Builder resourceConfig(Consumer<ResourceConfig.Builder> resourceConfig) {
            return resourceConfig(ResourceConfig.builder().applyMutation(resourceConfig).build());
        }

        /**
         * <p>
         * Specifies a limit to how long a model training job can run. When the job reaches the time limit, Amazon
         * SageMaker ends the training job. Use this API to cap model training costs.
         * </p>
         * <p>
         * To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120
         * seconds. Algorithms can use this 120-second window to save the model artifacts.
         * </p>
         * 
         * @param stoppingCondition
         *        Specifies a limit to how long a model training job can run. When the job reaches the time limit,
         *        Amazon SageMaker ends the training job. Use this API to cap model training costs.</p>
         *        <p>
         *        To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination
         *        for 120 seconds. Algorithms can use this 120-second window to save the model artifacts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stoppingCondition(StoppingCondition stoppingCondition);

        /**
         * <p>
         * Specifies a limit to how long a model training job can run. When the job reaches the time limit, Amazon
         * SageMaker ends the training job. Use this API to cap model training costs.
         * </p>
         * <p>
         * To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120
         * seconds. Algorithms can use this 120-second window to save the model artifacts.
         * </p>
         * This is a convenience that creates an instance of the {@link StoppingCondition.Builder} avoiding the need to
         * create one manually via {@link StoppingCondition#builder()}.
         *
         * When the {@link Consumer} completes, {@link StoppingCondition.Builder#build()} is called immediately and its
         * result is passed to {@link #stoppingCondition(StoppingCondition)}.
         * 
         * @param stoppingCondition
         *        a consumer that will call methods on {@link StoppingCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stoppingCondition(StoppingCondition)
         */
        default Builder stoppingCondition(Consumer<StoppingCondition.Builder> stoppingCondition) {
            return stoppingCondition(StoppingCondition.builder().applyMutation(stoppingCondition).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String trainingInputMode;

        private Map<String, String> hyperParameters = DefaultSdkAutoConstructMap.getInstance();

        private List<Channel> inputDataConfig = DefaultSdkAutoConstructList.getInstance();

        private OutputDataConfig outputDataConfig;

        private ResourceConfig resourceConfig;

        private StoppingCondition stoppingCondition;

        private BuilderImpl() {
        }

        private BuilderImpl(TrainingJobDefinition model) {
            trainingInputMode(model.trainingInputMode);
            hyperParameters(model.hyperParameters);
            inputDataConfig(model.inputDataConfig);
            outputDataConfig(model.outputDataConfig);
            resourceConfig(model.resourceConfig);
            stoppingCondition(model.stoppingCondition);
        }

        public final String getTrainingInputModeAsString() {
            return trainingInputMode;
        }

        @Override
        public final Builder trainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
            return this;
        }

        @Override
        public final Builder trainingInputMode(TrainingInputMode trainingInputMode) {
            this.trainingInputMode(trainingInputMode == null ? null : trainingInputMode.toString());
            return this;
        }

        public final void setTrainingInputMode(String trainingInputMode) {
            this.trainingInputMode = trainingInputMode;
        }

        public final Map<String, String> getHyperParameters() {
            return hyperParameters;
        }

        @Override
        public final Builder hyperParameters(Map<String, String> hyperParameters) {
            this.hyperParameters = HyperParametersCopier.copy(hyperParameters);
            return this;
        }

        public final void setHyperParameters(Map<String, String> hyperParameters) {
            this.hyperParameters = HyperParametersCopier.copy(hyperParameters);
        }

        public final Collection<Channel.Builder> getInputDataConfig() {
            return inputDataConfig != null ? inputDataConfig.stream().map(Channel::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder inputDataConfig(Collection<Channel> inputDataConfig) {
            this.inputDataConfig = InputDataConfigCopier.copy(inputDataConfig);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputDataConfig(Channel... inputDataConfig) {
            inputDataConfig(Arrays.asList(inputDataConfig));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputDataConfig(Consumer<Channel.Builder>... inputDataConfig) {
            inputDataConfig(Stream.of(inputDataConfig).map(c -> Channel.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setInputDataConfig(Collection<Channel.BuilderImpl> inputDataConfig) {
            this.inputDataConfig = InputDataConfigCopier.copyFromBuilder(inputDataConfig);
        }

        public final OutputDataConfig.Builder getOutputDataConfig() {
            return outputDataConfig != null ? outputDataConfig.toBuilder() : null;
        }

        @Override
        public final Builder outputDataConfig(OutputDataConfig outputDataConfig) {
            this.outputDataConfig = outputDataConfig;
            return this;
        }

        public final void setOutputDataConfig(OutputDataConfig.BuilderImpl outputDataConfig) {
            this.outputDataConfig = outputDataConfig != null ? outputDataConfig.build() : null;
        }

        public final ResourceConfig.Builder getResourceConfig() {
            return resourceConfig != null ? resourceConfig.toBuilder() : null;
        }

        @Override
        public final Builder resourceConfig(ResourceConfig resourceConfig) {
            this.resourceConfig = resourceConfig;
            return this;
        }

        public final void setResourceConfig(ResourceConfig.BuilderImpl resourceConfig) {
            this.resourceConfig = resourceConfig != null ? resourceConfig.build() : null;
        }

        public final StoppingCondition.Builder getStoppingCondition() {
            return stoppingCondition != null ? stoppingCondition.toBuilder() : null;
        }

        @Override
        public final Builder stoppingCondition(StoppingCondition stoppingCondition) {
            this.stoppingCondition = stoppingCondition;
            return this;
        }

        public final void setStoppingCondition(StoppingCondition.BuilderImpl stoppingCondition) {
            this.stoppingCondition = stoppingCondition != null ? stoppingCondition.build() : null;
        }

        @Override
        public TrainingJobDefinition build() {
            return new TrainingJobDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
