/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Identifies a model that you want to host and the resources to deploy for hosting it. If you are deploying multiple
 * models, tell Amazon SageMaker how to distribute traffic among the models by specifying variant weights.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProductionVariant implements SdkPojo, Serializable,
        ToCopyableBuilder<ProductionVariant.Builder, ProductionVariant> {
    private static final SdkField<String> VARIANT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProductionVariant::variantName)).setter(setter(Builder::variantName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VariantName").build()).build();

    private static final SdkField<String> MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProductionVariant::modelName)).setter(setter(Builder::modelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelName").build()).build();

    private static final SdkField<Integer> INITIAL_INSTANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ProductionVariant::initialInstanceCount)).setter(setter(Builder::initialInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialInstanceCount").build())
            .build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProductionVariant::instanceTypeAsString)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Float> INITIAL_VARIANT_WEIGHT_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .getter(getter(ProductionVariant::initialVariantWeight)).setter(setter(Builder::initialVariantWeight))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialVariantWeight").build())
            .build();

    private static final SdkField<String> ACCELERATOR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProductionVariant::acceleratorTypeAsString)).setter(setter(Builder::acceleratorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VARIANT_NAME_FIELD,
            MODEL_NAME_FIELD, INITIAL_INSTANCE_COUNT_FIELD, INSTANCE_TYPE_FIELD, INITIAL_VARIANT_WEIGHT_FIELD,
            ACCELERATOR_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String variantName;

    private final String modelName;

    private final Integer initialInstanceCount;

    private final String instanceType;

    private final Float initialVariantWeight;

    private final String acceleratorType;

    private ProductionVariant(BuilderImpl builder) {
        this.variantName = builder.variantName;
        this.modelName = builder.modelName;
        this.initialInstanceCount = builder.initialInstanceCount;
        this.instanceType = builder.instanceType;
        this.initialVariantWeight = builder.initialVariantWeight;
        this.acceleratorType = builder.acceleratorType;
    }

    /**
     * <p>
     * The name of the production variant.
     * </p>
     * 
     * @return The name of the production variant.
     */
    public String variantName() {
        return variantName;
    }

    /**
     * <p>
     * The name of the model that you want to host. This is the name that you specified when creating the model.
     * </p>
     * 
     * @return The name of the model that you want to host. This is the name that you specified when creating the model.
     */
    public String modelName() {
        return modelName;
    }

    /**
     * <p>
     * Number of instances to launch initially.
     * </p>
     * 
     * @return Number of instances to launch initially.
     */
    public Integer initialInstanceCount() {
        return initialInstanceCount;
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProductionVariantInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type.
     * @see ProductionVariantInstanceType
     */
    public ProductionVariantInstanceType instanceType() {
        return ProductionVariantInstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The ML compute instance type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link ProductionVariantInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The ML compute instance type.
     * @see ProductionVariantInstanceType
     */
    public String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * Determines initial traffic distribution among all of the models that you specify in the endpoint configuration.
     * The traffic to a production variant is determined by the ratio of the <code>VariantWeight</code> to the sum of
     * all <code>VariantWeight</code> values across all ProductionVariants. If unspecified, it defaults to 1.0.
     * </p>
     * 
     * @return Determines initial traffic distribution among all of the models that you specify in the endpoint
     *         configuration. The traffic to a production variant is determined by the ratio of the
     *         <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all
     *         ProductionVariants. If unspecified, it defaults to 1.0.
     */
    public Float initialVariantWeight() {
        return initialVariantWeight;
    }

    /**
     * <p>
     * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide on-demand
     * GPU computing for inference. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon SageMaker</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #acceleratorType}
     * will return {@link ProductionVariantAcceleratorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #acceleratorTypeAsString}.
     * </p>
     * 
     * @return The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
     *         on-demand GPU computing for inference. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
     *         SageMaker</a>.
     * @see ProductionVariantAcceleratorType
     */
    public ProductionVariantAcceleratorType acceleratorType() {
        return ProductionVariantAcceleratorType.fromValue(acceleratorType);
    }

    /**
     * <p>
     * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide on-demand
     * GPU computing for inference. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon SageMaker</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #acceleratorType}
     * will return {@link ProductionVariantAcceleratorType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #acceleratorTypeAsString}.
     * </p>
     * 
     * @return The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
     *         on-demand GPU computing for inference. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
     *         SageMaker</a>.
     * @see ProductionVariantAcceleratorType
     */
    public String acceleratorTypeAsString() {
        return acceleratorType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(variantName());
        hashCode = 31 * hashCode + Objects.hashCode(modelName());
        hashCode = 31 * hashCode + Objects.hashCode(initialInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(initialVariantWeight());
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProductionVariant)) {
            return false;
        }
        ProductionVariant other = (ProductionVariant) obj;
        return Objects.equals(variantName(), other.variantName()) && Objects.equals(modelName(), other.modelName())
                && Objects.equals(initialInstanceCount(), other.initialInstanceCount())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(initialVariantWeight(), other.initialVariantWeight())
                && Objects.equals(acceleratorTypeAsString(), other.acceleratorTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ProductionVariant").add("VariantName", variantName()).add("ModelName", modelName())
                .add("InitialInstanceCount", initialInstanceCount()).add("InstanceType", instanceTypeAsString())
                .add("InitialVariantWeight", initialVariantWeight()).add("AcceleratorType", acceleratorTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VariantName":
            return Optional.ofNullable(clazz.cast(variantName()));
        case "ModelName":
            return Optional.ofNullable(clazz.cast(modelName()));
        case "InitialInstanceCount":
            return Optional.ofNullable(clazz.cast(initialInstanceCount()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "InitialVariantWeight":
            return Optional.ofNullable(clazz.cast(initialVariantWeight()));
        case "AcceleratorType":
            return Optional.ofNullable(clazz.cast(acceleratorTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProductionVariant, T> g) {
        return obj -> g.apply((ProductionVariant) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProductionVariant> {
        /**
         * <p>
         * The name of the production variant.
         * </p>
         * 
         * @param variantName
         *        The name of the production variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variantName(String variantName);

        /**
         * <p>
         * The name of the model that you want to host. This is the name that you specified when creating the model.
         * </p>
         * 
         * @param modelName
         *        The name of the model that you want to host. This is the name that you specified when creating the
         *        model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelName(String modelName);

        /**
         * <p>
         * Number of instances to launch initially.
         * </p>
         * 
         * @param initialInstanceCount
         *        Number of instances to launch initially.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialInstanceCount(Integer initialInstanceCount);

        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * 
         * @param instanceType
         *        The ML compute instance type.
         * @see ProductionVariantInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The ML compute instance type.
         * </p>
         * 
         * @param instanceType
         *        The ML compute instance type.
         * @see ProductionVariantInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantInstanceType
         */
        Builder instanceType(ProductionVariantInstanceType instanceType);

        /**
         * <p>
         * Determines initial traffic distribution among all of the models that you specify in the endpoint
         * configuration. The traffic to a production variant is determined by the ratio of the
         * <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all ProductionVariants.
         * If unspecified, it defaults to 1.0.
         * </p>
         * 
         * @param initialVariantWeight
         *        Determines initial traffic distribution among all of the models that you specify in the endpoint
         *        configuration. The traffic to a production variant is determined by the ratio of the
         *        <code>VariantWeight</code> to the sum of all <code>VariantWeight</code> values across all
         *        ProductionVariants. If unspecified, it defaults to 1.0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialVariantWeight(Float initialVariantWeight);

        /**
         * <p>
         * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
         * on-demand GPU computing for inference. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         * SageMaker</a>.
         * </p>
         * 
         * @param acceleratorType
         *        The size of the Elastic Inference (EI) instance to use for the production variant. EI instances
         *        provide on-demand GPU computing for inference. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         *        SageMaker</a>.
         * @see ProductionVariantAcceleratorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantAcceleratorType
         */
        Builder acceleratorType(String acceleratorType);

        /**
         * <p>
         * The size of the Elastic Inference (EI) instance to use for the production variant. EI instances provide
         * on-demand GPU computing for inference. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         * SageMaker</a>.
         * </p>
         * 
         * @param acceleratorType
         *        The size of the Elastic Inference (EI) instance to use for the production variant. EI instances
         *        provide on-demand GPU computing for inference. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html">Using Elastic Inference in Amazon
         *        SageMaker</a>.
         * @see ProductionVariantAcceleratorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProductionVariantAcceleratorType
         */
        Builder acceleratorType(ProductionVariantAcceleratorType acceleratorType);
    }

    static final class BuilderImpl implements Builder {
        private String variantName;

        private String modelName;

        private Integer initialInstanceCount;

        private String instanceType;

        private Float initialVariantWeight;

        private String acceleratorType;

        private BuilderImpl() {
        }

        private BuilderImpl(ProductionVariant model) {
            variantName(model.variantName);
            modelName(model.modelName);
            initialInstanceCount(model.initialInstanceCount);
            instanceType(model.instanceType);
            initialVariantWeight(model.initialVariantWeight);
            acceleratorType(model.acceleratorType);
        }

        public final String getVariantName() {
            return variantName;
        }

        @Override
        public final Builder variantName(String variantName) {
            this.variantName = variantName;
            return this;
        }

        public final void setVariantName(String variantName) {
            this.variantName = variantName;
        }

        public final String getModelName() {
            return modelName;
        }

        @Override
        public final Builder modelName(String modelName) {
            this.modelName = modelName;
            return this;
        }

        public final void setModelName(String modelName) {
            this.modelName = modelName;
        }

        public final Integer getInitialInstanceCount() {
            return initialInstanceCount;
        }

        @Override
        public final Builder initialInstanceCount(Integer initialInstanceCount) {
            this.initialInstanceCount = initialInstanceCount;
            return this;
        }

        public final void setInitialInstanceCount(Integer initialInstanceCount) {
            this.initialInstanceCount = initialInstanceCount;
        }

        public final String getInstanceTypeAsString() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(ProductionVariantInstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Float getInitialVariantWeight() {
            return initialVariantWeight;
        }

        @Override
        public final Builder initialVariantWeight(Float initialVariantWeight) {
            this.initialVariantWeight = initialVariantWeight;
            return this;
        }

        public final void setInitialVariantWeight(Float initialVariantWeight) {
            this.initialVariantWeight = initialVariantWeight;
        }

        public final String getAcceleratorTypeAsString() {
            return acceleratorType;
        }

        @Override
        public final Builder acceleratorType(String acceleratorType) {
            this.acceleratorType = acceleratorType;
            return this;
        }

        @Override
        public final Builder acceleratorType(ProductionVariantAcceleratorType acceleratorType) {
            this.acceleratorType(acceleratorType == null ? null : acceleratorType.toString());
            return this;
        }

        public final void setAcceleratorType(String acceleratorType) {
            this.acceleratorType = acceleratorType;
        }

        @Override
        public ProductionVariant build() {
            return new ProductionVariant(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
