/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines how the algorithm is used for a training job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TrainingSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<TrainingSpecification.Builder, TrainingSpecification> {
    private static final SdkField<String> TRAINING_IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TrainingSpecification::trainingImage)).setter(setter(Builder::trainingImage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingImage").build()).build();

    private static final SdkField<String> TRAINING_IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(TrainingSpecification::trainingImageDigest)).setter(setter(Builder::trainingImageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingImageDigest").build())
            .build();

    private static final SdkField<List<HyperParameterSpecification>> SUPPORTED_HYPER_PARAMETERS_FIELD = SdkField
            .<List<HyperParameterSpecification>> builder(MarshallingType.LIST)
            .getter(getter(TrainingSpecification::supportedHyperParameters))
            .setter(setter(Builder::supportedHyperParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedHyperParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HyperParameterSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(HyperParameterSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SUPPORTED_TRAINING_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(TrainingSpecification::supportedTrainingInstanceTypesAsStrings))
            .setter(setter(Builder::supportedTrainingInstanceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedTrainingInstanceTypes")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> SUPPORTS_DISTRIBUTED_TRAINING_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(TrainingSpecification::supportsDistributedTraining))
            .setter(setter(Builder::supportsDistributedTraining))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportsDistributedTraining")
                    .build()).build();

    private static final SdkField<List<MetricDefinition>> METRIC_DEFINITIONS_FIELD = SdkField
            .<List<MetricDefinition>> builder(MarshallingType.LIST)
            .getter(getter(TrainingSpecification::metricDefinitions))
            .setter(setter(Builder::metricDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ChannelSpecification>> TRAINING_CHANNELS_FIELD = SdkField
            .<List<ChannelSpecification>> builder(MarshallingType.LIST)
            .getter(getter(TrainingSpecification::trainingChannels))
            .setter(setter(Builder::trainingChannels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingChannels").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ChannelSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(ChannelSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<HyperParameterTuningJobObjective>> SUPPORTED_TUNING_JOB_OBJECTIVE_METRICS_FIELD = SdkField
            .<List<HyperParameterTuningJobObjective>> builder(MarshallingType.LIST)
            .getter(getter(TrainingSpecification::supportedTuningJobObjectiveMetrics))
            .setter(setter(Builder::supportedTuningJobObjectiveMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedTuningJobObjectiveMetrics")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HyperParameterTuningJobObjective> builder(MarshallingType.SDK_POJO)
                                            .constructor(HyperParameterTuningJobObjective::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TRAINING_IMAGE_FIELD,
            TRAINING_IMAGE_DIGEST_FIELD, SUPPORTED_HYPER_PARAMETERS_FIELD, SUPPORTED_TRAINING_INSTANCE_TYPES_FIELD,
            SUPPORTS_DISTRIBUTED_TRAINING_FIELD, METRIC_DEFINITIONS_FIELD, TRAINING_CHANNELS_FIELD,
            SUPPORTED_TUNING_JOB_OBJECTIVE_METRICS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String trainingImage;

    private final String trainingImageDigest;

    private final List<HyperParameterSpecification> supportedHyperParameters;

    private final List<String> supportedTrainingInstanceTypes;

    private final Boolean supportsDistributedTraining;

    private final List<MetricDefinition> metricDefinitions;

    private final List<ChannelSpecification> trainingChannels;

    private final List<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics;

    private TrainingSpecification(BuilderImpl builder) {
        this.trainingImage = builder.trainingImage;
        this.trainingImageDigest = builder.trainingImageDigest;
        this.supportedHyperParameters = builder.supportedHyperParameters;
        this.supportedTrainingInstanceTypes = builder.supportedTrainingInstanceTypes;
        this.supportsDistributedTraining = builder.supportsDistributedTraining;
        this.metricDefinitions = builder.metricDefinitions;
        this.trainingChannels = builder.trainingChannels;
        this.supportedTuningJobObjectiveMetrics = builder.supportedTuningJobObjectiveMetrics;
    }

    /**
     * <p>
     * The Amazon ECR registry path of the Docker image that contains the training algorithm.
     * </p>
     * 
     * @return The Amazon ECR registry path of the Docker image that contains the training algorithm.
     */
    public String trainingImage() {
        return trainingImage;
    }

    /**
     * <p>
     * An MD5 hash of the training algorithm that identifies the Docker image used for training.
     * </p>
     * 
     * @return An MD5 hash of the training algorithm that identifies the Docker image used for training.
     */
    public String trainingImageDigest() {
        return trainingImageDigest;
    }

    /**
     * Returns true if the SupportedHyperParameters property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSupportedHyperParameters() {
        return supportedHyperParameters != null && !(supportedHyperParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters. This
     * is required if the algorithm supports automatic model tuning.&gt;
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSupportedHyperParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the <code>HyperParameterSpecification</code> objects, that define the supported
     *         hyperparameters. This is required if the algorithm supports automatic model tuning.&gt;
     */
    public List<HyperParameterSpecification> supportedHyperParameters() {
        return supportedHyperParameters;
    }

    /**
     * <p>
     * A list of the instance types that this algorithm can use for training.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSupportedTrainingInstanceTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the instance types that this algorithm can use for training.
     */
    public List<TrainingInstanceType> supportedTrainingInstanceTypes() {
        return TrainingInstanceTypesCopier.copyStringToEnum(supportedTrainingInstanceTypes);
    }

    /**
     * Returns true if the SupportedTrainingInstanceTypes property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public boolean hasSupportedTrainingInstanceTypes() {
        return supportedTrainingInstanceTypes != null && !(supportedTrainingInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the instance types that this algorithm can use for training.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSupportedTrainingInstanceTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the instance types that this algorithm can use for training.
     */
    public List<String> supportedTrainingInstanceTypesAsStrings() {
        return supportedTrainingInstanceTypes;
    }

    /**
     * <p>
     * Indicates whether the algorithm supports distributed training. If set to false, buyers can't request more than
     * one instance during training.
     * </p>
     * 
     * @return Indicates whether the algorithm supports distributed training. If set to false, buyers can't request more
     *         than one instance during training.
     */
    public Boolean supportsDistributedTraining() {
        return supportsDistributedTraining;
    }

    /**
     * Returns true if the MetricDefinitions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMetricDefinitions() {
        return metricDefinitions != null && !(metricDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the algorithm.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMetricDefinitions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
     *         algorithm.
     */
    public List<MetricDefinition> metricDefinitions() {
        return metricDefinitions;
    }

    /**
     * Returns true if the TrainingChannels property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasTrainingChannels() {
        return trainingChannels != null && !(trainingChannels instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the algorithm.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTrainingChannels()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
     *         algorithm.
     */
    public List<ChannelSpecification> trainingChannels() {
        return trainingChannels;
    }

    /**
     * Returns true if the SupportedTuningJobObjectiveMetrics property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public boolean hasSupportedTuningJobObjectiveMetrics() {
        return supportedTuningJobObjectiveMetrics != null
                && !(supportedTuningJobObjectiveMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
     * tuning job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSupportedTuningJobObjectiveMetrics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the metrics that the algorithm emits that can be used as the objective metric in a
     *         hyperparameter tuning job.
     */
    public List<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics() {
        return supportedTuningJobObjectiveMetrics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(trainingImage());
        hashCode = 31 * hashCode + Objects.hashCode(trainingImageDigest());
        hashCode = 31 * hashCode + Objects.hashCode(supportedHyperParameters());
        hashCode = 31 * hashCode + Objects.hashCode(supportedTrainingInstanceTypesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(supportsDistributedTraining());
        hashCode = 31 * hashCode + Objects.hashCode(metricDefinitions());
        hashCode = 31 * hashCode + Objects.hashCode(trainingChannels());
        hashCode = 31 * hashCode + Objects.hashCode(supportedTuningJobObjectiveMetrics());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TrainingSpecification)) {
            return false;
        }
        TrainingSpecification other = (TrainingSpecification) obj;
        return Objects.equals(trainingImage(), other.trainingImage())
                && Objects.equals(trainingImageDigest(), other.trainingImageDigest())
                && Objects.equals(supportedHyperParameters(), other.supportedHyperParameters())
                && Objects.equals(supportedTrainingInstanceTypesAsStrings(), other.supportedTrainingInstanceTypesAsStrings())
                && Objects.equals(supportsDistributedTraining(), other.supportsDistributedTraining())
                && Objects.equals(metricDefinitions(), other.metricDefinitions())
                && Objects.equals(trainingChannels(), other.trainingChannels())
                && Objects.equals(supportedTuningJobObjectiveMetrics(), other.supportedTuningJobObjectiveMetrics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TrainingSpecification").add("TrainingImage", trainingImage())
                .add("TrainingImageDigest", trainingImageDigest()).add("SupportedHyperParameters", supportedHyperParameters())
                .add("SupportedTrainingInstanceTypes", supportedTrainingInstanceTypesAsStrings())
                .add("SupportsDistributedTraining", supportsDistributedTraining()).add("MetricDefinitions", metricDefinitions())
                .add("TrainingChannels", trainingChannels())
                .add("SupportedTuningJobObjectiveMetrics", supportedTuningJobObjectiveMetrics()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TrainingImage":
            return Optional.ofNullable(clazz.cast(trainingImage()));
        case "TrainingImageDigest":
            return Optional.ofNullable(clazz.cast(trainingImageDigest()));
        case "SupportedHyperParameters":
            return Optional.ofNullable(clazz.cast(supportedHyperParameters()));
        case "SupportedTrainingInstanceTypes":
            return Optional.ofNullable(clazz.cast(supportedTrainingInstanceTypesAsStrings()));
        case "SupportsDistributedTraining":
            return Optional.ofNullable(clazz.cast(supportsDistributedTraining()));
        case "MetricDefinitions":
            return Optional.ofNullable(clazz.cast(metricDefinitions()));
        case "TrainingChannels":
            return Optional.ofNullable(clazz.cast(trainingChannels()));
        case "SupportedTuningJobObjectiveMetrics":
            return Optional.ofNullable(clazz.cast(supportedTuningJobObjectiveMetrics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TrainingSpecification, T> g) {
        return obj -> g.apply((TrainingSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TrainingSpecification> {
        /**
         * <p>
         * The Amazon ECR registry path of the Docker image that contains the training algorithm.
         * </p>
         * 
         * @param trainingImage
         *        The Amazon ECR registry path of the Docker image that contains the training algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingImage(String trainingImage);

        /**
         * <p>
         * An MD5 hash of the training algorithm that identifies the Docker image used for training.
         * </p>
         * 
         * @param trainingImageDigest
         *        An MD5 hash of the training algorithm that identifies the Docker image used for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingImageDigest(String trainingImageDigest);

        /**
         * <p>
         * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters.
         * This is required if the algorithm supports automatic model tuning.&gt;
         * </p>
         * 
         * @param supportedHyperParameters
         *        A list of the <code>HyperParameterSpecification</code> objects, that define the supported
         *        hyperparameters. This is required if the algorithm supports automatic model tuning.&gt;
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedHyperParameters(Collection<HyperParameterSpecification> supportedHyperParameters);

        /**
         * <p>
         * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters.
         * This is required if the algorithm supports automatic model tuning.&gt;
         * </p>
         * 
         * @param supportedHyperParameters
         *        A list of the <code>HyperParameterSpecification</code> objects, that define the supported
         *        hyperparameters. This is required if the algorithm supports automatic model tuning.&gt;
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedHyperParameters(HyperParameterSpecification... supportedHyperParameters);

        /**
         * <p>
         * A list of the <code>HyperParameterSpecification</code> objects, that define the supported hyperparameters.
         * This is required if the algorithm supports automatic model tuning.&gt;
         * </p>
         * This is a convenience that creates an instance of the {@link List<HyperParameterSpecification>.Builder}
         * avoiding the need to create one manually via {@link List<HyperParameterSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HyperParameterSpecification>.Builder#build()} is called
         * immediately and its result is passed to {@link #supportedHyperParameters(List<HyperParameterSpecification>)}.
         * 
         * @param supportedHyperParameters
         *        a consumer that will call methods on {@link List<HyperParameterSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportedHyperParameters(List<HyperParameterSpecification>)
         */
        Builder supportedHyperParameters(Consumer<HyperParameterSpecification.Builder>... supportedHyperParameters);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypesWithStrings(Collection<String> supportedTrainingInstanceTypes);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypesWithStrings(String... supportedTrainingInstanceTypes);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypes(Collection<TrainingInstanceType> supportedTrainingInstanceTypes);

        /**
         * <p>
         * A list of the instance types that this algorithm can use for training.
         * </p>
         * 
         * @param supportedTrainingInstanceTypes
         *        A list of the instance types that this algorithm can use for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTrainingInstanceTypes(TrainingInstanceType... supportedTrainingInstanceTypes);

        /**
         * <p>
         * Indicates whether the algorithm supports distributed training. If set to false, buyers can't request more
         * than one instance during training.
         * </p>
         * 
         * @param supportsDistributedTraining
         *        Indicates whether the algorithm supports distributed training. If set to false, buyers can't request
         *        more than one instance during training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportsDistributedTraining(Boolean supportsDistributedTraining);

        /**
         * <p>
         * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         * algorithm.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions);

        /**
         * <p>
         * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         * algorithm.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(MetricDefinition... metricDefinitions);

        /**
         * <p>
         * A list of <code>MetricDefinition</code> objects, which are used for parsing metrics generated by the
         * algorithm.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MetricDefinition>.Builder} avoiding the
         * need to create one manually via {@link List<MetricDefinition>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MetricDefinition>.Builder#build()} is called immediately and
         * its result is passed to {@link #metricDefinitions(List<MetricDefinition>)}.
         * 
         * @param metricDefinitions
         *        a consumer that will call methods on {@link List<MetricDefinition>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricDefinitions(List<MetricDefinition>)
         */
        Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions);

        /**
         * <p>
         * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         * algorithm.
         * </p>
         * 
         * @param trainingChannels
         *        A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingChannels(Collection<ChannelSpecification> trainingChannels);

        /**
         * <p>
         * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         * algorithm.
         * </p>
         * 
         * @param trainingChannels
         *        A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         *        algorithm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingChannels(ChannelSpecification... trainingChannels);

        /**
         * <p>
         * A list of <code>ChannelSpecification</code> objects, which specify the input sources to be used by the
         * algorithm.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ChannelSpecification>.Builder} avoiding the
         * need to create one manually via {@link List<ChannelSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ChannelSpecification>.Builder#build()} is called immediately
         * and its result is passed to {@link #trainingChannels(List<ChannelSpecification>)}.
         * 
         * @param trainingChannels
         *        a consumer that will call methods on {@link List<ChannelSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trainingChannels(List<ChannelSpecification>)
         */
        Builder trainingChannels(Consumer<ChannelSpecification.Builder>... trainingChannels);

        /**
         * <p>
         * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
         * tuning job.
         * </p>
         * 
         * @param supportedTuningJobObjectiveMetrics
         *        A list of the metrics that the algorithm emits that can be used as the objective metric in a
         *        hyperparameter tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTuningJobObjectiveMetrics(Collection<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics);

        /**
         * <p>
         * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
         * tuning job.
         * </p>
         * 
         * @param supportedTuningJobObjectiveMetrics
         *        A list of the metrics that the algorithm emits that can be used as the objective metric in a
         *        hyperparameter tuning job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedTuningJobObjectiveMetrics(HyperParameterTuningJobObjective... supportedTuningJobObjectiveMetrics);

        /**
         * <p>
         * A list of the metrics that the algorithm emits that can be used as the objective metric in a hyperparameter
         * tuning job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HyperParameterTuningJobObjective>.Builder}
         * avoiding the need to create one manually via {@link List<HyperParameterTuningJobObjective>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HyperParameterTuningJobObjective>.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #supportedTuningJobObjectiveMetrics(List<HyperParameterTuningJobObjective>)}.
         * 
         * @param supportedTuningJobObjectiveMetrics
         *        a consumer that will call methods on {@link List<HyperParameterTuningJobObjective>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #supportedTuningJobObjectiveMetrics(List<HyperParameterTuningJobObjective>)
         */
        Builder supportedTuningJobObjectiveMetrics(
                Consumer<HyperParameterTuningJobObjective.Builder>... supportedTuningJobObjectiveMetrics);
    }

    static final class BuilderImpl implements Builder {
        private String trainingImage;

        private String trainingImageDigest;

        private List<HyperParameterSpecification> supportedHyperParameters = DefaultSdkAutoConstructList.getInstance();

        private List<String> supportedTrainingInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private Boolean supportsDistributedTraining;

        private List<MetricDefinition> metricDefinitions = DefaultSdkAutoConstructList.getInstance();

        private List<ChannelSpecification> trainingChannels = DefaultSdkAutoConstructList.getInstance();

        private List<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics = DefaultSdkAutoConstructList
                .getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TrainingSpecification model) {
            trainingImage(model.trainingImage);
            trainingImageDigest(model.trainingImageDigest);
            supportedHyperParameters(model.supportedHyperParameters);
            supportedTrainingInstanceTypesWithStrings(model.supportedTrainingInstanceTypes);
            supportsDistributedTraining(model.supportsDistributedTraining);
            metricDefinitions(model.metricDefinitions);
            trainingChannels(model.trainingChannels);
            supportedTuningJobObjectiveMetrics(model.supportedTuningJobObjectiveMetrics);
        }

        public final String getTrainingImage() {
            return trainingImage;
        }

        @Override
        public final Builder trainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
            return this;
        }

        public final void setTrainingImage(String trainingImage) {
            this.trainingImage = trainingImage;
        }

        public final String getTrainingImageDigest() {
            return trainingImageDigest;
        }

        @Override
        public final Builder trainingImageDigest(String trainingImageDigest) {
            this.trainingImageDigest = trainingImageDigest;
            return this;
        }

        public final void setTrainingImageDigest(String trainingImageDigest) {
            this.trainingImageDigest = trainingImageDigest;
        }

        public final Collection<HyperParameterSpecification.Builder> getSupportedHyperParameters() {
            return supportedHyperParameters != null ? supportedHyperParameters.stream()
                    .map(HyperParameterSpecification::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder supportedHyperParameters(Collection<HyperParameterSpecification> supportedHyperParameters) {
            this.supportedHyperParameters = HyperParameterSpecificationsCopier.copy(supportedHyperParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedHyperParameters(HyperParameterSpecification... supportedHyperParameters) {
            supportedHyperParameters(Arrays.asList(supportedHyperParameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedHyperParameters(Consumer<HyperParameterSpecification.Builder>... supportedHyperParameters) {
            supportedHyperParameters(Stream.of(supportedHyperParameters)
                    .map(c -> HyperParameterSpecification.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSupportedHyperParameters(Collection<HyperParameterSpecification.BuilderImpl> supportedHyperParameters) {
            this.supportedHyperParameters = HyperParameterSpecificationsCopier.copyFromBuilder(supportedHyperParameters);
        }

        public final Collection<String> getSupportedTrainingInstanceTypesAsStrings() {
            return supportedTrainingInstanceTypes;
        }

        @Override
        public final Builder supportedTrainingInstanceTypesWithStrings(Collection<String> supportedTrainingInstanceTypes) {
            this.supportedTrainingInstanceTypes = TrainingInstanceTypesCopier.copy(supportedTrainingInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTrainingInstanceTypesWithStrings(String... supportedTrainingInstanceTypes) {
            supportedTrainingInstanceTypesWithStrings(Arrays.asList(supportedTrainingInstanceTypes));
            return this;
        }

        @Override
        public final Builder supportedTrainingInstanceTypes(Collection<TrainingInstanceType> supportedTrainingInstanceTypes) {
            this.supportedTrainingInstanceTypes = TrainingInstanceTypesCopier.copyEnumToString(supportedTrainingInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTrainingInstanceTypes(TrainingInstanceType... supportedTrainingInstanceTypes) {
            supportedTrainingInstanceTypes(Arrays.asList(supportedTrainingInstanceTypes));
            return this;
        }

        public final void setSupportedTrainingInstanceTypesWithStrings(Collection<String> supportedTrainingInstanceTypes) {
            this.supportedTrainingInstanceTypes = TrainingInstanceTypesCopier.copy(supportedTrainingInstanceTypes);
        }

        public final Boolean getSupportsDistributedTraining() {
            return supportsDistributedTraining;
        }

        @Override
        public final Builder supportsDistributedTraining(Boolean supportsDistributedTraining) {
            this.supportsDistributedTraining = supportsDistributedTraining;
            return this;
        }

        public final void setSupportsDistributedTraining(Boolean supportsDistributedTraining) {
            this.supportsDistributedTraining = supportsDistributedTraining;
        }

        public final Collection<MetricDefinition.Builder> getMetricDefinitions() {
            return metricDefinitions != null ? metricDefinitions.stream().map(MetricDefinition::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copy(metricDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(MetricDefinition... metricDefinitions) {
            metricDefinitions(Arrays.asList(metricDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions) {
            metricDefinitions(Stream.of(metricDefinitions).map(c -> MetricDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMetricDefinitions(Collection<MetricDefinition.BuilderImpl> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copyFromBuilder(metricDefinitions);
        }

        public final Collection<ChannelSpecification.Builder> getTrainingChannels() {
            return trainingChannels != null ? trainingChannels.stream().map(ChannelSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder trainingChannels(Collection<ChannelSpecification> trainingChannels) {
            this.trainingChannels = ChannelSpecificationsCopier.copy(trainingChannels);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder trainingChannels(ChannelSpecification... trainingChannels) {
            trainingChannels(Arrays.asList(trainingChannels));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder trainingChannels(Consumer<ChannelSpecification.Builder>... trainingChannels) {
            trainingChannels(Stream.of(trainingChannels).map(c -> ChannelSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTrainingChannels(Collection<ChannelSpecification.BuilderImpl> trainingChannels) {
            this.trainingChannels = ChannelSpecificationsCopier.copyFromBuilder(trainingChannels);
        }

        public final Collection<HyperParameterTuningJobObjective.Builder> getSupportedTuningJobObjectiveMetrics() {
            return supportedTuningJobObjectiveMetrics != null ? supportedTuningJobObjectiveMetrics.stream()
                    .map(HyperParameterTuningJobObjective::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder supportedTuningJobObjectiveMetrics(
                Collection<HyperParameterTuningJobObjective> supportedTuningJobObjectiveMetrics) {
            this.supportedTuningJobObjectiveMetrics = HyperParameterTuningJobObjectivesCopier
                    .copy(supportedTuningJobObjectiveMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTuningJobObjectiveMetrics(
                HyperParameterTuningJobObjective... supportedTuningJobObjectiveMetrics) {
            supportedTuningJobObjectiveMetrics(Arrays.asList(supportedTuningJobObjectiveMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedTuningJobObjectiveMetrics(
                Consumer<HyperParameterTuningJobObjective.Builder>... supportedTuningJobObjectiveMetrics) {
            supportedTuningJobObjectiveMetrics(Stream.of(supportedTuningJobObjectiveMetrics)
                    .map(c -> HyperParameterTuningJobObjective.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setSupportedTuningJobObjectiveMetrics(
                Collection<HyperParameterTuningJobObjective.BuilderImpl> supportedTuningJobObjectiveMetrics) {
            this.supportedTuningJobObjectiveMetrics = HyperParameterTuningJobObjectivesCopier
                    .copyFromBuilder(supportedTuningJobObjectiveMetrics);
        }

        @Override
        public TrainingSpecification build() {
            return new TrainingSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
