/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configures how labels are consolidated across human workers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AnnotationConsolidationConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<AnnotationConsolidationConfig.Builder, AnnotationConsolidationConfig> {
    private static final SdkField<String> ANNOTATION_CONSOLIDATION_LAMBDA_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(AnnotationConsolidationConfig::annotationConsolidationLambdaArn))
            .setter(setter(Builder::annotationConsolidationLambdaArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnnotationConsolidationLambdaArn")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(ANNOTATION_CONSOLIDATION_LAMBDA_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String annotationConsolidationLambdaArn;

    private AnnotationConsolidationConfig(BuilderImpl builder) {
        this.annotationConsolidationLambdaArn = builder.annotationConsolidationLambdaArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of a Lambda function implements the logic for annotation consolidation.
     * </p>
     * <p>
     * For the built-in bounding box, image classification, semantic segmentation, and text classification task types,
     * Amazon SageMaker Ground Truth provides the following Lambda functions:
     * </p>
     * <p>
     * <b>Bounding box</b> - Finds the most similar boxes from different workers based on the Jaccard index of the
     * boxes.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-BoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-BoundingBox</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Image classification</b> - Uses a variant of the Expectation Maximization approach to estimate the true class
     * of an image based on annotations from individual workers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClass</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Multi-label image classification</b> - Uses a variant of the Expectation Maximization approach to estimate the
     * true classes of an image based on annotations from individual workers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClassMultiLabel</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Semantic segmentation</b> - Treats each pixel in an image as a multi-class classification and treats pixel
     * annotations from workers as "votes" for the correct label.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-SemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-SemanticSegmentation</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Text classification</b> - Uses a variant of the Expectation Maximization approach to estimate the true class
     * of text based on annotations from individual workers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClass</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClass</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Multi-label text classification</b> - Uses a variant of the Expectation Maximization approach to estimate the
     * true classes of text based on annotations from individual workers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClassMultiLabel</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Named entity recognition</b> - Groups similar selections and calculates aggregate boundaries, resolving to
     * most-assigned label.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-NamedEntityRecognition</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-NamedEntityRecognition</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Bounding box verification</b> - Uses a variant of the Expectation Maximization approach to estimate the true
     * class of verification judgement for bounding box labels based on annotations from individual workers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationBoundingBox</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Semantic segmentation verification</b> - Uses a variant of the Expectation Maximization approach to estimate
     * the true class of verification judgment for semantic segmentation labels based on annotations from individual
     * workers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationSemanticSegmentation</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Bounding box adjustment</b> - Finds the most similar boxes from different workers based on the Jaccard index
     * of the adjusted annotations.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentBoundingBox</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>Semantic segmentation adjustment</b> - Treats each pixel in an image as a multi-class classification and
     * treats pixel adjusted annotations from workers as "votes" for the correct label.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * <p>
     * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentSemanticSegmentation</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-annotation-consolidation.html">Annotation
     * Consolidation</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of a Lambda function implements the logic for annotation
     *         consolidation.</p>
     *         <p>
     *         For the built-in bounding box, image classification, semantic segmentation, and text classification task
     *         types, Amazon SageMaker Ground Truth provides the following Lambda functions:
     *         </p>
     *         <p>
     *         <b>Bounding box</b> - Finds the most similar boxes from different workers based on the Jaccard index of
     *         the boxes.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-BoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-BoundingBox</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Image classification</b> - Uses a variant of the Expectation Maximization approach to estimate the
     *         true class of an image based on annotations from individual workers.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClass</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Multi-label image classification</b> - Uses a variant of the Expectation Maximization approach to
     *         estimate the true classes of an image based on annotations from individual workers.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClassMultiLabel</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Semantic segmentation</b> - Treats each pixel in an image as a multi-class classification and treats
     *         pixel annotations from workers as "votes" for the correct label.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-SemanticSegmentation</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Text classification</b> - Uses a variant of the Expectation Maximization approach to estimate the true
     *         class of text based on annotations from individual workers.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClass</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClass</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Multi-label text classification</b> - Uses a variant of the Expectation Maximization approach to
     *         estimate the true classes of text based on annotations from individual workers.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClassMultiLabel</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Named entity recognition</b> - Groups similar selections and calculates aggregate boundaries,
     *         resolving to most-assigned label.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-NamedEntityRecognition</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Bounding box verification</b> - Uses a variant of the Expectation Maximization approach to estimate
     *         the true class of verification judgement for bounding box labels based on annotations from individual
     *         workers.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationBoundingBox</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Semantic segmentation verification</b> - Uses a variant of the Expectation Maximization approach to
     *         estimate the true class of verification judgment for semantic segmentation labels based on annotations
     *         from individual workers.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationSemanticSegmentation</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Bounding box adjustment</b> - Finds the most similar boxes from different workers based on the Jaccard
     *         index of the adjusted annotations.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentBoundingBox</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Semantic segmentation adjustment</b> - Treats each pixel in an image as a multi-class classification
     *         and treats pixel adjusted annotations from workers as "votes" for the correct label.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         <p>
     *         <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentSemanticSegmentation</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-annotation-consolidation.html">Annotation
     *         Consolidation</a>.
     */
    public String annotationConsolidationLambdaArn() {
        return annotationConsolidationLambdaArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(annotationConsolidationLambdaArn());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AnnotationConsolidationConfig)) {
            return false;
        }
        AnnotationConsolidationConfig other = (AnnotationConsolidationConfig) obj;
        return Objects.equals(annotationConsolidationLambdaArn(), other.annotationConsolidationLambdaArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AnnotationConsolidationConfig")
                .add("AnnotationConsolidationLambdaArn", annotationConsolidationLambdaArn()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AnnotationConsolidationLambdaArn":
            return Optional.ofNullable(clazz.cast(annotationConsolidationLambdaArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AnnotationConsolidationConfig, T> g) {
        return obj -> g.apply((AnnotationConsolidationConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AnnotationConsolidationConfig> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of a Lambda function implements the logic for annotation consolidation.
         * </p>
         * <p>
         * For the built-in bounding box, image classification, semantic segmentation, and text classification task
         * types, Amazon SageMaker Ground Truth provides the following Lambda functions:
         * </p>
         * <p>
         * <b>Bounding box</b> - Finds the most similar boxes from different workers based on the Jaccard index of the
         * boxes.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-BoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-BoundingBox</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Image classification</b> - Uses a variant of the Expectation Maximization approach to estimate the true
         * class of an image based on annotations from individual workers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClass</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Multi-label image classification</b> - Uses a variant of the Expectation Maximization approach to estimate
         * the true classes of an image based on annotations from individual workers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClassMultiLabel</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Semantic segmentation</b> - Treats each pixel in an image as a multi-class classification and treats pixel
         * annotations from workers as "votes" for the correct label.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-SemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-SemanticSegmentation</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Text classification</b> - Uses a variant of the Expectation Maximization approach to estimate the true
         * class of text based on annotations from individual workers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClass</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClass</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Multi-label text classification</b> - Uses a variant of the Expectation Maximization approach to estimate
         * the true classes of text based on annotations from individual workers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClassMultiLabel</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Named entity recognition</b> - Groups similar selections and calculates aggregate boundaries, resolving to
         * most-assigned label.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-NamedEntityRecognition</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-NamedEntityRecognition</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Bounding box verification</b> - Uses a variant of the Expectation Maximization approach to estimate the
         * true class of verification judgement for bounding box labels based on annotations from individual workers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationBoundingBox</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Semantic segmentation verification</b> - Uses a variant of the Expectation Maximization approach to
         * estimate the true class of verification judgment for semantic segmentation labels based on annotations from
         * individual workers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationSemanticSegmentation</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Bounding box adjustment</b> - Finds the most similar boxes from different workers based on the Jaccard
         * index of the adjusted annotations.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentBoundingBox</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>Semantic segmentation adjustment</b> - Treats each pixel in an image as a multi-class classification and
         * treats pixel adjusted annotations from workers as "votes" for the correct label.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * <p>
         * <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentSemanticSegmentation</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-annotation-consolidation.html">Annotation
         * Consolidation</a>.
         * </p>
         * 
         * @param annotationConsolidationLambdaArn
         *        The Amazon Resource Name (ARN) of a Lambda function implements the logic for annotation
         *        consolidation.</p>
         *        <p>
         *        For the built-in bounding box, image classification, semantic segmentation, and text classification
         *        task types, Amazon SageMaker Ground Truth provides the following Lambda functions:
         *        </p>
         *        <p>
         *        <b>Bounding box</b> - Finds the most similar boxes from different workers based on the Jaccard index
         *        of the boxes.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-BoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-BoundingBox</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Image classification</b> - Uses a variant of the Expectation Maximization approach to estimate the
         *        true class of an image based on annotations from individual workers.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClass</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Multi-label image classification</b> - Uses a variant of the Expectation Maximization approach to
         *        estimate the true classes of an image based on annotations from individual workers.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-ImageMultiClassMultiLabel</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Semantic segmentation</b> - Treats each pixel in an image as a multi-class classification and
         *        treats pixel annotations from workers as "votes" for the correct label.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-SemanticSegmentation</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Text classification</b> - Uses a variant of the Expectation Maximization approach to estimate the
         *        true class of text based on annotations from individual workers.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClass</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClass</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Multi-label text classification</b> - Uses a variant of the Expectation Maximization approach to
         *        estimate the true classes of text based on annotations from individual workers.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-TextMultiClassMultiLabel</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Named entity recognition</b> - Groups similar selections and calculates aggregate boundaries,
         *        resolving to most-assigned label.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-NamedEntityRecognition</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Bounding box verification</b> - Uses a variant of the Expectation Maximization approach to estimate
         *        the true class of verification judgement for bounding box labels based on annotations from individual
         *        workers.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationBoundingBox</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Semantic segmentation verification</b> - Uses a variant of the Expectation Maximization approach to
         *        estimate the true class of verification judgment for semantic segmentation labels based on annotations
         *        from individual workers.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-VerificationSemanticSegmentation</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Bounding box adjustment</b> - Finds the most similar boxes from different workers based on the
         *        Jaccard index of the adjusted annotations.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentBoundingBox</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Semantic segmentation adjustment</b> - Treats each pixel in an image as a multi-class
         *        classification and treats pixel adjusted annotations from workers as "votes" for the correct label.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>arn:aws:lambda:us-east-1:432418664414:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-east-2:266458841044:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:us-west-2:081040173940:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-1:568282634449:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-1:477331159723:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-2:454466003867:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-south-1:565803892007:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-central-1:203001061592:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-northeast-2:845288260483:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:eu-west-2:487402164563:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ap-southeast-1:377565633583:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        <p>
         *        <code>arn:aws:lambda:ca-central-1:918755190332:function:ACS-AdjustmentSemanticSegmentation</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/sms-annotation-consolidation.html">Annotation
         *        Consolidation</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder annotationConsolidationLambdaArn(String annotationConsolidationLambdaArn);
    }

    static final class BuilderImpl implements Builder {
        private String annotationConsolidationLambdaArn;

        private BuilderImpl() {
        }

        private BuilderImpl(AnnotationConsolidationConfig model) {
            annotationConsolidationLambdaArn(model.annotationConsolidationLambdaArn);
        }

        public final String getAnnotationConsolidationLambdaArn() {
            return annotationConsolidationLambdaArn;
        }

        @Override
        public final Builder annotationConsolidationLambdaArn(String annotationConsolidationLambdaArn) {
            this.annotationConsolidationLambdaArn = annotationConsolidationLambdaArn;
            return this;
        }

        public final void setAnnotationConsolidationLambdaArn(String annotationConsolidationLambdaArn) {
            this.annotationConsolidationLambdaArn = annotationConsolidationLambdaArn;
        }

        @Override
        public AnnotationConsolidationConfig build() {
            return new AnnotationConsolidationConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
