/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A multi-expression that searches for the specified resource or resources in a search. All resource objects that
 * satisfy the expression's condition are included in the search results. You must specify at least one subexpression,
 * filter, or nested filter. A <code>SearchExpression</code> can contain up to twenty elements.
 * </p>
 * <p>
 * A <code>SearchExpression</code> contains the following components:
 * </p>
 * <ul>
 * <li>
 * <p>
 * A list of <code>Filter</code> objects. Each filter defines a simple Boolean expression comprised of a resource
 * property name, Boolean operator, and value. A <code>SearchExpression</code> can include only one
 * <code>Contains</code> operator.
 * </p>
 * </li>
 * <li>
 * <p>
 * A list of <code>NestedFilter</code> objects. Each nested filter defines a list of Boolean expressions using a list of
 * resource properties. A nested filter is satisfied if a single object in the list satisfies all Boolean expressions.
 * </p>
 * </li>
 * <li>
 * <p>
 * A list of <code>SearchExpression</code> objects. A search expression object can be nested in a list of search
 * expression objects.
 * </p>
 * </li>
 * <li>
 * <p>
 * A Boolean operator: <code>And</code> or <code>Or</code>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SearchExpression implements SdkPojo, Serializable,
        ToCopyableBuilder<SearchExpression.Builder, SearchExpression> {
    private static final SdkField<List<Filter>> FILTERS_FIELD = SdkField
            .<List<Filter>> builder(MarshallingType.LIST)
            .getter(getter(SearchExpression::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Filter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Filter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<NestedFilters>> NESTED_FILTERS_FIELD = SdkField
            .<List<NestedFilters>> builder(MarshallingType.LIST)
            .getter(getter(SearchExpression::nestedFilters))
            .setter(setter(Builder::nestedFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NestedFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NestedFilters> builder(MarshallingType.SDK_POJO)
                                            .constructor(NestedFilters::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<SearchExpression>> SUB_EXPRESSIONS_FIELD = SdkField
            .<List<SearchExpression>> builder(MarshallingType.LIST)
            .getter(getter(SearchExpression::subExpressions))
            .setter(setter(Builder::subExpressions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubExpressions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SearchExpression> builder(MarshallingType.SDK_POJO)
                                            .constructor(SearchExpression::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SearchExpression::operatorAsString)).setter(setter(Builder::operator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operator").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILTERS_FIELD,
            NESTED_FILTERS_FIELD, SUB_EXPRESSIONS_FIELD, OPERATOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Filter> filters;

    private final List<NestedFilters> nestedFilters;

    private final List<SearchExpression> subExpressions;

    private final String operator;

    private SearchExpression(BuilderImpl builder) {
        this.filters = builder.filters;
        this.nestedFilters = builder.nestedFilters;
        this.subExpressions = builder.subExpressions;
        this.operator = builder.operator;
    }

    /**
     * Returns true if the Filters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of filter objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFilters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of filter objects.
     */
    public List<Filter> filters() {
        return filters;
    }

    /**
     * Returns true if the NestedFilters property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasNestedFilters() {
        return nestedFilters != null && !(nestedFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of nested filter objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNestedFilters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of nested filter objects.
     */
    public List<NestedFilters> nestedFilters() {
        return nestedFilters;
    }

    /**
     * Returns true if the SubExpressions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubExpressions() {
        return subExpressions != null && !(subExpressions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of search expression objects.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubExpressions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of search expression objects.
     */
    public List<SearchExpression> subExpressions() {
        return subExpressions;
    }

    /**
     * <p>
     * A Boolean operator used to evaluate the search expression. If you want every conditional statement in all lists
     * to be satisfied for the entire search expression to be true, specify <code>And</code>. If only a single
     * conditional statement needs to be true for the entire search expression to be true, specify <code>Or</code>. The
     * default value is <code>And</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link BooleanOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return A Boolean operator used to evaluate the search expression. If you want every conditional statement in all
     *         lists to be satisfied for the entire search expression to be true, specify <code>And</code>. If only a
     *         single conditional statement needs to be true for the entire search expression to be true, specify
     *         <code>Or</code>. The default value is <code>And</code>.
     * @see BooleanOperator
     */
    public BooleanOperator operator() {
        return BooleanOperator.fromValue(operator);
    }

    /**
     * <p>
     * A Boolean operator used to evaluate the search expression. If you want every conditional statement in all lists
     * to be satisfied for the entire search expression to be true, specify <code>And</code>. If only a single
     * conditional statement needs to be true for the entire search expression to be true, specify <code>Or</code>. The
     * default value is <code>And</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operator} will
     * return {@link BooleanOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operatorAsString}.
     * </p>
     * 
     * @return A Boolean operator used to evaluate the search expression. If you want every conditional statement in all
     *         lists to be satisfied for the entire search expression to be true, specify <code>And</code>. If only a
     *         single conditional statement needs to be true for the entire search expression to be true, specify
     *         <code>Or</code>. The default value is <code>And</code>.
     * @see BooleanOperator
     */
    public String operatorAsString() {
        return operator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(filters());
        hashCode = 31 * hashCode + Objects.hashCode(nestedFilters());
        hashCode = 31 * hashCode + Objects.hashCode(subExpressions());
        hashCode = 31 * hashCode + Objects.hashCode(operatorAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SearchExpression)) {
            return false;
        }
        SearchExpression other = (SearchExpression) obj;
        return Objects.equals(filters(), other.filters()) && Objects.equals(nestedFilters(), other.nestedFilters())
                && Objects.equals(subExpressions(), other.subExpressions())
                && Objects.equals(operatorAsString(), other.operatorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SearchExpression").add("Filters", filters()).add("NestedFilters", nestedFilters())
                .add("SubExpressions", subExpressions()).add("Operator", operatorAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Filters":
            return Optional.ofNullable(clazz.cast(filters()));
        case "NestedFilters":
            return Optional.ofNullable(clazz.cast(nestedFilters()));
        case "SubExpressions":
            return Optional.ofNullable(clazz.cast(subExpressions()));
        case "Operator":
            return Optional.ofNullable(clazz.cast(operatorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SearchExpression, T> g) {
        return obj -> g.apply((SearchExpression) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SearchExpression> {
        /**
         * <p>
         * A list of filter objects.
         * </p>
         * 
         * @param filters
         *        A list of filter objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<Filter> filters);

        /**
         * <p>
         * A list of filter objects.
         * </p>
         * 
         * @param filters
         *        A list of filter objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Filter... filters);

        /**
         * <p>
         * A list of filter objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Filter>.Builder} avoiding the need to
         * create one manually via {@link List<Filter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Filter>.Builder#build()} is called immediately and its
         * result is passed to {@link #filters(List<Filter>)}.
         * 
         * @param filters
         *        a consumer that will call methods on {@link List<Filter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(List<Filter>)
         */
        Builder filters(Consumer<Filter.Builder>... filters);

        /**
         * <p>
         * A list of nested filter objects.
         * </p>
         * 
         * @param nestedFilters
         *        A list of nested filter objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nestedFilters(Collection<NestedFilters> nestedFilters);

        /**
         * <p>
         * A list of nested filter objects.
         * </p>
         * 
         * @param nestedFilters
         *        A list of nested filter objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nestedFilters(NestedFilters... nestedFilters);

        /**
         * <p>
         * A list of nested filter objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<NestedFilters>.Builder} avoiding the need
         * to create one manually via {@link List<NestedFilters>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<NestedFilters>.Builder#build()} is called immediately and
         * its result is passed to {@link #nestedFilters(List<NestedFilters>)}.
         * 
         * @param nestedFilters
         *        a consumer that will call methods on {@link List<NestedFilters>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nestedFilters(List<NestedFilters>)
         */
        Builder nestedFilters(Consumer<NestedFilters.Builder>... nestedFilters);

        /**
         * <p>
         * A list of search expression objects.
         * </p>
         * 
         * @param subExpressions
         *        A list of search expression objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subExpressions(Collection<SearchExpression> subExpressions);

        /**
         * <p>
         * A list of search expression objects.
         * </p>
         * 
         * @param subExpressions
         *        A list of search expression objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subExpressions(SearchExpression... subExpressions);

        /**
         * <p>
         * A list of search expression objects.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SearchExpression>.Builder} avoiding the
         * need to create one manually via {@link List<SearchExpression>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SearchExpression>.Builder#build()} is called immediately and
         * its result is passed to {@link #subExpressions(List<SearchExpression>)}.
         * 
         * @param subExpressions
         *        a consumer that will call methods on {@link List<SearchExpression>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subExpressions(List<SearchExpression>)
         */
        Builder subExpressions(Consumer<Builder>... subExpressions);

        /**
         * <p>
         * A Boolean operator used to evaluate the search expression. If you want every conditional statement in all
         * lists to be satisfied for the entire search expression to be true, specify <code>And</code>. If only a single
         * conditional statement needs to be true for the entire search expression to be true, specify <code>Or</code>.
         * The default value is <code>And</code>.
         * </p>
         * 
         * @param operator
         *        A Boolean operator used to evaluate the search expression. If you want every conditional statement in
         *        all lists to be satisfied for the entire search expression to be true, specify <code>And</code>. If
         *        only a single conditional statement needs to be true for the entire search expression to be true,
         *        specify <code>Or</code>. The default value is <code>And</code>.
         * @see BooleanOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BooleanOperator
         */
        Builder operator(String operator);

        /**
         * <p>
         * A Boolean operator used to evaluate the search expression. If you want every conditional statement in all
         * lists to be satisfied for the entire search expression to be true, specify <code>And</code>. If only a single
         * conditional statement needs to be true for the entire search expression to be true, specify <code>Or</code>.
         * The default value is <code>And</code>.
         * </p>
         * 
         * @param operator
         *        A Boolean operator used to evaluate the search expression. If you want every conditional statement in
         *        all lists to be satisfied for the entire search expression to be true, specify <code>And</code>. If
         *        only a single conditional statement needs to be true for the entire search expression to be true,
         *        specify <code>Or</code>. The default value is <code>And</code>.
         * @see BooleanOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BooleanOperator
         */
        Builder operator(BooleanOperator operator);
    }

    static final class BuilderImpl implements Builder {
        private List<Filter> filters = DefaultSdkAutoConstructList.getInstance();

        private List<NestedFilters> nestedFilters = DefaultSdkAutoConstructList.getInstance();

        private List<SearchExpression> subExpressions = DefaultSdkAutoConstructList.getInstance();

        private String operator;

        private BuilderImpl() {
        }

        private BuilderImpl(SearchExpression model) {
            filters(model.filters);
            nestedFilters(model.nestedFilters);
            subExpressions(model.subExpressions);
            operator(model.operator);
        }

        public final Collection<Filter.Builder> getFilters() {
            return filters != null ? filters.stream().map(Filter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder filters(Collection<Filter> filters) {
            this.filters = FilterListCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Filter... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<Filter.Builder>... filters) {
            filters(Stream.of(filters).map(c -> Filter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setFilters(Collection<Filter.BuilderImpl> filters) {
            this.filters = FilterListCopier.copyFromBuilder(filters);
        }

        public final Collection<NestedFilters.Builder> getNestedFilters() {
            return nestedFilters != null ? nestedFilters.stream().map(NestedFilters::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder nestedFilters(Collection<NestedFilters> nestedFilters) {
            this.nestedFilters = NestedFiltersListCopier.copy(nestedFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nestedFilters(NestedFilters... nestedFilters) {
            nestedFilters(Arrays.asList(nestedFilters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nestedFilters(Consumer<NestedFilters.Builder>... nestedFilters) {
            nestedFilters(Stream.of(nestedFilters).map(c -> NestedFilters.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setNestedFilters(Collection<NestedFilters.BuilderImpl> nestedFilters) {
            this.nestedFilters = NestedFiltersListCopier.copyFromBuilder(nestedFilters);
        }

        public final Collection<Builder> getSubExpressions() {
            return subExpressions != null ? subExpressions.stream().map(SearchExpression::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder subExpressions(Collection<SearchExpression> subExpressions) {
            this.subExpressions = SearchExpressionListCopier.copy(subExpressions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subExpressions(SearchExpression... subExpressions) {
            subExpressions(Arrays.asList(subExpressions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subExpressions(Consumer<Builder>... subExpressions) {
            subExpressions(Stream.of(subExpressions).map(c -> SearchExpression.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSubExpressions(Collection<BuilderImpl> subExpressions) {
            this.subExpressions = SearchExpressionListCopier.copyFromBuilder(subExpressions);
        }

        public final String getOperator() {
            return operator;
        }

        @Override
        public final Builder operator(String operator) {
            this.operator = operator;
            return this;
        }

        @Override
        public final Builder operator(BooleanOperator operator) {
            this.operator(operator == null ? null : operator.toString());
            return this;
        }

        public final void setOperator(String operator) {
            this.operator = operator;
        }

        @Override
        public SearchExpression build() {
            return new SearchExpression(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
