/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration to run a processing job in a specified container image.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AppSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<AppSpecification.Builder, AppSpecification> {
    private static final SdkField<String> IMAGE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageUri").getter(getter(AppSpecification::imageUri)).setter(setter(Builder::imageUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUri").build()).build();

    private static final SdkField<List<String>> CONTAINER_ENTRYPOINT_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ContainerEntrypoint")
            .getter(getter(AppSpecification::containerEntrypoint))
            .setter(setter(Builder::containerEntrypoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerEntrypoint").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CONTAINER_ARGUMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ContainerArguments")
            .getter(getter(AppSpecification::containerArguments))
            .setter(setter(Builder::containerArguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerArguments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_URI_FIELD,
            CONTAINER_ENTRYPOINT_FIELD, CONTAINER_ARGUMENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String imageUri;

    private final List<String> containerEntrypoint;

    private final List<String> containerArguments;

    private AppSpecification(BuilderImpl builder) {
        this.imageUri = builder.imageUri;
        this.containerEntrypoint = builder.containerEntrypoint;
        this.containerArguments = builder.containerArguments;
    }

    /**
     * <p>
     * The container image to be run by the processing job.
     * </p>
     * 
     * @return The container image to be run by the processing job.
     */
    public String imageUri() {
        return imageUri;
    }

    /**
     * Returns true if the ContainerEntrypoint property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasContainerEntrypoint() {
        return containerEntrypoint != null && !(containerEntrypoint instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The entrypoint for a container used to run a processing job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasContainerEntrypoint()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The entrypoint for a container used to run a processing job.
     */
    public List<String> containerEntrypoint() {
        return containerEntrypoint;
    }

    /**
     * Returns true if the ContainerArguments property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasContainerArguments() {
        return containerArguments != null && !(containerArguments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The arguments for a container used to run a processing job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasContainerArguments()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The arguments for a container used to run a processing job.
     */
    public List<String> containerArguments() {
        return containerArguments;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(imageUri());
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerEntrypoint() ? containerEntrypoint() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasContainerArguments() ? containerArguments() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AppSpecification)) {
            return false;
        }
        AppSpecification other = (AppSpecification) obj;
        return Objects.equals(imageUri(), other.imageUri()) && hasContainerEntrypoint() == other.hasContainerEntrypoint()
                && Objects.equals(containerEntrypoint(), other.containerEntrypoint())
                && hasContainerArguments() == other.hasContainerArguments()
                && Objects.equals(containerArguments(), other.containerArguments());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AppSpecification").add("ImageUri", imageUri())
                .add("ContainerEntrypoint", hasContainerEntrypoint() ? containerEntrypoint() : null)
                .add("ContainerArguments", hasContainerArguments() ? containerArguments() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ImageUri":
            return Optional.ofNullable(clazz.cast(imageUri()));
        case "ContainerEntrypoint":
            return Optional.ofNullable(clazz.cast(containerEntrypoint()));
        case "ContainerArguments":
            return Optional.ofNullable(clazz.cast(containerArguments()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AppSpecification, T> g) {
        return obj -> g.apply((AppSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AppSpecification> {
        /**
         * <p>
         * The container image to be run by the processing job.
         * </p>
         * 
         * @param imageUri
         *        The container image to be run by the processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUri(String imageUri);

        /**
         * <p>
         * The entrypoint for a container used to run a processing job.
         * </p>
         * 
         * @param containerEntrypoint
         *        The entrypoint for a container used to run a processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerEntrypoint(Collection<String> containerEntrypoint);

        /**
         * <p>
         * The entrypoint for a container used to run a processing job.
         * </p>
         * 
         * @param containerEntrypoint
         *        The entrypoint for a container used to run a processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerEntrypoint(String... containerEntrypoint);

        /**
         * <p>
         * The arguments for a container used to run a processing job.
         * </p>
         * 
         * @param containerArguments
         *        The arguments for a container used to run a processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerArguments(Collection<String> containerArguments);

        /**
         * <p>
         * The arguments for a container used to run a processing job.
         * </p>
         * 
         * @param containerArguments
         *        The arguments for a container used to run a processing job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerArguments(String... containerArguments);
    }

    static final class BuilderImpl implements Builder {
        private String imageUri;

        private List<String> containerEntrypoint = DefaultSdkAutoConstructList.getInstance();

        private List<String> containerArguments = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AppSpecification model) {
            imageUri(model.imageUri);
            containerEntrypoint(model.containerEntrypoint);
            containerArguments(model.containerArguments);
        }

        public final String getImageUri() {
            return imageUri;
        }

        @Override
        public final Builder imageUri(String imageUri) {
            this.imageUri = imageUri;
            return this;
        }

        public final void setImageUri(String imageUri) {
            this.imageUri = imageUri;
        }

        public final Collection<String> getContainerEntrypoint() {
            if (containerEntrypoint instanceof SdkAutoConstructList) {
                return null;
            }
            return containerEntrypoint;
        }

        @Override
        public final Builder containerEntrypoint(Collection<String> containerEntrypoint) {
            this.containerEntrypoint = ContainerEntrypointCopier.copy(containerEntrypoint);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerEntrypoint(String... containerEntrypoint) {
            containerEntrypoint(Arrays.asList(containerEntrypoint));
            return this;
        }

        public final void setContainerEntrypoint(Collection<String> containerEntrypoint) {
            this.containerEntrypoint = ContainerEntrypointCopier.copy(containerEntrypoint);
        }

        public final Collection<String> getContainerArguments() {
            if (containerArguments instanceof SdkAutoConstructList) {
                return null;
            }
            return containerArguments;
        }

        @Override
        public final Builder containerArguments(Collection<String> containerArguments) {
            this.containerArguments = ContainerArgumentsCopier.copy(containerArguments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder containerArguments(String... containerArguments) {
            containerArguments(Arrays.asList(containerArguments));
            return this;
        }

        public final void setContainerArguments(Collection<String> containerArguments) {
            this.containerArguments = ContainerArgumentsCopier.copy(containerArguments);
        }

        @Override
        public AppSpecification build() {
            return new AppSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
