/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Docker container image configuration object for the model explainability job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ModelExplainabilityAppSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<ModelExplainabilityAppSpecification.Builder, ModelExplainabilityAppSpecification> {
    private static final SdkField<String> IMAGE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageUri").getter(getter(ModelExplainabilityAppSpecification::imageUri))
            .setter(setter(Builder::imageUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageUri").build()).build();

    private static final SdkField<String> CONFIG_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigUri").getter(getter(ModelExplainabilityAppSpecification::configUri))
            .setter(setter(Builder::configUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigUri").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Environment")
            .getter(getter(ModelExplainabilityAppSpecification::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_URI_FIELD,
            CONFIG_URI_FIELD, ENVIRONMENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String imageUri;

    private final String configUri;

    private final Map<String, String> environment;

    private ModelExplainabilityAppSpecification(BuilderImpl builder) {
        this.imageUri = builder.imageUri;
        this.configUri = builder.configUri;
        this.environment = builder.environment;
    }

    /**
     * <p>
     * The container image to be run by the model explainability job.
     * </p>
     * 
     * @return The container image to be run by the model explainability job.
     */
    public String imageUri() {
        return imageUri;
    }

    /**
     * <p>
     * JSON formatted S3 file that defines explainability parameters. For more information on this JSON configuration
     * file, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/json-model-explainability-parameter-config.html">Configure
     * model explainability parameters</a>.
     * </p>
     * 
     * @return JSON formatted S3 file that defines explainability parameters. For more information on this JSON
     *         configuration file, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/json-model-explainability-parameter-config.html"
     *         >Configure model explainability parameters</a>.
     */
    public String configUri() {
        return configUri;
    }

    /**
     * Returns true if the Environment property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Sets the environment variables in the Docker container.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEnvironment()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Sets the environment variables in the Docker container.
     */
    public Map<String, String> environment() {
        return environment;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(imageUri());
        hashCode = 31 * hashCode + Objects.hashCode(configUri());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModelExplainabilityAppSpecification)) {
            return false;
        }
        ModelExplainabilityAppSpecification other = (ModelExplainabilityAppSpecification) obj;
        return Objects.equals(imageUri(), other.imageUri()) && Objects.equals(configUri(), other.configUri())
                && hasEnvironment() == other.hasEnvironment() && Objects.equals(environment(), other.environment());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ModelExplainabilityAppSpecification").add("ImageUri", imageUri()).add("ConfigUri", configUri())
                .add("Environment", hasEnvironment() ? environment() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ImageUri":
            return Optional.ofNullable(clazz.cast(imageUri()));
        case "ConfigUri":
            return Optional.ofNullable(clazz.cast(configUri()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ModelExplainabilityAppSpecification, T> g) {
        return obj -> g.apply((ModelExplainabilityAppSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ModelExplainabilityAppSpecification> {
        /**
         * <p>
         * The container image to be run by the model explainability job.
         * </p>
         * 
         * @param imageUri
         *        The container image to be run by the model explainability job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUri(String imageUri);

        /**
         * <p>
         * JSON formatted S3 file that defines explainability parameters. For more information on this JSON
         * configuration file, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/json-model-explainability-parameter-config.html">Configure
         * model explainability parameters</a>.
         * </p>
         * 
         * @param configUri
         *        JSON formatted S3 file that defines explainability parameters. For more information on this JSON
         *        configuration file, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/json-model-explainability-parameter-config.html"
         *        >Configure model explainability parameters</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configUri(String configUri);

        /**
         * <p>
         * Sets the environment variables in the Docker container.
         * </p>
         * 
         * @param environment
         *        Sets the environment variables in the Docker container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Map<String, String> environment);
    }

    static final class BuilderImpl implements Builder {
        private String imageUri;

        private String configUri;

        private Map<String, String> environment = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ModelExplainabilityAppSpecification model) {
            imageUri(model.imageUri);
            configUri(model.configUri);
            environment(model.environment);
        }

        public final String getImageUri() {
            return imageUri;
        }

        @Override
        public final Builder imageUri(String imageUri) {
            this.imageUri = imageUri;
            return this;
        }

        public final void setImageUri(String imageUri) {
            this.imageUri = imageUri;
        }

        public final String getConfigUri() {
            return configUri;
        }

        @Override
        public final Builder configUri(String configUri) {
            this.configUri = configUri;
            return this;
        }

        public final void setConfigUri(String configUri) {
            this.configUri = configUri;
        }

        public final Map<String, String> getEnvironment() {
            if (environment instanceof SdkAutoConstructMap) {
                return null;
            }
            return environment;
        }

        @Override
        public final Builder environment(Map<String, String> environment) {
            this.environment = MonitoringEnvironmentMapCopier.copy(environment);
            return this;
        }

        public final void setEnvironment(Map<String, String> environment) {
            this.environment = MonitoringEnvironmentMapCopier.copy(environment);
        }

        @Override
        public ModelExplainabilityAppSpecification build() {
            return new ModelExplainabilityAppSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
