/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A hosted endpoint for real-time inference.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> ENDPOINT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointName").getter(getter(Endpoint::endpointName)).setter(setter(Builder::endpointName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointName").build()).build();

    private static final SdkField<String> ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointArn").getter(getter(Endpoint::endpointArn)).setter(setter(Builder::endpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointArn").build()).build();

    private static final SdkField<String> ENDPOINT_CONFIG_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointConfigName").getter(getter(Endpoint::endpointConfigName))
            .setter(setter(Builder::endpointConfigName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointConfigName").build())
            .build();

    private static final SdkField<List<ProductionVariantSummary>> PRODUCTION_VARIANTS_FIELD = SdkField
            .<List<ProductionVariantSummary>> builder(MarshallingType.LIST)
            .memberName("ProductionVariants")
            .getter(getter(Endpoint::productionVariants))
            .setter(setter(Builder::productionVariants))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductionVariants").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ProductionVariantSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(ProductionVariantSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataCaptureConfigSummary> DATA_CAPTURE_CONFIG_FIELD = SdkField
            .<DataCaptureConfigSummary> builder(MarshallingType.SDK_POJO).memberName("DataCaptureConfig")
            .getter(getter(Endpoint::dataCaptureConfig)).setter(setter(Builder::dataCaptureConfig))
            .constructor(DataCaptureConfigSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataCaptureConfig").build()).build();

    private static final SdkField<String> ENDPOINT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointStatus").getter(getter(Endpoint::endpointStatusAsString))
            .setter(setter(Builder::endpointStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointStatus").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FailureReason").getter(getter(Endpoint::failureReason)).setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Endpoint::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(Endpoint::lastModifiedTime)).setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<List<MonitoringSchedule>> MONITORING_SCHEDULES_FIELD = SdkField
            .<List<MonitoringSchedule>> builder(MarshallingType.LIST)
            .memberName("MonitoringSchedules")
            .getter(getter(Endpoint::monitoringSchedules))
            .setter(setter(Builder::monitoringSchedules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MonitoringSchedules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MonitoringSchedule> builder(MarshallingType.SDK_POJO)
                                            .constructor(MonitoringSchedule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Endpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENDPOINT_NAME_FIELD,
            ENDPOINT_ARN_FIELD, ENDPOINT_CONFIG_NAME_FIELD, PRODUCTION_VARIANTS_FIELD, DATA_CAPTURE_CONFIG_FIELD,
            ENDPOINT_STATUS_FIELD, FAILURE_REASON_FIELD, CREATION_TIME_FIELD, LAST_MODIFIED_TIME_FIELD,
            MONITORING_SCHEDULES_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String endpointName;

    private final String endpointArn;

    private final String endpointConfigName;

    private final List<ProductionVariantSummary> productionVariants;

    private final DataCaptureConfigSummary dataCaptureConfig;

    private final String endpointStatus;

    private final String failureReason;

    private final Instant creationTime;

    private final Instant lastModifiedTime;

    private final List<MonitoringSchedule> monitoringSchedules;

    private final List<Tag> tags;

    private Endpoint(BuilderImpl builder) {
        this.endpointName = builder.endpointName;
        this.endpointArn = builder.endpointArn;
        this.endpointConfigName = builder.endpointConfigName;
        this.productionVariants = builder.productionVariants;
        this.dataCaptureConfig = builder.dataCaptureConfig;
        this.endpointStatus = builder.endpointStatus;
        this.failureReason = builder.failureReason;
        this.creationTime = builder.creationTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.monitoringSchedules = builder.monitoringSchedules;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the endpoint.
     * </p>
     * 
     * @return The name of the endpoint.
     */
    public String endpointName() {
        return endpointName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the endpoint.
     */
    public String endpointArn() {
        return endpointArn;
    }

    /**
     * <p>
     * The endpoint configuration associated with the endpoint.
     * </p>
     * 
     * @return The endpoint configuration associated with the endpoint.
     */
    public String endpointConfigName() {
        return endpointConfigName;
    }

    /**
     * Returns true if the ProductionVariants property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasProductionVariants() {
        return productionVariants != null && !(productionVariants instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the production variants hosted on the endpoint. Each production variant is a model.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProductionVariants()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the production variants hosted on the endpoint. Each production variant is a model.
     */
    public List<ProductionVariantSummary> productionVariants() {
        return productionVariants;
    }

    /**
     * Returns the value of the DataCaptureConfig property for this object.
     * 
     * @return The value of the DataCaptureConfig property for this object.
     */
    public DataCaptureConfigSummary dataCaptureConfig() {
        return dataCaptureConfig;
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointStatus}
     * will return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #endpointStatusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.
     * @see EndpointStatus
     */
    public EndpointStatus endpointStatus() {
        return EndpointStatus.fromValue(endpointStatus);
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointStatus}
     * will return {@link EndpointStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #endpointStatusAsString}.
     * </p>
     * 
     * @return The status of the endpoint.
     * @see EndpointStatus
     */
    public String endpointStatusAsString() {
        return endpointStatus;
    }

    /**
     * <p>
     * If the endpoint failed, the reason it failed.
     * </p>
     * 
     * @return If the endpoint failed, the reason it failed.
     */
    public String failureReason() {
        return failureReason;
    }

    /**
     * <p>
     * The time that the endpoint was created.
     * </p>
     * 
     * @return The time that the endpoint was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The last time the endpoint was modified.
     * </p>
     * 
     * @return The last time the endpoint was modified.
     */
    public Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * Returns true if the MonitoringSchedules property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasMonitoringSchedules() {
        return monitoringSchedules != null && !(monitoringSchedules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of monitoring schedules for the endpoint. For information about model monitoring, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor.html">Amazon SageMaker Model Monitor</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMonitoringSchedules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of monitoring schedules for the endpoint. For information about model monitoring, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor.html">Amazon SageMaker Model
     *         Monitor</a>.
     */
    public List<MonitoringSchedule> monitoringSchedules() {
        return monitoringSchedules;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the tags associated with the endpoint. For more information, see <a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a> in the <i>AWS
     * General Reference Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the tags associated with the endpoint. For more information, see <a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a> in the
     *         <i>AWS General Reference Guide</i>.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(endpointName());
        hashCode = 31 * hashCode + Objects.hashCode(endpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(endpointConfigName());
        hashCode = 31 * hashCode + Objects.hashCode(hasProductionVariants() ? productionVariants() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataCaptureConfig());
        hashCode = 31 * hashCode + Objects.hashCode(endpointStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasMonitoringSchedules() ? monitoringSchedules() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(endpointName(), other.endpointName()) && Objects.equals(endpointArn(), other.endpointArn())
                && Objects.equals(endpointConfigName(), other.endpointConfigName())
                && hasProductionVariants() == other.hasProductionVariants()
                && Objects.equals(productionVariants(), other.productionVariants())
                && Objects.equals(dataCaptureConfig(), other.dataCaptureConfig())
                && Objects.equals(endpointStatusAsString(), other.endpointStatusAsString())
                && Objects.equals(failureReason(), other.failureReason()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && hasMonitoringSchedules() == other.hasMonitoringSchedules()
                && Objects.equals(monitoringSchedules(), other.monitoringSchedules()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Endpoint").add("EndpointName", endpointName()).add("EndpointArn", endpointArn())
                .add("EndpointConfigName", endpointConfigName())
                .add("ProductionVariants", hasProductionVariants() ? productionVariants() : null)
                .add("DataCaptureConfig", dataCaptureConfig()).add("EndpointStatus", endpointStatusAsString())
                .add("FailureReason", failureReason()).add("CreationTime", creationTime())
                .add("LastModifiedTime", lastModifiedTime())
                .add("MonitoringSchedules", hasMonitoringSchedules() ? monitoringSchedules() : null)
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EndpointName":
            return Optional.ofNullable(clazz.cast(endpointName()));
        case "EndpointArn":
            return Optional.ofNullable(clazz.cast(endpointArn()));
        case "EndpointConfigName":
            return Optional.ofNullable(clazz.cast(endpointConfigName()));
        case "ProductionVariants":
            return Optional.ofNullable(clazz.cast(productionVariants()));
        case "DataCaptureConfig":
            return Optional.ofNullable(clazz.cast(dataCaptureConfig()));
        case "EndpointStatus":
            return Optional.ofNullable(clazz.cast(endpointStatusAsString()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "MonitoringSchedules":
            return Optional.ofNullable(clazz.cast(monitoringSchedules()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The name of the endpoint.
         * </p>
         * 
         * @param endpointName
         *        The name of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointName(String endpointName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the endpoint.
         * </p>
         * 
         * @param endpointArn
         *        The Amazon Resource Name (ARN) of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointArn(String endpointArn);

        /**
         * <p>
         * The endpoint configuration associated with the endpoint.
         * </p>
         * 
         * @param endpointConfigName
         *        The endpoint configuration associated with the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointConfigName(String endpointConfigName);

        /**
         * <p>
         * A list of the production variants hosted on the endpoint. Each production variant is a model.
         * </p>
         * 
         * @param productionVariants
         *        A list of the production variants hosted on the endpoint. Each production variant is a model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productionVariants(Collection<ProductionVariantSummary> productionVariants);

        /**
         * <p>
         * A list of the production variants hosted on the endpoint. Each production variant is a model.
         * </p>
         * 
         * @param productionVariants
         *        A list of the production variants hosted on the endpoint. Each production variant is a model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productionVariants(ProductionVariantSummary... productionVariants);

        /**
         * <p>
         * A list of the production variants hosted on the endpoint. Each production variant is a model.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ProductionVariantSummary>.Builder} avoiding
         * the need to create one manually via {@link List<ProductionVariantSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ProductionVariantSummary>.Builder#build()} is called
         * immediately and its result is passed to {@link #productionVariants(List<ProductionVariantSummary>)}.
         * 
         * @param productionVariants
         *        a consumer that will call methods on {@link List<ProductionVariantSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #productionVariants(List<ProductionVariantSummary>)
         */
        Builder productionVariants(Consumer<ProductionVariantSummary.Builder>... productionVariants);

        /**
         * Sets the value of the DataCaptureConfig property for this object.
         *
         * @param dataCaptureConfig
         *        The new value for the DataCaptureConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataCaptureConfig(DataCaptureConfigSummary dataCaptureConfig);

        /**
         * Sets the value of the DataCaptureConfig property for this object.
         *
         * This is a convenience that creates an instance of the {@link DataCaptureConfigSummary.Builder} avoiding the
         * need to create one manually via {@link DataCaptureConfigSummary#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataCaptureConfigSummary.Builder#build()} is called immediately
         * and its result is passed to {@link #dataCaptureConfig(DataCaptureConfigSummary)}.
         * 
         * @param dataCaptureConfig
         *        a consumer that will call methods on {@link DataCaptureConfigSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataCaptureConfig(DataCaptureConfigSummary)
         */
        default Builder dataCaptureConfig(Consumer<DataCaptureConfigSummary.Builder> dataCaptureConfig) {
            return dataCaptureConfig(DataCaptureConfigSummary.builder().applyMutation(dataCaptureConfig).build());
        }

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * 
         * @param endpointStatus
         *        The status of the endpoint.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder endpointStatus(String endpointStatus);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * 
         * @param endpointStatus
         *        The status of the endpoint.
         * @see EndpointStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EndpointStatus
         */
        Builder endpointStatus(EndpointStatus endpointStatus);

        /**
         * <p>
         * If the endpoint failed, the reason it failed.
         * </p>
         * 
         * @param failureReason
         *        If the endpoint failed, the reason it failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * The time that the endpoint was created.
         * </p>
         * 
         * @param creationTime
         *        The time that the endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The last time the endpoint was modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The last time the endpoint was modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * A list of monitoring schedules for the endpoint. For information about model monitoring, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor.html">Amazon SageMaker Model Monitor</a>.
         * </p>
         * 
         * @param monitoringSchedules
         *        A list of monitoring schedules for the endpoint. For information about model monitoring, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor.html">Amazon SageMaker Model
         *        Monitor</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoringSchedules(Collection<MonitoringSchedule> monitoringSchedules);

        /**
         * <p>
         * A list of monitoring schedules for the endpoint. For information about model monitoring, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor.html">Amazon SageMaker Model Monitor</a>.
         * </p>
         * 
         * @param monitoringSchedules
         *        A list of monitoring schedules for the endpoint. For information about model monitoring, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor.html">Amazon SageMaker Model
         *        Monitor</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoringSchedules(MonitoringSchedule... monitoringSchedules);

        /**
         * <p>
         * A list of monitoring schedules for the endpoint. For information about model monitoring, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor.html">Amazon SageMaker Model Monitor</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MonitoringSchedule>.Builder} avoiding the
         * need to create one manually via {@link List<MonitoringSchedule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MonitoringSchedule>.Builder#build()} is called immediately
         * and its result is passed to {@link #monitoringSchedules(List<MonitoringSchedule>)}.
         * 
         * @param monitoringSchedules
         *        a consumer that will call methods on {@link List<MonitoringSchedule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #monitoringSchedules(List<MonitoringSchedule>)
         */
        Builder monitoringSchedules(Consumer<MonitoringSchedule.Builder>... monitoringSchedules);

        /**
         * <p>
         * A list of the tags associated with the endpoint. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a> in the <i>AWS
         * General Reference Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of the tags associated with the endpoint. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a> in the
         *        <i>AWS General Reference Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of the tags associated with the endpoint. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a> in the <i>AWS
         * General Reference Guide</i>.
         * </p>
         * 
         * @param tags
         *        A list of the tags associated with the endpoint. For more information, see <a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a> in the
         *        <i>AWS General Reference Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of the tags associated with the endpoint. For more information, see <a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS resources</a> in the <i>AWS
         * General Reference Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String endpointName;

        private String endpointArn;

        private String endpointConfigName;

        private List<ProductionVariantSummary> productionVariants = DefaultSdkAutoConstructList.getInstance();

        private DataCaptureConfigSummary dataCaptureConfig;

        private String endpointStatus;

        private String failureReason;

        private Instant creationTime;

        private Instant lastModifiedTime;

        private List<MonitoringSchedule> monitoringSchedules = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            endpointName(model.endpointName);
            endpointArn(model.endpointArn);
            endpointConfigName(model.endpointConfigName);
            productionVariants(model.productionVariants);
            dataCaptureConfig(model.dataCaptureConfig);
            endpointStatus(model.endpointStatus);
            failureReason(model.failureReason);
            creationTime(model.creationTime);
            lastModifiedTime(model.lastModifiedTime);
            monitoringSchedules(model.monitoringSchedules);
            tags(model.tags);
        }

        public final String getEndpointName() {
            return endpointName;
        }

        @Override
        public final Builder endpointName(String endpointName) {
            this.endpointName = endpointName;
            return this;
        }

        public final void setEndpointName(String endpointName) {
            this.endpointName = endpointName;
        }

        public final String getEndpointArn() {
            return endpointArn;
        }

        @Override
        public final Builder endpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
            return this;
        }

        public final void setEndpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
        }

        public final String getEndpointConfigName() {
            return endpointConfigName;
        }

        @Override
        public final Builder endpointConfigName(String endpointConfigName) {
            this.endpointConfigName = endpointConfigName;
            return this;
        }

        public final void setEndpointConfigName(String endpointConfigName) {
            this.endpointConfigName = endpointConfigName;
        }

        public final Collection<ProductionVariantSummary.Builder> getProductionVariants() {
            if (productionVariants instanceof SdkAutoConstructList) {
                return null;
            }
            return productionVariants != null ? productionVariants.stream().map(ProductionVariantSummary::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder productionVariants(Collection<ProductionVariantSummary> productionVariants) {
            this.productionVariants = ProductionVariantSummaryListCopier.copy(productionVariants);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productionVariants(ProductionVariantSummary... productionVariants) {
            productionVariants(Arrays.asList(productionVariants));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder productionVariants(Consumer<ProductionVariantSummary.Builder>... productionVariants) {
            productionVariants(Stream.of(productionVariants)
                    .map(c -> ProductionVariantSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setProductionVariants(Collection<ProductionVariantSummary.BuilderImpl> productionVariants) {
            this.productionVariants = ProductionVariantSummaryListCopier.copyFromBuilder(productionVariants);
        }

        public final DataCaptureConfigSummary.Builder getDataCaptureConfig() {
            return dataCaptureConfig != null ? dataCaptureConfig.toBuilder() : null;
        }

        @Override
        public final Builder dataCaptureConfig(DataCaptureConfigSummary dataCaptureConfig) {
            this.dataCaptureConfig = dataCaptureConfig;
            return this;
        }

        public final void setDataCaptureConfig(DataCaptureConfigSummary.BuilderImpl dataCaptureConfig) {
            this.dataCaptureConfig = dataCaptureConfig != null ? dataCaptureConfig.build() : null;
        }

        public final String getEndpointStatus() {
            return endpointStatus;
        }

        @Override
        public final Builder endpointStatus(String endpointStatus) {
            this.endpointStatus = endpointStatus;
            return this;
        }

        @Override
        public final Builder endpointStatus(EndpointStatus endpointStatus) {
            this.endpointStatus(endpointStatus == null ? null : endpointStatus.toString());
            return this;
        }

        public final void setEndpointStatus(String endpointStatus) {
            this.endpointStatus = endpointStatus;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        public final Collection<MonitoringSchedule.Builder> getMonitoringSchedules() {
            if (monitoringSchedules instanceof SdkAutoConstructList) {
                return null;
            }
            return monitoringSchedules != null ? monitoringSchedules.stream().map(MonitoringSchedule::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder monitoringSchedules(Collection<MonitoringSchedule> monitoringSchedules) {
            this.monitoringSchedules = MonitoringScheduleListCopier.copy(monitoringSchedules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder monitoringSchedules(MonitoringSchedule... monitoringSchedules) {
            monitoringSchedules(Arrays.asList(monitoringSchedules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder monitoringSchedules(Consumer<MonitoringSchedule.Builder>... monitoringSchedules) {
            monitoringSchedules(Stream.of(monitoringSchedules).map(c -> MonitoringSchedule.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMonitoringSchedules(Collection<MonitoringSchedule.BuilderImpl> monitoringSchedules) {
            this.monitoringSchedules = MonitoringScheduleListCopier.copyFromBuilder(monitoringSchedules);
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
