/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the output location for the compiled model and the target device that the model runs on.
 * <code>TargetDevice</code> and <code>TargetPlatform</code> are mutually exclusive, so you need to choose one between
 * the two to specify your target device or platform. If you cannot find your device you want to use from the
 * <code>TargetDevice</code> list, use <code>TargetPlatform</code> to describe the platform of your edge device and
 * <code>CompilerOptions</code> if there are specific settings that are required or recommended to use for particular
 * TargetPlatform.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OutputConfig implements SdkPojo, Serializable, ToCopyableBuilder<OutputConfig.Builder, OutputConfig> {
    private static final SdkField<String> S3_OUTPUT_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3OutputLocation").getter(getter(OutputConfig::s3OutputLocation))
            .setter(setter(Builder::s3OutputLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3OutputLocation").build()).build();

    private static final SdkField<String> TARGET_DEVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetDevice").getter(getter(OutputConfig::targetDeviceAsString)).setter(setter(Builder::targetDevice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetDevice").build()).build();

    private static final SdkField<TargetPlatform> TARGET_PLATFORM_FIELD = SdkField
            .<TargetPlatform> builder(MarshallingType.SDK_POJO).memberName("TargetPlatform")
            .getter(getter(OutputConfig::targetPlatform)).setter(setter(Builder::targetPlatform))
            .constructor(TargetPlatform::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetPlatform").build()).build();

    private static final SdkField<String> COMPILER_OPTIONS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CompilerOptions").getter(getter(OutputConfig::compilerOptions)).setter(setter(Builder::compilerOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompilerOptions").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(OutputConfig::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_OUTPUT_LOCATION_FIELD,
            TARGET_DEVICE_FIELD, TARGET_PLATFORM_FIELD, COMPILER_OPTIONS_FIELD, KMS_KEY_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3OutputLocation;

    private final String targetDevice;

    private final TargetPlatform targetPlatform;

    private final String compilerOptions;

    private final String kmsKeyId;

    private OutputConfig(BuilderImpl builder) {
        this.s3OutputLocation = builder.s3OutputLocation;
        this.targetDevice = builder.targetDevice;
        this.targetPlatform = builder.targetPlatform;
        this.compilerOptions = builder.compilerOptions;
        this.kmsKeyId = builder.kmsKeyId;
    }

    /**
     * <p>
     * Identifies the S3 bucket where you want Amazon SageMaker to store the model artifacts. For example,
     * <code>s3://bucket-name/key-name-prefix</code>.
     * </p>
     * 
     * @return Identifies the S3 bucket where you want Amazon SageMaker to store the model artifacts. For example,
     *         <code>s3://bucket-name/key-name-prefix</code>.
     */
    public String s3OutputLocation() {
        return s3OutputLocation;
    }

    /**
     * <p>
     * Identifies the target device or the machine learning instance that you want to run your model on after the
     * compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
     * <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetDevice} will
     * return {@link TargetDevice#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetDeviceAsString}.
     * </p>
     * 
     * @return Identifies the target device or the machine learning instance that you want to run your model on after
     *         the compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
     *         <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
     * @see TargetDevice
     */
    public TargetDevice targetDevice() {
        return TargetDevice.fromValue(targetDevice);
    }

    /**
     * <p>
     * Identifies the target device or the machine learning instance that you want to run your model on after the
     * compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
     * <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #targetDevice} will
     * return {@link TargetDevice#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #targetDeviceAsString}.
     * </p>
     * 
     * @return Identifies the target device or the machine learning instance that you want to run your model on after
     *         the compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
     *         <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
     * @see TargetDevice
     */
    public String targetDeviceAsString() {
        return targetDevice;
    }

    /**
     * <p>
     * Contains information about a target platform that you want your model to run on, such as OS, architecture, and
     * accelerators. It is an alternative of <code>TargetDevice</code>.
     * </p>
     * <p>
     * The following examples show how to configure the <code>TargetPlatform</code> and <code>CompilerOptions</code>
     * JSON strings for popular target platforms:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Raspberry Pi 3 Model B+
     * </p>
     * <p>
     * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM_EABIHF"},</code>
     * </p>
     * <p>
     * <code> "CompilerOptions": {'mattr': ['+neon']}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Jetson TX2
     * </p>
     * <p>
     * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "NVIDIA"},</code>
     * </p>
     * <p>
     * <code> "CompilerOptions": {'gpu-code': 'sm_62', 'trt-ver': '6.0.1', 'cuda-ver': '10.0'}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * EC2 m5.2xlarge instance OS
     * </p>
     * <p>
     * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "X86_64", "Accelerator": "NVIDIA"},</code>
     * </p>
     * <p>
     * <code> "CompilerOptions": {'mcpu': 'skylake-avx512'}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * RK3399
     * </p>
     * <p>
     * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "MALI"}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * ARMv7 phone (CPU)
     * </p>
     * <p>
     * <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM_EABI"},</code>
     * </p>
     * <p>
     * <code> "CompilerOptions": {'ANDROID_PLATFORM': 25, 'mattr': ['+neon']}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * ARMv8 phone (CPU)
     * </p>
     * <p>
     * <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM64"},</code>
     * </p>
     * <p>
     * <code> "CompilerOptions": {'ANDROID_PLATFORM': 29}</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return Contains information about a target platform that you want your model to run on, such as OS,
     *         architecture, and accelerators. It is an alternative of <code>TargetDevice</code>.</p>
     *         <p>
     *         The following examples show how to configure the <code>TargetPlatform</code> and
     *         <code>CompilerOptions</code> JSON strings for popular target platforms:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Raspberry Pi 3 Model B+
     *         </p>
     *         <p>
     *         <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM_EABIHF"},</code>
     *         </p>
     *         <p>
     *         <code> "CompilerOptions": {'mattr': ['+neon']}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Jetson TX2
     *         </p>
     *         <p>
     *         <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "NVIDIA"},</code>
     *         </p>
     *         <p>
     *         <code> "CompilerOptions": {'gpu-code': 'sm_62', 'trt-ver': '6.0.1', 'cuda-ver': '10.0'}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EC2 m5.2xlarge instance OS
     *         </p>
     *         <p>
     *         <code>"TargetPlatform": {"Os": "LINUX", "Arch": "X86_64", "Accelerator": "NVIDIA"},</code>
     *         </p>
     *         <p>
     *         <code> "CompilerOptions": {'mcpu': 'skylake-avx512'}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RK3399
     *         </p>
     *         <p>
     *         <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "MALI"}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ARMv7 phone (CPU)
     *         </p>
     *         <p>
     *         <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM_EABI"},</code>
     *         </p>
     *         <p>
     *         <code> "CompilerOptions": {'ANDROID_PLATFORM': 25, 'mattr': ['+neon']}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ARMv8 phone (CPU)
     *         </p>
     *         <p>
     *         <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM64"},</code>
     *         </p>
     *         <p>
     *         <code> "CompilerOptions": {'ANDROID_PLATFORM': 29}</code>
     *         </p>
     *         </li>
     */
    public TargetPlatform targetPlatform() {
        return targetPlatform;
    }

    /**
     * <p>
     * Specifies additional parameters for compiler options in JSON format. The compiler options are
     * <code>TargetPlatform</code> specific. It is required for NVIDIA accelerators and highly recommended for CPU
     * compilations. For any other cases, it is optional to specify <code>CompilerOptions.</code>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CPU</code>: Compilation for CPU supports the following compiler options.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>mcpu</code>: CPU micro-architecture. For example, <code>{'mcpu': 'skylake-avx512'}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>mattr</code>: CPU flags. For example, <code>{'mattr': ['+neon', '+vfpv4']}</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>ARM</code>: Details of ARM CPU compilations.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NEON</code>: NEON is an implementation of the Advanced SIMD extension used in ARMv7 processors.
     * </p>
     * <p>
     * For example, add <code>{'mattr': ['+neon']}</code> to the compiler options if compiling for ARM 32-bit platform
     * with the NEON support.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>NVIDIA</code>: Compilation for NVIDIA GPU supports the following compiler options.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>gpu_code</code>: Specifies the targeted architecture.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>trt-ver</code>: Specifies the TensorRT versions in x.y.z. format.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cuda-ver</code>: Specifies the CUDA version in x.y format.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For example, <code>{'gpu-code': 'sm_72', 'trt-ver': '6.0.1', 'cuda-ver': '10.1'}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ANDROID</code>: Compilation for the Android OS supports the following compiler options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ANDROID_PLATFORM</code>: Specifies the Android API levels. Available levels range from 21 to 29. For
     * example, <code>{'ANDROID_PLATFORM': 28}</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>mattr</code>: Add <code>{'mattr': ['+neon']}</code> to compiler options if compiling for ARM 32-bit
     * platform with NEON support.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>INFERENTIA</code>: Compilation for target ml_inf1 uses compiler options passed in as a JSON string. For
     * example, <code>"CompilerOptions": "\"--verbose 1 --num-neuroncores 2 -O2\""</code>.
     * </p>
     * <p>
     * For information about supported compiler options, see <a
     * href="https://github.com/aws/aws-neuron-sdk/blob/master/docs/neuron-cc/command-line-reference.md"> Neuron
     * Compiler CLI</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CoreML</code>: Compilation for the CoreML <a>OutputConfig$TargetDevice</a> supports the following compiler
     * options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>class_labels</code>: Specifies the classification labels file name inside input tar.gz file. For example,
     * <code>{"class_labels": "imagenet_labels_1000.txt"}</code>. Labels inside the txt file should be separated by
     * newlines.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * 
     * @return Specifies additional parameters for compiler options in JSON format. The compiler options are
     *         <code>TargetPlatform</code> specific. It is required for NVIDIA accelerators and highly recommended for
     *         CPU compilations. For any other cases, it is optional to specify <code>CompilerOptions.</code> </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CPU</code>: Compilation for CPU supports the following compiler options.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>mcpu</code>: CPU micro-architecture. For example, <code>{'mcpu': 'skylake-avx512'}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>mattr</code>: CPU flags. For example, <code>{'mattr': ['+neon', '+vfpv4']}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ARM</code>: Details of ARM CPU compilations.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NEON</code>: NEON is an implementation of the Advanced SIMD extension used in ARMv7 processors.
     *         </p>
     *         <p>
     *         For example, add <code>{'mattr': ['+neon']}</code> to the compiler options if compiling for ARM 32-bit
     *         platform with the NEON support.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NVIDIA</code>: Compilation for NVIDIA GPU supports the following compiler options.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>gpu_code</code>: Specifies the targeted architecture.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>trt-ver</code>: Specifies the TensorRT versions in x.y.z. format.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cuda-ver</code>: Specifies the CUDA version in x.y format.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For example, <code>{'gpu-code': 'sm_72', 'trt-ver': '6.0.1', 'cuda-ver': '10.1'}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ANDROID</code>: Compilation for the Android OS supports the following compiler options:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ANDROID_PLATFORM</code>: Specifies the Android API levels. Available levels range from 21 to 29.
     *         For example, <code>{'ANDROID_PLATFORM': 28}</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>mattr</code>: Add <code>{'mattr': ['+neon']}</code> to compiler options if compiling for ARM 32-bit
     *         platform with NEON support.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INFERENTIA</code>: Compilation for target ml_inf1 uses compiler options passed in as a JSON string.
     *         For example, <code>"CompilerOptions": "\"--verbose 1 --num-neuroncores 2 -O2\""</code>.
     *         </p>
     *         <p>
     *         For information about supported compiler options, see <a
     *         href="https://github.com/aws/aws-neuron-sdk/blob/master/docs/neuron-cc/command-line-reference.md"> Neuron
     *         Compiler CLI</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CoreML</code>: Compilation for the CoreML <a>OutputConfig$TargetDevice</a> supports the following
     *         compiler options:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>class_labels</code>: Specifies the classification labels file name inside input tar.gz file. For
     *         example, <code>{"class_labels": "imagenet_labels_1000.txt"}</code>. Labels inside the txt file should be
     *         separated by newlines.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     */
    public String compilerOptions() {
        return compilerOptions;
    }

    /**
     * <p>
     * The AWS Key Management Service (AWS KMS) key that Amazon SageMaker uses to encrypt data on the storage volume
     * after compilation job. If you don't provide a KMS key ID, Amazon SageMaker uses the default KMS key for Amazon S3
     * for your role's account
     * </p>
     * <p>
     * The KmsKeyId can be any of the following formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name: <code>alias/ExampleAlias</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The AWS Key Management Service (AWS KMS) key that Amazon SageMaker uses to encrypt data on the storage
     *         volume after compilation job. If you don't provide a KMS key ID, Amazon SageMaker uses the default KMS
     *         key for Amazon S3 for your role's account</p>
     *         <p>
     *         The KmsKeyId can be any of the following formats:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name: <code>alias/ExampleAlias</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
     *         </p>
     *         </li>
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3OutputLocation());
        hashCode = 31 * hashCode + Objects.hashCode(targetDeviceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetPlatform());
        hashCode = 31 * hashCode + Objects.hashCode(compilerOptions());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OutputConfig)) {
            return false;
        }
        OutputConfig other = (OutputConfig) obj;
        return Objects.equals(s3OutputLocation(), other.s3OutputLocation())
                && Objects.equals(targetDeviceAsString(), other.targetDeviceAsString())
                && Objects.equals(targetPlatform(), other.targetPlatform())
                && Objects.equals(compilerOptions(), other.compilerOptions()) && Objects.equals(kmsKeyId(), other.kmsKeyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("OutputConfig").add("S3OutputLocation", s3OutputLocation())
                .add("TargetDevice", targetDeviceAsString()).add("TargetPlatform", targetPlatform())
                .add("CompilerOptions", compilerOptions()).add("KmsKeyId", kmsKeyId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3OutputLocation":
            return Optional.ofNullable(clazz.cast(s3OutputLocation()));
        case "TargetDevice":
            return Optional.ofNullable(clazz.cast(targetDeviceAsString()));
        case "TargetPlatform":
            return Optional.ofNullable(clazz.cast(targetPlatform()));
        case "CompilerOptions":
            return Optional.ofNullable(clazz.cast(compilerOptions()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OutputConfig, T> g) {
        return obj -> g.apply((OutputConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OutputConfig> {
        /**
         * <p>
         * Identifies the S3 bucket where you want Amazon SageMaker to store the model artifacts. For example,
         * <code>s3://bucket-name/key-name-prefix</code>.
         * </p>
         * 
         * @param s3OutputLocation
         *        Identifies the S3 bucket where you want Amazon SageMaker to store the model artifacts. For example,
         *        <code>s3://bucket-name/key-name-prefix</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3OutputLocation(String s3OutputLocation);

        /**
         * <p>
         * Identifies the target device or the machine learning instance that you want to run your model on after the
         * compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
         * <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
         * </p>
         * 
         * @param targetDevice
         *        Identifies the target device or the machine learning instance that you want to run your model on after
         *        the compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
         *        <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
         * @see TargetDevice
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetDevice
         */
        Builder targetDevice(String targetDevice);

        /**
         * <p>
         * Identifies the target device or the machine learning instance that you want to run your model on after the
         * compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
         * <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
         * </p>
         * 
         * @param targetDevice
         *        Identifies the target device or the machine learning instance that you want to run your model on after
         *        the compilation has completed. Alternatively, you can specify OS, architecture, and accelerator using
         *        <a>TargetPlatform</a> fields. It can be used instead of <code>TargetPlatform</code>.
         * @see TargetDevice
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TargetDevice
         */
        Builder targetDevice(TargetDevice targetDevice);

        /**
         * <p>
         * Contains information about a target platform that you want your model to run on, such as OS, architecture,
         * and accelerators. It is an alternative of <code>TargetDevice</code>.
         * </p>
         * <p>
         * The following examples show how to configure the <code>TargetPlatform</code> and <code>CompilerOptions</code>
         * JSON strings for popular target platforms:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Raspberry Pi 3 Model B+
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM_EABIHF"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'mattr': ['+neon']}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Jetson TX2
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "NVIDIA"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'gpu-code': 'sm_62', 'trt-ver': '6.0.1', 'cuda-ver': '10.0'}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * EC2 m5.2xlarge instance OS
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "X86_64", "Accelerator": "NVIDIA"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'mcpu': 'skylake-avx512'}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * RK3399
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "MALI"}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * ARMv7 phone (CPU)
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM_EABI"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'ANDROID_PLATFORM': 25, 'mattr': ['+neon']}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * ARMv8 phone (CPU)
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM64"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'ANDROID_PLATFORM': 29}</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param targetPlatform
         *        Contains information about a target platform that you want your model to run on, such as OS,
         *        architecture, and accelerators. It is an alternative of <code>TargetDevice</code>.</p>
         *        <p>
         *        The following examples show how to configure the <code>TargetPlatform</code> and
         *        <code>CompilerOptions</code> JSON strings for popular target platforms:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Raspberry Pi 3 Model B+
         *        </p>
         *        <p>
         *        <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM_EABIHF"},</code>
         *        </p>
         *        <p>
         *        <code> "CompilerOptions": {'mattr': ['+neon']}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Jetson TX2
         *        </p>
         *        <p>
         *        <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "NVIDIA"},</code>
         *        </p>
         *        <p>
         *        <code> "CompilerOptions": {'gpu-code': 'sm_62', 'trt-ver': '6.0.1', 'cuda-ver': '10.0'}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EC2 m5.2xlarge instance OS
         *        </p>
         *        <p>
         *        <code>"TargetPlatform": {"Os": "LINUX", "Arch": "X86_64", "Accelerator": "NVIDIA"},</code>
         *        </p>
         *        <p>
         *        <code> "CompilerOptions": {'mcpu': 'skylake-avx512'}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RK3399
         *        </p>
         *        <p>
         *        <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "MALI"}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ARMv7 phone (CPU)
         *        </p>
         *        <p>
         *        <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM_EABI"},</code>
         *        </p>
         *        <p>
         *        <code> "CompilerOptions": {'ANDROID_PLATFORM': 25, 'mattr': ['+neon']}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ARMv8 phone (CPU)
         *        </p>
         *        <p>
         *        <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM64"},</code>
         *        </p>
         *        <p>
         *        <code> "CompilerOptions": {'ANDROID_PLATFORM': 29}</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetPlatform(TargetPlatform targetPlatform);

        /**
         * <p>
         * Contains information about a target platform that you want your model to run on, such as OS, architecture,
         * and accelerators. It is an alternative of <code>TargetDevice</code>.
         * </p>
         * <p>
         * The following examples show how to configure the <code>TargetPlatform</code> and <code>CompilerOptions</code>
         * JSON strings for popular target platforms:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Raspberry Pi 3 Model B+
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM_EABIHF"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'mattr': ['+neon']}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Jetson TX2
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "NVIDIA"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'gpu-code': 'sm_62', 'trt-ver': '6.0.1', 'cuda-ver': '10.0'}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * EC2 m5.2xlarge instance OS
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "X86_64", "Accelerator": "NVIDIA"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'mcpu': 'skylake-avx512'}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * RK3399
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "LINUX", "Arch": "ARM64", "Accelerator": "MALI"}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * ARMv7 phone (CPU)
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM_EABI"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'ANDROID_PLATFORM': 25, 'mattr': ['+neon']}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * ARMv8 phone (CPU)
         * </p>
         * <p>
         * <code>"TargetPlatform": {"Os": "ANDROID", "Arch": "ARM64"},</code>
         * </p>
         * <p>
         * <code> "CompilerOptions": {'ANDROID_PLATFORM': 29}</code>
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link TargetPlatform.Builder} avoiding the need to
         * create one manually via {@link TargetPlatform#builder()}.
         *
         * When the {@link Consumer} completes, {@link TargetPlatform.Builder#build()} is called immediately and its
         * result is passed to {@link #targetPlatform(TargetPlatform)}.
         * 
         * @param targetPlatform
         *        a consumer that will call methods on {@link TargetPlatform.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetPlatform(TargetPlatform)
         */
        default Builder targetPlatform(Consumer<TargetPlatform.Builder> targetPlatform) {
            return targetPlatform(TargetPlatform.builder().applyMutation(targetPlatform).build());
        }

        /**
         * <p>
         * Specifies additional parameters for compiler options in JSON format. The compiler options are
         * <code>TargetPlatform</code> specific. It is required for NVIDIA accelerators and highly recommended for CPU
         * compilations. For any other cases, it is optional to specify <code>CompilerOptions.</code>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CPU</code>: Compilation for CPU supports the following compiler options.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>mcpu</code>: CPU micro-architecture. For example, <code>{'mcpu': 'skylake-avx512'}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>mattr</code>: CPU flags. For example, <code>{'mattr': ['+neon', '+vfpv4']}</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>ARM</code>: Details of ARM CPU compilations.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NEON</code>: NEON is an implementation of the Advanced SIMD extension used in ARMv7 processors.
         * </p>
         * <p>
         * For example, add <code>{'mattr': ['+neon']}</code> to the compiler options if compiling for ARM 32-bit
         * platform with the NEON support.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>NVIDIA</code>: Compilation for NVIDIA GPU supports the following compiler options.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>gpu_code</code>: Specifies the targeted architecture.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>trt-ver</code>: Specifies the TensorRT versions in x.y.z. format.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cuda-ver</code>: Specifies the CUDA version in x.y format.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For example, <code>{'gpu-code': 'sm_72', 'trt-ver': '6.0.1', 'cuda-ver': '10.1'}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ANDROID</code>: Compilation for the Android OS supports the following compiler options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ANDROID_PLATFORM</code>: Specifies the Android API levels. Available levels range from 21 to 29. For
         * example, <code>{'ANDROID_PLATFORM': 28}</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>mattr</code>: Add <code>{'mattr': ['+neon']}</code> to compiler options if compiling for ARM 32-bit
         * platform with NEON support.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>INFERENTIA</code>: Compilation for target ml_inf1 uses compiler options passed in as a JSON string. For
         * example, <code>"CompilerOptions": "\"--verbose 1 --num-neuroncores 2 -O2\""</code>.
         * </p>
         * <p>
         * For information about supported compiler options, see <a
         * href="https://github.com/aws/aws-neuron-sdk/blob/master/docs/neuron-cc/command-line-reference.md"> Neuron
         * Compiler CLI</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CoreML</code>: Compilation for the CoreML <a>OutputConfig$TargetDevice</a> supports the following
         * compiler options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>class_labels</code>: Specifies the classification labels file name inside input tar.gz file. For
         * example, <code>{"class_labels": "imagenet_labels_1000.txt"}</code>. Labels inside the txt file should be
         * separated by newlines.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param compilerOptions
         *        Specifies additional parameters for compiler options in JSON format. The compiler options are
         *        <code>TargetPlatform</code> specific. It is required for NVIDIA accelerators and highly recommended
         *        for CPU compilations. For any other cases, it is optional to specify <code>CompilerOptions.</code>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CPU</code>: Compilation for CPU supports the following compiler options.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>mcpu</code>: CPU micro-architecture. For example, <code>{'mcpu': 'skylake-avx512'}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>mattr</code>: CPU flags. For example, <code>{'mattr': ['+neon', '+vfpv4']}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ARM</code>: Details of ARM CPU compilations.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NEON</code>: NEON is an implementation of the Advanced SIMD extension used in ARMv7 processors.
         *        </p>
         *        <p>
         *        For example, add <code>{'mattr': ['+neon']}</code> to the compiler options if compiling for ARM 32-bit
         *        platform with the NEON support.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NVIDIA</code>: Compilation for NVIDIA GPU supports the following compiler options.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>gpu_code</code>: Specifies the targeted architecture.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>trt-ver</code>: Specifies the TensorRT versions in x.y.z. format.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cuda-ver</code>: Specifies the CUDA version in x.y format.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For example, <code>{'gpu-code': 'sm_72', 'trt-ver': '6.0.1', 'cuda-ver': '10.1'}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ANDROID</code>: Compilation for the Android OS supports the following compiler options:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ANDROID_PLATFORM</code>: Specifies the Android API levels. Available levels range from 21 to 29.
         *        For example, <code>{'ANDROID_PLATFORM': 28}</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>mattr</code>: Add <code>{'mattr': ['+neon']}</code> to compiler options if compiling for ARM
         *        32-bit platform with NEON support.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INFERENTIA</code>: Compilation for target ml_inf1 uses compiler options passed in as a JSON
         *        string. For example, <code>"CompilerOptions": "\"--verbose 1 --num-neuroncores 2 -O2\""</code>.
         *        </p>
         *        <p>
         *        For information about supported compiler options, see <a
         *        href="https://github.com/aws/aws-neuron-sdk/blob/master/docs/neuron-cc/command-line-reference.md">
         *        Neuron Compiler CLI</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CoreML</code>: Compilation for the CoreML <a>OutputConfig$TargetDevice</a> supports the
         *        following compiler options:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>class_labels</code>: Specifies the classification labels file name inside input tar.gz file. For
         *        example, <code>{"class_labels": "imagenet_labels_1000.txt"}</code>. Labels inside the txt file should
         *        be separated by newlines.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compilerOptions(String compilerOptions);

        /**
         * <p>
         * The AWS Key Management Service (AWS KMS) key that Amazon SageMaker uses to encrypt data on the storage volume
         * after compilation job. If you don't provide a KMS key ID, Amazon SageMaker uses the default KMS key for
         * Amazon S3 for your role's account
         * </p>
         * <p>
         * The KmsKeyId can be any of the following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name: <code>alias/ExampleAlias</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param kmsKeyId
         *        The AWS Key Management Service (AWS KMS) key that Amazon SageMaker uses to encrypt data on the storage
         *        volume after compilation job. If you don't provide a KMS key ID, Amazon SageMaker uses the default KMS
         *        key for Amazon S3 for your role's account</p>
         *        <p>
         *        The KmsKeyId can be any of the following formats:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID: <code>1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN: <code>arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name: <code>alias/ExampleAlias</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias name ARN: <code>arn:aws:kms:us-west-2:111122223333:alias/ExampleAlias</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);
    }

    static final class BuilderImpl implements Builder {
        private String s3OutputLocation;

        private String targetDevice;

        private TargetPlatform targetPlatform;

        private String compilerOptions;

        private String kmsKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(OutputConfig model) {
            s3OutputLocation(model.s3OutputLocation);
            targetDevice(model.targetDevice);
            targetPlatform(model.targetPlatform);
            compilerOptions(model.compilerOptions);
            kmsKeyId(model.kmsKeyId);
        }

        public final String getS3OutputLocation() {
            return s3OutputLocation;
        }

        @Override
        public final Builder s3OutputLocation(String s3OutputLocation) {
            this.s3OutputLocation = s3OutputLocation;
            return this;
        }

        public final void setS3OutputLocation(String s3OutputLocation) {
            this.s3OutputLocation = s3OutputLocation;
        }

        public final String getTargetDevice() {
            return targetDevice;
        }

        @Override
        public final Builder targetDevice(String targetDevice) {
            this.targetDevice = targetDevice;
            return this;
        }

        @Override
        public final Builder targetDevice(TargetDevice targetDevice) {
            this.targetDevice(targetDevice == null ? null : targetDevice.toString());
            return this;
        }

        public final void setTargetDevice(String targetDevice) {
            this.targetDevice = targetDevice;
        }

        public final TargetPlatform.Builder getTargetPlatform() {
            return targetPlatform != null ? targetPlatform.toBuilder() : null;
        }

        @Override
        public final Builder targetPlatform(TargetPlatform targetPlatform) {
            this.targetPlatform = targetPlatform;
            return this;
        }

        public final void setTargetPlatform(TargetPlatform.BuilderImpl targetPlatform) {
            this.targetPlatform = targetPlatform != null ? targetPlatform.build() : null;
        }

        public final String getCompilerOptions() {
            return compilerOptions;
        }

        @Override
        public final Builder compilerOptions(String compilerOptions) {
            this.compilerOptions = compilerOptions;
            return this;
        }

        public final void setCompilerOptions(String compilerOptions) {
            this.compilerOptions = compilerOptions;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public OutputConfig build() {
            return new OutputConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
