/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the Docker container for the model package.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ModelPackageContainerDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<ModelPackageContainerDefinition.Builder, ModelPackageContainerDefinition> {
    private static final SdkField<String> CONTAINER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerHostname").getter(getter(ModelPackageContainerDefinition::containerHostname))
            .setter(setter(Builder::containerHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerHostname").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Image")
            .getter(getter(ModelPackageContainerDefinition::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Image").build()).build();

    private static final SdkField<String> IMAGE_DIGEST_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageDigest").getter(getter(ModelPackageContainerDefinition::imageDigest))
            .setter(setter(Builder::imageDigest))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageDigest").build()).build();

    private static final SdkField<String> MODEL_DATA_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelDataUrl").getter(getter(ModelPackageContainerDefinition::modelDataUrl))
            .setter(setter(Builder::modelDataUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelDataUrl").build()).build();

    private static final SdkField<String> PRODUCT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProductId").getter(getter(ModelPackageContainerDefinition::productId))
            .setter(setter(Builder::productId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductId").build()).build();

    private static final SdkField<Map<String, String>> ENVIRONMENT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Environment")
            .getter(getter(ModelPackageContainerDefinition::environment))
            .setter(setter(Builder::environment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Environment").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ModelInput> MODEL_INPUT_FIELD = SdkField.<ModelInput> builder(MarshallingType.SDK_POJO)
            .memberName("ModelInput").getter(getter(ModelPackageContainerDefinition::modelInput))
            .setter(setter(Builder::modelInput)).constructor(ModelInput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelInput").build()).build();

    private static final SdkField<String> FRAMEWORK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Framework").getter(getter(ModelPackageContainerDefinition::framework))
            .setter(setter(Builder::framework))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Framework").build()).build();

    private static final SdkField<String> FRAMEWORK_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FrameworkVersion").getter(getter(ModelPackageContainerDefinition::frameworkVersion))
            .setter(setter(Builder::frameworkVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FrameworkVersion").build()).build();

    private static final SdkField<String> NEAREST_MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NearestModelName").getter(getter(ModelPackageContainerDefinition::nearestModelName))
            .setter(setter(Builder::nearestModelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NearestModelName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_HOSTNAME_FIELD,
            IMAGE_FIELD, IMAGE_DIGEST_FIELD, MODEL_DATA_URL_FIELD, PRODUCT_ID_FIELD, ENVIRONMENT_FIELD, MODEL_INPUT_FIELD,
            FRAMEWORK_FIELD, FRAMEWORK_VERSION_FIELD, NEAREST_MODEL_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String containerHostname;

    private final String image;

    private final String imageDigest;

    private final String modelDataUrl;

    private final String productId;

    private final Map<String, String> environment;

    private final ModelInput modelInput;

    private final String framework;

    private final String frameworkVersion;

    private final String nearestModelName;

    private ModelPackageContainerDefinition(BuilderImpl builder) {
        this.containerHostname = builder.containerHostname;
        this.image = builder.image;
        this.imageDigest = builder.imageDigest;
        this.modelDataUrl = builder.modelDataUrl;
        this.productId = builder.productId;
        this.environment = builder.environment;
        this.modelInput = builder.modelInput;
        this.framework = builder.framework;
        this.frameworkVersion = builder.frameworkVersion;
        this.nearestModelName = builder.nearestModelName;
    }

    /**
     * <p>
     * The DNS host name for the Docker container.
     * </p>
     * 
     * @return The DNS host name for the Docker container.
     */
    public final String containerHostname() {
        return containerHostname;
    }

    /**
     * <p>
     * The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored.
     * </p>
     * <p>
     * If you are using your own custom algorithm instead of an algorithm provided by Amazon SageMaker, the inference
     * code must meet Amazon SageMaker requirements. Amazon SageMaker supports both
     * <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats. For more
     * information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own
     * Algorithms with Amazon SageMaker</a>.
     * </p>
     * 
     * @return The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored.</p>
     *         <p>
     *         If you are using your own custom algorithm instead of an algorithm provided by Amazon SageMaker, the
     *         inference code must meet Amazon SageMaker requirements. Amazon SageMaker supports both
     *         <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
     *         with Amazon SageMaker</a>.
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * An MD5 hash of the training algorithm that identifies the Docker image used for training.
     * </p>
     * 
     * @return An MD5 hash of the training algorithm that identifies the Docker image used for training.
     */
    public final String imageDigest() {
        return imageDigest;
    }

    /**
     * <p>
     * The Amazon S3 path where the model artifacts, which result from model training, are stored. This path must point
     * to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).
     * </p>
     * <note>
     * <p>
     * The model artifacts must be in an S3 bucket that is in the same region as the model package.
     * </p>
     * </note>
     * 
     * @return The Amazon S3 path where the model artifacts, which result from model training, are stored. This path
     *         must point to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).</p> <note>
     *         <p>
     *         The model artifacts must be in an S3 bucket that is in the same region as the model package.
     *         </p>
     */
    public final String modelDataUrl() {
        return modelDataUrl;
    }

    /**
     * <p>
     * The Amazon Web Services Marketplace product ID of the model package.
     * </p>
     * 
     * @return The Amazon Web Services Marketplace product ID of the model package.
     */
    public final String productId() {
        return productId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Environment property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEnvironment() {
        return environment != null && !(environment instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The environment variables to set in the Docker container. Each key and value in the <code>Environment</code>
     * string to string map can have length of up to 1024. We support up to 16 entries in the map.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnvironment} method.
     * </p>
     * 
     * @return The environment variables to set in the Docker container. Each key and value in the
     *         <code>Environment</code> string to string map can have length of up to 1024. We support up to 16 entries
     *         in the map.
     */
    public final Map<String, String> environment() {
        return environment;
    }

    /**
     * <p>
     * A structure with Model Input details.
     * </p>
     * 
     * @return A structure with Model Input details.
     */
    public final ModelInput modelInput() {
        return modelInput;
    }

    /**
     * <p>
     * The machine learning framework of the model package container image.
     * </p>
     * 
     * @return The machine learning framework of the model package container image.
     */
    public final String framework() {
        return framework;
    }

    /**
     * <p>
     * The framework version of the Model Package Container Image.
     * </p>
     * 
     * @return The framework version of the Model Package Container Image.
     */
    public final String frameworkVersion() {
        return frameworkVersion;
    }

    /**
     * <p>
     * The name of a pre-trained machine learning benchmarked by Amazon SageMaker Inference Recommender model that
     * matches your model. You can find a list of benchmarked models by calling <code>ListModelMetadata</code>.
     * </p>
     * 
     * @return The name of a pre-trained machine learning benchmarked by Amazon SageMaker Inference Recommender model
     *         that matches your model. You can find a list of benchmarked models by calling
     *         <code>ListModelMetadata</code>.
     */
    public final String nearestModelName() {
        return nearestModelName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(containerHostname());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(imageDigest());
        hashCode = 31 * hashCode + Objects.hashCode(modelDataUrl());
        hashCode = 31 * hashCode + Objects.hashCode(productId());
        hashCode = 31 * hashCode + Objects.hashCode(hasEnvironment() ? environment() : null);
        hashCode = 31 * hashCode + Objects.hashCode(modelInput());
        hashCode = 31 * hashCode + Objects.hashCode(framework());
        hashCode = 31 * hashCode + Objects.hashCode(frameworkVersion());
        hashCode = 31 * hashCode + Objects.hashCode(nearestModelName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModelPackageContainerDefinition)) {
            return false;
        }
        ModelPackageContainerDefinition other = (ModelPackageContainerDefinition) obj;
        return Objects.equals(containerHostname(), other.containerHostname()) && Objects.equals(image(), other.image())
                && Objects.equals(imageDigest(), other.imageDigest()) && Objects.equals(modelDataUrl(), other.modelDataUrl())
                && Objects.equals(productId(), other.productId()) && hasEnvironment() == other.hasEnvironment()
                && Objects.equals(environment(), other.environment()) && Objects.equals(modelInput(), other.modelInput())
                && Objects.equals(framework(), other.framework()) && Objects.equals(frameworkVersion(), other.frameworkVersion())
                && Objects.equals(nearestModelName(), other.nearestModelName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ModelPackageContainerDefinition").add("ContainerHostname", containerHostname())
                .add("Image", image()).add("ImageDigest", imageDigest()).add("ModelDataUrl", modelDataUrl())
                .add("ProductId", productId()).add("Environment", hasEnvironment() ? environment() : null)
                .add("ModelInput", modelInput()).add("Framework", framework()).add("FrameworkVersion", frameworkVersion())
                .add("NearestModelName", nearestModelName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerHostname":
            return Optional.ofNullable(clazz.cast(containerHostname()));
        case "Image":
            return Optional.ofNullable(clazz.cast(image()));
        case "ImageDigest":
            return Optional.ofNullable(clazz.cast(imageDigest()));
        case "ModelDataUrl":
            return Optional.ofNullable(clazz.cast(modelDataUrl()));
        case "ProductId":
            return Optional.ofNullable(clazz.cast(productId()));
        case "Environment":
            return Optional.ofNullable(clazz.cast(environment()));
        case "ModelInput":
            return Optional.ofNullable(clazz.cast(modelInput()));
        case "Framework":
            return Optional.ofNullable(clazz.cast(framework()));
        case "FrameworkVersion":
            return Optional.ofNullable(clazz.cast(frameworkVersion()));
        case "NearestModelName":
            return Optional.ofNullable(clazz.cast(nearestModelName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ModelPackageContainerDefinition, T> g) {
        return obj -> g.apply((ModelPackageContainerDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ModelPackageContainerDefinition> {
        /**
         * <p>
         * The DNS host name for the Docker container.
         * </p>
         * 
         * @param containerHostname
         *        The DNS host name for the Docker container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerHostname(String containerHostname);

        /**
         * <p>
         * The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored.
         * </p>
         * <p>
         * If you are using your own custom algorithm instead of an algorithm provided by Amazon SageMaker, the
         * inference code must meet Amazon SageMaker requirements. Amazon SageMaker supports both
         * <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path formats. For
         * more information, see <a href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using
         * Your Own Algorithms with Amazon SageMaker</a>.
         * </p>
         * 
         * @param image
         *        The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored.</p>
         *        <p>
         *        If you are using your own custom algorithm instead of an algorithm provided by Amazon SageMaker, the
         *        inference code must meet Amazon SageMaker requirements. Amazon SageMaker supports both
         *        <code>registry/repository[:tag]</code> and <code>registry/repository[@digest]</code> image path
         *        formats. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms.html">Using Your Own Algorithms
         *        with Amazon SageMaker</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * An MD5 hash of the training algorithm that identifies the Docker image used for training.
         * </p>
         * 
         * @param imageDigest
         *        An MD5 hash of the training algorithm that identifies the Docker image used for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageDigest(String imageDigest);

        /**
         * <p>
         * The Amazon S3 path where the model artifacts, which result from model training, are stored. This path must
         * point to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).
         * </p>
         * <note>
         * <p>
         * The model artifacts must be in an S3 bucket that is in the same region as the model package.
         * </p>
         * </note>
         * 
         * @param modelDataUrl
         *        The Amazon S3 path where the model artifacts, which result from model training, are stored. This path
         *        must point to a single <code>gzip</code> compressed tar archive (<code>.tar.gz</code> suffix).</p>
         *        <note>
         *        <p>
         *        The model artifacts must be in an S3 bucket that is in the same region as the model package.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelDataUrl(String modelDataUrl);

        /**
         * <p>
         * The Amazon Web Services Marketplace product ID of the model package.
         * </p>
         * 
         * @param productId
         *        The Amazon Web Services Marketplace product ID of the model package.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productId(String productId);

        /**
         * <p>
         * The environment variables to set in the Docker container. Each key and value in the <code>Environment</code>
         * string to string map can have length of up to 1024. We support up to 16 entries in the map.
         * </p>
         * 
         * @param environment
         *        The environment variables to set in the Docker container. Each key and value in the
         *        <code>Environment</code> string to string map can have length of up to 1024. We support up to 16
         *        entries in the map.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder environment(Map<String, String> environment);

        /**
         * <p>
         * A structure with Model Input details.
         * </p>
         * 
         * @param modelInput
         *        A structure with Model Input details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelInput(ModelInput modelInput);

        /**
         * <p>
         * A structure with Model Input details.
         * </p>
         * This is a convenience that creates an instance of the {@link ModelInput.Builder} avoiding the need to create
         * one manually via {@link ModelInput#builder()}.
         *
         * When the {@link Consumer} completes, {@link ModelInput.Builder#build()} is called immediately and its result
         * is passed to {@link #modelInput(ModelInput)}.
         * 
         * @param modelInput
         *        a consumer that will call methods on {@link ModelInput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #modelInput(ModelInput)
         */
        default Builder modelInput(Consumer<ModelInput.Builder> modelInput) {
            return modelInput(ModelInput.builder().applyMutation(modelInput).build());
        }

        /**
         * <p>
         * The machine learning framework of the model package container image.
         * </p>
         * 
         * @param framework
         *        The machine learning framework of the model package container image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder framework(String framework);

        /**
         * <p>
         * The framework version of the Model Package Container Image.
         * </p>
         * 
         * @param frameworkVersion
         *        The framework version of the Model Package Container Image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder frameworkVersion(String frameworkVersion);

        /**
         * <p>
         * The name of a pre-trained machine learning benchmarked by Amazon SageMaker Inference Recommender model that
         * matches your model. You can find a list of benchmarked models by calling <code>ListModelMetadata</code>.
         * </p>
         * 
         * @param nearestModelName
         *        The name of a pre-trained machine learning benchmarked by Amazon SageMaker Inference Recommender model
         *        that matches your model. You can find a list of benchmarked models by calling
         *        <code>ListModelMetadata</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nearestModelName(String nearestModelName);
    }

    static final class BuilderImpl implements Builder {
        private String containerHostname;

        private String image;

        private String imageDigest;

        private String modelDataUrl;

        private String productId;

        private Map<String, String> environment = DefaultSdkAutoConstructMap.getInstance();

        private ModelInput modelInput;

        private String framework;

        private String frameworkVersion;

        private String nearestModelName;

        private BuilderImpl() {
        }

        private BuilderImpl(ModelPackageContainerDefinition model) {
            containerHostname(model.containerHostname);
            image(model.image);
            imageDigest(model.imageDigest);
            modelDataUrl(model.modelDataUrl);
            productId(model.productId);
            environment(model.environment);
            modelInput(model.modelInput);
            framework(model.framework);
            frameworkVersion(model.frameworkVersion);
            nearestModelName(model.nearestModelName);
        }

        public final String getContainerHostname() {
            return containerHostname;
        }

        public final void setContainerHostname(String containerHostname) {
            this.containerHostname = containerHostname;
        }

        @Override
        @Transient
        public final Builder containerHostname(String containerHostname) {
            this.containerHostname = containerHostname;
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        @Transient
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final String getImageDigest() {
            return imageDigest;
        }

        public final void setImageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
        }

        @Override
        @Transient
        public final Builder imageDigest(String imageDigest) {
            this.imageDigest = imageDigest;
            return this;
        }

        public final String getModelDataUrl() {
            return modelDataUrl;
        }

        public final void setModelDataUrl(String modelDataUrl) {
            this.modelDataUrl = modelDataUrl;
        }

        @Override
        @Transient
        public final Builder modelDataUrl(String modelDataUrl) {
            this.modelDataUrl = modelDataUrl;
            return this;
        }

        public final String getProductId() {
            return productId;
        }

        public final void setProductId(String productId) {
            this.productId = productId;
        }

        @Override
        @Transient
        public final Builder productId(String productId) {
            this.productId = productId;
            return this;
        }

        public final Map<String, String> getEnvironment() {
            if (environment instanceof SdkAutoConstructMap) {
                return null;
            }
            return environment;
        }

        public final void setEnvironment(Map<String, String> environment) {
            this.environment = EnvironmentMapCopier.copy(environment);
        }

        @Override
        @Transient
        public final Builder environment(Map<String, String> environment) {
            this.environment = EnvironmentMapCopier.copy(environment);
            return this;
        }

        public final ModelInput.Builder getModelInput() {
            return modelInput != null ? modelInput.toBuilder() : null;
        }

        public final void setModelInput(ModelInput.BuilderImpl modelInput) {
            this.modelInput = modelInput != null ? modelInput.build() : null;
        }

        @Override
        @Transient
        public final Builder modelInput(ModelInput modelInput) {
            this.modelInput = modelInput;
            return this;
        }

        public final String getFramework() {
            return framework;
        }

        public final void setFramework(String framework) {
            this.framework = framework;
        }

        @Override
        @Transient
        public final Builder framework(String framework) {
            this.framework = framework;
            return this;
        }

        public final String getFrameworkVersion() {
            return frameworkVersion;
        }

        public final void setFrameworkVersion(String frameworkVersion) {
            this.frameworkVersion = frameworkVersion;
        }

        @Override
        @Transient
        public final Builder frameworkVersion(String frameworkVersion) {
            this.frameworkVersion = frameworkVersion;
            return this;
        }

        public final String getNearestModelName() {
            return nearestModelName;
        }

        public final void setNearestModelName(String nearestModelName) {
            this.nearestModelName = nearestModelName;
        }

        @Override
        @Transient
        public final Builder nearestModelName(String nearestModelName) {
            this.nearestModelName = nearestModelName;
            return this;
        }

        @Override
        public ModelPackageContainerDefinition build() {
            return new ModelPackageContainerDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
