/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.sagemaker.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Gets the Amazon EC2 Container Registry path of the docker image of the model that is hosted in this
 * <a>ProductionVariant</a>.
 * </p>
 * <p>
 * If you used the <code>registry/repository[:tag]</code> form to specify the image path of the primary container when
 * you created the model hosted in this <code>ProductionVariant</code>, the path resolves to a path of the form
 * <code>registry/repository[@digest]</code>. A digest is a hash value that identifies a specific version of an image.
 * For information about Amazon ECR paths, see <a
 * href="https://docs.aws.amazon.com/AmazonECR/latest/userguide/docker-pull-ecr-image.html">Pulling an Image</a> in the
 * <i>Amazon ECR User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeployedImage implements SdkPojo, Serializable, ToCopyableBuilder<DeployedImage.Builder, DeployedImage> {
    private static final SdkField<String> SPECIFIED_IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SpecifiedImage").getter(getter(DeployedImage::specifiedImage)).setter(setter(Builder::specifiedImage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpecifiedImage").build()).build();

    private static final SdkField<String> RESOLVED_IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResolvedImage").getter(getter(DeployedImage::resolvedImage)).setter(setter(Builder::resolvedImage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolvedImage").build()).build();

    private static final SdkField<Instant> RESOLUTION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ResolutionTime").getter(getter(DeployedImage::resolutionTime)).setter(setter(Builder::resolutionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResolutionTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SPECIFIED_IMAGE_FIELD,
            RESOLVED_IMAGE_FIELD, RESOLUTION_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String specifiedImage;

    private final String resolvedImage;

    private final Instant resolutionTime;

    private DeployedImage(BuilderImpl builder) {
        this.specifiedImage = builder.specifiedImage;
        this.resolvedImage = builder.resolvedImage;
        this.resolutionTime = builder.resolutionTime;
    }

    /**
     * <p>
     * The image path you specified when you created the model.
     * </p>
     * 
     * @return The image path you specified when you created the model.
     */
    public final String specifiedImage() {
        return specifiedImage;
    }

    /**
     * <p>
     * The specific digest path of the image hosted in this <code>ProductionVariant</code>.
     * </p>
     * 
     * @return The specific digest path of the image hosted in this <code>ProductionVariant</code>.
     */
    public final String resolvedImage() {
        return resolvedImage;
    }

    /**
     * <p>
     * The date and time when the image path for the model resolved to the <code>ResolvedImage</code>
     * </p>
     * 
     * @return The date and time when the image path for the model resolved to the <code>ResolvedImage</code>
     */
    public final Instant resolutionTime() {
        return resolutionTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(specifiedImage());
        hashCode = 31 * hashCode + Objects.hashCode(resolvedImage());
        hashCode = 31 * hashCode + Objects.hashCode(resolutionTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeployedImage)) {
            return false;
        }
        DeployedImage other = (DeployedImage) obj;
        return Objects.equals(specifiedImage(), other.specifiedImage()) && Objects.equals(resolvedImage(), other.resolvedImage())
                && Objects.equals(resolutionTime(), other.resolutionTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeployedImage").add("SpecifiedImage", specifiedImage()).add("ResolvedImage", resolvedImage())
                .add("ResolutionTime", resolutionTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SpecifiedImage":
            return Optional.ofNullable(clazz.cast(specifiedImage()));
        case "ResolvedImage":
            return Optional.ofNullable(clazz.cast(resolvedImage()));
        case "ResolutionTime":
            return Optional.ofNullable(clazz.cast(resolutionTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeployedImage, T> g) {
        return obj -> g.apply((DeployedImage) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeployedImage> {
        /**
         * <p>
         * The image path you specified when you created the model.
         * </p>
         * 
         * @param specifiedImage
         *        The image path you specified when you created the model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder specifiedImage(String specifiedImage);

        /**
         * <p>
         * The specific digest path of the image hosted in this <code>ProductionVariant</code>.
         * </p>
         * 
         * @param resolvedImage
         *        The specific digest path of the image hosted in this <code>ProductionVariant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolvedImage(String resolvedImage);

        /**
         * <p>
         * The date and time when the image path for the model resolved to the <code>ResolvedImage</code>
         * </p>
         * 
         * @param resolutionTime
         *        The date and time when the image path for the model resolved to the <code>ResolvedImage</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resolutionTime(Instant resolutionTime);
    }

    static final class BuilderImpl implements Builder {
        private String specifiedImage;

        private String resolvedImage;

        private Instant resolutionTime;

        private BuilderImpl() {
        }

        private BuilderImpl(DeployedImage model) {
            specifiedImage(model.specifiedImage);
            resolvedImage(model.resolvedImage);
            resolutionTime(model.resolutionTime);
        }

        public final String getSpecifiedImage() {
            return specifiedImage;
        }

        @Override
        public final Builder specifiedImage(String specifiedImage) {
            this.specifiedImage = specifiedImage;
            return this;
        }

        public final void setSpecifiedImage(String specifiedImage) {
            this.specifiedImage = specifiedImage;
        }

        public final String getResolvedImage() {
            return resolvedImage;
        }

        @Override
        public final Builder resolvedImage(String resolvedImage) {
            this.resolvedImage = resolvedImage;
            return this;
        }

        public final void setResolvedImage(String resolvedImage) {
            this.resolvedImage = resolvedImage;
        }

        public final Instant getResolutionTime() {
            return resolutionTime;
        }

        @Override
        public final Builder resolutionTime(Instant resolutionTime) {
            this.resolutionTime = resolutionTime;
            return this;
        }

        public final void setResolutionTime(Instant resolutionTime) {
            this.resolutionTime = resolutionTime;
        }

        @Override
        public DeployedImage build() {
            return new DeployedImage(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
